/*
        linalg::grad -- computes the vector gradient of an expression

        grad(x, m, <ogSystem, <c>>)

        x       : arithmetical expression
        m       : list of (index) identifiers
        ogSystem: (optional) name or list of the scale factors of an 
                  orthogonally curvilinear coordinate system
        c       : the parameter for the coordinate systems
                  'Torus' and 'EllipticCylindrical'

        This function computes the gradient of x in respect to m in form of 
        a list. If only two arguments are given then 'linalg::grad' operates 
        on cartesian coordinate system. 
        Hence the result is the vector v of partial derivatives whose i-th 
        entry is defined by v[i]:=diff( x,m[i] ).

        The vector is of type Dom::Matrix().

        Otherwise 'linalg::grad' operates on the orthogonally curvilinear
        coordinate system specified by a list of scale factors or by a name 
        which scale factors are defined in table 'linalg::ogCoordTab'.

        The scale factors g_i are defined by g_i := |diff( T(u),u_i )|,
        where T describes the orthogonal transformation x=T(u) with x 
        the cartesian coordinates and u the orthogonally curvilinear
        coordinates.
*/

linalg::grad:= proc(f,x)
    local g, i, n;
begin
    // type checking:
    if testargs() then
        if args(0) < 2 then
            error("expecting at least 2 arguments")
        elif args(0) > 4 then
            error("expecting no more than 4 arguments")
        elif not testtype( f,Type::Arithmetical ) then
            error("1st argument must be of 'Type::Arithmetical'")
        elif not testtype( x,Type::ListOf(Type::Unknown) ) then 
            error("2nd argument must be a list of (indexed) identifiers")
        end_if;

        // n = the dimension
        n:= nops(x);

        if args(0) = 3 then
            if domtype(args(3)) <> DOM_LIST 
                 and type(linalg::ogCoordTab[args(3),hold(Scales)]) = "_index" 
            then
                error("illegal or undefined coordinate system")
            end_if;
            // presently only 3-dim transformations are installed
            // in ogCoordTab:
            if domtype(args(3)) <> DOM_LIST and n <> 3 then
               error("unknown orthogonal coordinate system")
            end_if;
        end_if
    end_if;

    // n = the dimension
    n:= nops(x);

    if args(0) >= 3 then
        g:= args(3);
        if domtype(g) <> DOM_LIST then
           if args(0) = 4 then
             g:= linalg::ogCoordTab[g,hold(Scales)](op(x), args(4))
           else
             g:= linalg::ogCoordTab[g,hold(Scales)](op(x))
           end_if;
        elif nops(g) <> n then
           error("number of variables and number of base vectors do not match");
        end_if;
        return(Dom::Matrix()(n, 1, [diff(f,x[i])/g[i] $ i=1..n]))
    else
        return(Dom::Matrix()(n, 1, [diff(f,x[i]) $ i=1..n]))
    end_if
end_proc:

linalg::grad := funcenv(linalg::grad):
linalg::grad::Content := (Out, data) ->
                         if nops(data) <> 2 then
                           return(Out::stdFunc(data));
                         else
                           Out::Capply(Out::Cgrad,  
                                       Out(op(data, 1)),
                                       map(op(op(data,2)), Out)):
                         end_if:
