/*
 linalg::hermiteForm  --  compute the Hermite normal form of a matrix

 hermiteForm( A [,All] )

 A   - An (m x n) Matrix
 All - Additional option. One may als compute the left transform
       (see below for details) 

 hermiteForm(A) computes the Hermite normal form of the matrix A,
 i.e. an upper-triangular matrix such that
 val(A[i,j])<=beta*val(A[j,j]) for j>i, and if A is a square matrix,
 the product of the diagonal elements equals det(A) (up to its sign).

 The Hermite normal form is obtained from A by:

 (i) subtracting the product of a row by a coefficient to another row
 (ii) exchanging two rows
 (iii) multiplying a row by -1.

 This algorithm needs a pseudo-division method Div such that 
 if q=Div(a,b), then val(a-q*b)<=beta*val(b).
 For integer entries, val(k)=abs(k), beta=1/2 and Div(a,b)=round(a/b).
   
 Use the additional option 'All' to calculate the left transform,
 i.e. the (m x m) matrix U, so that 

                            U * A = H

 and H is the hermite canonical form of A. 

 Examples:
 >> use(Dom):
 >> MZ:= Matrix(Integer):
 >> A:=MZ( [[9,-36,30], [-36,192,-180], [30,-180,180]] ):
 >> linalg::hermiteForm( A );

     +-         -+
     | 3,  0, 30 |
     |           |
     | 0, 12,  0 |
     |           |
     | 0,  0, 60 |
     +-         -+

 >> linalg::hermiteForm( A,All );
  
       -- +-           -+  +-            -+ --
       |  |  3,  0, 30  |  |  13,  9,  7  |  |
       |  |             |  |              |  |
       |  |  0, 12,  0  |, |   6,  4,  3  |  |
       |  |             |  |              |  |
       |  |  0,  0, 60  |  |  20, 15, 12  |  |
       -- +-           -+  +-            -+ --

 >> %[2] * A;

             +-           -+
             |  3,  0, 30  |
             |             |
             |  0, 12,  0  |
             |             |
             |  0,  0, 60  |
             +-           -+

 >> B:= MZ([[3,7,11], [13,17,19]]):
 >> linalg::hermiteForm( B );

             +-           -+
             | 1, -11, -25 |
             |             |
             | 0,  40,  86 |
             +-           -+

*/

linalg::hermiteForm := proc(A)
    local i,j,k,m,n,i0,v0,q,do_conversion,Mat,changed;
begin
  // if args(0) <> 1 then error("expecting one argument") end_if;
  Mat:= A::dom;
  if testargs() then
    if A::dom::hasProp( Cat::Matrix ) <> TRUE then
      error("expecting matrix of 'Cat::Matrix'")
    end_if
  end_if;

  if Mat::coeffRing <> Dom::Integer then 
    do_conversion:= Mat::coerce; // keep type of matrix
    Mat:= Dom::Matrix(Dom::Integer);
    if (A:= Mat::coerce(A)) = FAIL then
      error("expecting matrix with integer coefficients")
    end_if
  else
    do_conversion:= id
  end_if;

  // NOTE: 
  // New code concerning the computation of the 
  // matrix transform.

  if args(0) = 2 and args(2) <> hold(All) then
    error("2nd argument must be the option 'All'")
  elif args(0) > 2 then
    error("too many arguments given")
  end_if;
  k:= Mat::matdim( A );
  n:= k[1]; m:= k[2];
  
  if args(0) = 2 and args(2) = hold(All) then 
    // if args(2) = hold(All) then 
    // A is an (m x n) matrix. Create an (m x m) identity matrix I
    // an use 'linalg::concat(A,I)' to create the matrix (A|I). 
    A:= linalg::concatMatrix(A, Mat::identity(n));
    m:= m+n; //new matrix dimension: 'number of colums' + 'number of rows' 
    // If second argument is not 'All', just use the routine as it was! 
  end_if;

  for j from 1 to min(n,m) do
    repeat
    // choose the non-zero entry of smallest valuation:
    v0:= infinity; 
    changed:=FALSE;
    for i from j to n do
      if not iszero(A[i,j]) then
        if specfunc::abs(A[i,j]) < v0 then // v(A[i,j]) < v0
          i0:=i; 
          v0:=specfunc::abs(A[i,j]) // v0:= v( A[i,j] )
        end_if
      end_if
    end_for;
    if v0<>infinity and not iszero(A[i0,j]) then
      // now reduce the other entries:
      for i from 1 to n do
        if i=i0 then next end_if;
        q:= round( A[i,j]/A[i0,j] ); // divide( A[i,j],A[i0,j] )
        if not iszero(q) then 
          // subtract q * (row i0) to row i
          changed:=TRUE;
          for k from j to m do A[i,k]:=A[i,k]-q*A[i0,k] end_for
        end_if
      end_for;
    end_if
    until changed=FALSE end_repeat;

    if v0<>infinity and i0<>j then 
      // exchange row i0 and row j:
      A:= Mat::swapRow( A,i0,j,j..m )
    end_if;
      if specfunc::sign(A[j,j]) = -1 then 
        // multiply by -1:
        for k from j to m do A[j,k]:= -A[j,k] end_for
      end_if
    end_for;

    if args(0) = 2 and args(2) = hold(All) then 
      // A:= do_conversion(A);
      // Undo manipulation on the number of columns of A! 
      m:= m-n;
      // Return the hermite canonical form A[1..n, 1..m]
      // and it's left transform A[1..n, m+1..m+n] in a 
      // list.
      return([do_conversion(A[1..n, 1..m]), 
              do_conversion(A[1..n, m+1..m+n])]);
    else 
      return( do_conversion(A) );
    end_if;
end_proc:

