/*++
    hilbert - Hilbert matrix

    Call:

    linalg::hilbert( n <, CoefficientDomain> );
                                                                   
    Parameter:

    n  --  a positve integer
    CoefficientDomain -- 
             constructor of the coeffient domain for the output matrix.
             Admissible choices are
             Dom::Real, Dom::Float, Dom::Numerical,
             Dom::ExpressionField(), Dom::Integer.
                                                                   
    Synopsis:

    The n x n -Hilbert matrix H given by H[i,j]=1/(i+j-1)
    is returned by hilbert(n). It is of the domain type 
    Dom::Matrix().

    The Hilbert matrices are notoriously ill-conditioned
    positive definite symmetric matrices. Their spectral
    condition number, i.e., the ratio of the largest by 
    the smallest eigenvalue, is approximately exp(3.5*n)
    for large n.

    The optional second argument is the constructor of the
    output matrix, if a domain type different from 
    Dom::Matrix(Dom::ExpressionField()) is desired.

    Examples:

    >> linalg::hilbert(2);

                              +-          -+
                              |   1,  1/2  |
                              |            |
                              |  1/2, 1/3  |
                              +-          -+

    See also:  linalg::invhilbert, Dom::Matrix
++*/
linalg::hilbert:= proc(n,R=Dom::ExpressionField())
   local i,j;
begin
   if testargs() then
       if args(0) = 0 then error("expecting positive integer") end_if;
       if args(0) > 2 then error("too many arguments") end_if;
       if not testtype( n,Type::PosInt ) then error("expecting positive integer") end_if;
       if R::hasProp(Cat::Rng) <> TRUE then error("component domain must be of category 'Cat::Rng'") end_if;
   end_if;

   n:= array(1..n,1..n,[[1/(i+j-1) $ j=1..n] $ i=1..n]);
   case R
    of Dom::Complex do
    of Dom::Real do
    of Dom::Numerical do
    of Dom::Rational do
    of Dom::ExpressionField() do
       // conversion of the components is not necessary!
       return((Dom::Matrix(R))::create( n ))
   otherwise
       return(Dom::Matrix(R)( n ))
   end_case
end_proc: 

