/*
    linalg::intBasis  --  returns a set of linear independent vectors
                          for the intersection of vector spaces

    intBasis(S1, S2 [,S3,...])

    S1, S2, S3, ...: list or set of vectors

    intBasis returns a basis for the intersection of the vector 
    spaces spanned by the vectors in Si.
*/

linalg::intBasis:= proc(S)
    local l, s, t, n, v, k, K, R, T, ns, nt, tmp, returnList, j;
begin
  if testargs() then
    if args(0) = 0 then error("no arguments given") end_if;
    n:= FAIL;
    for s in [args()] do
      case domtype(s)
      of DOM_LIST do
        T:= Type::ListOf; break
      of DOM_SET do
        T:= Type::SetOf; break
      otherwise
        error("expecting only sets or lists of vectors")
      end_case;

      if nops(S) = 0 then next end_if;
      v:= op(s,1);
      if not testtype(v,linalg::vectorOf(Type::AnyType)) then
        error("expecting sets or lists of vectors")
      elif n = FAIL then
        n:= nops(v);
        R:= v::dom::coeffRing;
        if not R::hasProp( Cat::Field ) then
          error("expecting vectors over a 'Cat::Field'")
        end_if
      end_if;
      if not testtype( s,T(linalg::vectorOf(R,n)) ) then
        error("vectors are not compatible")
      end_if
    end_for
  end_if;

  if args(0) = 1 then return( linalg::basis(S) )
    elif nops(S) = 0 then return( S )
  end_if;

  v:= op( S,1 );
  n:= nops( v ); // dimension of the vectors in S
  T:= v::dom; // type of the vectors in S

  s:= linalg::basis( S );

  if domtype(S) = DOM_LIST then returnList:= TRUE
  else returnList:= FALSE
  end_if;

  R:= T::coeffRing;
  for t in [args(2..args(0))] do
    if nops(t) = 0 then 
      if returnList then return( [] ) else return( {} ) end_if
    end_if;
    ns:= nops(s);
    t:= linalg::basis( t );
    nt:= nops(t);

    K:= array( 1..n, 1..ns+nt );
    ( (K[k,j]:= op(s,j)[k]) $ k=1..n ) $ j=1..ns;
    ( (K[k,j]:= op(t,j-ns)[k]) $ k=1..n ) $ j=ns+1..ns+nt;

    K:= linalg::nullspace( T::create( K ) );

    v:= [NIL $ nops(K)];
    for l from 1 to nops(K) do
      tmp:= op(s,1) * K[l][1];
      ( tmp:= tmp + K[l][k] * op(s,k) ) $ k=2..ns;
      v[l]:= tmp
    end_for;

    if returnList then s:= map(v,_negate) else s:= map({op(v)},_negate) end_if
  end_for;
  return( s )
end_proc:

