/*++
    invhilbert - inverse of a Hilbert matrix

    Call:

    linalg::invhilbert( n <, CoefficientDomain> )

    Parameter:

    n      --  a positve integer
    CoefficientDomain -- 
             constructor of the coeffient domain for the output matrix.
             Admissible choices are
             Dom::Real, Dom::Float, Dom::Numerical,
             Dom::ExpressionField(), Dom::Integer.
                                                                   
    Synopsis:

    The n x n -Hilbert matrix H is given by H[i,j]=1/(i+j-1).
    invhilbert(n) returns its inverse invH given by

            invH[i,j] = (-1)^(i+j)*c[i]*c[j]/(i+j-1)
    with
            c[i]= (n+i-1)! / ( (i-1)! )^2/ (n-i)!

    It is of the domain type Dom::Matrix(), if no second argument is given. 
    The optional second argument is the CoefficientDomain of the output 
    matrix, if a domain type different from Dom::Matrix(Dom::ExpressionField()) 
    is desired. Note that the inverse of a Hilbert matrix has integer
    components, i.e., any component ring encompassing the integers
    is admissible.

    Examples:

    >> A:= linalg::invhilbert(2)

                               +-        -+
                               |   4, -6  |
                               |          |
                               |  -6, 12  |
                               +-        -+

    >> A, A^(-1)

                       +-        -+  +-          -+
                       |   4, -6  |  |   1,  1/2  |
                       |          |, |            |
                       |  -6, 12  |  |  1/2, 1/3  |
                       +-        -+  +-          -+

    >> linalg::invhilbert(2, Dom::Float);

                             +-            -+
                             |   4.0, -6.0  |
                             |              |
                             |  -6.0, 12.0  |
                             +-            -+


    See also:  linalg::hilbert, Dom::Matrix
------------------------------------------------------------------*/
linalg::invhilbert:= proc(n,R=Dom::ExpressionField())
  local i,j,c,invH;
begin
  if testargs() then
      if args(0) = 0 then error("expecting positive integer") end_if;
      if args(0) > 2 then error("too many arguments") end_if;
      if not testtype(n,Type::PosInt) then error("expecting positive integer") end_if;
      if R::hasProp(Cat::Rng) <> TRUE then error("component domain must be of category 'Cat::Rng'") end_if;
   end_if;

   c:=[( (-1)^i*((n+i-1)!)/((i-1)!)^2/((n-i)!) ) $ i = 1..n];
   invH:=array(1..n,1..n):
   ((invH[j,i]:=(invH[i,j]:= c[i]*c[j]/(i+j-1);):) $ j = 1..i-1) $ i = 1..n;
   (invH[i,i]:= c[i]^2/(2*i-1);) $ i = 1 .. n;

   case R
   of Dom::Complex do
   of Dom::Real do
   of Dom::Numerical do
   of Dom::Rational do
   of Dom::Integer do
   of Dom::ExpressionField() do
       // conversion of the components is not necessary!
       return((Dom::Matrix(R))::create( invH ))
   otherwise
       return(Dom::Matrix(R)( invH ))
   end_case
end_proc:

