// W.Oevel, 18/06/02 

/*++
    invpascal - the inverse Pascal matrix

    Call:

    linalg::invpascal( n <, CoefficientDomain> );
                                                                   
    Parameter:

    n  --  a positve integer
    CoefficientDomain -- 
             constructor of the coeffient domain for the output matrix.
             Admissible choices are
             Dom::Real, Dom::Float, Dom::Numerical,
             Dom::ExpressionField(), Dom::Integer.
                                                                   
    Synopsis:

    The n x n -Pascal matrix P is given by P[i,j]= binomial(i + j - 2, j - 1).
    Its inverse is computed by invpascal(n) with a runtim of O(n^2).
    Both P and its inverse are symmetric, positive definite, have
    determinant = 1 and consist of integer entries.

    The optional second argument is the constructor of the
    output matrix, if a domain type different from 
    Dom::Matrix(Dom::ExpressionField()) is desired.

    Examples:

3.0.0 > linalg::pascal(4), linalg::invpascal(4)

          +-              -+  +-                  -+
          |  1, 1,  1,  1  |  |   4,  -6,  4,  -1  |
          |                |  |                    |
          |  1, 2,  3,  4  |  |  -6,  14, -11,  3  |
          |                |, |                    |
          |  1, 3,  6, 10  |  |   4, -11,  10, -3  |
          |                |  |                    |
          |  1, 4, 10, 20  |  |  -1,  3,   -3,  1  |
          +-              -+  +-                  -+

    See also:  linalg::pascal, Dom::Matrix
++*/
linalg::invpascal:= proc(n, R=Dom::ExpressionField())
   local a, b, s, i, j;
begin
   if testargs() then
       if args(0) = 0 then 
          error("expecting a positive integer (the dimension)") 
       end_if;
       if args(0) > 2 then 
          error("too many arguments") 
       end_if;
       if not testtype(n, Type::PosInt ) then 
          error("expecting a positive integer (the dimension)") 
       end_if;
       if domtype(R) <> DOM_DOMAIN or 
          R::hasProp(Cat::Rng) <> TRUE then 
          error("the component domain must be of category 'Cat::Rng'")
       end_if;
   end_if;

   /* ---------------------------------------
      invpascal[i, j] = a[i,j] = 
        (-1)^(i+j)* _plus(binomial(k-1, i-1)*
                          binomial(k-1, j-1) 
                          $ k=max(i, j)..n);
      The following recursion is satisfied:
        a[i-1,j-1] - a[i-1,j] - a[i,j-1] = (-1)^(i+j)*b(i)*b(j)
      with b(j) = binomial(n, j-1)
   ----------------------------------------*/

   //-------------------------------------------------------
   // b = container for b[i] = binomial(n-1, i-1) and (later)
   // b[i] = binomial(n, i - 1)
   b:= [1 $ n+1];
   for i from 2 to ceil(n/2) do
       b[i]:= b[i-1]*(n+1-i)/(i-1);
       b[n+1-i]:= b[i];
   end_for;
   // now, b[i] = binomial(n-1, i-1)
   //-------------------------------------------------------

   // a = the inverse Pascal matrix
   a:= array(1..n,1..n);
   // fill in last row and last column
   s:= (-1)^n;
   for j from 1 to n-1 do
     s:= -s; // s = (-1)^(n + j);
     a[n, j]:= s * b[j];
     a[j, n]:= a[n, j];
   end_for; 
   a[n, n]:= 1;

   //-------------------------------------------------------
   // Replace b[i] = binomial(n-1, i-1) by
   // b[i] = binomial(n, i-1) by
   for i from n downto 2 do
       b[i]:= b[i-1] + b[i];
   end_for;
   // now, b[i] = binomial(n, i-1)
   //-------------------------------------------------------

   // fill in row by row and column by column using
   // a[i-1, j-1] - a[i-1, j] - a[i, j-1] = (-1)^(i + j)*b(n+1, j)*b(n+1, i)
   for i from n-1 downto 1 do
     a[i, i] := 2*a[i, i+1] + b[i+1]^2;
     for j from i-1 downto 1 do 
         a[i, j] := a[i, j+1] + a[i+1, j] + (-1)^(i+j)*b[i+1]*b[j+1];
         a[j, i] := a[i, j];
     end_for;
   end_for;

   //-------------------------------------------------------
   // Conversion of the array  to a suitable matrix
   //-------------------------------------------------------
   case R
    of Dom::Complex do
    of Dom::Real do
    of Dom::Numerical do
    of Dom::Rational do
    of Dom::ExpressionField() do
       // conversion of the components is not necessary!
       return((Dom::Matrix(R))::create( a ))
   otherwise
       return(Dom::Matrix(R)( a ))
   end_case
end_proc:
