// W.Oevel, 18/06/02 

/*++
    invvandermonde - the inverse Pascal matrix

    Call:

    linalg::invvandermonde( n <, CoefficientDomain> );
                                                                   
    Parameter:

    n  --  a positve integer
    CoefficientDomain -- 
             constructor of the coeffient domain for the output matrix.
             Admissible choices are
             Dom::Real, Dom::Float, Dom::Numerical,
             Dom::ExpressionField(), Dom::Integer.
                                                                   
    Synopsis:

    The n x n -Pascal matrix P is given by P[i,j]= binomial(i + j - 2, j - 1).
    Its inverse is computed by invvandermonde(n) with a runtim of O(n^2).
    Both P and its inverse are symmetric, positive definite, have
    determinant = 1 and consist of integer entries.

    The optional second argument is the constructor of the
    output matrix, if a domain type different from 
    Dom::Matrix(Dom::ExpressionField()) is desired.

    Examples:

3.0.0 > linalg::vandermonde(4), linalg::invvandermonde(4)

          +-              -+  +-                  -+
          |  1, 1,  1,  1  |  |   4,  -6,  4,  -1  |
          |                |  |                    |
          |  1, 2,  3,  4  |  |  -6,  14, -11,  3  |
          |                |, |                    |
          |  1, 3,  6, 10  |  |   4, -11,  10, -3  |
          |                |  |                    |
          |  1, 4, 10, 20  |  |  -1,  3,   -3,  1  |
          +-              -+  +-                  -+

    See also:  linalg::vandermonde, Dom::Matrix
++*/
linalg::invvandermonde:= proc(v, R=Dom::ExpressionField())
   local RR, n, a, t, i, j, k, use_expand;
begin
   if testargs() then
       if args(0) = 0 then 
          error("expecting at least one argument (a list of nodes)");
       end_if;
       if args(0) > 2 then 
          error("too many arguments") 
       end_if;
       if not testtype(v,linalg::vectorOf(Type::AnyType) ) and
          not testtype(v, DOM_LIST ) then
            error("the first argument should be a vector of category 'Cat::Matrix' ".
                  "or a list");
       end_if;
       if R::hasProp(Cat::Rng) <> TRUE then 
          error("the component domain must be of category 'Cat::Rng'")
       end_if;
   end_if;

   if domtype(v) <> DOM_LIST then
     RR:= v::dom::coeffRing;
     v:= [op(v)]:
     if args(0) = 1 then
       // no coeff ring requested by the user.
       R:= RR;
     end_if;
   else
       RR:= Dom::ExpressionField();
   end_if;

   n:= nops(v);

   if nops({op(v)}) < n then
     //the Vandermonde nodes must be distinct
     return(FAIL);
   end_if;

   if indets(v) <> {} then
        use_expand:= TRUE;
   else use_expand:= FALSE;
   end_if; 

   /* ---------------------------------------
      invvandermonde[i, j] = a[i,j] 
   ----------------------------------------*/
   //------------------------------------------
   // compute the denominator
   // t[j] = product(v[j]-v[k]) $ k=1..n, k<>j)
   // for all elements of the j-th column
   //------------------------------------------
   t:= array(1..n, [1$n]);
   for j from 1 to n do 
     for i from 1 to j-1 do
       t[j]:= t[j]*(v[j]-v[i]);
     end_for;
     for i from j+1 to n do
       t[j]:= t[j]*(v[j]-v[i]);
    end_for:
   end_for;

   //--------------------------------------
   // initialize a = invvandermonde:
   //--------------------------------------

   a:= array(1..n, 1..n, [[0 $ n] $ n]):

   //-----------------------------------------------------
   // compute the first column of the inverse 
   // (without denominator t[1])
   // 1) initialize a[n+1-i,1] = v[i]+v[i+1]+...+v[n], i=2..n
   //-----------------------------------------------------
   a[1,1]:= v[n];
   for i from 2 to n-1 do
       a[i, 1]:= a[i-1, 1] + v[n+1-i]
   end_for:
   a[n, 1]:= 1;

   //-------------------------------------------------
   // 2) construct the symmetric polys in v[2],..,v[n]
   //-------------------------------------------------
   
   for k from n-1 downto 2 do
     a[1, 1]:= v[k]*a[1,1];
     for i from 2 to k-1 do
         a[i,1]:= v[k+1-i]*a[i,1] + a[i-1,1];
         if use_expand then
            a[i, 1]:= expand(a[i, 1]);
         end_if;
     end_for:
   end_for:

   //-----------------------------------------------------
   // Use 
   // a[i,j1]*v[j1] - a[i,j2]*v[j2] = a[i-1,j1] - a[i-1,j2]
   // together with a[n, j] = 1 to compute the other rows
   // recursively from the first row
   //-----------------------------------------------------
   for j from 2 to n do
     a[n, j]:= 1;
     for i from n-1 downto 1 do
       a[i, j]:= a[i, 1] + a[i+1, 1]*v[1] - a[i+1,j]*v[j];
       if use_expand then
          a[i, j]:= expand(a[i, j]);
       end_if;
     end_for;
   end_for;

   //-----------------------------------------------------
   // divide the entries by the denominators t[j]
   // common to all elements of the j-th column
   //-----------------------------------------------------
   for i from 1 to n do
    for j from 1 to n do
       a[i, j]:= (-1)^(n-i)*a[i, j]/t[j];
    end_for;
   end_for;

   //-------------------------------------------------------
   // Conversion of the array  to a suitable matrix
   //-------------------------------------------------------
   if R = RR then // output type = input type
        // conversion of the components is not necessary!
        return((Dom::Matrix(R))::create( a ))
   else return(Dom::Matrix(R)( a ))
   end_if;
end_proc:
