//      
// K. F. Gehrs, 06/06/00

/*++      
        linalg::jordanForm -- computes the Jordan form of a square matrix

        jordanForm( A [,All] )

        A  : square matrix over a Cat::Field
        All: option

        jordanForm( A ) computes the Jordan form of A, hence it
        determines a transformation matrix P such that the matrix

            J = P^(-1) * A * P

        has Jordan form.
        If option 'All' is given then the list [J,P] is returned.

        Note: The secondary diagonal of J is above the main diagonal.
        Use
    
            transpose(J) = transpose(P)*transpose(A)*transpose(P)^(-1)

        to get this diagonal below the main diagonal.
    
        Reference: "Calcul exact des formes de Jordan et de Frobenius d'une 
                   matrice", by Patrick Ozello, "Docteur de I'Universite 
                   Scientifique", pages 10 to 25.    
++*/
linalg::jordanForm:= proc(A)
     local B, R, Mat, u, i, j, arr, a, n, F, J, Q, P, calculPJ, tmp, 
           calculJ, Rplus, Rnegate, Rmult, Rzero, Rdivide, Rcoerce,
           Rone, Riszero, polyCoeff, L, Res;
begin
    if testargs() then
        if args(0) < 1 or args(0) > 2 then
            error("expecting 1 or 2 arguments")
        end_if;
    
        Mat:= A::dom;

        if Mat::hasProp( Cat::Matrix ) <> TRUE then
            error("first argument is not of 'Cat::Matrix'")
        end_if;

        R:= Mat::coeffRing;
        if not R::hasProp( Cat::Field ) then
            error("expecting matrix over 'Cat::Field'");
        end_if;
    
        n:= Mat::matdim(A);
    
        if n[1] <> n[2] then
            error("not a square matrix")
        else 
            n:= n[1]
        end_if;
        arr:= args(0);
        if arr = 2 and args(2) <> hold(All) then
            error("2nd argument must be the option 'All'")
        elif arr > 2 then
            error("too many arguments given")
        end_if;
    else
        Mat:= A::dom;
        R:= Mat::coeffRing;
        n:= Mat::matdim(A)[1];
    end_if;

    
    //=============================================================

    if (numeric::indets(A) <> {} and n < 7) or 
       (numeric::indets(A) = {} and n < 13) then 
      if args(0) = 1 then 
        Res:= linalg::symbolicEigenvectors(A,"JordanForm");
        if Res <> FAIL then 
          return(Res);
        end_if;
      elif args(0) = 2 then 
        Res:= linalg::symbolicEigenvectors(A,"JordanForm","All");
        if Res <> FAIL then 
          return(Res);
        end_if;
      end_if;  
    end_if;
        
        /* ev:= linalg::eigenvalues(A);
        if nops(ev) = n and min(map(ev,length)) > 100 then 
        L:= linalg::symbolicEigenvectors(A);
        if args(0) = 1 then 
          return(A::dom(n,n,[L[i][1] $ i = 1..n],Diagonal));
        else 
          return([A::dom(n,n,[L[i][1] $ i = 1..n],Diagonal),
                  A::dom::concatMatrix(L[i][3][1] $ i = 1..n)]);  
        end_if;*/
    
    //=============================================================
    
    Rplus:= R::_plus;
    Rnegate:= R::_negate;
    Rmult:= R::_mult;
    Rzero:= R::zero;
    Rdivide:= R::_divide;
    Rone:= R::one;
    Riszero:= R::iszero;
    Rcoerce:= R::coerce;

    // ------------- SUBROUTINES -------------------

        //----------------------------------------------------------
        // "calculJ" calculates the Jordan Canonical Form out of the 
        // Frobenius Canonical Form of the matrix A 
        //----------------------------------------------------------

        calculJ:= proc(F, arr)
            name linalg::jordanForm;
            local z, NumberofBlocks, Jidx, t, i, j, k, p, num, pow, q, qt, 
                  roots, calculBlock, extractCo, m, converted;
        begin
    
            // ------------- SUBROUTINES -------------------
       
            // "calculBlock" caculates one of the block matrices in J
            // according to the block-structure of the Frobenius Form 
            // It places the eigenvalues on the main diagonal and 
            // fills the entries above the main diagonal with 1, 
            // according to the degree of the minimal polynomial 
            // of the block matrix, the other entries with 0.
            // Note, that these polynomials are uniquely determined 
            // by the former caculation of the Frobenius Canonical Form
            // and thereby determine the structure of the block matrices in the
            // Jordan Canonical Form.
        
            calculBlock := proc(n, lambda, size, Jindex)
                name linalg::jordanForm;
            //  global J, Jindx
                local i, j, tmp;
            begin
                tmp:= size + Jindex - 1;
            
                for i from Jindex to tmp - 1 do 
                    J[i, i]:= lambda; 
                    J[i, i + 1]:= Rone;
                end_for;
            
                J[tmp, tmp]:= lambda;
                
                for i from Jindex to tmp do
                    for j from Jindex to i - 1 do 
                        J[i, j]:= Rzero; 
                    end_for;
                end_for;
            
                for i from Jindex to tmp - 2 do
                    for j from i + 2 to tmp do
                        J[i, j]:= Rzero;
                    end_for;
                end_for;
                
                for i from Jindex to size + Jindex - 1 do
                    for j from Jindex + size to n do 
                        J[i, j]:= Rzero; 
                        J[j, i]:= Rzero;
                    end_for;
                end_for;
                
                return ( Jindex + size );
            end_proc:
        
            // "extractCo" determines the characteristic polynomials 
            // of the companion matrices, given by the Frobenius Form.
            // Thereby we are able to determine the eigenvalues and 
            // their algebraic multiplicities. 
        
            extractCo := proc(F, q, arr, z)
                name linalg::jordanForm;
                local c, i, nb, pow;
            begin
                nb:= arr[0];
                arr[0]:= 0;
                
                for c from 1 to nb do
                    q[c]:= poly(-F[arr[c - 1] + 1, arr[c]],[z],polyCoeff);
                    pow:= 1; 
                    for i from arr[c - 1] + 2 to arr[c] do 
                        // Do not use 'poly( F[i,arr[c]]*z^pow, [z], polyCoeff )',
                        // because for some component domains such as
                        // Dom::ExpressionField(normal), the product
                        // F[...]*z^pow and therefore the factor z is 
                        // converted first into an element of the component 
                        // domain! The result is then not a polynomial in z!
                        q[c]:= q[c] - poly([[F[i,arr[c]],pow]],[z],polyCoeff); 
                        pow:= pow + 1;
                    end_for;
                    q[c]:= q[c] + poly(z^pow,[z],polyCoeff);
                end_for;

                return( [q,arr] )
            end_proc:

            // ------------- MAIN of "calculJ" -------------------
            NumberofBlocks:= arr[0];
            J:= array(1 .. n, 1 .. n);
            p:= array(1 .. NumberofBlocks);
            Jidx:= 1;
            // z:= `#z`;
            z:= genident("z");
            [p,arr]:= extractCo(F, p, arr, z);
    
            for i from 1 to NumberofBlocks do
                L:= map({op(map(poly2list(p[i]), ind -> ind[1]))}, domtype);
                if contains(L, DOM_FLOAT) then 
	          p[i]:= numeric::rationalize(p[i]);
		  q:= polylib::sqrfree( p[i] );
		  converted:= TRUE:
		  // q:= map(q, ind -> mapcoeffs(ind, Dom::Float));
		else
		  q:= polylib::sqrfree( p[i] );
		  converted:= FALSE:
                end_if:

                qt:= type(Factored::expr(q));
                if nops(q) = 1 then
                    // p[i] is a constant polynomial!?
                    error("should not happen")
                elif nops(q) > 3 or qt = "_mult" then
                    m:= nops(q) div 2;
                elif qt = "_power" or qt = "_plus" or qt = DOM_IDENT then
                    m:= 1;
                else
                    error("should not happen");
                end_if;
    
                for j from 1 to m do
		    if converted = FALSE then 
                      t:= op(q, 2*j); //q[2*j]; 
		    else 
		      t:= mapcoeffs(op(q, 2*j), Dom::Float); //mapcoeffs(q[2*j], Dom::Float);
		    end_if;
                    pow:= op(q, 2*j+1); // q[2*j+1];
                                      
                    if (k:= degree(t)) = 0 then  // not has(t, z) 
                        next // j
                    end_if;

                    if polyCoeff = hold(Expr) then
                        roots:= solve(t, z, hold(MaxDegree)=4, hold(Domain)=R, 
			              IgnoreSpecialCases);
                    else
                        roots:= solve(t, z, hold(MaxDegree)=4, IgnoreSpecialCases);
                    end_if;

                    if type(roots) = "solve" then
                        warning("the system solver 'solve' is not able to determine roots of the minimal polynomial");
                        return( FAIL )
                    elif hastype(roots, RootOf) then
                        warning("found roots of the minimal polynomial that cannot be determined in terms of radicals");
                        return( FAIL )
                    elif (num:= nops(roots)) <> k then
                        warning("the Jordan normal form does not exist over the component domain of the input matrix");
                        return( FAIL )
                    else
                        ( Jidx:= calculBlock(n, roots[k], pow, Jidx) ) $ k=1..num;
                    end_if
                end_for
            end_for;
            return( J )
        end_proc:


        //----------------------------------------------------------
        // "calculPJ" calculates the Jordan Canonical Form out of the 
        // Frobenius Canonical Form of the matrix A and also the 
        // Transformation Matrix Q, so that later P := B * Q^-1 is the 
        // Matrix which serves the condition J := P^-1 * A * P 
        // Note: If the computation over R is not possible then FAIL
        //       is returned!
        //----------------------------------------------------------
        calculPJ:= proc(F, arr)
            name linalg::jordanForm;
            local z, NumberofBlocks, Qi, Qipos, Qijpos, Jidx, Qidx, t, i, j, k, n,
                  m, p, Li, Qij, dq, num, pow, q, qt, roots,  
                  calculBlockJ, extractCo, calculHelpMAT, addBlockMatrix,
                  calculBlockP, converted;    
        begin
            // ------------- SUBROUTINES -------------------
    
            // "calculHelpMAT" caculates one block matrix, which will later 
            // be used to construct the matrix Q.
            // Note: If the computation over R is not possible then FAIL
            //       is returned!
            calculHelpMAT :=  proc(i, n, zz)
                name linalg::jordanForm;
                local L, h, k, y, tmp;
            begin
                L:= array( 1..i, 1..n);
                // y:= `#y`;
                y:= genident("y");
                if Riszero(zz) then 
                    for h from 1 to i do
                        for k from 1 to n do
                            tmp:= poly( binomial(k - 1, i - h)*y^(k - 1 + h - i), [y]);
                            if ( L[h,k]:= Rcoerce(evalp(tmp, y = 0)) ) = FAIL then
                                return( FAIL )
                            end_if;
                        end_for;
                    end_for;
                else 
                    for h from 1 to i do
                        for k from 1 to n do
                            if ( L[h,k]:= Rcoerce(binomial(k - 1, i - h)*zz^(k - 1 + h - i)) ) = FAIL then
                                return( FAIL )
                            end_if;
                        end_for;
                    end_for; 
                end_if;
                return (L);
            end_proc:
    
            // "addBlockMatrix" adds the block matrix B to the matrix A
            // according to the position given by the parameter pos
            addBlockMatrix:= proc(A, B, pos)
                name linalg::jordanForm;
                local m, n, i, j, ps;
            begin
                ps:= pos - 1; 
                m:= op( (op(B,0)[2]), 2); 
                n:= op( (op(B,0)[3]), 2);
                for i from 1 to m do
                    for j from 1 to n do
                        A[ps+i, j]:= B[i, j];
                    end_for;
                end_for;
                m:= pos + m;
                return( [m,A] );
            end_proc;

            // "calculBlockP" caculates one of the block matrices in Q itself
            calculBlockP := proc(n, Qmat, Qindex, Qisize)
                name linalg::jordanForm;
                // global Q
                local i, j, tmp, a, b;
            begin
                tmp:= Qindex + Qisize - 1;
                for i from Qindex to tmp do
                    for j from Qindex to tmp do
                        a:= i - Qindex + 1;
                        b:= j - Qindex + 1;
                        Q[i, j]:= Qmat[a, b];
                    end_for;
                end_for;
                for i from Qindex to tmp do
                    for j from Qisize+Qindex to n do
                        Q[i, j]:= Rzero;
                        Q[j, i]:= Rzero;
                    end_for;
                end_for;                   
                return( Qisize + Qindex );
            end_proc:

            // "calculBlock" caculates one of the block matrices in J
            // according to the block-structure of the Frobenius Form 
            // It places the eigenvalues on the main diagonal and 
            // fills the entries above the main diagonal with 1, 
            // according to the degree of the minimal polynomial 
            // of the block matrix, the other entries with 0.
            // Note, that these polynomials are uniquely determined 
            // by the former caculation of the Frobenius Canonical Form
            // and thereby determine the structure of the block matrices in the
            // Jordan Canonical Form.
            calculBlockJ:= proc(n, lambda, size, Jindex)
                name linalg::jordanForm;
                // global J, Jidx, Q
                local i, j, tmp, converted;
            begin
                tmp:= size + Jindex - 1;
                for i from Jindex to tmp - 1 do 
                    J[i, i]:= lambda; 
                    J[i, i + 1]:= Rone;
                end_for;
                J[tmp, tmp]:= lambda;
                for i from Jindex to tmp do
                    for j from Jindex to i - 1 do 
                        J[i, j]:= Rzero; 
                    end_for;
                end_for;
                for i from Jindex to tmp - 2 do
                    for j from i + 2 to tmp do
                        J[i, j]:= Rzero;
                    end_for;
                end_for;
                for i from Jindex to size + Jindex - 1 do
                    for j from Jindex + size to n do 
                        J[i, j]:= Rzero; 
                        J[j, i]:= Rzero;
                    end_for;
                end_for;

                return ( Jindex + size );
            end_proc:

            // "extractCo" determines the characteristic polynomials 
            // of the companion matrices, given by the Frobenius Form
            // Thereby we are able to determine the eigenvalues. 
            extractCo:= proc(F, q, arr, z)
                name linalg::jordanForm;
                local c, i, nb, pow;
            begin
                nb:= arr[0];
                arr[0]:= 0;
                for c from 1 to nb do
                    q[c]:= poly(-F[arr[c - 1] + 1, arr[c]], [z], polyCoeff);
                    pow:= 1;
                    for i from arr[c - 1] + 2 to arr[c] do 
                        // Do not use 'poly(F[i,arr[c]]*z^pow, [z], polyCoeff)',
                        // because for some component domains such as
                        // Dom::ExpressionField(normal), the product
                        // F[...]*z^pow and therefore the factor z is 
                        // converted first into an element of the component 
                        // domain! The result is then not a polynomial in z!
                        q[c]:= q[c] - poly([[F[i,arr[c]],pow]],[z],polyCoeff); 
                        pow:= pow + 1;
                    end_for;
                    q[c]:= q[c] + poly(z^pow,[z],polyCoeff)
                end_for;

                return( [q,arr] )
            end_proc:

            // ------------- MAIN of "calculPJ" -------------------
            NumberofBlocks:= arr[0];
            n:= linalg::matdim(F);
            n:= n[1];
            J:= array(1 .. n, 1 .. n);
            p:= array(1 .. NumberofBlocks);
            Jidx:= 1;
            Qidx:= 1;
            Q:= array(1 .. n, 1 .. n);

            //z:= `#z`;
            z:= genident("z");
    
            ([p,arr]):= extractCo(F, p, arr, z);
        
            for i from 1 to NumberofBlocks do
                L:= map({op(map(poly2list(p[i]), ind -> ind[1]))}, domtype);
                if contains(L, DOM_FLOAT) then 
	          p[i]:= numeric::rationalize(p[i]);
		  q:= polylib::sqrfree( p[i] );
		  converted:= TRUE:
		  // q:= map(q, ind -> mapcoeffs(ind, Dom::Float));
		else
		  q:= polylib::sqrfree( p[i] );
		  converted:= FALSE:
                end_if:
                qt:= type(Factored::expr(q));
                if nops(q) = 1 then
                    // p[i] is a constant polynomial!?
                    error("should not happen")
                elif nops(q) > 3 or qt = "_mult" then
                    m:= nops(q) div 2;
                elif qt = "_power" or qt = "_plus" or qt = DOM_IDENT then
                    m:= 1;
                else
                    error("should not happen");
                end_if;

                dq:= degree( p[i] );
                Qi:= array( 1..dq , 1..dq );
                Qipos:= 1;
    
                for j from 1 to m do
		    if converted = FALSE then 
                      t:= op(q, 2*j); //q[2*j]; 
		    else 
		      t:= mapcoeffs(op(q, 2*j), Dom::Float); // mapcoeffs(q[2*j], Dom::Float);
		    end_if;
                    pow:= op(q, 2*j+1); //q[2*j+1];
                    if (k:= degree(t)) = 0 then // not has(t, z) 
                        next // j
                    end_if;
                     
                    Qij:= array( 1..pow*degree(t), 1..dq);
                    Qijpos:= 1;
        
                    if polyCoeff = hold(Expr) then
                        roots:= solve(t, z, hold(MaxDegree)=4, hold(Domain)=R, IgnoreSpecialCases);
                    else
                        roots:= solve(t, z, hold(MaxDegree)=4, IgnoreSpecialCases);
                    end_if;

                    if type(roots) = "solve" then
                        warning("the system solver 'solve' is not able to determine roots of the minimal polynomial");
                        return( FAIL )
                    elif hastype(roots, RootOf) then
                        warning("found roots of the minimal polynomial that cannot be determined in terms of radicals");
                        return( FAIL )
                    elif (num:= nops(roots)) <> k then
		        warning("the Jordan normal form does not exist over the component domain of the input matrix");
                        return( FAIL )
                    else
                        for k from 1 to num do
                            Jidx:= calculBlockJ(n, roots[k], pow, Jidx);
                            if ( Li:= calculHelpMAT(pow,dq,roots[k]) ) = FAIL then
                                return( FAIL )
                            end_if;
                    
                            [Qijpos,Qij]:= addBlockMatrix(Qij,Li,Qijpos);
                        end_for;
                        [Qipos,Qi]:= addBlockMatrix(Qi,Qij,Qipos);
                    end_if
                end_for;
                Qidx:= calculBlockP(n,Qi,Qidx,dq);
            end_for;

            return( [J,Q] )
        end_proc:

    // ------------- MAIN OF "jordanForm" -------------------

    // choose an appropriate coefficient ring for polynomial 
    // computations:
    if R::hasProp(Ax::systemRep) then
        polyCoeff:= hold(Expr)
    else
        polyCoeff:= R
    end_if;
    
    arr:= array(0 .. n);
    arr[0]:= 0;
    i:= 0;
    if args(0) = 2 then
        ([F,B]) := linalg::frobeniusFormOzello(A,hold(All));
        a:= [ F[u + 1, u] $ u=1 .. n-1, 0 ];
        for j from 1 to n do
            if Riszero(a[j]) then
                i:= i + 1; 
                arr[i]:= j;
            end_if;
        end_for;
        arr[0]:= i;
 
        if (tmp:= calculPJ(F,arr)) = FAIL then
            return( FAIL );
        end_if;

        J:= Mat::create(tmp[1]);
        P:= Mat::create(tmp[2]);

        if R::normal <> FAIL then
            P:= map( Mat::_mult( B,Mat::_invert(P) ),R::normal );
        else
            P:= Mat::_mult( B,Mat::_invert(P) );
        end_if;
        L:= map({op( J )}, domtype); 
        if contains(L, DOM_FLOAT) then 
          return( [float(J), float(P)] );
        else
          return([J,P])
        end_if:
    else
        F:= linalg::frobeniusFormOzello(A);
        a:= [ F[u + 1, u] $ u=1 .. n-1, 0 ];
        for j from 1 to n do
            if Riszero(a[j]) then
                i:= i + 1; 
                arr[i]:= j;
            end_if;
        end_for;
        arr[0]:= i;
        J:= calculJ(F,arr);
        if J = FAIL then
            return( FAIL )
        else
	    J:= Mat::create(J);
            L:= map({op( J )}, domtype); 
	    if contains(L, DOM_FLOAT) then 
	      return( float(J) );
	    else
              return( J )
	    end_if:
        end_if
    end_if
end_proc:

