/*++
    linalg::matlinsolve  --  solve a linear system of equations

    matlinsolve( A [,B] )
    matlinsolve( A {,v,l} )
    matlinsolve( A {,v,Special,Unique} )

    A, B            - matrices
    v               - vector
    l               - list with exactly ncols(A) elements
    Special, Unique - identifier (options)

    matlinsolve(A,B) solves the systems A*x=col(B,i), i=1..ncols(B) 
    and returns a matrix representing the special solutions of the 
    ncols(B) systems.
    If B is missing the the system A[1..r,1..c-1]*c=col(A,c)
    will be solved.

    matlinsolve(A,v,l) solves the system A*x=B. If the system
    is undetermined, then the general solution of the system 
    will be expressed by the sum of a special solution and a
    linear combination of the general solution of the corresponding
    homogeneous system with scalars in l. Only those scalars are
    used which correspond to a kernel vector of A.
    If v is missing then the system A[1..r,1..c-1]*c=col(A,c)
    will be solved.


    matlinsolve(A,v,Special) returns a special solution of the system.
    matlinsolve(A,v,Unique) returns an unique solution if one exits,
    otherwise FAIL is returned if more than one solution exists.
    If v is missing then the system A[1..r,1..c-1]*c=col(A,c)
    will be solved.


    matlinsolve returns [] if the system has no solution,
    and FAIL if the solution could not be computed over the 
    coefficient domain of A.

    Solutions are always elements of the domain Matrix(A::coeffRing).
++*/

linalg::matlinsolve := proc(A,B)
    local r, c, t, R, i, j, k, ns, rs, cB, s, v, l, Flag,
          RisField, create, Rdivide, Rone, Rnegate, Riszero, Rmult, 
          Rcoerce, Rzero, Afloated, Bfloated, L, Mat, 
          arguments, useproc;
begin
    if A::dom::hasProp( Cat::Matrix ) <> TRUE then
        error("the first argument is not of category 'Cat::Matrix'")
    end_if;

    Mat:= A::dom;

    R := A::dom::coeffRing;
    if R::hasProp( Cat::IntegralDomain ) <> TRUE then
        error("expecting a matrix over Cat::IntegralDomain")
    end_if;

    // New cases: 
    // check whether the matrix contains floats and can be converted 
    // to 'float(Matrix)'. If this is the case, call the corresponding 
    // 'numeric'-routine as in the case above. 

    if args(0) > 1 and B::dom::hasProp(Cat::Matrix) = TRUE then 
      if (Afloated:= linalg::checkForFloats(A) <> FALSE) or  
         (Bfloated:= linalg::checkForFloats(B) <> FALSE) then 
        if Afloated::dom::hasProp(Cat::Matrix) <> TRUE then Afloated:= A end_if;
        if B::dom::hasProp(Cat::Matrix) = TRUE then Bfloated:= B end_if;
        L:= numeric::matlinsolve(Afloated, Bfloated);
        if iszero(L[2]) then 
          L:= Mat::coerce(L[1]);
          if L <> FAIL then  
            return(L);
          else  
            return([]);
          end_if;
        elif (L[1]:= Mat::coerce(L[1])) <> FAIL and 
             (L[2]:= Mat::coerce(L[2])) <> FAIL then 
          L[2]:= linalg::col(L[2], 1..linalg::matdim(L[2])[2]);
          if contains({args(i) $ i = 1..args(0)}, hold(Special)) = TRUE then  
            return(L[1]);
          elif contains({args(i) $ i = 1..args(0)}, hold(Unique)) = TRUE then 
            return(NIL);
          end_if; 
          return(L);
        else
          return([]);
        end_if;
      end_if;
    elif (Afloated:= linalg::checkForFloats(A)) <> FALSE then 
      B:= Afloated::dom::col(Afloated, Afloated::dom::matdim(Afloated)[2]);
      Afloated:= Afloated::dom::delCol(Afloated, Afloated::dom::matdim(Afloated)[2]);
      // L:= numeric::matlinsolve(Afloated, [0 $ linalg::matdim(Afloated)[1]]);
      L:= numeric::matlinsolve(Afloated, B);
      if iszero(L[2]) then 
        L:= Mat::coerce(L[1]);
        if L <> FAIL then 
          return(L)
        else  
          return([]);
        end_if;
      elif (L[1]:= Mat::coerce(L[1])) <> FAIL and 
           (L[2]:= Mat::coerce(L[2])) <> FAIL then 
        L[2]:= linalg::col(L[2], 1..linalg::matdim(L[2])[2]);
        if contains({args(i) $ i = 1..args(0)}, hold(Special)) = TRUE then 
          return(L[1]); 
        elif contains({args(i) $ i = 1..args(0)}, hold(Unique)) = TRUE then
          return(NIL);
        end_if;
        return(L);
      else
        return([]);
      end_if;
    end_if;
    
    // There is no other chance than to skip to the 
    // symbolic computation. 
    
    t:= A::dom::matdim(A);
    r:= t[1]; c:= t[2];

    if A::dom::constructor <> Dom::DenseMatrix and 
       A::dom::constructor <> Dom::Matrix then
        t:= Dom::Matrix(R);
        A:= t::coerce(A);
        if A = FAIL then
           error("cannot convert matrix to the domain 'Dom::Matrix(R)'")
        end_if
    else
        t:= A::dom
    end_if;
    create:= t::create;

    if args(0) = 3 then Flag:= args(3) else Flag:= NIL end_if;

    if testargs() and args(0) > 1 then
        if not contains( {hold(Unique),hold(Special)},B )
             and domtype(B) <> DOM_LIST 
        then
            if args(0) > 3 then
                error("wrong no of args")
            end_if;

            B:= t::coerce( B );
            if B = FAIL then 
                error("types of matrices don't match")
            end_if;
            s:= B::dom::matdim(B);
            if s[1] <> r then error("invalid operands") end_if;
            if args(0) = 3 and not contains( {hold(Unique),hold(Special)},Flag ) and domtype(Flag) <> DOM_LIST then
                error("invalid 3rd argument")
            end_if
        elif args(0) > 2 then 
            error("wrong no of args")
        end_if
    end_if;

  RisField:= R::hasProp( Cat::Field );
  arguments:= {args(i) $ i = 1..args(0)};
  // If one of the arguments is 'IgnoreHeuristic', then 
  // ignore the heuristics listed in the following
  // procedure.
  if RisField and not contains(arguments, hold(IgnoreHeuristic)) and nops(indets(A)) > 0 then 
//  if linalg::symbolicity(A) = 1 then 
//    useproc:= linalg::SSS;
//  else 
//    useproc:= linalg::SSS2;
//  end_if;
    useproc:= linalg::SSSGaussJordan;

    if args(0) = 1 then  
      B:= A::dom::col(A, A::dom::matdim(A)[2]); // frueher: Mat(A::dom::matdim(A)[1], 1);
      A:= A::dom::delCol(A, A::dom::matdim(A)[2]);
    end_if;
    if B::dom::matdim(B)[2] = 1 then 
      if contains({args(i) $ i = 1..args(0)}, hold(Special)) = TRUE then 
        L:= useproc(A,B);
        if L = [] then return(NIL) else return(L[1]); end_if;
      elif domtype(Flag) = DOM_LIST then 
        L:= useproc(A,B);
        if L = [] then 
          return(NIL) 
        elif L[2] = [] then  
          return(L[1]);
        else
          return(L[1] + _plus(Flag[L[3][i]] * L[2][i] $ i = 1..nops(L[3])));
        end_if;            
      end_if;
      if contains({args(i) $ i = 1..args(0)}, hold(Unique)) = TRUE then 
        L:= useproc(A,B); 
        if L = [] then return([]) elif L[2] <> [] then return(NIL) else return(L[1]); end_if;
      end_if;
      L:= useproc(A,B);
      if L = [] then 
        return(L);
      elif L[2] = [] then 
        return(L[1]);
      else 
        return([L[1],L[2]]);
      end_if;
    else // B::dom::matdim(B)[2] > 1 
      // If B has several columns, neither the 'Flag' nor
      // the options 'Special' and 'Unique' are allowed.
/*    The following loop is necessary for those SSSxyz routines that do
      not cope with multi-column RightSides (such as linalg::SSS2)

      RightSides:= [B::dom::col(B,i) $ i = 1..B::dom::matdim(B)[2]];
      for i from 1 to nops(RightSides) do 
        L:= useproc(A, RightSides[i]);
        if L = [] then
           return([])
        end_if;
        if L[2] <> [] then 
          return([]);
        else 
          RightSides[i]:= L[1];
        end_if;
      end_for;
      return(A::dom::concatMatrix(RightSides[i] $ i = 1..nops(RightSides)));
*/
      L:= useproc(A, B);
      if L = [] then 
         return([])  // no solution
      elif L[2] <> [] then 
         return([]); // no unique solution
      else 
         return(L[1]); // the unique solution
      end_if;
    end_if; // if B::dom::matdim(B)[2] = 1
  end_if; // if RisField and not contains(arguments, hold(IgnoreHeuristic)) and nops(indets(A)) > 0

    //================================================================================
    // Here, either R is not a field or IgnoreHeuristic is specified or indets(A) = {}
    //================================================================================
    if args(0) > 1 then
       if not contains( {hold(Unique),hold(Special)},B ) and 
          domtype(B) <> DOM_LIST then
          B:= t::coerce(B);
          cB:= B::dom::matdim(B)[2];
          A:= A::dom::concatMatrix( A,B );
       else
          Flag:= B;
          c:= c - 1; cB:= 1
       end_if
    else
       c:= c - 1; cB:= 1;
       if c = 0 then error("invalid matrix dimension") end_if;
    end_if;

    // ========================================
    userinfo(1,"perform Gaussian elimination");
    // ========================================
  
    Riszero:= R::iszero;

    s:= A::dom::gaussElim( A );
    rs:= s[2]; // rank of (A,B)
    s:= s[1];

    // rank(A,B) = rank(A)?
    if rs > 0 then
        for t from rs to c do
            if not Riszero( s[rs,t] ) then break end_if
        end_for;
        if t > c then return( [] ) end_if
    end_if;

    userinfo(1,"compute row reduced form");
    s:= linalg::gaussJordan(s, hold(All));

    A:= s[1];

    // each col-no > c does not belong to ker(A)
    ns:= { $ 1..c } minus s[4];

    // ========================================
    // Special case: the rank is maximal, i.e., 
    // there is a unique solution:
    // ========================================
    if nops(ns) = 0 then  // rank(A) = c
       if RisField then
          // the right side of A is the solution
          return( A[1..c,c+1..cB+c] )
       else
          Rdivide:= R::_divide;
          s:= A[1..c,c+1..cB+c]; // right-hande side
          for i from 1 to c do
            for j from 1 to cB do
              if (t:= Rdivide( s[i,j],A[i,i] )) = FAIL then
                 // no solution
                 return( [] )
              else
                 s[i,j]:= t
              end_if
             end_for
            end_for;
            return( s )
        end_if
    elif cB > 1 then
        return( [] )
    elif Flag = hold(Unique) then
        return( NIL )
    end_if;

    //====================================================================
    // Here, we know that the rank is not maximal, i.e., there is a kernel.
    // Note that A had been processed by gaussElim and gaussJordan before.
    //=====================================================================

    // =======================================
    userinfo(1,"search for special solution");
    // =======================================

    if RisField then
        // compute special solution over fields
        s:= create(c,1);

        j:= 1;
        for i from 1 to c do
            if not contains( ns,i ) then
                s[i,1]:= A[j,c+1];
                j:= j + 1;
                if j > r then break /* i */ end_if
            end_if
        end_for;

        if Flag = hold(Special) then
            return( s )
        end_if

    elif Flag <> hold(Special) then
        error("sorry, can determine a basis for the null space only over fields")
    else
        // compute special solution over integral domains
        Rdivide:= R::_divide;

        j:= 1;
        s:= create(c,1); // the zero column
        for i from 1 to c do
            if not contains( ns,i ) then
                if (t:= Rdivide( A[j,c+1],A[j,i] )) = FAIL then
                    // no solution:
                    return( [] )
                else
                    s[i,1]:= t
                end_if;

                j:= j + 1;
                if j > r then break /* i */ end_if
            end_if
        end_for;

        return( s )
    end_if;

    // ==================
    // Compute the kernel
    // ==================
    userinfo(1,"solve the homogeneous system");

    Rnegate:= R::_negate;
    Rone:= R::one;
    Rzero:= R::zero;

    rs:= [];
    ns:= sort( [op(ns)] );
    l:= min( r,c );
    if domtype(Flag) <> DOM_LIST then
        for i from 1 to nops(ns) do
            k:= ns[i];
            v:= map( [op(A[1..l,k..k])], Rnegate );
            for j from 1 to i do
                if j = i then
                    v:= [op(v,1..k-1),Rone,op(v,k..l+j-1)];
                else
                    t:= ns[j];
                    v:= [op(v,1..t-1),Rzero,op(v,t..l+j-1)]
                end_if
            end_for;

            rs:= rs . [create( c,1,[op(v,1..min(c,l+i))] )]
        end_for;

        return([s, rs]);
    else
        if nops(Flag) < c-1 then
            Flag:= Flag . [genident() $ i = nops(Flag)+1..c-1];
        end_if;
        Rmult:= R::_mult;
        Rcoerce:= R::coerce;

        for i from 1 to nops(ns) do
            k:= ns[i];
            if (R:= Rcoerce(op(Flag,k))) = FAIL then
                return( FAIL )
            end_if;
            v:= map( [op(A[1..l,k..k])], x-> Rmult( Rnegate(x),R ) );
            for j from 1 to i do
                if j = i then
                    v:= [op(v,1..k-1),R,op(v,k..l+j-1)]
                else
                    t:= ns[j];
                    v:= [op(v,1..t-1),Rzero,op(v,t..l+j-1)]
                end_if
            end_for;

            rs:= rs . [create( c,1,[op(v,1..min(c,l+i))] )]
        end_for;

        return(A::dom::_plus( s,op(rs) ));
    end_if

end_proc:
