// K. F. Gehrs, 2001/08/10
// Reversion: K. F. Gehrs, 2002/09/13
/*
    linalg::minpoly -- returns the minimal polynomial of a square matrix

    linalg::minpoly(A,x)

    A: square matrix over a field
    x: identifier
     
    Computes the minimal polynomial of the matrix A in x, i.e. the polynomial, 
    not zero, of the smallest degree, which annihilates the matrix A.
    The algorithm mentioned below from the book Matrix Analysis is used in 
    a modified form here. In special cases we switch to the tactic of 
    computing the frobenius form of the input matrix first and use this,
    to compute the minimal polynomial of A. 

    The coefficient domain of A has to be a field, i.e. a domain of category 
    'Cat::Field'.

    Reference: Matrix Analysis,
               by Roger A. Horn (The Johns Hopkins University) 
               by Charles R. Johnson (Clemson University)
               algorithm page 148/149 problem 5
               Cambridge University Press 1985 
*/

linalg::minpoly:= proc(A, x)
  local Mat, T, L, B, h, i, j, l, minp;
begin
  if args(0) <> 2 then
    error("expecting two arguments, a matrix of 'Cat::Matrix' and an identifier")
  end_if;
  Mat := A::dom;
  if testargs() then
    if Mat::hasProp(Cat::Matrix) <> TRUE then
      error("expecting a matrix of 'Cat::Matrix'")
    else
      if not Mat::coeffRing::hasProp(Cat::Field) then
        error("expecting matrix over 'Cat::Field'")
      end_if
    end_if;
    if domtype(x) <> DOM_IDENT then
      error("2nd argument must be an identifier")
    end_if
  end_if;
  l := Mat::matdim(A);
  if l[1] <> l[2] then
    error("not a square matrix")
  end_if;
  if l[1] = 1 then 
    return(Dom::DistributedPolynomial([x], Mat::coeffRing)(poly(x - A[1,1], [x], Mat::coeffRing)))
  end_if;
  l := l[1];
  B := Mat::identity(l);
  T := array(1..l^2, 1..l^2 + 1, [[Mat::coeffRing::zero $ l^2 + 1] $ l^2]);
  for h from 0 to l do
    ((T[(j - 1)*l + i, h + 1] := B[i, j]) $ i = 1..l) $ j = 1..l;
    B := B*A
  end_for;
  L := linalg::nullspace(Dom::Matrix(Mat::coeffRing)(T));
  minp:= Dom::DistributedPolynomial([x], Mat::coeffRing)(poly([[L[1][i], i - 1] $ i = 1..l^2], 
                                                         [x], Mat::coeffRing)):
  if degree(minp) > l then 
    warning("unable to compute minimal polynomial");
    return(FAIL);
  end_if:
  return(minp);
end_proc

/*
 VERSION OF linalg::minpoly USING THE FROBENIUS FROM (CODE IS O.K.) AND THE 
 CURRENT TESTFILE minpoly.tst ACCEPTS IT.

linalg::minpoly := proc(A,x)
    local Mat, i, j, l, minp, frob, Rzero, Rone, Rnegate, r, c, deg, L,	T, B, h, ind;
begin
  if args(0) <> 2 then
    error("expecting two arguments, a matrix of 'Cat::Matrix' and an identifier")
  end_if;
  Mat := A::dom;
  Rzero:= Mat::coeffRing::zero;
  Rone:= Mat::coeffRing::one;
  Rnegate:= Mat::coeffRing::_negate;
  if testargs() then
    if Mat::hasProp( Cat::Matrix ) <> TRUE then
      error("expecting a matrix of 'Cat::Matrix'")
    elif not (Mat::coeffRing)::hasProp( Cat::Field ) then
      error("expecting matrix over 'Cat::Field'")
    end_if;
    if domtype(x) <> DOM_IDENT then
      error("2nd argument must be an identifier")
    end_if;
  end_if;
  l:= Mat::matdim(A);
  if l[1] <> l[2] then
    error("not a square matrix")
  end_if;
  if A = Mat(l[1], l[1], [A[ind, ind] $ ind = 1..l[1]], Diagonal) then  // A is a diagonal matrix
    L:= { A[ind, ind] $ ind = 1..l[1] };
    minp:= poly( [[ Rone, 0 ]], [x], Mat::coeffRing );
    for ind in L do 
      minp:= minp * poly([[Rone, 1], [Rnegate(ind), 0]], [x], Mat::coeffRing );
    end_for;
    minp:= Dom::DistributedPolynomial([x],Mat::coeffRing)(minp);
    return(minp);
  else
    frob:= linalg::frobeniusForm(A);
    if Mat::coeffRing = Dom::Float or has({map(op(A), type)}, DOM_FLOAT) then 
      //old routine making us auf gaussian elimination
      l:= l[1];
      B:= Mat::identity(l);
      T:= array( 1..l^2,1..l^2+1, [[ Mat::coeffRing::zero $ l^2+1 ] $ l^2] );
      for h from 0 to l do
        ( ( T[(j-1)*l+i,h+1]:= B[i,j] ) $ i=1..l ) $ j=1..l;
        B:= B * A
      end_for;
      L:= linalg::nullspace( Mat::create(T) );
      minp:= Dom::DistributedPolynomial([x],Mat::coeffRing)(
                poly( [[ L[1][i],i-1 ] $ i=1..l^2],[x],Mat::coeffRing )
              );
      if degree(minp) > l then 
        warning("unable to compute minimal polynomial");
        return(FAIL);
      else 
        return(minp);
      end_if;
      //end of the old routine 
    end_if; 
    i:= l[1];
    j:= l[1] - 1;
    while (iszero(frob[i, j])) do 
      i:= i - 1;
      j:= j - 1;
      if i = 0 or j = 0 then 
        break;
      end_if;
    end_while;
    r:= i;
    c:= j + 1;
    while (frob[i, j] = Rone and frob[i-1, j] = Rzero) do
      i:= i - 1;
      j:= j - 1;
      if i = 0 or j = 0 then 
        j:= j + 1;
        break;
      end_if;
    end_while;
    minp:= poly( [[ Rone, c ]], [x], Mat::coeffRing );
    deg:= c - 1;
    for ind from r downto i do
      minp:= minp - poly([[ frob[ind, c], deg ]], [x], Mat::coeffRing);
      deg:= deg - 1;
    end_for;
    minp:= Dom::DistributedPolynomial([x],Mat::coeffRing)(minp);
    return(minp);
  end_if;    	
end_proc:
*/

