/*
        linalg::nullspace  --  compute the kernel of a matrix

        nullspace(A)

        A: matrix

        linalg::nullspace(A) computes a basis for the kernel of A.
        A must belong to the category Cat::Field.
*/

linalg::nullspace:= proc(A)
    local R, c, r, ns, rs, k, v, i, j, m, Mat, Rzero, Rone, Rnegate;
begin
    if args(0) <> 1 and (args(0) = 2 and args(2) <> `#FloatCheck`) then 
      error("expecting 1 argument") 
    end_if;

    Mat:= A::dom;
    if testargs() then
        if Mat::hasProp( Cat::Matrix ) <> TRUE then
            error("argument is not of 'Cat::Matrix'")
        end_if;
        if not (A::dom::coeffRing)::hasProp( Cat::Field ) then
            error("expecting matrix over 'Cat::Field'")
        end_if
    end_if;

    R:= Mat::coeffRing;

    // kernel vectors are of type Dom::Matrix(R):
    if Mat::constructor = Dom::SquareMatrix then
        Mat:= Dom::Matrix(R)
    end_if;

    k:= Mat::matdim(A);
    r:= k[1]; c:= k[2];

    userinfo(2,"compute Gauss-Jordan form");
    
    if args(0) = 2 and args(2) = `#FloatCheck` then 
      // print("In 'linalg::nullspace'");
      // At this time, the following case is only relevant, when we try to 
      // compute eigenvectors of the Matrix A and 'linalg::nullspace' is 
      // called from 'linalg::eigenvectors' with additional argument `#FloatCheck`. 
      // We proceed as follows: 'linalg::nullspace' calls 'linalg::gaussJordan'
      // with additional argument `#FloatCheck`. Then 'linalg::gaussJordan' 
      // calls 'A::dom::gaussElim' with additional argument `#FloatCheck`. 
      // Then, 'A::dom::gaussElim' uses a special heuristic for zeroTest
      // and we will be better in finding 'eigenvectors' than in the general 
      // case where A perhaps contains symbolic entries. 
      A:= linalg::gaussJordan( A, hold(All), `#FloatCheck`);
    else
      A:= linalg::gaussJordan( A,hold(All) );
    end_if;

    ns:= { $ 1..c } minus A[4];
    if nops( ns ) = 0 then return( [] ) end_if;
    A:= A[1]; // reduced row echolon form of A

    userinfo(2,"solve the homogeneous system");

    Rnegate:= R::_negate;
    Rone:= R::one;
    Rzero:= R::zero;

    rs:= [ NIL $ nops(ns) ];
    ns:= sort( [op(ns)] );
    m:= min( r,c );
    for i from 1 to nops(ns) do
        k:= ns[i];
        v:= map( [op(A[1..m,k..k])], Rnegate );
        for j from 1 to i do
            if j = i then
                v:= [op(v,1..k-1),Rone,op(v,k..m+j-1)];
            else
                R:= ns[j];
                v:= [op(v,1..R-1),Rzero,op(v,R..m+j-1)]
            end_if
        end_for;

        rs[i]:= Mat::create( c,1,[op(v,1..min(c,m+i))] )
    end_for;

    return( rs )
end_proc:

