/*++
  ogsystab.mu

        linalg::ogCoordTab -- a table of orthogonally curvilinear coordinate 
			      system including scale factors

Calls:

 linalg::ogCoordTab[ogName, Transformation](u1,u2,u3,<c>)
 linalg::ogCoordTab[ogName, InverseTransformation](x, y, z, <c>)
 linalg::ogCoordTab[ogName, Dimension]
 linalg::ogCoordTab[ogName, Ranges]
 linalg::ogCoordTab[ogName, UnitVectors](u1, u2, u3, <c>)
 linalg::ogCoordTab[ogName, Scales](u1, u2, u3, <c>)
 linalg::ogCoordTab[ogName](u1, u2, u3, <c>)
                  =linalg::ogCoordTab[ogName, UnitVectors](u1, u2, u3, <c>)

Parameters:
        ogName    : name of an orthogonally curvilinear coordinate system
	Scales	  : ident
	u1,u2,u3  : variables
        x, y, z   : cartesian coordinates
	<c>	  : an (optional) constant (for Torus and EllipticCylindrical).
                    The default value is c = 1.

	The 'UnitVectors' and the 'Scales' are expressed in the
        orthogonal variables u1, u2, u3.

	The following systems are predefined:
	(0 <= thet <= Pi, 0 <= phi <= 2*Pi, u,v real)

	- Cartesian:		x,y,z

	- Spherical:		x = r*sin(thet)*cos(phi),
				y = r*sin(thet)*sin(phi),
				z = r*cos(thet)

	- Cylindrical:		x = r*cos(phi)
				y = r*sin(phi)
				z = z

	- ParabolicCylindrical:	x = 1/2*(u^2-v^2)
				y = u*v
				z = z
                                (u >=0)

	- Torus:		x = (c-r*cos(thet))*cos(phi)
				y = (c-r*cos(thet))*sin(phi)
				z = r*sin(thet), 
                                0 <= r < c 

	- RotationParabolic:	x = u*v*cos(phi)
				y = u*v*sin(phi)
				z = 1/2*(u^2-v^2)

	- EllipticCylindrical:	x = c*cosh(u)*cos(v)
				y = c*sinh(u)*sin(v)
				z = z,
                                c > 0 constant

	The vectors of the orthogonally curvilinear coordinate systems are 
	of type DOM_LIST and not elements of category 'Matrix'. 
	Therefore, the vectors must be converted into elements of 
	category 'Matrix' explicitly, if desired.

	The user is able to add new orthogonally curvilinear coordinate 
	systems (see help pages for 'table').
        // comment by Walter: Wirklich? Und wie? Man finde dazu erst einmal
        // die Hilfeseite zu unprotect, dann finde heraus, dass 'linalg',
        // und nicht 'linalg::ogCoordTab' zu unprotecten ist.

        The scale factors g_i are defined by g_i := |diff( T(u),u_i )|,
        where T describes the orthogonal transformation x=T(u) with x 
        the cartesian coordinates and u the orthogonally cuvilinear 
	coordinates.

	Examples:
	> linalg::ogCoordTab[Spherical](r,thet,Phi);

         [[cos(Phi) sin(thet), sin(Phi) sin(thet), cos(thet)],
          [cos(Phi) cos(thet), sin(Phi) cos(thet), -sin(thet)],
          [-sin(Phi), cos(Phi), 0]]
	
	> linalg::ogCoordTab[Cylindrical,Scales](r,thet,Phi);

          [1, r, 1]
++*/

linalg::ogCoordTab := table(
    //---------------------------------------------------------------
    (hold(Cartesian), hold(Transformation)) =
         proc(u, v, w) begin 
           if args(0) < 3 then error("expecting at least 3 parameters") end_if;
           [u, v, w]
         end_proc ,

    (hold(Cartesian), hold(InverseTransformation)) =
         proc(x, y, z) begin 
           if args(0) < 3 then error("expecting at least 3 parameters") end_if;
           [x, y, z]
         end_proc ,

    (hold(Cartesian), hold(Dimension)) = 3,

    (hold(Cartesian), hold(Ranges)) = (
      () -> [-infinity..infinity, -infinity..infinity, -infinity..infinity]
    ),

    (hold(Cartesian), hold(UnitVectors)) = 
       proc() begin [[1,0,0], [0,1,0], [0,0,1]] end_proc,

    (hold(Cartesian)) = 
       proc() begin [[1,0,0], [0,1,0], [0,0,1]] end_proc,

    (hold(Cartesian), hold(Scales)) = proc() begin [1,1,1] end_proc,

    //---------------------------------------------------------------
    (hold(Spherical), hold(Transformation)) =
      proc(r, phi, thet) begin
        if args(0) < 3 then error("expecting at least 3 parameters") end_if;
        [r*cos(phi)*sin(thet), r*sin(phi)*sin(thet), r*cos(thet)] 
      end_proc,

    (hold(Spherical), hold(InverseTransformation)) =
      proc(x, y, z) 
      local r, a, s;
      begin
        if args(0) < 3 then error("expecting at least 3 parameters") end_if;
        r:= sqrt(x^2 + y^2 + z^2):
        if iszero(r) then
           return([abs(z), 0, 0]);
        end_if:
        if iszero(x^2 + y^2) then 
           return([r, 0, arccos(z/r)]);
        end_if;
        a:= arccos(x/sqrt(x^2 + y^2));
        s:= sign(y):
        [r, a + s*(1-s)*(a - PI), arccos(z/r)]
      end_proc,

    (hold(Spherical), hold(Dimension)) = 3,

    (hold(Spherical), hold(Ranges)) = (
      () -> [0..infinity, 0..2*PI, 0..PI]
    ),

    (hold(Spherical), hold(UnitVectors)) = 
  proc(r, phi, thet) 
  begin 
    if args(0) < 3 then error("expecting at least 3 parameters") end_if;
    [[cos(phi)*sin(thet), sin(phi)*sin(thet), cos(thet)],
     [-sin(phi), cos(phi), 0],
     [cos(phi)*cos(thet), sin(phi)*cos(thet), -sin(thet)]]
  end_proc,

    (hold(Spherical)) = 
      proc() begin 
        if args(0) < 3 then error("expecting at least 3 parameters") end_if;
        [[sin(args(2))*cos(args(3)),sin(args(2))*sin(args(3)),cos(args(2))],
         [cos(args(2))*cos(args(3)),cos(args(2))*sin(args(3)),-sin(args(2))],
         [-sin(args(3)),cos(args(3)),0]
      ] 
      end_proc ,

    (hold(Spherical), hold(Scales))=proc(r, phi, thet) begin 
      if args(0) < 3 then error("expecting at least 3 parameters") end_if;
      [1,r*sin(thet),r]
      end_proc ,
    //---------------------------------------------------------------
    (hold(Cylindrical), hold(Transformation)) =
            proc(r, phi, z) begin
              if args(0) < 3 then error("expecting at least 3 parameters") end_if;
              [r*cos(phi), r*sin(phi), z] 
            end_proc,

    (hold(Cylindrical), hold(InverseTransformation)) =
            proc(x, y, z) 
            local r, a, s;
            begin
              if args(0) < 3 then error("expecting at least 3 parameters") end_if;
              if iszero(x^2 + y^2) then
                 // set polar angle = 0 at the origin
                 return([0, 0, z]) 
              end_if:
              r:= sqrt(x^2 + y^2):
              a:= arccos(x/r);
              s:= sign(y):
              [r, a + s*(1-s)*(a - PI), z]
            end_proc,

    (hold(Cylindrical), hold(Dimension)) = 3,

    (hold(Cylindrical), hold(Ranges)) = (
       () -> [0..infinity, 0..2*PI, -infinity .. infinity]
    ),

    (hold(Cylindrical), hold(UnitVectors)) = 
        proc() begin 
          if args(0) < 2 then error("expecting at least 2 parameters") end_if;
          [[ cos(args(2)),sin(args(2)),0],
           [-sin(args(2)),cos(args(2)),0],
           [   0,             0,       1]
          ] 
        end_proc ,

    (hold(Cylindrical)) = 
        proc() begin 
          if args(0) < 2 then error("expecting at least 2 parameters") end_if;
           [[ cos(args(2)),sin(args(2)),0],
            [-sin(args(2)),cos(args(2)),0],
            [   0,             0,       1]
           ] 
        end_proc ,

    (hold(Cylindrical),hold(Scales)) = 
        proc() begin 
            if args(0) < 1 then error("expecting at least 1 parameter") end_if;
            [1,args(1),1] 
        end_proc,
    //---------------------------------------------------------------
    (hold(ParabolicCylindrical), hold(Transformation)) =
            proc(u, v, w) begin
              if args(0) < 3 then error("expecting at least 3 parameters") end_if;
              [u^2/2 - v^2/2, u*v, w] 
            end_proc,

    (hold(ParabolicCylindrical), hold(InverseTransformation)) =
            proc(x, y, z) 
            local r;
            begin
              if args(0) < 3 then error("expecting at least 3 parameters") end_if;
              r:= sqrt(x^2 + y^2):
              [sqrt(r+x), (1+sign(y)-sign(y)^2)*sqrt(r-x), z]
            end_proc,

    (hold(ParabolicCylindrical), hold(Dimension)) = 3,

    (hold(ParabolicCylindrical), hold(Ranges)) = (
      () -> [0 .. infinity, -infinity .. infinity, -infinity .. infinity]
    ),

    (hold(ParabolicCylindrical), hold(UnitVectors)) =
     proc(u, v, w)
       local r;
       begin 
        if args(0) < 2 then error("expecting at least 2 parameters") end_if;
        r:= sqrt(u^2 + v^2):
        if iszero(r) then
           return([[ 1, 0, 0],
                   [ 0, 1, 0],
                   [ 0, 0, 1]
                  ]);
        end_if;
	[[ u/r, v/r, 0],
	 [-v/r, u/r, 0],
         [  0,   0 , 1]
        ]
     end_proc ,

    (hold(ParabolicCylindrical)) =
     proc(u, v, w)
       local r;
       begin 
        if args(0) < 2 then error("expecting at least 2 parameters") end_if;
        r:= sqrt(u^2 + v^2):
        if iszero(r)
        then return([[ 1, 0, 0],
                     [ 0, 1, 0],
                     [ 0, 0, 1]
                    ]);
        else return([[ u/r, v/r, 0],
                     [-v/r, u/r, 0],
                     [  0,   0 , 1]
                    ]);
        end_if;
     end_proc ,

    (hold(ParabolicCylindrical), hold(Scales)) =
     proc(u, v, w)
       local r;
       begin 
        if args(0) < 2 then error("expecting at least 2 parameters") end_if;
        r:= sqrt(u^2 + v^2):
       	[r,r,1]
     end_proc ,
    //---------------------------------------------------------------
    (hold(RotationParabolic), hold(Transformation)) = 
     proc(u, v, w) begin 
        if args(0) < 3 then error("expecting at least 3 parameters") end_if;
        [u*v*cos(w), u*v*sin(w), u^2/2 - v^2/2]
     end_proc,

    (hold(RotationParabolic), hold(InverseTransformation)) = 
     proc(x, y, z)
     local r1, r2, s, w;
     begin 
        if args(0) < 3 then error("expecting at least 3 parameters") end_if;
        r1:= sqrt(x^2 + y^2);
        r2:= sqrt(x^2 + y^2 + z^2):
        if iszero(r1) 
        then w:= 0;  // arbitrary choice of w
        else w:= arccos(x/r1);
             s:= sign(y);
             w:= w + s*(1-s)*(w - PI);
        end_if:
        [sqrt(r2 + z), sqrt(r2 - z), w];
     end_proc,

    (hold(RotationParabolic), hold(Dimension)) = 3,

    (hold(RotationParabolic), hold(Ranges)) = (
     () -> [0 .. infinity, 0 .. infinity, 0 .. 2*PI]
    ),
                     	
    (hold(RotationParabolic), hold(UnitVectors)) =
     proc() begin 
        if args(0) < 3 then error("expecting at least 3 parameters") end_if;
       	[[args(2)*cos(args(3))/(args(2)^2+args(1)^2)^(1/2),
 	  args(2)*sin(args(3))/(args(2)^2+args(1)^2)^(1/2),
	     args(1)          /(args(2)^2+args(1)^2)^(1/2)
         ],
	 [args(1)*cos(args(3))/(args(1)^2+args(2)^2)^(1/2),
	  args(1)*sin(args(3))/(args(1)^2+args(2)^2)^(1/2),
               -args(2)       /(args(1)^2+args(2)^2)^(1/2)
         ],
	 [-sin(args(3)), cos(args(3)), 0]
        ]
     end_proc ,

    (hold(RotationParabolic)) = 
      proc() begin 
        if args(0) < 3 then error("expecting at least 3 parameters") end_if;
       	[[args(2)*cos(args(3))/(args(2)^2+args(1)^2)^(1/2),
 	  args(2)*sin(args(3))/(args(2)^2+args(1)^2)^(1/2),
	     args(1)          /(args(2)^2+args(1)^2)^(1/2)
         ],
	 [args(1)*cos(args(3))/(args(1)^2+args(2)^2)^(1/2),
	  args(1)*sin(args(3))/(args(1)^2+args(2)^2)^(1/2),
               -args(2)       /(args(1)^2+args(2)^2)^(1/2)
         ],
	 [-sin(args(3)), cos(args(3)), 0]
        ]
     end_proc ,

    (hold(RotationParabolic),hold(Scales)) =
      proc() begin 
        if args(0) < 2 then error("expecting at least 2 parameters") end_if;
	[(args(2)^2+args(1)^2)^(1/2),(args(2)^2+args(1)^2)^(1/2),args(1)*args(2)]
     end_proc ,

    //---------------------------------------------------------------
    (hold(EllipticCylindrical), hold(Transformation)) = 
       proc(u, v, z)
       local c;
       begin
          if args(0) < 3 then error("expecting at least 3 parameters") end_if;
          if args(0) = 3 then c:= 1 else c:= args(4); end_if;
          [c*cosh(u)*cos(v), c*sinh(u)*sin(v), z]:
       end_proc,

    (hold(EllipticCylindrical), hold(InverseTransformation)) = 
       proc(x, y, z)
       local c, coshu, u, v, s;
       begin
          if args(0) < 3 then error("expecting at least 3 parameters") end_if;
          if args(0) = 3 then c:= 1 else c:= args(4); end_if;
          if iszero(c) then
             error("expecting a positive parameter");
          end_if;
          coshu:= sqrt(1/2+x^2/c^2/2+y^2/c^2/2 + 1/2*sqrt( ((x/c+1)^2+y^2/c^2)*((x/c-1)^2+y^2/c^2) )); 
          u:= arccosh(coshu):
          // beware of numerical instability. This may produce
          // non-real numbers, if coshu is a bit smaller than 1:
          if domtype(u) = DOM_COMPLEX then
             u:= numeric::complexRound(u, 10.0^(-DIGITS/2));
          end_if:
          v:= arccos(x/c/coshu);
          // beware of numerical instability. This may produce
          // non-real numbers, if x/c/cosh is a bit larger than 1:
          if domtype(v) = DOM_COMPLEX then
             v:= numeric::complexRound(v, 10.0^(-DIGITS/2));
          end_if:
          s:= sign(y):
          [u, v + s*(1-s)*(v - PI), z]:
       end_proc,

    (hold(EllipticCylindrical), hold(Dimension)) = 3,

    (hold(EllipticCylindrical), hold(Ranges)) = (
       () -> [0 .. infinity, 0 .. 2*PI, -infinity .. infinity]
    ),
                     	
    (hold(EllipticCylindrical), hold(UnitVectors)) =
      proc(u, v, w)
      local r;
      begin 
        if args(0) < 3 then 
           error("expecting at least 3 parameters")
        end_if;
        if iszero(u) then
           r:= sin(v);
           if iszero(r) then
                r:= 1
           else r:= sign(r);
           end_if;
           return([[ 0, r, 0],
                   [-r, 0, 0],
                   [ 0, 0, 1]]);
        end_if;

        r:= sqrt(cosh(u)^2 - cos(v)^2);
	[[ sinh(u)*cos(v)/r, cosh(u)*sin(v)/r, 0],
	 [-cosh(u)*sin(v)/r, sinh(u)*cos(v)/r, 0],
	 [     0           ,        0        , 1]]
      end_proc,

    (hold(EllipticCylindrical)) =
      proc(u, v, w)
      local r;
      begin 
        if args(0) < 3 then error("expecting at least 3 parameters") end_if;
        if iszero(u) then
           r:= sin(v);
           if iszero(r) then
                r:= 1
           else r:= sign(r);
           end_if;
           return([[ 0, r, 0],
                   [-r, 0, 0],
                   [ 0, 0, 1]]);
        end_if;

        r:= sqrt(cosh(u)^2 - cos(v)^2);
	[[ sinh(u)*cos(v)/r, cosh(u)*sin(v)/r, 0],
	 [-cosh(u)*sin(v)/r, sinh(u)*cos(v)/r, 0],
	 [     0           ,        0        , 1]]
      end_proc,

    (hold(EllipticCylindrical),hold(Scales)) =
     proc(u, v, w, c) 
     local r;
     begin 
        if args(0) < 3 then error("expecting at least 3 parameters") end_if;
        if args(0) = 3 then c:= 1 else c:= args(4); end_if;
        r:= c*sqrt(cosh(u)^2 - cos(v)^2);
	[r, r, 1]
     end_proc ,
    //---------------------------------------------------------------
    //---------------------------------------------------------------
    (hold(Torus), hold(Transformation)) = 
       proc(r, thet, phi)
       local c;
       begin
          if args(0) < 3 then error("expecting at least 3 parameters") end_if;
          if args(0) = 3 then c:= 1 else c:= args(4); end_if;
          [(c-r*cos(thet))*cos(phi), (c-r*cos(thet))*sin(phi), r*sin(thet)]
       end_proc,

    (hold(Torus), hold(InverseTransformation)) = 
       proc(x, y, z)
       local c, tmp, r, thet, phi, s;
       begin
          if args(0) < 3 then error("expecting at least 3 parameters") end_if;
          if args(0) = 3 then c:= 1 else c:= args(4); end_if;
          if iszero(c) then
             error("expecting a positive parameter");
          end_if;

          tmp := c - sqrt(x^2 + y^2);
 
          //----------------------------------------
          //  r^2 = x^+y^2+z^2+c^2-2*c*sqrt(x^2+y^2) 
          //      = (c - sqrt(x^2+y^2))^2 + z^2
          //----------------------------------------
          r := sqrt( tmp^2 + z^2 );

          //----------------------------------------
          // cos(thet) = (c - sqrt(x^2 + y^2)) / r
          //----------------------------------------
          if iszero(r) 
            then thet:= 0:
            else thet:= arccos(tmp/r);
          end_if;
          s:= sign(z):
          thet:= thet + s*(1-s)*(thet - PI);

          //----------------------------------------
          // cos(phi) =  x/(c - r*cos(thet)) = x / sqrt(x^2 + y^2)
          //----------------------------------------
          tmp:= sqrt(x^2 + y^2);
          if iszero(tmp)
             then phi:= 0;
             else phi:=  arccos(x/ sqrt(x^2 + y^2));
          end_if;
          s:= sign(y):
          phi:= phi + s*(1-s)*(phi - PI);

          [r, thet, phi];
       end_proc,

    (hold(Torus), hold(Dimension)) = 3,

    (hold(Torus), hold(Ranges)) = 
       proc()
       local c;
       begin
         if args(0) = 0 then c:= 1 else c:= args(1); end_if;
         [0..c, 0..2*PI, 0..2*PI]
       end_proc,
                     	
    (hold(Torus), hold(UnitVectors)) =
      proc(r, thet, phi)
      local c;
      begin 
          if args(0) < 3 then error("expecting at least 3 parameters") end_if;
          if args(0) = 3 then c:= 1 else c:= args(4); end_if;
          [[-cos(thet)*cos(phi), -cos(thet)*sin(phi), sin(thet)],
           [ sin(thet)*cos(phi),  sin(thet)*sin(phi), cos(thet)],
           [         -  sin(phi),             cos(phi), 0]
          ]
     end_proc ,

    (hold(Torus)) = 
      proc(r, thet, phi)
      local c;
      begin 
          if args(0) < 3 then error("expecting at least 3 parameters") end_if;
          if args(0) = 3 then c:= 1 else c:= args(4); end_if;
          [[-cos(thet)*cos(phi), -cos(thet)*sin(phi), sin(thet)],
           [ sin(thet)*cos(phi),  sin(thet)*sin(phi), cos(thet)],
           [         -  sin(phi),             cos(phi), 0]
          ]
     end_proc ,

    (hold(Torus),hold(Scales)) = 
      proc(r, thet, phi)
      local c;
      begin 
        if args(0) < 3 then error("expecting at least 3 parameters") end_if;
        if args(0) = 3 then c:= 1 else c:= args(4); end_if;
        [1, r, c - r*cos(thet)]
     end_proc 
    //---------------------------------------------------------------
):

// end of file 
