// W.Oevel, 18/06/02 

/*++
    pascal - the Pascal matrix

    Call:

    linalg::pascal( n <, CoefficientDomain> );
                                                                   
    Parameter:

    n  --  a positve integer
    CoefficientDomain -- 
             constructor of the coeffient domain for the output matrix.
             Admissible choices are
             Dom::Real, Dom::Float, Dom::Numerical,
             Dom::ExpressionField(), Dom::Integer.
                                                                   
    Synopsis:

    The n x n -Pascal matrix P given by P[i,j]= binomial(i + j - 2, j - 1)
    is returned by pascal(n). It is of domain type Dom::Matrix().

    The Pascal matrices are notoriously ill-conditioned
    positive definite symmetric matrices.
    The determinant is 1 for any dimension n.
    The inverse of pascal(n) has integer entries.
    It is implemented by linalg::invpascal

    The Cholesky factor of the Pascal matrix is a lower 
    Pascal triangle

    The optional second argument is the constructor of the
    output matrix, if a domain type different from 
    Dom::Matrix(Dom::ExpressionField()) is desired.

    Examples:

3.0.0 > linalg::pascal(4)

                      +-              -+
                      |  1, 1,  1,  1  |
                      |                |
                      |  1, 2,  3,  4  |
                      |                |
                      |  1, 3,  6, 10  |
                      |                |
                      |  1, 4, 10, 20  |
                      +-              -+

3.0.0 > linalg::factorCholesky(linalg::pascal(4))

                       +-            -+
                       |  1, 0, 0, 0  |
                       |              |
                       |  1, 1, 0, 0  |
                       |              |
                       |  1, 2, 1, 0  |
                       |              |
                       |  1, 3, 3, 1  |
                       +-            -+

3.0.0 > 1/%

                      +-               -+
                      |   1,  0,  0, 0  |
                      |                 |
                      |  -1,  1,  0, 0  |
                      |                 |
                      |   1, -2,  1, 0  |
                      |                 |
                      |  -1,  3, -3, 1  |
                      +-               -+

    See also:  linalg::invpascal, Dom::Matrix
++*/
linalg::pascal:= proc(n, R=Dom::ExpressionField())
   local a, i,j;
begin
   if testargs() then
       if args(0) = 0 then 
          error("expecting a positive integer (the dimension)") 
       end_if;
       if args(0) > 2 then 
          error("too many arguments") 
       end_if;
       if not testtype(n, Type::PosInt ) then 
          error("expecting a positive integer (the dimension)") 
       end_if;
       if domtype(R) <> DOM_DOMAIN or 
          R::hasProp(Cat::Rng) <> TRUE then 
          error("the component domain must be of category 'Cat::Rng'")
       end_if;
   end_if;

   a:= array(1..n,1..n);
   for j from 1 to n do  // fill in 
      a[1, j]:= 1;       // the first row and
      a[j, 1]:= 1;       // the first column
   end_for:
   for i from 2 to n do
     // fill in the i-th row using Pascal's triangle
     a[i, i]:= a[i-1, i] + a[i, i-1];
     for j from i+1 to n do
       a[i, j]:= a[i-1, j] + a[i, j-1];
       a[j, i]:= a[i, j];
     end_for;
   end_for;
   case R
    of Dom::Complex do
    of Dom::Real do
    of Dom::Numerical do
    of Dom::Rational do
    of Dom::ExpressionField() do
       // conversion of the components is not necessary!
       return((Dom::/*Dense*/Matrix(R))::create( a ))
   otherwise
       return(Dom::/*Dense*/Matrix(R)( a ))
   end_case
end_proc: 
