/*
 linalg::permanent(x)

 x - square matrix over a commutative ring

 computes the permanent of a matrix (Note: the permanent is defined like the 
 determinant, but with the signs of the permutations not entering the definition.)

 An alternative way to define the permament of a matrix A is the following: we use 
 a formula by H. J. Ryser (1963): one computes (for every subset s = {j_1, j_2,..., j_k}
 of {1,2,...,n}) the product 

    a({j_1, j_2,..., j_k}) := (-1)^k prod_{i=1}^n (A[i, j_1] + A[i, j_2] + ... + A[i, j_k]),

 adds the values a(s) for each such s and multiplies the result by (-1)^n. 

 For a proof of this formula see 

   M. Hall, Combinatorial Theory, John Wiley and Sons 1986, 2nd Edition, 
            p. 56

   D. E. Knuth, The Art of Computer Programming II, Addison-Wesley 1998, 3rd Edition, 
                pp. 515 and 699.

 This implementation uses time O(n^2*2^n). 

*/
linalg::permanent := proc(A)
  local n, rowsums, sum, M, m, p, q, j;
begin
   if testargs() then
    if args(0) <> 1 then
      error("expecting 1 argument")
    end_if;
    if A::dom::hasProp(Cat::Matrix) <> TRUE then
      error("argument is not of 'Cat::Matrix'")
    end_if;
    if  not (A::dom::coeffRing)::hasProp(Cat::CommutativeRing) then
      error("expecting matrix over a 'Cat::CommutativeRing'")
    end_if;
  end_if;
  n:=A::dom::matdim(A);
  if n[1] <> n[2] then
     error("not a square matrix")
  end_if;

  n := n[1];
  rowsums := [0 $ n];
  sum := 0;
  M := 2^n - 2;

  for m from 1 to M step 2 do // einfach bei ungeradem m  
    if modp(m - 1,4) = 0 then    
      (rowsums[j] := rowsums[j] + A[j,1]) $ j = 1..n;
    else
      (rowsums[j] := rowsums[j] - A[j,1]) $ j = 1..n;
    end_if;
    sum := sum - _mult(op(rowsums));
    q := m + 1; // ein wenig mehr Aufwand bei geradem m
    p := 1;
    while modp(q,2) = 0 do
      q := q div 2;
      p := p + 1
    end_while;
    if modp(q - 1,4) = 0 then
      (rowsums[j] := rowsums[j] + A[j,p]) $ j = 1..n;
    else
      (rowsums[j] := rowsums[j] - A[j,p]) $ j = 1..n;
    end_if;
    sum := sum + _mult(op(rowsums));
  end_for;
  if modp(n,2) = 0 then
    return(expand(sum - _mult(A[j,n] $ j = 1..n)))
  else
    return(expand(-sum + _mult(A[j,n] $ j = 1..n)))
  end_if
end_proc:

/*
 ==================================================================
   Old version of an implementation for 'linalg::permanent' still 
   here for safety!
 ==================================================================

linalg::permanent:= proc(x)
  local R,n,i;
begin
   if testargs() then
    if args(0) <> 1 then
      error("expecting 1 argument")
    end_if;
    if x::dom::hasProp(Cat::Matrix) <> TRUE then
      error("argument is not of 'Cat::Matrix'")
    end_if;
    if  not (x::dom::coeffRing)::hasProp(Cat::CommutativeRing) then
      error("expecting matrix over a 'Cat::CommutativeRing'")
    end_if;
    n:=x::dom::matdim(x);
    if n[1] <> n[2] then
      error("not a square matrix")
    end_if
  end_if;

  R:=x::dom::coeffRing;
  n:= x::dom::matdim(x)[1];
  case n
    of 1 do
      return( x[1,1] )
    of 2 do
      return(R::_plus( R::_mult(x[1,1], x[2,2]), R::_mult(x[1,2], x[2,1]) ))
    of 3 do
      return(R::_plus(
        R::_mult(x[1, 1], x[2, 2], x[3, 3]), 
        R::_mult(x[1, 1], x[2, 3], x[3, 2]), 
        R::_mult(x[2, 1], x[1, 2], x[3, 3]),
        R::_mult(x[2, 1], x[1, 3], x[3, 2]), 
        R::_mult(x[3, 1], x[1, 2], x[2, 3]),
        R::_mult(x[3, 1], x[1, 3], x[2, 2])
      ))
    otherwise
      return(R::_plus( R::_mult(x[i,1], 
                  linalg::permanent(x::dom::delRow(x::dom::delCol(x,1), i))) 
             $ i = 1..n
      ) )
  end_case
end_proc:

*/

