/*
   linalg::potential -- computes the (scalar) potential of 
          a vector function (a gradient field)

   linalg::potential(f,x <, x0> <,Test>)

   f:    the vector field: a list of expressions 
   x:    the variables: a list of Type::Unknown's
   x0:   the 'base point': a list of Type::Arithmetical's
   Test: (optional) ident

   This function determines whether the scalar potential of the vector 
   function 'f' exists or not. FALSE is returned if this does not exist.

   (The scalar potential of f exists iff the curl of f is zero).

   If the potential of f exists and the option 'Test' is given	
   then TRUE is returned.
   Otherwise the potential p(x) of 'f' is computed.

   If a base point x0 is given the returned potential p, say,
   is chosen to satisfy p(x0) = 0.

*/

linalg::potential:= proc(f, x, x0)
    local n, test, basegiven, i, j, rot, X, pot, d, done, tmp, R, Rnormal;
begin
    if args(0) < 2 then
        error("expecting at least 2 arguments")
    end_if;
    if args(0) > 4 then
        error("expecting no more than 4 arguments")
    end_if;

    if testtype( f,linalg::vectorOf(Type::AnyType) ) then
        R:= f::dom::coeffRing;
        if not (f::dom::coeffRing)::hasProp( Cat::Field ) then
            error("1st argument: expecting 3-dimensional vector over a 'Cat::Field'")
        end_if
    elif not testtype( f, Type::ListOf(Type::Arithmetical) ) then
        error("1st argument: expecting a list of arithmetical expressions")
    else
        R:= Dom::ExpressionField();
    end_if;
    if (Rnormal:= R::normal) = FAIL then
       if R::hasProp(Ax::systemRep) then
          Rnormal:= normal
       else
          Rnormal:= () -> args(1);
       end_if;
    end_if;

    if not testtype( x, Type::ListOf(Type::Unknown) ) then
       error("2nd argument: expecting a list of (indexed) identifiers")
    end_if;

    if domtype(f) <> DOM_LIST then
       f:= [op(f)];
    end_if;
    if domtype(x) <> DOM_LIST then
       x:= [op(x)];
    end_if;
    n:= nops(x);
    if n <> nops(f) then
       error("1st argument: expecting ".expr2text(n). " expressions in ".
             expr2text(n). " unknowns");
    end_if;

    basegiven:= FALSE;
    test:= FALSE;
    if args(0) = 3 then
       if args(3) = hold(Test) then
            // the 3rd argument is not a base point
            test:= TRUE:
            x0:= [0 $ n];
       else // the 3rd argument is a base point
            basegiven:= TRUE;
       end_if;
    else 
       x0:= [0 $ n];
    end_if;
    if args(0) = 4 then
       if args(4) = hold(Test) then
            test:= TRUE;
       else error("4th argument: invalid option ('Test' expected)")
       end_if;
    end_if;

    if basegiven and not testtype(x0, Type::ListOf(Type::Arithmetical, n, n) ) then
       error("3rd argument: expecting a list of ".expr2text(n).
             " arithmetical expressions");
    end_if;
       
    //------------------------------
    // argument testing finished
    // Now, check curl(f) = 0
    //------------------------------

    if has(f, diff) then
       f:= map(f, rewrite, D);
    end_if:
    
    for i from 1 to n do
     for j from 1 to i-1 do
       rot:= diff(f[i], x[j]) - diff(f[j], x[i]):
       if has(rot, diff) then
          rot:= rewrite(rot, D);
       end_if;
       if not iszero(Rnormal(rot)) then
         return( FALSE )
       end_if;
     end_for:
    end_for:
    if test then 
      return( TRUE ) 
    end_if;
     
    //------------------------------
    // The potential exists.
    // Compute it via integration:
    //------------------------------

    // First, we try to integrate the field without
    // using a base point x0. This has the advantage
    // of avoiding (potential) singularities at the
    // base point.
    // Problem: int does not accept indexed identifiers
    //          as the integration variable. Arrggrr!
    // Solution: integrate w.r.t. a dummy variable X.

    X:= genident("X"); // the integration variable

    done:= TRUE;
    pot:= int(subs(f[1], x[1] = X), X):
    if not has(pot, int) then
       pot:= subs(pot, X = x[1]);
    end_if;
    for i from 2 to n do
       d:= Rnormal(f[i] - diff(pot, x[i]));
       if has(d, {x[j] $ j=1..i-1}) then
          done:= FALSE;
          break;
       end_if;
       if not iszero(d) then
          tmp:= int(subs(d, x[i] = X), X);
          if has(tmp, int) then 
             done:= FALSE;
             break;
          else
             pot:= pot + subs(tmp, X = x[i], EvalChanges);
          end_if;
       end_if;
    end_for;
    if done then
       if basegiven then
            return(pot - subs(pot, [x[i]=x0[i] $ i=1..n], EvalChanges));
       else return(pot);
       end_if;
    end_if;

    // Something went wrong. Now, use a general
    // integral representation, that is valid
    // even if symbolic integrals remain:
    // We could choose the contour integral
    //    potential = int(x * f(lambda*x), lambda = 0..1)
    // form the origin to the point x.
    // To make it easier for int, we first integrate along the
    // first coordinate line (putting x[2] = x0[2], x[3] = x0[3], .. ), then
    // we integrate along the second coordinate line
    // (putting x[1] = constant, x[3] = x0[3], x[4] = x0[4], .. ) etc.
    pot:= 0:
    for i from 1 to n do
        // We only compute a 'local' potential, ignoring discontinuities.
        // Hence, we may call int with the option 'Continuous' which
        // makes it **much** easier for int to get a result
        pot:= pot + 
          int(subs(f[i], [x[i]=X, (x[j] = x0[j] $ j= i+1..n)],EvalChanges), 
              X = x0[i]..x[i], Continuous):
    end_for:
    return(pot);
end_proc:

