/*
        linalg::randomMatrix -- generates a random matrix

        randomMatrix( m, n<, R><, size><, modus> )

        m, n : positive integers
        R    : (optional) component ring (of category 'Cat::Rng')
        size : maximal norm of the components
        modus: (optional) Unimodular or Diagonal

        randomMatrix generates a random matrix A of dimension m by n.

        If a coefficient Ring R is not given then the random matrix
        will be of type Matrix(ExpressionField). Otherwise R has
        to provide the method "random".

        Modus: 
        Unimodular: see below.
        Diagonal  : creates a random diagonal matrix
*/

linalg::randomMatrix := proc(m, n, R=Dom::ExpressionField(), size=null(), modus=hold(Standard))
    local i, j, modi, Rrandom;
begin
    modi:= {hold(Standard),hold(Unimodular),hold(Diagonal)}: // Known modi
    if contains({args(i) $ i = 1..args(0)}, hold(Unimodular)) then 
      if m <> n then 
        error("option 'Unimodular' is available only for square matrices");
      elif m = 1 then 
        if R::dom <> DOM_DOMAIN then 
          return(matrix([[1]]));
        else 
          return(Dom::Matrix(R)([[1]]));
        end_if;
      end_if;
    end_if;
    case args(0)
    of 0 do
    of 1 do
        error("expecting at least 2 arguments")
    of 2 do
        break
    of 3 do
       if R::dom <> DOM_DOMAIN then
           if contains( modi,R ) then
               modus:= R;
               R:= Dom::ExpressionField();
           else
               size:= R;
               R:= Dom::ExpressionField();
           end_if
       end_if;
       break
    of 4 do
       if R::dom = DOM_DOMAIN then
          if contains( modi,size ) then
               modus:= size;
               size:= null()
          end_if
       elif contains( modi,R ) then
           modus:= R;
           if size::dom = DOM_DOMAIN then
               R:= size;
               size:= null()
           else
               R:= Dom::ExpressionField()
           end_if
       else
           if contains( modi,size ) then
               ([size,modus]):= [R,size];
               R:= Dom::ExpressionField()
           else
               ([R,size]):= [size,R]
           end_if
       end_if;
       break
    of 5 do
       if R::dom = DOM_DOMAIN then
           if contains( modi,size ) then
               ([size,modus]):= [modus,size]
           end_if
       elif contains( modi,R ) then
           if size::dom = DOM_DOMAIN then
               ([R,size,modus]):= [size,modus,R]
           else
               ([R,modus]):= [modi,R]
           end_if
       else
           if contains( modi,modus ) then
               ([R,size]):= [size,R]
           else
               ([R,size,modus]):= [size,modus,R]
           end_if
       end_if;
       break
    otherwise
       error("too many arguments")
    end_case;
            
    if testargs() then
        if not testtype(m,Type::PosInt) or not testtype(n,Type::PosInt) then
            error("expecting dimension as positive integers")
        end_if;
        if R::hasProp( Cat::Rng ) <> TRUE then
            error("component ring must be of 'Cat::Rng'")
        end_if;
        if not contains(modi,modus) then
            error("unknown modus")
        end_if
    end_if;

    case modus
    of hold(Unimodular) do
        return( linalg::randomUnimodularMatrix(m,n,R,size) )
    of hold(Diagonal) do
        Rrandom := R::random;
        if Rrandom = FAIL then
            error("Domain attribute \"random\" missing")
        end_if;

        return( (Dom::Matrix(R))::create(
            m,n,[Rrandom(size) $ j=1..min(n,m)],hold(Diagonal)
        ) )
    otherwise
        Rrandom := R::random;
        if Rrandom = FAIL then
            error("Domain attribute \"random\" missing")
        end_if;

        return( (Dom::Matrix(R))::create(
            m,n,[[Rrandom(size) $ j=1..n] $ i=1..m]
        ) )
    end_case
end_proc:

/*--
        linalg::randomUnimodularMatrix -- create randomly unimodular matrices over rings

        randomUnimodularMatrix( m,n,R<, size> )

        m, n: positive integers (dimension of the matrix)
        R   : component ring
        size: positive integer
 

        genExample( m,n,b ) creates randomly an unimodular matrix over
        Z, i.e. a matrix A with det(A) = +- 1.

        If a coefficient ring R is given, then the matrix A will
        be defined over R and det(A) = R* (i.e. an unit in R).

        Reference: Juergen Hansen: "Generating Problems in Linear Algebra"
                   MapleTech, Volume 1, No.2, 1994.
--*/
linalg::randomUnimodularMatrix:= proc( m,n,R,size=10 )
    local p, r, c, A, i, j, k, v, l, s, tooBig;
begin
    if not testtype( size,Type::PosInt ) then
        error("invalid value for the size of the components")
    elif R::norm = FAIL then
        error("component ring does not define the entry \"norm\"")
    end_if;

    p:= random(1..3):
    v:= [R::_negate(R::one),R::zero,R::one];

    r:= random(1..m):
    c:= random(1..n):

    userinfo(2,"(1) Build up an upper triangular matrix");
    A:= Dom::Matrix(R)( m,n,[R::one $ min(m,n)],hold(Diagonal) );
    for i from 1 to m do
        for j from i+1 to n do A[i,j]:= v[p()] end_for
    end_for;

    userinfo(2,"(2) Interchange rows");
    for i from 1 to m do
        p:= random(i..m)();
        if p <> i then A:= A::dom::swapRow( A,i,p ) end_if
    end_for;

    userinfo(2,"(3) perform row / column addition");
    l:= n*(trunc(ln(size)/ln(1.2))+2);

    for p from 1 to l do
        i:= r();
        j:= r();
        while i = j do j:= r() end_while;
        s:= (-1)^random(0..1)();
        tooBig:= FALSE;
        v:= [ NIL $ n ];
        for k from 1 to n do
            v[k]:= A[j,k] + s*A[i,k];
            if traperror( (tooBig:= bool(R::norm(v[k]) > size)) ) <> 0 then
                error("can't determine size of the components")
            elif tooBig then
                break
            end_if
        end_for;
        if not tooBig then A:= linalg::setRow( A,j,v ) end_if;

        i:= c();
        j:= c();
        while i = j do j:= c() end_while;
        s:= (-1)^random(0..1)();
        tooBig:= FALSE;
        v:= [ NIL $ m ];
        for k from 1 to m do
            v[k]:= A[k,j] + s*A[k,i];
            if traperror( (tooBig:= bool(R::norm(v[k]) > size)) ) <> 0 then
                error("can't determine size of the components")
            elif tooBig then
                break
            end_if
        end_for;
        if not tooBig then A:= linalg::setCol( A,j,v ) end_if
    end_for;

    return( A )
end_proc:

