/*
   linalg::scalarProduct(u,v) -- returns the Euclidian scalar product
                                 of u and v where u and v are
                                 vectors over the same coeffRing. 
   linalg::scalarProduct(u,v,Real) -- assume all components of u, v 
                                      to be real (i.e., do not use
                                      conjugate)

   The Euclidian scalar product of two vectors with respect to
   the standard basis defined by

    u[1]*conjugate(v[1]) + u[2]*conjugate(v[2]) + ...

   If the component ring of the vectors do not define
   an entry "conjugate" the scalar product

           u[1]*v[1] + u[2]*v[2] + ...

   is used.
*/

linalg::scalarProduct := proc(u,v, real)
    local i, R, Rmult, Rnormal;
begin
 if testargs() then
   if args(0) < 2 then error("expecting at least 2 arguments") end_if;
   if args(0) > 3 then error("expecting no more than 3 arguments") end_if;
   if not testtype( u,linalg::vectorOf(Type::AnyType) ) then
     error("expecting two vectors")
   end_if;
   if not testtype(v,linalg::vectorOf(u::dom::coeffRing,u::dom::nops(u))) then
     error("vectors are not compatible")
   end_if;
   if args(0) = 3 and real <> Real then
     error("unexpected 3rd argument: expecting 'Real'"):
   end_if:
 end_if;
 if args(0) = 2 then
    real:= NIL;
 end_if:

 R:= u::dom::coeffRing;
 Rmult:= R::_mult;
 if (Rnormal:= R::normal) = FAIL then
   if R::hasProp(Ax::systemRep) then 
      Rnormal:= normal 
   else 
      Rnormal:= () -> args(1);
   end_if
 end_if;

 if R::conjugate = FAIL or real = Real then
     return( Rnormal(R::_plus(
         Rmult(u[i],v[i]) $ i=1..nops(u)
     )) )
 else
     return( Rnormal(R::_plus( 
         Rmult(u[i],R::conjugate(v[i])) $ i=1..nops(u)
     )) )
 end_if
end_proc:

linalg::scalarProduct := funcenv(linalg::scalarProduct):
linalg::scalarProduct::Content := stdlib::genOutFunc("Cscalarproduct", 2):
