//      
// K. F. Gehrs, 2000/03/12

/*
    linalg::smithForm -- returns the the Smith Normal Form (SNF) 
                         of a matrix 

    linalg::smithForm(A);
    
    A: a non-singular square matrix over Cat::EuclideanDomain
 
    Computes the Smith Normal Form (SNF) of the matrix A
    i.e. the matrix B, which is a diagonal matrix with coefficients 
    such that B[i,i] mod B[i+1,i+1] = 0  (i < ncols(A))
    The coefficient domain of A has to be Cat::EuclideanDomain.
    
    The algorithm requires the comuptation of linalg::det(A) als well 
    the method gcdex(a,b) to compute d:=ggT(a,b) and u,v such that 
    d = u*a + v*b. 

    Reference: A Course in Computational Algebraic Number Theory,
               by Henri Cohen, GTM 138, Springer Verlag, 
               algorithm 2.4.14 page 77
               (library signature: TFF 2242)

    Note that the algorithm in the reference above requires the
    matrix to be invertible. In principle, invertibility is not
    required. There are some relatively straightforward algorithms
    to compute the Smith normal form that do not require invertibility.
    E.g., Proposition 3.0.5 in  
    http://modular.fas.harvard.edu/papers/ant/html/node9.html
*/

linalg::smithForm := proc(A) 
    local a,b,R,i,j,n,k,l,Mat,changed1,helpCol,helpRow,c,List,helpMat,di,colI,colJ,
          rowJ,rowI,Result,Rmult,Rdivide,Rplus,Rnegate,Rzero,Rone,Riszero,Rmod,Det,
          Rcoerce,Matcoerce,smithZ,L;
begin
    if args(0) = 0 then error("expecting a matrix of category 'Cat::Matrix'") end;
    Mat:= A::dom;

    if testargs() then
       if  Mat::hasProp( Cat::Matrix ) <> TRUE then
           error("argument is not of 'Cat::Matrix'")
       elif not (Mat::coeffRing)::hasProp( Cat::EuclideanDomain ) then
           error("expecting matrix over 'Cat::EuclideanDomain'")
       end_if;      
    end_if;

    n:= Mat::matdim(A);
    if n[1] <> n[2] then
        error("not a square matrix")
    else 
        n:= n[1]
    end_if;
    
    R:= A::dom::coeffRing;
    Det:=linalg::det(A);

    Riszero:= R::iszero;
    if Riszero(Det) then error("matrix is singular") end;

    smithZ:= FALSE; 
    Matcoerce:= id;
    Rcoerce:= id;

    if R = Dom::Integer then
        smithZ:= TRUE;
    elif not R::hasProp(Cat::UnivariatePolynomial) then
        // is A a matrix with integer coefficients?
        helpMat:= Dom::Matrix(Dom::Integer);
        a:= helpMat::coerce(A);
        if a <> FAIL then
            // keep methods for converting the result back to a matrix of type Mat:
            Rcoerce:= R::coerce;
            Matcoerce:= Mat::coerce;
    
            A:= a;
            R:= Dom::Integer;
            Mat:= helpMat;
            Det:= R::coerce(Det);
            Riszero:= R::iszero;
            smithZ:= TRUE;
        end_if;
        if not smithZ then
            if R::constructor = Dom::ExpressionField then
                b:= indets(A);
                if nops(b) <> 1 then
                    error("can't convert matrix to a matrix over univariate polynomials")
                else
                    helpMat:= Dom::Matrix(Dom::DistributedPolynomial([op(b,1)],R));
                    a:= helpMat::coerce(A);
                    if a = FAIL then
                        error("can't convert matrix to a matrix over univariate polynomials")
                    else
                        // keep methods for converting the result back to a matrix of type Mat:
                        Rcoerce:= R::coerce;
                        Matcoerce:= Mat::coerce;
    
                        A:= a;
                        Mat:= helpMat;
                        R:= Mat::coeffRing;
                        Det:= R::coerce(Det);
                        Riszero:= R::iszero;
                    end_if
                end_if
            else
                error("expecting an integer matrix, or a matrix defined over a 'Cat::UnivariatePolynomial'")
            end_if
        end_if
    end_if;

    Result:=Mat::identity(n);

    colI:=Mat(n,1);
    colJ:=colI;
    rowI:=Mat(1,n);
    rowJ:=rowI;

    if smithZ then
        //****************************************************//
        //* smithForm for matrices with integer coefficients *//
        //****************************************************//
        if n=1 then 
            return(Rcoerce(Det))
        else 
            i:=n; 
            changed1:=TRUE;
            while changed1=TRUE do 
                //step 2
                j:=i; 
                c:=0;
                //step 3
                while j<>1 do 
                    j:=j-1; 
                    if A[i,j]<>0 then 
                        //step 4
                        a:=A[i,i]; 
                        b:=A[i,j];
                        List:=igcdex(a,b);
                        if a<>0 then 
                            if b mod a = 0 then
                                List[2]:=1;
                                List[3]:=0;
                            end_if;
                        end_if;
                        helpMat:=[Mat::col(A,i),Mat::col(A,j)];
                        helpCol:=_plus(_mult(List[2],helpMat[1]),_mult(List[3],helpMat[2]));
                        di:=List[1];
                        a:=A[i,i]/di;
                        b:=A[i,j]/di;
                        colJ:=_plus(_mult(a,helpMat[2]),_negate(_mult(b,helpMat[1])));
                        for k from 1 to n do
                            colJ[k,1]:=colJ[k,1] mod Det;
                            colI[k,1]:=helpCol[k,1] mod Det;
                        end_for;
                        A:=Mat::setCol(A,j,colJ);
                        A:=Mat::setCol(A,i,colI);
                    end_if;
                end_while;
                
                //step 5
                j:=i; 
                while j<>1 do 
                    //step 6
                    j:=j-1; 
                    if A[j,i]<>0 then 
                        //step 7
                        a:=A[i,i]; 
                        b:=A[j,i];
                        List:=igcdex(a,b);
                        if a<>0 then 
                            if b mod a = 0 then
                                List[2]:=1;
                                List[3]:=0;
                            end_if;
                        end_if;
                        helpMat:=[Mat::row(A,i),Mat::row(A,j)];
                        helpRow:=_plus(_mult(List[2],helpMat[1]),_mult(List[3],helpMat[2]));
                        di:=List[1];
                        a:=A[i,i]/di;
                        b:=A[j,i]/di;
                        rowJ:=_plus(_mult(a,helpMat[2]),_negate(_mult(b,helpMat[1])));
                        for l from 1 to n do
                            rowJ[1,l]:=rowJ[1,l] mod Det;
                            rowI[1,l]:=helpRow[1,l] mod Det;
                        end_for;
                        A:=Mat::setRow(A,j,rowJ); 
                        A:=Mat::setRow(A,i,rowI); 
                        c:=c+1;
                    end_if;
                end_while;
    
                //step 8
                if c>0 then  
                    changed1:=TRUE;
                else
                    //step 9
                    changed1:=FALSE;
                    b:=A[i,i]; 
                    for j from i-1 downto 1 do
                        for l from i-1 downto 1 do
                            if b<>0 then 
                                if A[j,l] mod b <>0 then
                                    helpMat:=[Mat::row(A,i),Mat::row(A,j)];
                                    A:=Mat::setRow(A,i,_plus(helpMat[1],helpMat[2]));
                                    changed1:=TRUE;
                                    break;
                                end_if;
                                if A[l,j] mod b <>0 then
                                    helpMat:=[Mat::row(A,i),Mat::row(A,l)];
                                    A:=Mat::setRow(A,i,_plus(helpMat[1],helpMat[2]));
                                    changed1:=TRUE;
                                    break;
                                end_if;
                                if A[l,l] mod b <>0 then
                                    // L2:=[Mat::row(A,i),Mat::row(A,l)]; $$$ changed on 23.07.2003 by Kai $$$
                                    helpMat:= [Mat::row(A,i),Mat::row(A,l)];
                                    A:=Mat::setRow(A,i,_plus(helpMat[1],helpMat[2]));
                                    changed1:=TRUE;
                                    break;
                                end_if;
                            end_if;
                        end_for;
                        if l > 0 then  
                            break; 
                        end_if;
                    end_for;
                    //step 10
                    if changed1=FALSE then 
                        Result[n-i+1,n-i+1]:=igcd(A[i,i],Det); 
                        Det:=Det/Result[n-i+1,n-i+1];
                        if i=2 then 
                            changed1:=FALSE;
                            Result[n,n]:=igcd(A[1,1],Det);
                            //===========================================
                            for i from 1 to n-1 do 
                              if Result[i+1,i+1] div Result[i,i] = 0 then 
                                L:= [Result[j,j] $ j = 1..n];
                                (Result[n-j+1,n-j+1]:= L[j]) $ j = 1..n;
                                break;
                              end_if;
                            end_for;
                            //===========================================
                            return( Matcoerce(Result) );
                        else
                            i:=i-1;
                            changed1:=TRUE;
                        end_if
                    end_if
                end_if
            end_while
            // should never be reached ...
        end_if
    else
        //***************************************************//
        //* smithForm for matrices over an Euclidean domain *//
        //***************************************************//
        Rplus:= R::_plus;
        Rnegate:= R::_negate;
        Rmult:= R::_mult;
        Rzero:= R::zero;
        Rdivide:= R::_divide;
        Rone:= R::one;
        Rmod:=R::rem;
        if n=1 then 
            return(Rcoerce(Det))
        else 
            i:=n; 
            changed1:=TRUE;
            while changed1=TRUE do 
            //step 2
                j:=i; 
                c:=0;
                //step 3
                while j<>1 do 
                    j:=j-1; 
                    if not Riszero(A[i,j]) then 
                        //step 4
                        a:=A[i,i]; 
                        b:=A[i,j];
                        List:=gcdex(a,b);
                        if not Riszero(a) then 
                            if Riszero(Rmod(b,a)) then
                                List[2]:=Rone;
                                List[3]:=Rzero;
                            end_if;
                        end_if;
                        helpMat:=[Mat::col(A,i),Mat::col(A,j)];
                        helpCol:=map(helpMat[1],Rmult,List[2]) + map(helpMat[2],Rmult,List[3]);
                        di:=List[1];
                        a:=Rdivide(A[i,i],di);
                        b:=Rdivide(A[i,j],di);
                        colJ:= map(helpMat[2],Rmult,a) - map(helpMat[1],Rmult,b);
                        for k from 1 to n do
                            colJ[k,1]:=Rmod(colJ[k,1],Det);
                            colI[k,1]:=Rmod(helpCol[k,1],Det);
                        end_for;
                        A:=Mat::setCol(A,j,colJ);
                        A:=Mat::setCol(A,i,colI);
                    end_if;
                end_while;
                
                //step 5
                j:=i; 
                while j<>1 do 
                    //step 6
                    j:=j-1; 
                    if not Riszero(A[j,i]) then 
                        //step 7
                        a:=A[i,i]; 
                        b:=A[j,i];
                        List:=gcdex(a,b);
                        if not Riszero(a) then 
                            if Riszero(Rmod(b,a)) then
                                List[2]:=Rone;
                                List[3]:=Rzero;
                            end_if;
                        end_if;
                        helpMat:=[Mat::row(A,i),Mat::row(A,j)];
                        helpRow:= map(helpMat[1],Rmult,List[2]) + map(helpMat[2],Rmult,List[3]);
                        di:=List[1];
                        a:=Rdivide(A[i,i],di);
                        b:=Rdivide(A[j,i],di);
                        rowJ:= map(helpMat[2],Rmult,a) - map(helpMat[1],Rmult,b);
                        for l from 1 to n do
                            rowJ[1,l]:=Rmod(rowJ[1,l],Det);
                            rowI[1,l]:=Rmod(helpRow[1,l],Det);
                        end_for;
                        A:=Mat::setRow(A,j,rowJ);
                        A:=Mat::setRow(A,i,rowI); 
                        c:=c+1;
                    end_if;
                end_while;
    
                //step 8
                if c>0 then  
                    changed1:=TRUE;
                else
                //step 9
                changed1:=FALSE;
                    b:=A[i,i]; 
                    for j from i-1 downto 1 do
                        for l from i-1 downto 1 do
                            if not Riszero(b) then 
                                if not Riszero(Rmod(A[j,l],b)) then
                                    helpMat:=[Mat::row(A,i),Mat::row(A,j)];
                                    A:=Mat::setRow(A,i,_plus(helpMat[1],helpMat[2]));
                                    changed1:=TRUE;
                                    break;
                                end_if;
                                if not Riszero(Rmod(A[l,j],b)) then
                                helpMat:=[Mat::row(A,i),Mat::row(A,l)];
                                A:=Mat::setRow(A,i,_plus(helpMat[1],helpMat[2]));
                                changed1:=TRUE;
                                break;
                                end_if;
                                if not Riszero(Rmod(A[l,l],b)) then
                                    helpMat:=[Mat::row(A,i),Mat::row(A,l)];
                                    A:=Mat::setRow(A,i,_plus(helpMat[1],helpMat[2]));
                                    changed1:=TRUE;
                                break;
                                end_if;
                            end_if;
                        end_for;
                        if l > 0 then  
                            break; 
                        end_if;
                    end_for;
    
                    //step 10
                    if changed1=FALSE then 
                        Result[n-i+1,n-i+1]:=gcd(A[i,i],Det); 
                        Det:=Rdivide(Det,Result[n-i+1,n-i+1]);
                        if i=2 then 
                            changed1:=FALSE;
                            Result[n,n]:=gcd(A[1,1],Det);
                            //===========================================
                            for i from 1 to n-1 do 
                              if iszero(Rdivide(Result[i+1,i+1], Result[i,i])) then 
                                L:= [Result[j,j] $ j = 1..n];
                                (Result[n-j+1,n-j+1]:= L[j]) $ j = 1..n;
                                break;
                              end_if;
                            end_for;
                            //===========================================
                            return(Matcoerce(Result))
                        else
                            i:=i-1;
                            changed1:=TRUE;
                        end_if
                    end_if
                end_if
            end_while
        end_if
    end_if
end_proc:

