
/*
   sqrtMatrix(A<, sqrtfunc>)

   A - square matrix
   sqrtfunc - a function satisfying sqrtfunc(a)^2 = a for every element
              a of the coefficient ring of A

   returns a matrix B with B^2 = A, such that the eigenvalues of B are the square roots of the eigenvalues of A

   the notion of square root must exist in the coefficient ring of A,
   and A must possess a Jordan normal form

*/


linalg::sqrtMatrix:=
proc(A, sqrtfunc = sqrt)
  local n: DOM_INT, a, k, j, i: DOM_INT, l, D, T, B, coefflist, Mat, F, twoa, u,
           Cols, b;
begin
  if args(0) > 2 then
    error("Wrong number of arguments")
  end_if;

  Mat:= A::dom;

  if testargs() then
    if Mat::hasProp( Cat::Matrix ) <> TRUE then
      error("first argument is not of 'Cat::Matrix'")
    end_if;


    F:= Mat::coeffRing;
    if not F::hasProp( Cat::Field ) then
      error("expecting matrix over 'Cat::Field'");
    end_if;

    if traperror((u:= sqrtfunc(F::zero))) <> 0 then
      error("square root function cannot handle elements of the coefficient ring")
    end_if;
    if u <> F::zero then
      error("Square root function does not work correctly")
    end_if;
    
    n:= Mat::matdim(A);

    if n[1] <> n[2] then
      error("not a square matrix")
    else
      n:= n[1]
    end_if
  else
    F:= Mat::coeffRing;
    n:= Mat::matdim(A)[1]
  end_if;

 
  
  l:= linalg::jordanForm(A, All);

  if l = FAIL then
    return(FAIL)
  end_if;

  [D, T]:= l;
  B:= Mat(n, n, 0); // will be the squre root of D
  
  // initially, we are in the first row of a Jordan box
  coefflist:=[];
  
  for i from 1 to n do
    if (k:= nops(coefflist)) = 0 then
      // a new box starts, with another eigenvalue
      a:= sqrtfunc(D[i, i]);
      if a = FAIL then
        warning("Eigenvalue has no square root");
        return(FAIL)
      end_if;
      u:= F::one
    else
      u:= F::zero
    end_if;
    B[i, i]:= a;
    if i = n or iszero(D[i, i+1]) then
      // last row of a Jordan box
      coefflist:=[]
    else
      if iszero((twoa:= a+a)) then
        warning("square root of matrix does not exist");
        return(FAIL)
      end_if;
    // D[i, i+1] = 1
    // the Jordan box continues

    // add new coefficient
      coefflist:= coefflist.
                  [twoa^(-1)*
                     (u - _plus(coefflist[j]*
                                     coefflist[k+1-j] $j=1..k) )];
      for j from 1 to nops(coefflist) do
        B[i-j+1, i+1]:= coefflist[j]
      end_for;
    end_if                    
  end_for;
  
  if n < 7 and max(map({op(T)},length)) > 100 then 
    Cols:= [0 $ n];
    for i from 1 to n do 
      b:= matrix(n,1);
      b[i]:= 1;
      Cols[i]:= linalg::SSS(T,b)[1];
    end_for;  
    return(T*B*(T::dom::concatMatrix(op(Cols))));
  else   
    return(T*B*T^(-1))
  end_if;

end_proc:

