/*++
 linalg::symbolicAdjoint_Fu(M)
   -- return the adjoint of the square matrix M
      that involves symbolic entries


 Computes the adjoint of a matrix over a commutative ring.
 This is the matrix whose (i,j)-th entry is the (j,i)-th
 cofactor of M.
 The (j,i)-th cofactor of M is defined by 

        (-1)^(i+j)*det(M(i,j))

 where M(i,j) is the submatrix of M obtained from M by
 deleting the i-th row and j-th column.

 Calls:  
     linalg::symbolicAdjoint_Fu(M)
     linalg::symbolicAdjoint_Fu(M, All)


 Parameters: M -- a square matrix

 Options: All - return the list [adjoint(M), det(M)]

 Reference:

     Modified Gauss Algorithm for Matrices with Symbolic Entries,
     B. Fuchssteiner, ACM Communications in Computer Algebra, 42(3),
     3. September 2008

   The algorithm makes use of the subroutine 'linalg::otimes_Fu'

++*/

linalg::symbolicAdjoint_Fu := proc(M)
local m, n, d, Mat, MList, i, j, k, jj, P, PP, eval_Fu;
begin
  [m, n]:= M::dom::matdim(M);
  if m <> n then
     error("expecting a square matrix");
  end_if;
  // Special Cases for 1x1 and 2x2 matrices
  if n = 1 then
     d:= M[1,1]:
     M[1,1]:= 1: // convert 1 into the matrix ring
     if has([args()], All) then
       return([M, d]);
     else
       return(M);
     end_if;
  elif n = 2 then
     d:= M[1,1]*M[2,2] - M[1,2]*M[2,1];
     [M[1,1], M[1,2],M[2,1],M[2,2]]:= [M[2,2],-M[1,2],-M[2,1],M[1,1]]:
     if has([args()], All) then
       return([M, d]);
     else
       return(M);
     end_if:
  end_if;
  Mat:= M::dom;
  m:= 2*n:
  M:= M . Mat::identity(n);
  M:= array(1..n, 1..m, [op(M)]);
  MList:= [0 $ n]:
  for k from 1 to n do
    MList[k]:= M[k, k];
    for i from 1 to n do
      if i = k then 
         next;
      end_if:
      P:= poly(M[i,k], [#Z[jj] $ jj = 0..k-1]);
      M[i, k]:= 0:
      for j from k+1 to m do
        PP:= poly(M[k,j], [#Z[jj] $ jj = 0..k-1]);
        M[i,j]:= M[i,j] - #Z[k]*linalg::otimes_Fu(P,PP, 1, k-1);
      end_for:
    end_for;
    M[k, k]:= 1:
    for j from k+1 to m do
      M[k, j]:= #Z[k]*M[k, j];
    end_for:
  end_for;
  M:= array(1..n, 1..n, [ [M[i, j] $ j =  n+1..m] $ i = 1..n]):

  eval_Fu:= proc(d)
  local k;
  begin
    if iszero(d) then 
       return(0);
    end_if:
    for k from n downto 1 do
      d:= poly(d, [#Z[k]]); // this is important for efficiency!!!
      d:= MList[k]*coeff(d, #Z[k], 0) + coeff(d, #Z[k], 1);
    end_for:
    return(expand(d, ArithmeticOnly));
  end_proc:

  M:= map(M, eval_Fu);

  if has([args()], All) then 
    // return [adjoint(M), det(M)]. Note that
    // the determinant is eval_Fu(1).
    return([Mat(M), eval_Fu(1)]):
  else
    return(Mat(M));
  end_if:
end_proc:

// end of file 
