// linalg::symbolicity(A) checks the number of symbols and
// their location in the matrix A.
// If Gauss elimination would involve many symbols, then the
// number 2 is returned to indicate that the matrix is 'heavily symbolic'.
// The number 1 is returned when the matrix is 'slightly symbolic'.
// The number 1/2 is returned when the matrix is 'very light' in symbols.
// The number 0 is returned when the matrix contains only numerical
// entries (DOM_INT, DOM_RAT, DOM_COMPLEX, DOM_FLOAT, DOM_INTERVAL).
//
// The call linalg::symbolicity(A, "MatrixIsRationalized") assumes
// that A had been rationalized by linalg::rationalizeMatrix and
// does no further rationalization.

linalg::symbolicity:= proc(A)
 local nvars, m, n, AA, pvars, nZeroes, i, j;
begin
  nvars:= nops(indets(A)); // number of variables in A

  if nvars = 0 and
     map({A::dom::nonZeroOperands(A)}, domtype) minus 
     {DOM_INT, DOM_RAT, DOM_COMPLEX, DOM_FLOAT, DOM_INTERVAL}
     = {} then
     // the matrix contains only numerical data
     return(0);
  end_if;
  [m, n]:= A::dom::matdim(A);
  if m*n <= 9 then
     return(2);
  end_if:

  // Beware: exact numbers such as sin(1) or sqrt(2) are just as bad
  // as identifiers. Take them into account by rationalizing the matrix:
  if has([args()], "MatrixIsRationalized") then
     AA:= A:
  else
     AA:= [linalg::rationalizeMatrix(A, ReturnType = matrix)][1]:
     nvars:= nops(indets(AA)); // number of variables in A
  end_if:
  AA:= map(AA, nops@indets); // AA contains the number of
                             // symbols in each entry of A
  pvars:= AA::dom::nonZeroes(AA); // number of entries in A that contain
                                  // symbols

  if nvars <= 2 and m*n >= 100 then
     return(1/2);  // this is a contradiction like 'Bratwurst light': 
                   // the matrix is symbolic, but not too bad. Indicate
                   // that we can use an algorithm using 'normal'
  end_if:

  if pvars <= 10 and nvars <=10 then 
    // Only few components of A contain variables. Since linalg::SSS uses
    // a pivot strategy based on the pivot size of elements, we can be
    // sure that the algorithm will choose 'small' pivot elements within
    // the elimination. Hence, it should be faster than 'linalg::SSS2'.  
    // Note: If R = Dom::ExpressionField() then R::pivotSive = length.
    return(1); // 1 indicates that linalg::SSS should be used
  end_if;

  nZeroes:= 0; // number of non-zero components below the main diagonal 
  for j from 1 to n - 1 do
    for i from j+1 to m do
      if not iszero(A[i,j]) then 
        nZeroes:= nZeroes + 1;
      end_if;
    end_for;
  end_for;
  if nZeroes in {0,1,2} or 
     nZeroes < floor(2/3*(m+n)) then 
    // A is nearly upper triangular. Hence, use linalg::SSS since there
    // are only few elimination steps to be done. The backsubstitution 
    // is done by linalg::SSS following Cramer's rule avoiding expression 
    // swell by using only n elimination for the computation of each 
    // determinant.
    return(1); // 1 indicates that linalg::SSS should be used
  end_if;

  if nvars >= 2 and m*n < 100 then
     return(2); // 2 indicates that linalg::SSS2 should be used
  end_if;

  return(1); // 1 indicates that linalg::SSS should be used
end_proc:
