//   
// walter, 25/09/98

/*
  linalg::vandermondeSolve( c, y <, Transposed> )

  Parameters:

    c          -- a vector or list [c[1],..,c[n]] with distinct elements c[i]
    y          -- a vector or list [y[1],..,y[n]]
    Transposed -- option

  Synopsis:

  Let V be the Vandermonde matrix generated by [c[1],..,c[n]]:
            ( 1  c[1] c[1]^2 .. c[1]^(n-1) ) 
        V = ( ..  ..   ..    ..  ..        )
            ( ..  ..   ..    ..  ..        )
            ( 1  c[n] c[n]^2 .. c[n]^(n-1) )

  Return Value: a vector or list [x[1],..,x[n]]. The return type
                coincided with the type of the input vector/list y

  The call 

     linalg::vandermondeSolve([c[1],..,c[n]], [y[1],..,y[n]])

  returns the vector or list [x[1],..,x[n]] representing the solution of
  the linear system Vx=y, i.e,.

            sum(c[i]^(j-1)*x[j], j=1..n) = y[i], i=1..n.

  The call 

    linalg::vandermondeSolve([c[1],..,c[n]], [y[1],..,y[n]], Transposed)

  returns the vector or list [x[1],..,x[n]] representing the solution of
  the linear system V^T x=y, i.e,.

            sum(c[j]^(i-1)*x[j], j=1..n) = y[i], i=1..n.




  The implemented algorithm needs O(n^2) operations.

  Remark: the list [x[0],..,x[m]] returned by

      linalg::vandermondeSolve([c[i] $ i=0..m],[y[i] $ i=0..m])

  contains the coefficients of the interpolating polynomial

        P(c) = x[0] + x[1]*c + .. + x[m]*c^m

  with respect to the data table (c[0],y[0]),..,(c[m],y[m]), i.e.

        P(c[0])=y[0], .. , P(c[m])=y[m].

  Example:

  >>  m:= 2: x:= linalg::vandermondeSolve(
         [c.i $ i=0..m], [y.i $ i=0..m]):
  >>  map(%, factor);

--
|          2           2           2        2
|  (- c0 c1  y2 + c0 c2  y1 - c1 c2  y0 + c0  c1 y2 -
--

     2           2
   c0  c2 y1 + c1  c2 y0) / ((c1 - c2) (c0 - c2) (c0 - c1)),

     2        2        2        2        2        2
   c0  y1 - c1  y0 - c0  y2 + c2  y0 + c1  y2 - c2  y1
   ---------------------------------------------------,
              (c1 - c2) (c0 - c2) (c0 - c1)

                                                   --
     c0 y1 - c1 y0 - c0 y2 + c2 y0 + c1 y2 - c2 y1  |
   - ---------------------------------------------  |
             (c0 - c2) (c1 - c2) (c0 - c1)         --


  >>  InterPoly:= c -> _plus(x[i+1]*c^i $ i=0..m):
  >>  normal(InterPoly(c.i)) $ i=0..m;

                            y0, y1, y2

  >>  n:= 3: linalg::vandermondeSolve(
              matrix([c.i $ i=1..n]),
              matrix([y.i $ i=1..n]), Transposed):
  >>  map(%, factor);

             +-                                 -+
             |   y3 - c2 y2 - c3 y2 + c2 c3 y1   |
             |   -----------------------------   |
             |        (c1 - c2) (c1 - c3)        |
             |                                   |
             |    y3 - c1 y2 - c3 y2 + c1 c3 y1  |
             |  - -----------------------------  |
             |         (c1 - c2) (c2 - c3)       |
             |                                   |
             |   y3 - c1 y2 - c2 y2 + c1 c2 y1   |
             |   -----------------------------   |
             |        (c1 - c3) (c2 - c3)        |
             +-                                 -+

 --------------------------------------------------------------
 --------------------------------------------------------------
 The algorithm is based on the following factorization of
                            ( 1 c.1 c.1^2 .. )
 the Vandermonde Matrix V = ( 1 ..   ..   .. )
                            ( 1 c.n c.n^2 .. )

 |------------ compute divided differences ----------------------|
 |                                                               |
    ( 1                )(1         )( 1             )( 1         )
    (   1              )(  1       )(  1/(c2-c1)    )(-1 1       )
 ...(    1/(c3-c1)     )( -1 1     )(    1/(c3-c2)  )(  -1 1     )  V
    (        1/(c4-c2) )(   -1 1   )(     1/(c4-c3) )(    -1 1   )
    (              ... )(     .. ..)(         ..    )(      .. ..)

    (1 c1         )(1            )(1            )
    (  1 c2       )(  1 c1       )(  1          )
 =  (    1 c3     )(    1 c2     )(    1 c1     ) ...
    (      1 c4   )(      1 c3   )(      1 c2   )
    (        .. ..)(        .. ..)(        .. ..)

*/

linalg::vandermondeSolve:= proc(c,y)
local n, mn, R, k, j, listinput, dtype;
begin 
 if testargs() then
   if args(0)<2 or args(0)>3 then error("wrong number of arguments") end_if;
   if not testtype( c,linalg::vectorOf(Type::AnyType) ) and
      not testtype( c, DOM_LIST ) then
        error("the first argument should be a vector of category 'Cat::Matrix' ".
              "or a list");
   end_if;
   n:= nops(c);
   R:= c::dom::coeffRing:
   if (R = FAIL and not testtype(y, DOM_LIST)) and
      (R <> FAIL and not testtype(y, linalg::vectorOf(R))) then
        error("the second argument should be a vector of same type as the first argument ".
              "or a list");
   end_if;
   if nops({op(c)})<n then error("the Vandermonde nodes must be distinct") end_if;
 else 
   n:= nops(c)
 end_if;

 // convert vectors (matrices) c, y to lists for faster
 // indexed reading and writing.
 // Remember the dimension of y (can be 1 x n or n x 1)
 // and the coefficient ring.
 if domtype(c) <> DOM_LIST then
    c:= [op(c)]:
 end_if;
 if domtype(y) <> DOM_LIST then
    listinput:= FALSE:
    mn:= op(y::dom::matdim(y)):
    R:= y::dom::coeffRing;
    dtype:= domtype(y): // Dom::DenseMatrix(R) or Dom::Matrix(R)
    y:= [op(y)]:
 else
    listinput:= TRUE:
 end_if;
 if args(0)=3 then 
      if args(3)<>Transposed then error("unknown option") end_if;
      for k from n-1 downto 1 do 
         (y[j]:=y[j]-c[j-k]*y[j-1];)$ j=k+1..n;
      end_for;
      for k from n-1 downto 1 do
         (y[j]:=y[j]/(c[j]-c[j-k]);
          y[j-1]:=y[j-1]-y[j];
         )$j=k+1..n
      end_for;
 else (for j from n downto k+1 do 
           y[j]:= (y[j]-y[j-1])/(c[j]-c[j-k]); 
       end_for:) $ k=1..n-1:  // replace list y by divided differences
      // now y = [ y[1], [1,2], y[1,2,3], ..], where y[1,..,k] is the 
      //(k-1)-th divided difference of the original elements y[i] 
      (for j from n-1 downto k do 
            y[j]:=y[j]-c[j-k+1]*y[j+1] 
       end_for:) $ k=1..n-1;
 end_if;
 if listinput then
      return(y)
 else return(dtype(mn, y));
 end_if:
end_proc:

