//   
// stefanw

/*
        linalg::wiedemann(A, b <,mult> <,Probabilistic>)
  
        solves linear system Ax=b 

        A             - square matrix 
        b             - column vector
        mult          - function that returns A*x for each column vector x 
        Probabilistic - use a probabilistic algorithm (TRUE)
                        or a deterministic one (FALSE)
    
        Source: Douglas Wiedemann, Solving Sparse Linear equations over 
        Finite Fields, IEEE Transactions on Information Theory,
        vol. 32, no.1, jan. 1986

        Return Values

        - either a list [x, TRUE] if a solution for Ax=b has been found
        - or a list [x, FALSE] if a solution for Ax=0 has been found
*/

linalg::wiedemann:= proc(A,b,mult=_mult,prob=TRUE)
    local i,X,newf,m,x,abpow,F,d,j,k,n,u,a,y,f,rndgen,
          bk,st,pol,oldy,Mat;
begin
    if testargs() then
        if args(0) < 2 or args(0) > 4 then
            error("invalid number of arguments")
        end_if;
        Mat:= A::dom;
        if Mat::hasProp(Cat::Matrix) <> TRUE then
            error("expecting matrix and vector of 'Cat::Matrix'")
        elif b::dom <> Mat then
            error("matrix and vector type differ")
        end_if;

        F:= Mat::coeffRing; 
        if not F::hasProp(Cat::Field) then
            error("coefficient ring must be of 'Cat::Field'")
        elif b::dom::matdim(b)[1] <> Mat::matdim(A)[1] then
            error("incompatible vector dimension")
        end_if;

        if args(0) = 4 then
            if not contains( {TRUE,FALSE},prob ) then
                error("4th argument must be either TRUE or FALSE")
            end_if
        end_if
    else
        Mat:= A::dom;
        F:= Mat::coeffRing;
    end_if;

    if args(0) = 3 then
        if domtype(mult) = DOM_BOOL then
            prob:= mult;
            mult:= _mult
        end_if
    end_if;

    k:= Mat::matdim(A);
    m:= k[1]; n:=k[2];
    x:= genident();

    // handle non-square case


    if m<n then
        // too few equations: matrix must be singular
        // just fill with zeroes, ie. with equations 0=0

        userinfo(5, "Adding zero rows");
        A:= Mat::stackMatrix(A, Mat::create(n-m, n, 0));
        b:= Mat::stackMatrix(b, Mat::create(n-m, 1, 0));
      
        // instead of using stackMatrix, the multiplication can be
        // redefined: newmult:= proc(AA, bb) begin 
        //   M(m, 1, [op(mult(AA,bb)), (F::zero $m-n)]) end:

        return(linalg::wiedemann(A, b, mult, prob))
    end_if;

    if n<m then
        // too many equations
        // homogenous case : 
        // simply add zero colums and set the additional variables to zero    

        userinfo(5, "Adding zero columns");
        A:= Mat::concatMatrix(A, Mat::create(m, m-n, 0));
        b:= Mat::stackMatrix(b, Mat::create(m-n,1, 0));

        // perhaps not necessary, redefine multiplication 
        // newmult:= (AA, bb) -> mult(AA, M(n,1,[op(bb,1..n)]));
    
        while iszero(b) do
            b:= Mat::create(m,1,(i,j) -> F::random() );
        end_while;

        y:= linalg::wiedemann(A, b, mult, prob);

        if y=FAIL then
            userinfo(5, "Wiedemann algorithm failed");
            return(FAIL)
        else
            userinfo(5, "Wiedemann algorithm succeeded");
            userinfo(20, "Solution is ".expr2text(y));
            if Mat::isSparse <> TRUE then
                return( [Mat::create(n, 1, [op(op(y,1),1..n)]), op(y,2)] )
            else
                return( [Mat::create(n, 1, [op((op(y,1))::dom::body(op(y,1)), 1..n)]), op(y,2)] )
            end_if
        end_if
    end_if;


    rndgen:= (if testtype(F::size, DOM_INT) then F::random else random(100) end_if); 

    // do not use F::random e.g. in the case of F=ExpressionField 

    if iszero(b) then
        userinfo(3, "Singularity check ");

        for i from 1 to 5 do // 5 tries
            userinfo(3, expr2text(i)."th attempt");
            repeat
                b:=Mat::new(n,1, rndgen)
            until not iszero(b) end_repeat;
            userinfo(10, "Using b= ".expr2text(b));
            y:= linalg::wiedemann(A, b, mult, prob);
            userinfo(10, "Result of Wiedemann algorithm: ".expr2text(y));
            if y<>FAIL then         
                return( [op(y,1), FALSE] )
            end_if;
        end_for;   
    
        return( [Mat::create(n, 1, F::zero), TRUE] )

    end_if;


    if prob=TRUE then
        // algorithm 1 of Wiedemann's paper

        // step 1
        bk:= b; 
        // The line 'y:= Mat(n,1)' was diabled and replaced by Kai
        y:= Mat::create(n,1,F::zero);
        d:= 0;

        // step 2
        while not iszero(bk) do

            if d >= n then
                error("d exceeds degree bound");
            end_if;

            // step 4: compute all uA^ib
    
            userinfo(10, "Computing the A^i*b");
            st:=time():
            abpow:=array(0..2*n-1);
            abpow[0]:=bk;
            for j from 0 to 2*n-2 do
                // here abpow=A^j b
                abpow[j+1]:=mult(A,abpow[j]); 
            end_for;

            userinfo(5, "Time for computing the A^i*b :".expr2text(time()-st));
            userinfo(30, "The A^i*b are ".expr2text(abpow));

            // step 3
            repeat
                if Mat::isSparse = TRUE then
                    // The line 'u:=Mat::randmatrix(1,n)' was diabled and replaced by Kai
                    if Mat = Dom::SparseMatrixF2 then 
                       u:=Mat::randmatrix(1,n);
                    else 
                       u:=Mat::new(1,n,[[rndgen() $ j=1..n]]);
                    end_if;
                else
                    u:=Mat::new(1,n,[[rndgen() $ j=1..n]]);
                end_if;

                userinfo(3,"Trying u=",u);
                userinfo(10, "Computing the u*A^i*b");
                st:=time():
                a:=array(0..2*(n-d)-1);
                for j from 0 to 2*(n-d)-1 do
                    // here abpow=A^j*b;
                    // u*abpow is a 1x1-Matrix
                    a[j]:=(u*abpow[j])[1,1]; // gives a field element
                end_for;
            until has(map(a, iszero), FALSE) end_repeat;

            userinfo(5, "Time for computing the u*A^i*b :".
            expr2text(time()-st));
            userinfo(30, "The u*A^i*b are ".expr2text(a));

            // step 5
            st:=time():
            f:= linalg::minpoly_sequence_Euclidean([op(a)], x, F);

            userinfo(5, "Time for computing minimal polynomial :".
            expr2text(time()-st));
            userinfo(20, "Minimal polynomial is ".expr2text(f));
            if iszero(coeff(f,0)) then
                // handle singular case :
                userinfo(5, "Returning a solution to the homogenous system");

                X:=poly(x,[x], F);
                while TRUE do
                    newf:=f;
                    repeat
                      newf:=newf/X
                    until not iszero(coeff(newf,0)) end_repeat;
          
                    userinfo(20, "Division by maximal power of indet gives ".
                             expr2text(newf));
                    m:=degree(newf);
                    oldy:=(y:=_plus(coeff(newf, i)*abpow[i] $i=0..m));


                    if iszero(y) then
                      userinfo(10, "Found zero linear combination just".
                               "by applying the minimal polynomial");
                      // we have \sum_{i=0}^m newf[i]*A^i*b =0
                      // hence \sum_{i=1}^m newf[i]*A^i*b = -newf[0]*b
                      // hence x=(-1/newf[0])*\sum_{i=1}^m newf[i]*A^(i-1)b
                      // is the desired solution
                      return(
                      [-_invert(coeff(newf,0))*
                      (_plus(coeff(newf,i)*abpow[i-1] $i=1..m)),
                       TRUE])
                    end_if;
                      
                    userinfo(10, "Applying minimal polynomial gives ".
                             expr2text(y));
                    while not iszero(y) do
                        userinfo(20, "m= ".expr2text(m));
                        if m>degree(f) then
                            m:=FAIL;
                           break
                        end_if;
                        oldy:=y;
                        y:=mult( A, y );
                        m:=m+1;           
                    end_while;
 
                    if m<>FAIL then
                        userinfo(10, "Element of nullspace found");
                        return( [oldy, FALSE] ); 
                    // else
                       // try another u, see below
                    end_if;
    
                    userinfo(10, "Minimal polynomial failed");
                    repeat
                        userinfo(20, "Creating random vector");
                        if Mat::isSparse = TRUE then
                            u:= Mat::randmatrix(1,n);
                        else
                            u:=Mat::new(1,n,[[rndgen() $ j=1..n]]);
                        end_if;
        
                        userinfo(3,"Trying u=",u);
                        userinfo(10, "Computing the u*A^i*b");
                        st:=time():
                        a:=array(0..2*(n-d)-1);
                        for j from 0 to 2*(n-d)-1 do
                            // here abpow=A^j b
                            a[j]:=(u*abpow[j])[1,1];   // gives a field element
                            // u*abpow is a 1x1-Matrix
                        end_for;
                    until has(map(a, iszero), FALSE) end_repeat;
 
                    userinfo(5, "Time for computing the u*A^i*b :".expr2text(time()-st));
                    userinfo(30, "The u*A^i*b are ".expr2text(a));
                    // repeat step 5

                    st:=time():
                    f:=lcm(f, linalg::minpoly_sequence_Euclidean([op(a)], x, F));
                end_while;
            end_if;

            f:=multcoeffs(f, F::_invert(coeff(f,0)));
            userinfo(30, "Normalization gives ".expr2text(f));

            // step 6
            d:=d+degree(f); 
            userinfo(30, "d = ".expr2text(d));

            // fbar:=(f-poly(coeff(f,0),[x],F))/poly(x,[x],F);
            // need not be computed 
            y:= y + _plus(coeff(f,j+1)*abpow[j] $j=0..degree(f)-1);
            userinfo(30, "y = ".expr2text(y));

            bk:= b + mult(A,y);
            userinfo(30, "bk = ".expr2text(bk));

            // step 7: unnecessary
        end_while;
        return( [-y, TRUE] )
    else
        // algorithm 2 of Wiedemann's paper

        // step1
        userinfo(10, "Computing the A^i*b");
        st:=time():
        abpow:=array(0..2*n-1);
        abpow[0]:=b;
        for j from 0 to 2*n-2 do
            // here abpow=A^j b
            abpow[j+1]:=mult(A,abpow[j]); 
        end_for;

        userinfo(5, "Time for computing the A^i*b :".expr2text(time()-st));
        userinfo(30, "The A^i*b are ".expr2text(abpow));

        // step 2
        k:=0;
        f:=poly(1, [x], F);   

        while degree(f) < n and k < n do
            // u = (k+1) st unit vector
            // uA^ib = (k+1) st entry in A^ib

            // step 3 + step 4
            a:=map(abpow,_index,k+1);

            // step 5
            pol:=poly([[a[2*n-1-i], i] $i=0..2*n-1],[x], F);
            a:=pol*f;
            a:=[coeff(a,2*n-1-i) $i=0..2*n-1];
 
            // step 6 + step 7
            if has(map(a, iszero), FALSE) then
                f:= f* linalg::minpoly_sequence_Euclidean(a, x, F);
            else
                userinfo(5, "Zero sequence encountered")
            end_if;
 
            userinfo(10, "Minimal polynomial is ".expr2text(f));

            // handle singular case
            if iszero(coeff(f,0)) then 
              j:=1;
              X:=poly(x,[x], F);
              f;
              repeat
                j:=j+1;
                f:=f/X
              until not iszero(coeff(f, 0)) end_repeat;
 
              y:=_plus(coeff(f, i)*abpow[i] $i=0..degree(f));
              oldy:=y;

              if iszero(y) then
                      userinfo(10, "Found zero linear combination just ".
                               "by applying the minimal polynomial");
                      // we have \sum_{i=0}^m newf[i]*A^i*b =0
                      // hence \sum_{i=1}^m newf[i]*A^i*b = -newf[0]*b
                      // hence x=(-1/newf[0])*\sum_{i=1}^m newf[i]*A^(i-1)b
                      // is the desired solution
                      return([-_invert(coeff(f,0))*
                                (_plus(coeff(f,i)*abpow[i-1] $i=1..degree(f))),
                              TRUE])
              end_if;
              
              userinfo(20, "y= ".expr2text(y));
              while not iszero(y) do
                oldy:=y;
                y:=mult(A,oldy);
                j:=j-1;
                if j=0 then
                  return(FAIL)
                end_if;
              end_while;
              return( [oldy, FALSE] );
            end_if;
     
            // step 8
            k:=k+1;
        end_while;
    
        // step 9
        return([-1/coeff(f, 0)*_plus(coeff(f,i)*abpow[i-1] $i=1..degree(f)), TRUE] )
    end_if
end_proc:

/*--
        linalg::minpoly_sequence_Euclidean : computes the minpoly of
        a sequence, using the Euclidean algorithm
        should be significantly slower than Berlekamp's algorithm 
--*/
linalg::minpoly_sequence_Euclidean:= proc(sequence:DOM_LIST, x:DOM_IDENT, F):DOM_POLY
    local n,r,rnew,t,tnew,aux,q,e, j, reverse;
begin
    /*--- 
       reverse(f) - compute the reverse of the polynomial f

       The reverse of a polynomial f= an*x^n + ... + a0 is defined
       to be a0*x^n + ... + an
    ---*/
    reverse:= proc(f:DOM_POLY):DOM_POLY
       local j;
    begin
        _plus(multcoeffs(poly(op(f,[2,1])^(degree(f)-j),
        op(f,2..3)),coeff(f,j) ) $ j=0..degree(f))
    end_proc :

// ----------- MAIN -----------------

    userinfo(3, "minpoly_sequence called for sequence ".expr2text(sequence));
    if not contains({op(map(sequence,iszero))},FALSE) then 
        userinfo(1,"Result not reliable because zero sequence passed");
        return(poly(0,[x],F));
    end_if;
    n:=nops(sequence) div 2;
    if F=Dom::IntegerMod(2) and type(gf2fac)=DOM_DOMAIN then
        r:=poly(x^(2*n),[x], IntMod(2));
        rnew:=poly([[expr(sequence[j+1]),j]
                  $ j=0..2*n-1 ], [x], IntMod(2));
        ([rnew, tnew]):=[gf2fac::pade(r, rnew, n)];
    else
        r:=poly(x^(2*n),F);
        rnew:=_plus(poly(sequence[j+1],[x],F)*poly(x^j,[x],F) $ j=0..2*n-1);
        t:=poly(0,[x],F); 
        tnew:=poly(1,[x],F);
        while degree(rnew)>=n do
            aux:=divide(r,rnew);   // gives quotient and remainder
            r:=rnew;
            rnew:=op(aux,2);
            q:=op(aux,1);
            aux:=t;
            t:=tnew;
            tnew:=aux-q*t;
        end_while;
    end_if;
    e:=max(degree(rnew)+1-degree(reverse(tnew)),0);
    return(poly(x^e,[x],F)*poly(reverse(tnew), F));
end_proc:

prog::setcheckglobals(linalg::minpoly_sequence_Euclidean, {gf2fac}): 

