// 

// Find all maximum matches in the bipartite graph
// generated by the syntactic matching process.

// input:
//     A table where the keys are two-element lists of indices (vertices)
//     and the values are lists of return values for these edges
//
// output:
//     A list of tables where each key is a vertex in a maximum matching
//     and the corresponding value is one of the values of this key
//     in the input.  All such tables are contained in the output.
//     If the value of a key is [], it is interpreted as [[]].

// The algorithm is from Takeaki UNO, "Algorithms for Enumerating All
// Perfect, Maximum and Maximal Matchings in Bipartite Graphs"

matchlib::solve_bipartite :=
proc(graph, n)
  local t, retval, edges;
begin
  /*
   if domtype(countme) <> DOM_TABLE then
     countme := table();
   end_if;
  if contains(countme, nops([op(graph)])) then
    countme[nops([op(graph)])] := countme[nops([op(graph)])] + 1;
  else
    countme[nops([op(graph)])] := 1;
    end_if;
   */
  if nops(graph) = 0 then
    if n = 0 then return([[]]); end_if;
    return([]);
  end_if;
  if nops(graph) = 1 and n < 2 then return([[op(graph, [1,1]) = t]
					    $ t in op(graph, [1,2])]); end_if;
  edges := map([op(graph)], lhs);
  retval := matchlib::find_one_maximum_matching(edges);
  if nops(retval) < n then return([]); end_if;
  retval := matchlib::enum_maximum_matching(retval, edges, n);
  retval := matchlib::all_rhs_combos(retval, graph);
end_proc:

matchlib::find_one_maximum_matching :=
proc(graph)
  local nodes, N, i, flow, result;
begin
  // we need different lables for the two parts of the graph, so just
  // negate the second ones:
  graph := map(graph, x->[x[1], -x[2]]);

  nodes := [{op(map(graph, op, 1))},
            {op(map(graph, op, 2))}];
  N := Graph([op(nodes[1] union nodes[2]), "in", "out"],
             graph.[["in", i]$i in nodes[1],
                    [i, "out"]$i in nodes[2]], Directed);
  flow := Graph::maxFlow(N, ["in"], ["out"]);
  result := map([op(select(op(flow, 2),
                           x->((rhs(x) = 1
                                and op(x,[1,1]) <> "in"
                                and op(x,[1,2]) <> "out"))))],
                lhs);
  result := map(result, x->[x[1], -x[2]]);
end_proc:

matchlib::enum_maximum_matching :=
proc(match, edges, n)
  local G, graph, graphs, not_in_cycles, vertices;
begin
  edges := map(edges, x->[x[1], -x[2]]);
  match := map(match, x->[x[1], -x[2]]);
  vertices := {op(map(edges, op))};

  // orient the edges: Unmatched edges must be reversed
  edges := map(edges,
               proc(e)
               begin
                 if contains(match, e) > 0 then
                   e
                 else
                   revert(e);
                 end_if
               end);

  graph := Graph([op(vertices)], edges, Directed);
  G := graph;

  graphs := select(Graph::stronglyConnectedComponents(graph),
                   g->nops(Graph::getVertices(g)) > 1);

  not_in_cycles := vertices minus {op(Graph::getVertices(graph)) $ graph in graphs};
  not_in_cycles := select(edges, e -> e[1] in not_in_cycles);

  graphs := [match] .
            matchlib::enum_maximum_matching_iter(graphs, G, not_in_cycles);

//  graphs := select(graphs, g->nops(g)=n);

//  graphs := [[op({op(not_in_cycles)} intersect {op(match)})]];
//  for graph in subsolutions do
//    graphs := map(graphs,
//                  g->op(map(graph, _concat, g)));
//  end_for;
  map(graphs,
      g->map(g, e->(assert(e[1] > 0 and e[2] < 0);
                    [e[1], -e[2]])));
end_proc:


matchlib::enum_maximum_matching_iter :=
proc(subgraphs, G, not_in_cycles)
  local gplus, gminus, e, e1, e2, solutions, t, subgraph, new_subgraphs,
  rem, found;
begin
  solutions := [];
  if nops(subgraphs) > 0 then
    subgraph := op(subgraphs, 1);
    subgraphs := [op(subgraphs, 2.. nops(subgraphs))];
    if nops(subgraphs) > 0 then
      t := _union(op(subgraphs));
      t := Graph::addVertices(t, map(not_in_cycles, op));
      t := Graph::addEdges(t, not_in_cycles);
    else
      t := Graph([op({op(map(not_in_cycles, op))})],
                 not_in_cycles, Directed);
    end_if;
    solutions := [t union revert(subgraph)];
    // TODO: good strategies for choosing e?
    e := select(Graph::getEdges(subgraph), e->e[1]>0)[1];

    gplus := Graph::removeVertex(G, e); // interpreting e as a list of nodes
    new_subgraphs := select(Graph::stronglyConnectedComponents(gplus),
                            g->nops(Graph::getVertices(g)) > 1);
    rem := {op(Graph::getVertices(gplus))}
           minus {op(Graph::getVertices(t)) $ t in new_subgraphs};
    rem := select(Graph::getEdges(gplus), e -> e[1] in rem) . not_in_cycles;
    rem := listlib::removeDuplicates(rem);
    solutions := solutions . map(matchlib::enum_maximum_matching_iter
                                 (new_subgraphs, gplus, rem),
                                 s->s.[e]);

    // revert the edges in the cycle containing e
    gminus := Graph::removeEdge(G, Graph::getEdges(subgraph));
    gminus := Graph::addEdges(gminus,
                              map(Graph::getEdges(subgraph),
                                  revert));
    gminus := Graph::removeEdge(gminus, [revert(e)]);
    new_subgraphs := select(Graph::stronglyConnectedComponents(gminus),
                            g->nops(Graph::getVertices(g)) > 1);
    rem := {op(Graph::getVertices(gminus))}
           minus {op(Graph::getVertices(t)) $ t in new_subgraphs};
    rem := [op({op(select(Graph::getEdges(gminus), e -> e[1] in rem)
                   . not_in_cycles)})];
    rem := listlib::removeDuplicates(rem);
    solutions := solutions . map(matchlib::enum_maximum_matching_iter
                                 (new_subgraphs, gminus, rem),
                                 s->s.not_in_cycles);
  else
    // uncovered vertices
    t := select(map([op(select(Graph::inDegree(G), iszero@rhs))], lhs),
                t->t < 0);
    t := t . select(map([op(select(Graph::outDegree(G), iszero@rhs))], lhs),
                    t->t > 0);

    if nops(t)>0 then
      // find a feasible path of length 2
      rem := Graph::getEdgesLeaving(G);
      for e1 in t do
        if not contains(rem, e1) then next; end_if;
        found := FALSE;
        for e2 in rem[e1] do
          if not contains(rem, e2) or rem[e2] = [] then next; end_if;
          // there is one starting at e1.  revert it.
          if e1<0 then
            e := [e1, e2];
          else
            e := [e2, rem[e2][1]];
          end_if;

          gminus := Graph::removeEdge(G, [e]);

          G := Graph::removeEdge(G, [[e1, e2], [e2, rem[e2][1]]]);
          G := Graph::addEdges(G, [[e2, e1], [rem[e2][1], e2]]);

//          gplus := Graph([op({op(map(not_in_cycles, op))})],
//                       not_in_cycles, Directed);
          gplus := Graph([op({op(map(not_in_cycles, op)), e1, e2, rem[e2][1]})],
                         not_in_cycles, Directed);
          gplus := Graph::removeEdge(gplus, [[e1, e2], [e2, rem[e2][1]]]);
          gplus := Graph::addEdges(gplus, [[e2, e1], [rem[e2][1], e2]]);
          solutions := solutions.[gplus];

          rem := {op(Graph::getVertices(gminus))};
          rem := select(Graph::getEdges(gminus), e -> e[1] in rem)
                . not_in_cycles;
          rem := listlib::removeDuplicates(rem);
          solutions := solutions . map(matchlib::enum_maximum_matching_iter
                                       ([], gminus, rem),
                                       /* s->s.not_in_cycles */ id);

          gplus := Graph::removeVertex(G, revert(e));
          rem := {op(Graph::getVertices(gplus))};
          rem := select(Graph::getEdges(gplus), e -> e[1] in rem) .
                 select(not_in_cycles, t->{t[1], t[2]} intersect {e1, e2} = {});
          rem := listlib::removeDuplicates(rem);
          solutions := solutions . map(matchlib::enum_maximum_matching_iter
                                       ([], gplus, rem),
                                       s->s.[revert(e)]);

          found := TRUE;
          break;
        end_for;
        if found then break; end_if;
      end_for;
    end_if;
  end_if;

  solutions := map(solutions, s-> if testtype(s, Graph) then
                            Graph::getEdges(s)
                          else s end_if);
  solutions := map(solutions, select, e->e[1]>0);

  solutions;
end_proc:

matchlib::all_rhs_combos :=
proc(matches, graph)
  local retval, match;
begin
  graph := map(graph, x->if x = [] then [[]] else x end);
  retval := [];
  for match in matches do
    retval := retval . map(combinat::cartesianProduct
                            (op(map(match, m->graph[m]))),
                           z -> zip(match, z, `=`));
  end_for;
  retval;
end_proc:
