// 
//      
// 
// walter, 13/03/98 
//----------------------------------------------------------------
// The abscissae c[j] of Gauss-Legendre quadrature                
//   int(f(c),c=0..1) == sum(b[j]*f(c[j]),j=1..n)                 
// are computed by a Newton search for the roots of               
// the n.th Legendre-polynomial P(c)=(c-c[1])..(c-c[n])           
// The weights b[j] are obtained by the formula                   
//  b[j] = (n!)^4/((2n)!)^2 / Pp(c[j])/c[j]/(1-c[j])              
// where Pp(c)=diff(P(c),c)                                       
// Remark:  Let L be the j.th Lagrange-polynomial of              
//          c[1],..,c[n], i.e. L(c)=P(c)/Pp(c[j])/(c-c[j])        
//          With Lp=diff(L,c) one has                             
//          L(c)^2                                                
//           =diff((c-c[j])*L(c)^2,c)-2*(c-c[j])*L(c)*Lp(c)       
//           =diff(P(c)^2/Pp(c[j])^2/(c-c[j]),c)                  
//                     -2/Pp(c[j])*P(c)*Lp(c)                     
//          Integration yields                                    
//            b[j]=int(L(c)^2,c=0..1)                             
//                = Pp(c[j])^(-2)*( P(1)^2/(1-c[j])+P(0)^2/c[j])  
//                = Pp(c[j])^(-2)/c[j]/(1-c[j]) * P(0)^2          
//          (note that  int(P(c)*Lp(c),c=0..1)=0 due to           
//          orthogonality).                                       
//                                                                
// The data are returned as a list [b,c] of lists b and c.        
//----------------------------------------------------------------
// The present implementation follows Press/Teukolsky,Vetterling, 
// Flannery: Numerical Recipes in Fortran, 1992, Chapter 4.5,     
// SUBROUTINE gauleg.                                             
// It uses a different normalization of the Legendre polynomials  
// on the base interval [-1,1] instead of [0,1]                   
//----------------------------------------------------------------
// gldata(n,digits) -- computes numerical values of the weights   
// and abscissae of Gauss-Legendre quadrature with n nodes.       
//                                                                
// return value: list of lists [ b , c ],                         
//               where b=[ b[1],..,b[n] ] , c=[c[1],..,c[n]]      
//               are the weights and abscissae auf Gaussian       
//               quadrature:                                      
//               int(f(x),x=0..1)= sum( b[i]*f(c[i]), i=1..n )    
//               (exact for polynomials f(x) through degree 2n-1) 
// Example:                                                       
// >>  ([b,c]):=gldata(50,DIGITS);                                
//----------------------------------------------------------------

numeric::gldata:=proc(n,digits) 
  local nn,P1,P2,P3,Pp,Ppp,Pppp,Pi1,i,j,b,c,cc,z,dz,macheps,k;
  save DIGITS;
  option remember;
begin
  if args(0)<>2 then error("expecting two arguments") end_if;
  if domtype(n)<>DOM_INT 
     then error("expecting positive integer as first argument"):
  end_if;
  if n<1 then error("expecting positive integer as first argument"):
  end_if;
  if domtype(digits)<>DOM_INT 
     then error("expecting integer number of required DIGITS")
  end_if;
  if digits<1
     then error("number of required DIGITS must be positive")
  end_if;
  if digits<=205 then
     if n=20  then return(numeric::_gldata20)  end_if;
     if n=40  then return(numeric::_gldata40)  end_if;
     if n=80  then return(numeric::_gldata80)  end_if;
     if n=160 then return(numeric::_gldata160) end_if;
  end_if;
  userinfo(1,"Computing weights and abscissae of ".n."th Gauss formula with ".digits." digits");
  nn:=trunc((n+1)/2):
  if digits > 20               // set starting values c=[c[1],c[2],..] for the Newton search 
    then DIGITS:=digits+3+round(n/100):
         ([b,cc]):=numeric::gldata(n,trunc(digits/2)):// makes sure that starting points are 
         b:=array(1..n):c:=array(1..n);               // computed fast using lower precision 
         (c[i]:=1-2*cc[i];) $i=1..nn;           // use arrays (faster than lists)      
    else DIGITS:=10: b:=array(1..n): c:=array(1..n): Pi1:=float(PI/(n+1/2)):
         (c[i]:=cos((i-1/4)*Pi1);) $ i=1..nn;
         DIGITS:=digits+3+round(n/100):
  end_if;
  if n mod 2 = 1 then c[(n+1)/2]:=0: end_if;      // this is the exact value 
  macheps:=float(10^(1-DIGITS));
  for i from 1 to nn do
      z:=c[i]:      // for digits>20 the approximation c[i] of the Legendre root is 
                    // computed by gldata(n,digits/2),i.e., abserr < 10^(-digits/2) 
      while TRUE do // Newton search for root z of Legendre Polynomial. On average  
                    // one needs 2 Newton steps to guarantee abserr < 10^(-digits ) 
                    // if the stopping criterion                                    
                    //         if abs(dz)<macheps then break end_if;                
                    // is used. Instead we we use                                   
                    //   if abs(Ppp*dz^2)<=2*abs(z*Pp)* macheps then break end_if;  
                    // reducing the costs to an average of 1 Newton step.           
                    // However, we need to update Pp, whose precision determines    
                    // the precision of b[i]                                        
         P1:=1:P2:=0:
         for j from 1 to n do                  // compute Legendre polynomials      
                                               // P1=P(n,z),P2=P(n-1,z),P3=P(n-2,z) 
                 P3:=P2:P2:=P1:P1:=((2*j-1)*z*P2-(j-1)*P3)/j;
         end_for;
         Pp:= n*(z*P1-P2)/(z^2-1);            // Pp= diff(P1,z)    
         Ppp:=(n*(n+1)*P1-2*z*Pp)/(z^2-1);    // Ppp= diff(P1,z,z) 
         dz:=P1/Pp; z:=z-dz;
         // important to have <= in the following: watch out for z=0 ! 
         if abs(Ppp*dz^2)<=2*abs(z*Pp)* macheps then break end_if;
         /* if abs(dz)<macheps then break end_if; */   // this was the original criterion 
      end_while;  // root z found 
      Pppp:= ( (n+2)*(n-1)*Pp-4*z*Ppp )/(z^2-1); // Pppp= diff(P1,z,z,z) 
        // Pp is diff(P1,z) at the second but last point z. Use Taylor   
        // expansion to estimate Pp at the very last point, otherwise    
        //  b[i] will be accurate only to  digits/2 decimal places.      
        // Need to include Pppp, otherwise there are cases where b[i] is 
        // (a bit) less accurate than digits decimal places.             
      Pp:=Pp-Ppp*dz+dz^2/2*Pppp;
      c[i]:=    (1-z)/2;b[i]:=1/(1-z^2)/Pp^2;
      c[n+1-i]:=(1+z)/2;b[n+1-i]:=b[i]:
      if i>1 and abs(c[i]-c[i-1])<macheps 
        then error("Legendre root found twice");
      end_if;
  end_for:
  if digits>200 or (n<>20 and n<>40 and n<>80 and n<>160)
     then userinfo(1,"Gauss data computed and remembered"): 
  end_if:
  [[b[k]$k=1..n],[c[k]$k=1..n]];
 end_proc:

// --- the following data were generated and stored: --- 
// DIGITS:=220:                                          
// data20 :=numeric::gldata(20,DIGITS):  print(done20);  
// data40 :=numeric::gldata(40,DIGITS):  print(done40);  
// data80 :=numeric::gldata(80,DIGITS):  print(done80);  
// data160:=numeric::gldata(160,DIGITS): print(done160); 
// write(Text,"gldata",data20,data40,data80,data160);    
// ------------------------------------------------------
numeric::_gldata20:=
                 hold([[0.008807003569576059155930981175926408181071552771\
66836626217466333867420962981092390870155277107304883435185811375628500410\
41614596086783578040550048278105689640330529777692042324908802967413082647\
22831072625368577108090734, 0.02030071490019347066551997613746605493954531\
99949757684088034272809161483754936641477694603115221924880949128548375375\
11305019421029463162122763509478138152119710884781155138010659683195759701\
42621525166417073320780367548, 0.03133602416705453178475326759352080317580\
05382892181820497921727189874055168328393222818830284161017563016266997960\
36630878605045317852874265282181768968602608728256644047832154809891239164\
12514110222681231146275753959817, 0.04163837078835237436237907161102310305\
00889142915816453724410303928465414470397097356875954218954196745480580559\
66381885299574638544451391145511109384959659308035018219563756898984170654\
52300012349680834709220721650798254, 0.05096505990862021751837506774017493\
80833458280116962781309858080984261610126971732376746578847399291068792951\
77627416329661720386098599147373405594289761348279079217274020838113238030\
45614211910750729150353285034089775296, 0.05909726598075920865618868885569\
11435025206097744843877234449760100873741752557581528643439129095087230313\
37715461582956694340896871104956127115487855925669192814545989284422100232\
49757416079436894259012195942431653788491, 0.06584431922458831344924724987\
40815674580552555734917634982182468544271782147404665717789875919863146225\
52990027318123508200158415303769352476466288844566100951010590833181439897\
57085676256440201526270546269401313239182161, 0.07104805465919102566464916\
25335824665172577066960101516686835414919140437489671838084746121416002913\
00665342868331007009470121863637109136861812686224379428381063834413296135\
53134884624973347992875211844225145037111228156, 0.07458649323630187339391\
43685009847183463399520406841582481056089049222112977933903469806630176052\
40525854569272836690034328783889608932471547086171233376602317823638711435\
47803474019166198570512503775098694558518585851550, 0.07637669356536292534\
90421659775487967459743225561892986373505249087987265813688907677862439182\
51952967720009214068939130057513983980039150276501966092113867877700960156\
24575502592495190922536449896042558641690489424239651, 0.07637669356536292\
53490421659775487967459743225561892986373505249087987265813688907677862439\
18251952967720009214068939130057513983980039150276501966092113867877700960\
15624575502592495190922536449896042558641690489424239651, 0.07458649323630\
18733939143685009847183463399520406841582481056089049222112977933903469806\
63017605240525854569272836690034328783889608932471547086171233376602317823\
63871143547803474019166198570512503775098694558518585851550, 0.07104805465\
91910256646491625335824665172577066960101516686835414919140437489671838084\
74612141600291300665342868331007009470121863637109136861812686224379428381\
06383441329613553134884624973347992875211844225145037111228156, 0.06584431\
92245883134492472498740815674580552555734917634982182468544271782147404665\
71778987591986314622552990027318123508200158415303769352476466288844566100\
95101059083318143989757085676256440201526270546269401313239182161, 0.05909\
72659807592086561886888556911435025206097744843877234449760100873741752557\
58152864343912909508723031337715461582956694340896871104956127115487855925\
66919281454598928442210023249757416079436894259012195942431653788491, 0.05\
09650599086202175183750677401749380833458280116962781309858080984261610126\
97173237674657884739929106879295177627416329661720386098599147373405594289\
76134827907921727402083811323803045614211910750729150353285034089775296, 0\
.0416383707883523743623790716110231030500889142915816453724410303928465414\
47039709735687595421895419674548058055966381885299574638544451391145511109\
38495965930803501821956375689898417065452300012349680834709220721650798254\
, 0.0313360241670545317847532675935208031758005382892181820497921727189874\
05516832839322281883028416101756301626699796036630878605045317852874265282\
18176896860260872825664404783215480989123916412514110222681231146275753959\
817, 0.0203007149001934706655199761374660549395453199949757684088034272809\
16148375493664147769460311522192488094912854837537511305019421029463162122\
76350947813815211971088478115513801065968319575970142621525166417073320780\
367548, 0.0088070035695760591559309811759264081810715527716683662621746633\
38674209629810923908701552771073048834351858113756285004104161459608678357\
80405500482781056896403305297776920423249088029674130826472283107262536857\
7108090734], [0.0034357004074525376069388057643398608886764345491720519259\
07934391007641186123109580275298751713895362635529826377904930992569806300\
61116438079316035481639907608183040582626927767660383392720755027939364932\
4070770183567306, 0.018014036361043104366166934401361389043969836096905571\
96823120305397920960780847150999093737201764218434478254201788473556979460\
27024579041993551513071937278818638897980327535003430808400873541332105561\
3463918216202913310, 0.043882785874337047066123779398350943475407601288154\
11260205889042096455439564046053177763690353930631061980412267698679413156\
83085305817593931134463617919730354569134056007728081654241134987443437419\
7462281927735005882860, 0.080441514088890588302735469149239657335185317467\
18131337375363723356945300033759504038532971702117538969788982346542952721\
27866164830219421607721536064174348192621139808788769207812671617191797122\
5793036722835899248052901, 0.126834046769924603692847464822179204844634660\
21541177793022704696573232248092246765944794318967623969380754967416171925\
18442845135985546605312263799865392908051389965518355510470574895225628493\
339278757324254021439544406, 0.1819731596367424872735816518868570316283054\
41600315768030276688726729368707284933720648402252119346708941445311137021\
31897945948514539837972765026302781747313340468392067556029214937626217089\
19019897310241072484141336202, 0.24456649902458645099781797452237450078725\
43353987865832638256900526325148046171359279841584745661104008352846557823\
68821717372988715442604365219130677469699808293323016351759226654502171924\
2949454204994521227568036934066, 0.313146955642290219663725911487536381302\
12683914715864408602569324217711781347023605205215818273331052761613895573\
59249234703249465777863022749668053200961561683941865419652983921631625864\
898168406577269783159014044472871, 0.3861070744291774609597519023157126876\
28455531158536263842682130396414329068222086102523937404515645984113134342\
19784891285047956179359393257363452634482448844939556145554492365501360653\
43524176360416884888958233881257110, 0.46173673943325133312267979530058089\
44976018665932512495976023778078718289758315108792274429092218921965007471\
76817933647779731087714368557644405641361593307525964348268038388971334251\
5516011710687998936545845059778677509, 0.538263260566748666877320204699419\
10550239813340674875040239762219212817102416848912077255709077810780349925\
28231820663522202689122856314423555943586384066924740356517319616110286657\
484483988289312001063454154940221322490, 0.6138929255708225390402480976842\
87312371544468841463736157317869603585670931777913897476062595484354015886\
86565780215108714952043820640606742636547365517551155060443854445507634498\
63934656475823639583115111041766118742889, 0.68685304435770978033627408851\
24636186978731608528413559139743067578228821865297639479478418172666894723\
83861044264075076529675053422213697725033194679903843831605813458034701607\
8368374135101831593422730216840985955527128, 0.755433500975413549002182025\
47762549921274566460121341673617430994736748519538286407201584152543388959\
91647153442176311782826270112845573956347808693225303001917066769836482407\
733454978280757050545795005478772431963065933, 0.8180268403632575127264183\
48113142968371694558399684231969723311273270631292715066279351597747880653\
29105855468886297868102054051485460162027234973697218252686659531607932443\
97078506237378291080980102689758927515858663798, 0.87316595323007539630715\
25351778207951553653397845882220697729530342676775190775323405520568103237\
60306192450325838280748155715486401445339468773620013460709194861003448164\
4489529425104774371506660721242675745978560455593, 0.919558485911109411697\
26453085076034266481468253281868662624636276643054699966240495961467028297\
88246103021101765345704727872133835169780578392278463935825651807378860191\
211230792187328382808202877420696327716410075194710, 0.9561172141256629529\
33876220601649056524592398711845887397941109579035445604359539468222363096\
46069368938019587732301320586843169146941824060688655363820802696454308659\
43992271918345758865012556562580253771807226499411714, 0.98198596363895689\
56338330655986386109560301639030944280317687969460207903921915284900090626\
27982357815655217457982115264430205397297542095800644848692806272118136110\
2019672464996569191599126458667894438653608178379708669, 0.996564299592547\
46239306119423566013911132356545082794807409206560899235881387689041972470\
12482861046373644701736220950690074301936993888356192068396451836009239181\
695941737307223233961660727924497206063506759292298164327]]):

numeric::_gldata40:=
                 hold([[0.002260638549266595629235866439092666363915555099\
85299535017097117133197350097411256564381791394501546975337347823810498999\
45179377024438002555406402102698315098574119414254982737902207169719768952\
83021920112980565963847163, 0.00524914226557640680737108553363982618839631\
06578983898578375618058126494484554153502684152370161220729354883801937797\
86486230885105427120200311429973311425119224838678107492279685672159208034\
532877368235479220469532422793, 0.0082105291909539443564317424411819636461\
71146673478880985951957756795219650605136834327686131191941754533704240683\
10338488567608244694215628114576551901402615377600629085819171121172194275\
2651135809787209287234291609075921, 0.011122924597083478630752162092104286\
60351659833967721394036712000284565576007898749712410725965659951898728830\
53273579007760444230403920891779448671492911206543249054517841889289556881\
2786885654178492232125249474216326956, 0.013968503490011700549244578753860\
53865127543102538319922404295822518744974084546972085417410453207861822246\
10565531510338876140444076205144940318940923765169765997369390985544048340\
8226125681202600177323151773666777017061, 0.016730097641273923696339091543\
20542448862089332688217520659558834952619988563257613215010630354710579588\
87069074687410351648659754411585660570702044547243653222744081562691917009\
9111140658553545130809623488825455899456647, 0.019391083987236008819986015\
64522308112672960561601292891863344445976936507184537690805539984271136171\
07880299268037054689948423547393572840542005063410606080751078405643625917\
7711263529417934312983815445170920419075374108, 0.021935454092836635995837\
34302085774790550341858511844487926295432490089290295896114448110384776975\
56711174024719710125093165009005874328899111178780010732703381788701430127\
2914122228666381262201676676143484796106953065415, 0.024347903817536116030\
71708022407319403392151368855988757032302563758921420874043751040640610979\
80966260647350475719082225279414138925287086098008374171927323437566774259\
6326579422632632992061228016290449785370795016156944, 0.026613923491968412\
17749823988613025227766058591100322538218044026326040751093553739622251547\
03271330875787351184076432813140178042013212235188814050401557846650941443\
5087276724558885583750198749737114281370582139214353351, 0.028719884549695\
77568330886545521299280005241792722681325334063346277418085569500837692838\
14076867434217035281487671554217876858295568334069137610273730369735558563\
6906147857572527429187138894507159061660418798164170312393, 0.030653121246\
46446958326899820419929795129688175558716099344881934343407791155872968446\
47499105380716258273961624348831817131010137505642179433974788344973447931\
6956612277980332615547489530559479466503588967616408452819748, 0.032402006\
72830051903727726478337636501634648210424445318159762213564198646284281301\
65116336747984880562291957100568111759435724231085555295423024738214593323\
0719468800733651962041172841706734865470760916846091524864990123, 0.033956\
02290761695191284505411596199299209861918964297735894810179652039081343968\
68415135094483603096751829508859585778198563585162085005608013742970948165\
6691919077171567874998139212153743888667326561670517768207957186919, 0.035\
30582369564338984774181542764341617977955199779304865534213448133184865597\
42388773000680207978925057919927088113837722295215555132869719067441341296\
4593407144863511952405284001628900223914498178275790586239947473650500, 0.\
03644329119790202953025534172125891792877954049289931523812740841649527786\
53717010948543908749737005075355783486950370503138512197415049229726557438\
8391792736697204583143394070696774250788648300660341520591330035532125723,\
 0.03736158452898413210009466813066233659560146721017991086567044570882670\
15163617777195741530045178530561825881752414111135391252020611376850251265\
12812438214630894530926014902787892336553361015202161321192834003834799079\
76, 0.03805518095031312118577903796124741150627977692253555352709122951464\
96203179162915426548128644326145361190533753990726759826692224147711229846\
93188331194168496579534161883247852132227159789557240607148782985749721577\
58424, 0.03851990908212398279415376714190512426221987708197020774459044621\
62708215452322186002541943661304181451859905241290732292212874514710623468\
02716198835438332093223141031637441951186757776597779729781213227060131713\
12485178, 0.03875297398921240563186198147916316348183432639405321786647713\
69270360663745355231864103742072686981443718010272880869460645757446091588\
82547183516685085524607275878080478293950952429538866813137631332967187055\
94736477944, 0.03875297398921240563186198147916316348183432639405321786647\
71369270360663745355231864103742072686981443718010272880869460645757446091\
58882547183516685085524607275878080478293950952429538866813137631332967187\
05594736477944, 0.03851990908212398279415376714190512426221987708197020774\
45904462162708215452322186002541943661304181451859905241290732292212874514\
71062346802716198835438332093223141031637441951186757776597779729781213227\
06013171312485178, 0.03805518095031312118577903796124741150627977692253555\
35270912295146496203179162915426548128644326145361190533753990726759826692\
22414771122984693188331194168496579534161883247852132227159789557240607148\
78298574972157758424, 0.03736158452898413210009466813066233659560146721017\
99108656704457088267015163617777195741530045178530561825881752414111135391\
25202061137685025126512812438214630894530926014902787892336553361015202161\
32119283400383479907976, 0.03644329119790202953025534172125891792877954049\
28993152381274084164952778653717010948543908749737005075355783486950370503\
13851219741504922972655743883917927366972045831433940706967742507886483006\
60341520591330035532125723, 0.03530582369564338984774181542764341617977955\
19977930486553421344813318486559742388773000680207978925057919927088113837\
72229521555513286971906744134129645934071448635119524052840016289002239144\
98178275790586239947473650500, 0.03395602290761695191284505411596199299209\
86191896429773589481017965203908134396868415135094483603096751829508859585\
77819856358516208500560801374297094816566919190771715678749981392121537438\
88667326561670517768207957186919, 0.03240200672830051903727726478337636501\
63464821042444531815976221356419864628428130165116336747984880562291957100\
56811175943572423108555529542302473821459332307194688007336519620411728417\
06734865470760916846091524864990123, 0.03065312124646446958326899820419929\
79512968817555871609934488193434340779115587296844647499105380716258273961\
62434883181713101013750564217943397478834497344793169566122779803326155474\
89530559479466503588967616408452819748, 0.02871988454969577568330886545521\
29928000524179272268132533406334627741808556950083769283814076867434217035\
28148767155421787685829556833406913761027373036973555856369061478575725274\
29187138894507159061660418798164170312393, 0.02661392349196841217749823988\
61302522776605859110032253821804402632604075109355373962225154703271330875\
78735118407643281314017804201321223518881405040155784665094144350872767245\
58885583750198749737114281370582139214353351, 0.02434790381753611603071708\
02240731940339215136885598875703230256375892142087404375104064061097980966\
26064735047571908222527941413892528708609800837417192732343756677425963265\
79422632632992061228016290449785370795016156944, 0.02193545409283663599583\
73430208577479055034185851184448792629543249008929029589611444811038477697\
55671117402471971012509316500900587432889911117878001073270338178870143012\
72914122228666381262201676676143484796106953065415, 0.01939108398723600881\
99860156452230811267296056160129289186334444597693650718453769080553998427\
11361710788029926803705468994842354739357284054200506341060608075107840564\
36259177711263529417934312983815445170920419075374108, 0.01673009764127392\
36963390915432054244886208933268821752065955883495261998856325761321501063\
03547105795888706907468741035164865975441158566057070204454724365322274408\
15626919170099111140658553545130809623488825455899456647, 0.01396850349001\
17005492445787538605386512754310253831992240429582251874497408454697208541\
74104532078618222461056553151033887614044407620514494031894092376516976599\
73693909855440483408226125681202600177323151773666777017061, 0.01112292459\
70834786307521620921042866035165983396772139403671200028456557600789874971\
24107259656599518987288305327357900776044423040392089177944867149291120654\
32490545178418892895568812786885654178492232125249474216326956, 0.00821052\
91909539443564317424411819636461711466734788809859519577567952196506051368\
34327686131191941754533704240683103384885676082446942156281145765519014026\
153776006290858191711211721942752651135809787209287234291609075921, 0.0052\
49142265576406807371085533639826188396310657898389857837561805812649448455\
41535026841523701612207293548838019377978648623088510542712020031142997331\
1425119224838678107492279685672159208034532877368235479220469532422793, 0.\
00226063854926659562923586643909266636391555509985299535017097117133197350\
09741125656438179139450154697533734782381049899945179377024438002555406402\
10269831509857411941425498273790220716971976895283021920112980565963847163\
], [0.00088114514472039982518864878970675383211480920247704669833194145158\
05684901360974610569061343958578266514018655936951925660512247007867094012\
52816877704912280570079848686894607747124334772009288528355526931552392904\
6016534, 0.004636880650271496773472823889313922518888959324456731058461877\
84796054838072770168544474700874891609507521586197013318940142267835569070\
58342521831257241567834893652996977984991802261358691607020453356042454636\
22297366167, 0.01137002500811286866831485814354809651066603398100695178536\
89919944683927778927714061553495556350706638468284637354923454934699866476\
50131771989848508173890624744990795242172759855098868989095543415875461793\
09685669218945, 0.02104159039310417209772950027362035745255825469862648421\
54730782091208221882581353397896941821814894790191848760132727625483549183\
93570659409766420135290873878871333232219891646437462609993424910627123807\
68108835264979036, 0.03359359586066173331957391657739714178262321235865377\
53853284922921797380383268852330726563291077268888167724825377851249677534\
92945352825519149890781502474098703599797414557326199206453272453247595069\
09426185056382463215, 0.04895059651556285163587333456575344820775595947116\
52661373699442298149092995076463020530061329597008986738538358882223886702\
73947558039696186355597992383294609489661109282010721809765219447563065868\
34071443080427874404468, 0.06702024839387024808960909582269001821472672349\
44491752700298725966589605764587223495417188915402668422898324966798896754\
15984842855041269499846619298394923379378405729536252265377541736164719022\
88713548766454883886712357, 0.08769388458334416840183988466695061304637980\
78785061001541045184235964140786521377837752374839501079789238505307213467\
87540952186262666010612412969000645815609670312911634096460482950785512814\
05787156769803575299990238045, 0.11084717428674030615251422724675257598965\
43419365541028773289696853440290419097142793999541772022949524084973102383\
37624615067901472763857642007183924692349247011788028914274094175940880367\
5680552163626713495593608040919, 0.136340872405036448359501774122534725721\
31066323333593122622437790703544267695576369622944698751948141128141396583\
74424339659846008996110720410538074278701181571750499961550969443958764304\
667210287125055764353260758153170, 0.1640216576929102258103227425192529450\
14837009308074424964453515506668762021690117026984027529287625792815316757\
66798752947830162126657861591377219636770555500485004450213553018440858130\
65567871228217162254422483360879013, 0.19372305516600988102369377488465256\
13099381091584251944323262187349045137877808823899390201902849457155851861\
67124602232271757042280802518382519219491076834876727468716128128683883221\
0369143417692716831027359491649554391, 0.225266437452435898962034347235241\
01488301244920217426754871726413907212744325820476411835387718276835765921\
90791114451000038768039559636715773603720617601771322862515070149459664479\
774382087283904652998578588339264229727, 0.2584620991569106435457167128775\
88497700488802233442949605332378387028278629046570377062447589421260696440\
58939541483476685624505673129221906436229006084502164548158303898080545571\
85158635745929799772841064542678206979282, 0.29311039781419749923756012709\
81431585129501879735408808527988744656786681953896342810819705642560632117\
08593628390008593519572723826244269650878161679836372655099745610484517783\
2958447584110424910770226186680580815263546, 0.329002954587120763496253759\
41040284496652318998633710176829520795092814552893955833840226667599986749\
51408243725341983222725953161951426322228213697602486115933373751853208126\
054480661445984252643202143961187773298236842, 0.3659239074963731594294078\
27595701908287597813381876509271370012563089403403132887823904812400734758\
94724844212075966856149973103223913935973907014837143727523759737311834629\
86587707384628258144133538894955171354586577556, 0.40365120964931445014224\
15739674250525929539894473955625572662038817486391681782942250376896286992\
40721844393951923733174796127057850848411082381284413640315942769821043504\
6717808903841999799184810714690349758793936012684, 0.441957964662372395758\
27435779598794311563573457289202548485128391921041802606641324231144326945\
30786101315989658644986301769085476715783668690631496856272880020099719710\
836751759011072052243630415817287618142339668954290, 0.4806137912469745890\
33403277987688352660317682808429200415681409280996712499539621895697863096\
82425780496296395315829602730133396097548086655244099447629168693420800253\
99869458075573460992466763489218447789697317785672494, 0.51938620875302541\
09665967220123116473396823171915707995843185907190032875004603781043021369\
03175742195037036046841703972698666039024519133447559005523708313065791997\
4600130541924426539007533236510781552210302682214327505, 0.558042035337627\
60424172564220401205688436426542710797451514871608078958197393358675768855\
67305469213898684010341355013698230914523284216331309368503143727119979900\
280289163248240988927947756369584182712381857660331045710, 0.5963487903506\
85549857758426032574947407046010552604437442733796118251360831821705774962\
31037130075927815560604807626682520387294214915158891761871558635968405723\
01789564953282191096158000200815189285309650241206063987315, 0.63407609250\
36268405705921724042980917124021866181234907286299874369105965968671121760\
95187599265241052751557879240331438500268967760860640260929851628562724762\
4026268816537013412292615371741855866461105044828645413422443, 0.670997045\
41287923650374624058959715503347681001366289823170479204907185447106044166\
15977333240001325048591756274658016777274046838048573677771786302397513884\
066626248146791873945519338554015747356797856038812226701763157, 0.7068896\
02185802500762439872901856841487049812026459119147201125534321331804610365\
71891802943574393678829140637160999140648042727617375573034912183832016362\
73449002543895154822167041552415889575089229773813319419184736454, 0.74153\
79008430893564542832871224115022995111977665570503946676216129717213709534\
29622937552410578739303559410604585165233143754943268707780935637709939154\
9783545184169610191945442814841364254070200227158935457321793020717, 0.774\
73356254756410103796565276475898511698755079782573245128273586092787255674\
17952358816461228172316423407809208885548999961231960440363284226396279382\
398228677137484929850540335520225617912716095347001421411660735770272, 0.8\
06276944833990118976306225115347438690061890841574805567673781265095486212\
21911761006097980971505428441481383287539776772824295771919748161748078050\
89231651232725312838718713161167789630856582307283168972640508350445608, 0\
.8359783423070897741896772574807470549851629906919255750355464844933312379\
78309882973015972470712374207184683242332012470521698378733421384086227803\
6322944449951499554978644698155914186934432128771782837745577516639120987,\
 0.86365912759496355164049822587746527427868933676666406877377562209296455\
73230442363037705530124805185887185860341625575660340153991003889279589461\
92572129881842824950003844903055604123569533278971287494423564673924184682\
9, 0.889152825713259693847485772753247424010345658063445897122671030314655\
97095809028572060004582279770504759150268976166237538493209852723614235799\
28160753076507529882119710857259058240591196324319447836373286504406391959\
080, 0.9123061154166558315981601153330493869536201921214938998458954815764\
03585921347862216224762516049892021076149469278653212459047813737333989387\
58703099935418439032968708836590353951704921448718594212843230196424700009\
76195, 0.93297975160612975191039090417730998178527327650555082472997012740\
33410394235412776504582811084597331577101675033201103245840151571449587305\
00153380701605076620621594270463747734622458263835280977112864512335451161\
1328764, 0.951049403484437148364126665434246551792244040528834733862630055\
77018509070049235369794699386704029910132614616411177761132972605244196030\
38136444020076167053905103388907179892781902347805524369341316592855691957\
212559553, 0.9664064041393382666804260834226028582173767876413462246146715\
07707820261961673114766927343670892273111183227517462214875032246507054647\
17448085010921849752590129640020258544267380079354672754675240493090573814\
94361753678, 0.97895840960689582790227049972637964254744174530137351578452\
69217908791778117418646602103058178185105209808151239867272374516450816064\
29340590233579864709126121128666767780108353562537390006575089372876192318\
9116473502096, 0.988629974991887131331685141856451903489333966018993048214\
63100800553160722210722859384465044436492933615317153626450765450653001335\
23498682280101514918261093752550092047578272401449011310109044565841245382\
069031433078105, 0.9953631193497285032265271761106860774811110406755432689\
41538122152039451619272298314555252991251083904924784138029866810598577321\
64430929416574781687427584321651063470030220150081977386413083929795466439\
57545363777026338, 0.99911885485527960017481135121029324616788519079752295\
33016680585484194315098639025389430938656041421733485981344063048074339487\
75299213290598747183122295087719429920151313105392252875665227990711471644\
4730684476070953983]]):

numeric::_gldata80:=
                 hold([[0.000572475001593470767272085970657818059349696202\
78795143416312053706697756051137859270826040242253309254660277890987122188\
91156639605823200176287725205163143643799245384586372544159518301864474347\
287485038822610968105237887, 0.0013317667947563408346467679158342277328722\
77121193574281536783929704187643927967018127941479215671700039852854875540\
66691027958237548242901018947996543476105486465900957264476928424864357777\
7963353245228708033623683877041, 0.002090156562347447618369652100840675661\
77474868655321976102940120929980847391567159319078843344509816407618946892\
93780007214217418965919249878035337671551285750459899750118910111354794487\
14992136453665444176032448186488706, 0.00284546122570159932463455355858100\
92384626319173385662609839505855502536704456336104362584404759241669917931\
04045212638182070245125973566195609257362313718188312559984895050053569600\
148241131249628498751048768125060932636, 0.0035964523840586563763377854339\
78253738253519096182650022315882001509932667807725814541751103800468539167\
70771248098666284416177263300579376996597152876411840832694059185950192472\
0851247708340843688511075645506332868053756, 0.004341972634630429213204726\
10201714067620302147753270485384138989825756994782220869184420684732438769\
44825647135887598994708296437371311246957736569067292820733280929219124230\
85066642980204623202575074477377701318250457018, 0.00508088302055153226041\
59251762034718320228909398202626470343423378780892813409240864055917365117\
51194453861924810805668498696714732342097295185275093331549479642614451293\
767270546094151256352401085122135270156491573521927, 0.0058120570603989134\
58233384997716317429756590751481066482545021410417577194114321148952714857\
29664299521829091395246718807396233398195875351663589319540300058988450320\
7797844007877754063760078270787600690418914174154115426, 0.006534380796200\
66964689341294852817015520931719120721677217394695563377840750059578816920\
30401630023088138259061411451212756057529539445389351654742196487419287776\
06303362070940789167950569109002273828772162056385797808373, 0.00724675402\
02545380584810372917302750279784360775577565282234655292445366210654122903\
60856569803616641812931221102155905627864354371147807059409598377079748667\
319755084646197633712604431547436798812713573919224297225577412, 0.0079480\
91791862844022451454614589262885486046302860966363428671735967047635682899\
48463956092753452009797436479350216925375458723990747731797895606014896818\
0305703275890349664522406080427332645208473234496002841444401398066, 0.008\
63732602813465317929210356564549990015551465200517050358789107881962589689\
56914781253353925557727603664384487650756727702168417425603402988908318090\
57234801103272680366685979556374932280010447707303527983861787171772417, 0\
.0093134071041495157143677070760786045042238831680822117251796472692131785\
06677550601988789877026758541021039987634283060394475436063398081427854981\
51340193708621192992615847063684534834414620890844039090194911890086924234\
3, 0.009975305439070999464445964357556781680250532142507965333373629231029\
40223087055337591453278197575279952241710209222541070655374655912183568487\
43157129980128506960526201194953932432577206326903544692338128901188605072\
39022, 0.01062201305789100319435536862530656427323446212165399025415468843\
41000009559416516799302061833602599120648189501245148132685752816461289504\
01065554721193122123083625549223504072059500733446482475934683179139897433\
16979581, 0.01125254512316623096311079484308436951028979418591853149225921\
21522041811166917762434154335590211474213709589475588861650778529336378777\
17311694140550648875162885940133493793258861808490635819045168379867654369\
09560291634, 0.01186594143296505064659626230678420814617126455416094788579\
59555124372688802703985111464280128202571076069213453917221925508432366453\
23753353209068147695479906078020707718374760799569931743338214290071586104\
80144798156263, 0.01246126788205774555255892350160990117855124493778084775\
72961839490348098466893700844734215311404086368890624032214489027662176977\
38126751433211046644188494089504402528305791158239502335102332321127887609\
09433966655892838, 0.01303761788378255895148437180013464356284873791461965\
49756746490059785798517215082895369080551056166275498115449979812934316358\
67362326475816392242053768567902070718490414380860818619507852155281155521\
03002549514049596792, 0.01359411375024319033720935334027212991493858785005\
35352000422090145900463291546504485108550509529754669288160762348305211482\
06841633027864605937634086728074117576901828112612495744298636441044123519\
57355675300371206111916, 0.01412990802863843119837659898250726514713274918\
65698959001434448316969898812968919785236378395709599258505371926174043753\
79265458391622351436657862696453458207527399946638730853120137184539097777\
36860579180331987931095238, 0.01464418479163392384638379300978956983060371\
55723153975949428749347751707411308936875323687173804557219078534806035932\
45156234372311237024377276953409818614686318366920574659747009252537787178\
72144118927347331030501728434, 0.01513616087977899033061000504545058737367\
10337798446390378905138649578117661883432849249913134056096599642956641695\
47896553057483203757586480874692739601845693205697870873959927372531803989\
79514467152561068607445912253127, 0.01560508709405735082122143336030177593\
29998604101413830047739045577302854873191514616153922476139733493626879886\
74202575161027229405957168680777594316785412561480001390451348750038849260\
97368428310618865805039091033270623, 0.01605024933674388657402824514362534\
80447583819162968249133822192699086776854714570147505892351134381820588096\
35302947136243926451249585064663566241955050753058596689752603407719441720\
00645689137846941885406205030932600144, 0.01647096969882270069141809045097\
97680640135188463643088320997714205758570104334561153268262611577900459704\
77513128486687650953708940483029989561381451667284365140222853970096651729\
11787075686915645519417442525647245138371, 0.01686660749230576140833758153\
21193642229132519240833513998661316241535286306718371720277009117050593381\
92095593735077807579969156879305124834934368046582999625575212394178220634\
35679954621591164008032307095733045621944561, 0.01723656022587696439718211\
33655149160383903983714764899516462426868764498496470490442779813315097684\
08985271470479269147235261089917493664460537026436751538306564774280295315\
02525468850800880135831016606292845562975498284, 0.01758026452237379674776\
32961943484406145812261552567492348815961821368088561967158411466640658033\
20496531777441765262320332695600293861667390687718651217441660044140850540\
14917899321598462576171595083815149426171447327404, 0.01789719697670802730\
14307944080772271201180676152876353969000673031096556273502733579839822759\
41124694834507298173738167344802610622616632246686489757087350933167647452\
01867983133094989561670749995969240154533965955917174, 0.01818687495291798\
90219824955232614068300314108938050859728114808782018831678249459818786427\
15554419550474598777882679411215693231845939557682845074419198369813841678\
49591911725685161715957833710908583286913419081954666414, 0.01844885731913\
80044195754982867026096342840505659143129206501613539745862476008176800870\
41373612450457413981077446677996151863716304905924078746272900573256855211\
61317552792259075771954804976396833892321249591921534896490, 0.01868274511\
93652450133526885289193345824034539747136898523924069179942919658287013007\
85503435460298928053541617633191802004686643109345346084561760585231580748\
59838597203117032367351331999856729229683593862121957586489837, 0.01888818\
21810006987448874882131605273853509620097847573806454237351991597342532925\
70249876271032133801993135468494340540010989165179711120486402204036319306\
80224313175724038543075722614419698920448537333937229916423287962, 0.01906\
48556572388191721033957828681009570719619532795170264439134800880472478209\
31111262265484669668979726585133513076782049164913854313950558775340293059\
59576760880365995836445185770025130274943270309566528690965473203676, 0.01\
92124965034797115926062181647450692155109381354551231940509912683082962122\
98515475879304686012577424195363232699477378314655264727592753866686289173\
33660436237334322139300769120971664625065242917354534271565152720962702, 0\
.0193308798870382316635385551335783456304504639199322100087418110100813132\
72223429215362231367565088635248860155734565314330468793478022806869555647\
65100681921540968179524476298931072843459321976467876824482935266851773492\
, 0.0194198255295259844658870913343935829454401146702429271455168089286319\
85419943876685765536564531493001250162283966525077635207504866583461827573\
94903544782701524073321326101284752825712885315711313020797063968250338775\
006, 0.0194791979813847655993127623861304111574660057931347807992357461916\
55980620442748803770730721340763711115082211912407724808799259438907911809\
92164756417467224875610553005863633852651283772412153341577229925771033628\
097803, 0.0195089068281533274056402196263770241647752370148021298983444477\
82006367857008783384429688899067039817972106563708068355478105417275228408\
37791869655561092639829958304414825067118698997510272779320521484801477295\
687943438, 0.0195089068281533274056402196263770241647752370148021298983444\
47782006367857008783384429688899067039817972106563708068355478105417275228\
40837791869655561092639829958304414825067118698997510272779320521484801477\
295687943438, 0.0194791979813847655993127623861304111574660057931347807992\
35746191655980620442748803770730721340763711115082211912407724808799259438\
90791180992164756417467224875610553005863633852651283772412153341577229925\
771033628097803, 0.0194198255295259844658870913343935829454401146702429271\
45516808928631985419943876685765536564531493001250162283966525077635207504\
86658346182757394903544782701524073321326101284752825712885315711313020797\
063968250338775006, 0.0193308798870382316635385551335783456304504639199322\
10008741811010081313272223429215362231367565088635248860155734565314330468\
79347802280686955564765100681921540968179524476298931072843459321976467876\
824482935266851773492, 0.0192124965034797115926062181647450692155109381354\
55123194050991268308296212298515475879304686012577424195363232699477378314\
65526472759275386668628917333660436237334322139300769120971664625065242917\
354534271565152720962702, 0.0190648556572388191721033957828681009570719619\
53279517026443913480088047247820931111262265484669668979726585133513076782\
04916491385431395055877534029305959576760880365995836445185770025130274943\
270309566528690965473203676, 0.0188881821810006987448874882131605273853509\
62009784757380645423735199159734253292570249876271032133801993135468494340\
54001098916517971112048640220403631930680224313175724038543075722614419698\
920448537333937229916423287962, 0.0186827451193652450133526885289193345824\
03453974713689852392406917994291965828701300785503435460298928053541617633\
19180200468664310934534608456176058523158074859838597203117032367351331999\
856729229683593862121957586489837, 0.0184488573191380044195754982867026096\
34284050565914312920650161353974586247600817680087041373612450457413981077\
44667799615186371630490592407874627290057325685521161317552792259075771954\
804976396833892321249591921534896490, 0.0181868749529179890219824955232614\
06830031410893805085972811480878201883167824945981878642715554419550474598\
77788267941121569323184593955768284507441919836981384167849591911725685161\
715957833710908583286913419081954666414, 0.0178971969767080273014307944080\
77227120118067615287635396900067303109655627350273357983982275941124694834\
50729817373816734480261062261663224668648975708735093316764745201867983133\
094989561670749995969240154533965955917174, 0.0175802645223737967477632961\
94348440614581226155256749234881596182136808856196715841146664065803320496\
53177744176526232033269560029386166739068771865121744166004414085054014917\
899321598462576171595083815149426171447327404, 0.0172365602258769643971821\
13365514916038390398371476489951646242686876449849647049044277981331509768\
40898527147047926914723526108991749366446053702643675153830656477428029531\
502525468850800880135831016606292845562975498284, 0.0168666074923057614083\
37581532119364222913251924083351399866131624153528630671837172027700911705\
05933819209559373507780757996915687930512483493436804658299962557521239417\
822063435679954621591164008032307095733045621944561, 0.0164709696988227006\
91418090450979768064013518846364308832099771420575857010433456115326826261\
15779004597047751312848668765095370894048302998956138145166728436514022285\
397009665172911787075686915645519417442525647245138371, 0.0160502493367438\
86574028245143625348044758381916296824913382219269908677685471457014750589\
23511343818205880963530294713624392645124958506466356624195505075305859668\
975260340771944172000645689137846941885406205030932600144, 0.0156050870940\
57350821221433360301775932999860410141383004773904557730285487319151461615\
39224761397334936268798867420257516102722940595716868077759431678541256148\
000139045134875003884926097368428310618865805039091033270623, 0.0151361608\
79778990330610005045450587373671033779844639037890513864957811766188343284\
92499131340560965996429566416954789655305748320375758648087469273960184569\
320569787087395992737253180398979514467152561068607445912253127, 0.0146441\
84791633923846383793009789569830603715572315397594942874934775170741130893\
68753236871738045572190785348060359324515623437231123702437727695340981861\
468631836692057465974700925253778717872144118927347331030501728434, 0.0141\
29908028638431198376598982507265147132749186569895900143444831696989881296\
89197852363783957095992585053719261740437537926545839162235143665786269645\
345820752739994663873085312013718453909777736860579180331987931095238, 0.0\
13594113750243190337209353340272129914938587850053535200042209014590046329\
15465044851085505095297546692881607623483052114820684163302786460593763408\
672807411757690182811261249574429863644104412351957355675300371206111916, \
0.013037617883782558951484371800134643562848737914619654975674649005978579\
85172150828953690805510561662754981154499798129343163586736232647581639224\
20537685679020707184904143808608186195078521552811555210300254951404959679\
2, 0.012461267882057745552558923501609901178551244937780847757296183949034\
80984668937008447342153114040863688906240322144890276621769773812675143321\
10466441884940895044025283057911582395023351023323211278876090943396665589\
2838, 0.011865941432965050646596262306784208146171264554160947885795955512\
43726888027039851114642801282025710760692134539172219255084323664532375335\
32090681476954799060780207077183747607995699317433382142900715861048014479\
8156263, 0.011252545123166230963110794843084369510289794185918531492259212\
15220418111669177624341543355902114742137095894755888616507785293363787771\
73116941405506488751628859401334937932588618084906358190451683798676543690\
9560291634, 0.010622013057891003194355368625306564273234462121653990254154\
68843410000095594165167993020618336025991206481895012451481326857528164612\
89504010655547211931221230836255492235040720595007334464824759346831791398\
9743316979581, 0.009975305439070999464445964357556781680250532142507965333\
37362923102940223087055337591453278197575279952241710209222541070655374655\
91218356848743157129980128506960526201194953932432577206326903544692338128\
90118860507239022, 0.00931340710414951571436770707607860450422388316808221\
17251796472692131785066775506019887898770267585410210399876342830603944754\
36063398081427854981513401937086211929926158470636845348344146208908440390\
901949118900869242343, 0.0086373260281346531792921035656454999001555146520\
05170503587891078819625896895691478125335392555772760366438448765075672770\
21684174256034029889083180905723480110327268036668597955637493228001044770\
7303527983861787171772417, 0.007948091791862844022451454614589262885486046\
30286096636342867173596704763568289948463956092753452009797436479350216925\
37545872399074773179789560601489681803057032758903496645224060804273326452\
08473234496002841444401398066, 0.00724675402025453805848103729173027502797\
84360775577565282234655292445366210654122903608565698036166418129312211021\
55905627864354371147807059409598377079748667319755084646197633712604431547\
436798812713573919224297225577412, 0.0065343807962006696468934129485281701\
55209317191207216772173946955633778407500595788169203040163002308813825906\
14114512127560575295394453893516547421964874192877760630336207094078916795\
0569109002273828772162056385797808373, 0.005812057060398913458233384997716\
31742975659075148106648254502141041757719411432114895271485729664299521829\
09139524671880739623339819587535166358931954030005898845032077978440078777\
54063760078270787600690418914174154115426, 0.00508088302055153226041592517\
62034718320228909398202626470343423378780892813409240864055917365117511944\
53861924810805668498696714732342097295185275093331549479642614451293767270\
546094151256352401085122135270156491573521927, 0.0043419726346304292132047\
26102017140676203021477532704853841389898257569947822208691844206847324387\
69448256471358875989947082964373713112469577365690672928207332809292191242\
3085066642980204623202575074477377701318250457018, 0.003596452384058656376\
33778543397825373825351909618265002231588200150993266780772581454175110380\
04685391677077124809866628441617726330057937699659715287641184083269405918\
59501924720851247708340843688511075645506332868053756, 0.00284546122570159\
93246345535585810092384626319173385662609839505855502536704456336104362584\
40475924166991793104045212638182070245125973566195609257362313718188312559\
984895050053569600148241131249628498751048768125060932636, 0.0020901565623\
47447618369652100840675661774748686553219761029401209299808473915671593190\
78843344509816407618946892937800072142174189659192498780353376715512857504\
5989975011891011135479448714992136453665444176032448186488706, 0.001331766\
79475634083464676791583422773287227712119357428153678392970418764392796701\
81279414792156717000398528548755406669102795823754824290101894799654347610\
54864659009572644769284248643577777963353245228708033623683877041, 0.00057\
24750015934707672720859706578180593496962027879514341631205370669775605113\
78592708260402422533092546602778909871221889115663960582320017628772520516\
3143643799245384586372544159518301864474347287485038822610968105237887], [\
0.000223088674184685059959750452716407504153221157074091091814390066339426\
58218908944754403303468391725557814385720916790580728615655682382423782173\
00551884420167271558723743413270721510315767012210170750748533916735506463\
502, 0.0011750678008811555502528959084385073344625828017181892590579851913\
00347906039078863187405711092601144001838389734027054803910863492166772870\
71530138232146402153189653395526399683893781255266136853847286679223410543\
3933944, 0.002886229517155861053968248167544150955595697522643982090028368\
20572442508846625930663965586126787943108195674627950020623465597351140909\
99497946173616669401060331841746167507114203750336716541813143493535226572\
46317769860, 0.00535434875012223448674841643168430735892411108454919273128\
30611527973039312815021697140884685295916747661908029125612795300322216123\
36736134454071054593777167158320546416918692352002755002323067158347914361\
625449711169345, 0.0085757136306854647908559861454417632156529705925600813\
96230071015412073185263012644558809672941148306971029975070375325767196064\
33358592788293296469452249878247158686009944950638624361844388271619348837\
6704537313343899281, 0.012545429707136103307177384965431861877520141981847\
76157314921627210897727994449700791980068036089012510041885499578378589457\
07590875920922298298784520758750636949761061655107045112829407326293013821\
2883350189438203650660, 0.017257455478100374273863422164272500748548257075\
69434776335525431770529297633848500588158251516910741000417156594148716277\
90492624989372650565550922821056895289770247445733494637977841820925227828\
5105689243150716884557438, 0.022704616828182547253259241489485245608176592\
64795039974032940585832917581063789323240545442573929928580443168575825980\
92034541163402500124427046837062521645046708309408055967338908651263590261\
0143569852091314928070799243, 0.028878619345063662623866997749999132464583\
01687901031825798038672912750911401812148451881621040325677492422446733234\
44899060326688692068727858726554067832242167416653149499496573101268816415\
4216955473182072197596679130680, 0.035770061413777102023477020462273433103\
88250558020899397259849524843069741350680804986784462534223195901948135667\
36782382244917956166222461707138239286777756932336490141650009419605080644\
5568909959539242453883323044650007, 0.043368448714121172917633171924526260\
94427376584461263284294819769344003924877496537433724380203429725090678012\
40769226056286507795082518992453353111155742434955852301725617716003352478\
0076873495977985476596681127817641920, 0.051662210280614658402837964016302\
00684635014897400650740197021781783047518986118008878393380383931831453262\
16798538543533640663938634110368256628162536168037737692537774376335795317\
7502684481097043972791495339021740031424, 0.060638716160893085648113328180\
43779603207411166161686630368836087878439731342848177293950109688983729173\
59988825382867726009568346875480492513071710682657685666865888802852232367\
1761496130837277780460475375187321711155415, 0.070284296668444451511403938\
25417175358003425519267787227721851173367426695430209967992472876406844157\
99598388691536582818294764651067849553677555748925132070295180539590081898\
7424841753639440797347085655787879667982574905, 0.080584263209872362191688\
47804856646760318829369409614252164773719863836594627759727349229724380890\
25654021333112926784424432731725992103725400266301135462886253757503328320\
9150693037719952871350777478911435862095539561192, 0.091522930659268264814\
43750299385214612857939966426687534727660544601861282729443457447755871926\
12676983977445409767938379990285726252702405759038767323497617126485238777\
4112739694685091270011391574493227700978464163310190, 0.103083641247697275\
02568034413077282069469812462361630620094048779324304316316999464899048262\
34293700694046520575196500893714863462740632826660358518278507909201436539\
194841231205250636159682594148364810403469463496446582, 0.1152487899324793\
13067191965625486958007297027332358030726361974113231853851840864639918721\
84297919261078984110344372390123612188746258227652386375952191077351805799\
16896441385035809359420871372431606393181229307743504057, 0.12799985120820\
13638417297360345431630956604576655408926555256479047398640009060388750703\
37712857278955583447740230011710868385054171209217894818298752281178758873\
3430799015206595957206103396168951528283597822746153692286, 0.141317407318\
95005987296587085309236071658472353709445046465083713538079286741614269924\
71534449466559245632153544291166553458645219934205213990382504616063198257\
522720399216759169541019413961238992963427146543336308689048, 0.1551811778\
28986199614396193780532366955138527264893597331217452938171125116422326829\
86589966442343544748367730455648344135598382036246089402795428482919979308\
86736966751066709046708464795680653563084556753173298467894013, 0.16957005\
05069400991320164385778413825974699406422404025624866150294717802284206129\
33008558718422568335481865238221773536730182712847610338273706227438534298\
6013727624568872733302824777958940012351292358904007055375417798, 0.184462\
11347656401687603580635533156845546826496333040553081950580075881542868876\
00992605385090526406983429661605135387364786583415381879284152507047735651\
915636969255640760930048037950630560953676156733629480380194968487, 0.1998\
34688585124128422626850417996575784972452593106629157173985840727002014631\
86081894657833286524375509948365791446113560239993763199502449686748919781\
67990605312399982134824474823101962608919452827617611849323088723281, 0.21\
56643659386451076372571066875864206290032131701935692713095279116587244381\
21710167589972721675269958743845966788829013783470064663835931432164164329\
3883433052269015118305054473849508850593953045516402599502885754375762, 0.\
23192703955143403399007137343729954754415343746042153863662735874932453179\
82994435364684845467425957229600020534518384542739519030500027127639992079\
152621246547077923794800492818655470710764786876907514166635671228328603, \
0.248597944055607506203163624816215998217529663750200920058664791313665714\
99127854525948885264908234023912570261652346903549487575068467787200508638\
12376103370174870641494942144448471551129343587181678194168774714213499519\
, 0.2656516924147277614819608175320956713530219967558665988412912160690814\
40660186867223008108871471801058432135969701616707125559398212818483355656\
13308361593126533034603104433100179143505373297766771789728417778392987575\
38, 0.28306231458412195346880664981840902084894584782740262733174593564705\
74447317799062561911487332976556533550813615332844091370866016192075066931\
74957649399522556990617244444485696283510438982785655234869178428579667678\
7907, 0.300803297059015386487810178741233121441375437643213670625651155911\
46220274627038486981149331790411693532494420088771078020437913304275227303\
28216772167456683936056649094066941258491450278358013889296345422510445361\
572540, 0.3188476232502563421904783568205182059914451874381933140872561528\
25591643488329499498485080399751775815870583669535366960410420610183760812\
68298188417641272726392334720360948723546724874158550503839330928783004690\
52195123, 0.33716781462614904269044352818632065248150762244545980161934109\
50979713788795124381405918734619494895241242388243461598814238601830396218\
57847152913121116997793527921306580827448452940400546067878304688998191863\
4184084641, 0.355735972557744073445430349282643050751797355493117031789476\
78535094642673375635146565571743641660565030457808533218736487692102000652\
57743322559424630868645052612654702952429150974786405365672039167648203955\
283365763935, 0.3745238208038639397534205919824976013184755362181003934675\
82316928098748669762020358571014432479270195649879492264938152325969086151\
70069675661370165469341346065994452316672560354308934241175896983139218213\
33990255331405, 0.39350274857116693371380573066683908845260562458261314892\
57016470931167760849772283448738873109864863157552995763001479052431602469\
23452789545494198220590196859000056804887830210670022177163435306890151059\
7687311982071870, 0.412643854083676593720330475994356902140594612511468710\
24799544024181454124521850678664318207989607566432508558853818854241894549\
86983665914637873410255673586490252257134726490076802324182800178716153211\
634308087520180208, 0.4319179885954280567203794609996414660334318751825584\
44735485293963368638862655402551311735612086311364804656854202287934382940\
56447933297995762466527601614786565453601269807043184065091160712790337729\
32954487530770086503, 0.45129580077920770046836077494753154899146690778161\
69537045761823846662210747500132298672521372335178895922022630545172775551\
19609451241515479081442996517324072610427179045704445453746786249504001098\
2757949868293053127595, 0.470747781423789665685503339058291102787300662649\
04280089560223018110479583647368751249143828179601102950725763895242986301\
84651629786471257655453727574384699503735908591054203357404766156265422233\
144152158525229543306965, 0.4902443083716030011728243829462727260332243491\
76501092432714776209057813877303500067992436607413310999285917573635635640\
36555753356881394709426170462980572732720853523130366238767318278000262331\
81617540962956902215752887, 0.50975569162839699882717561705372727396677565\
08234989075672852237909421861226964999320075633925866890007140824263643643\
59634442466431186052905738295370194272672791464768696337612326817219997376\
6818382459037043097784247112, 0.529252218576210334314496660941708897212699\
33735095719910439776981889520416352631248750856171820398897049274236104757\
01369815348370213528742344546272425615300496264091408945796642595233843734\
577766855847841474770456693034, 0.5487041992207922995316392250524684510085\
33092218383046295423817615333778925249986770132747862766482110407797736945\
48272244488039054875848452091855700348267592738957282095429555454625321375\
04959989017242050131706946872404, 0.56808201140457194327962053900035853396\
65681248174415552645147060366313611373445974486882643879136886351953431457\
97712065617059435520667020042375334723983852134345463987301929568159349088\
3928720966227067045512469229913496, 0.587356145916323406279669524005643097\
85940538748853128975200455975818545875478149321335681792010392433567491441\
14618114575810545013016334085362126589744326413509747742865273509923197675\
817199821283846788365691912479819792, 0.6064972514288330662861942693331609\
11547394375417386851074298352906883223915022771655126112689013513684244700\
42369985209475683975307654721045450580177940980314099994319511216978932997\
78228365646931098489402312688017928129, 0.62547617919613606024657940801750\
23986815244637818996065324176830719012513302379796414289855675207298043501\
20507735061847674030913848299303243386298345306586539340055476833274396456\
9106575882410301686078178666009744668594, 0.644264027442255926554569650717\
35694924820264450688296821052321464905357326624364853434428256358339434969\
54219146678126351230789799934742256677440575369131354947387345297047570849\
025213594634327960832351796044716634236064, 0.6628321853738509573095564718\
13679347518492377554540198380658904902028621120487561859408126538050510475\
87576117565384011857613981696037814215284708687888300220647207869341917255\
15470595994539321216953110018081365815915358, 0.68115237674974365780952164\
31794817940085548125618066859127438471744083565116705005015149196002482241\
84129416330464633039589579389816239187317018115823587272736076652796390512\
7645327512584144949616066907121699530947804877, 0.699196702940984613512189\
82125876687855862456235678632937434884408853779725372961513018850668209588\
30646750557991122892197956208669572477269671783227832543316063943350905933\
058741508549721641986110703654577489554638427460, 0.7169376854158780465311\
93350181590979151054152172597372668254064352942555268220093743808851266702\
34434664491863846671559086291339838079249330682504235060047744300938275555\
55143037164895610172143447651308215714203323212092, 0.73434830758527223851\
80391824679043286469780032441334011587087839309185593398131327769918911285\
28198941567864030298383292874440601787181516644343866916384068734669653968\
9556689982085649462670223322821027158222160701242461, 0.751402055944392493\
79683637518378400178247033624979907994133520868633428500872145474051114735\
09176597608742973834765309645051242493153221279949136187623896629825129358\
505057855551528448870656412818321805831225285786500481, 0.7680729604485659\
66009928626562700452455846562539578461363372641250675468201700556463531515\
45325740427703999794654816154572604809694999728723600079208473787534529220\
76205199507181344529289235213123092485833364328771671396, 0.78433563406135\
48923627428933124135793709967868298064307286904720883412755618782898324100\
27278324730041256154033211170986216529935336164068567835835670611656694773\
0984881694945526150491149406046954483597400497114245624237, 0.800165311414\
87587157737314958200342421502754740689337084282601415927299798536813918105\
34216671347562449005163420855388643976000623680049755031325108021832009394\
687600017865175525176898037391080547172382388150676911276718, 0.8155378865\
23435983123964193644668431544531735036669594469180494199241184571311239900\
73946149094735930165703383948646126352134165846181207158474929522643480843\
63030744359239069951962049369439046323843266370519619805031512, 0.83042994\
94930599008679835614221586174025300593577595974375133849705282197715793870\
66991441281577431664518134761778226463269817287152389661726293772561465701\
3986272375431127266697175222041059987648707641095992944624582201, 0.844818\
82217101380038560380621946763304486147273510640266878254706182887488357767\
31701341003355765645525163226954435165586440161796375391059720457151708002\
069113263033248933290953291535204319346436915443246826701532105986, 0.8586\
82592681049940127034129146907639283415276462905549535349162864619207132583\
85730075284655505334407543678464557088334465413547800657947860096174953839\
36801742477279600783240830458980586038761007036572853456663691310951, 0.87\
20001487917986361582702639654568369043395423344591073444743520952601359990\
93961124929662287142721044416552259769988289131614945828790782105181701247\
7188212411266569200984793404042793896603831048471716402177253846307713, 0.\
88475121006752068693280803437451304199270297266764196927363802588676814614\
81591353600812781570208073892101588965562760987638781125374177234761362404\
780892264819420083103558614964190640579128627568393606818770692256495942, \
0.896916358752302724974319655869227179305301875376383693799059512206756956\
83683000535100951737657062993059534794248034991062851365372593671733396414\
81721492090798563460805158768794749363840317405851635189596530536503553418\
, 0.9084770693407317351855624970061478538714206003357331246527233945539813\
87172705565425522441280738732301602255459023206162000971427374729759424096\
12326765023828735147612225887260305314908729988608425506772299021535836689\
81, 0.91941573679012763780831152195143353239681170630590385747835226280136\
16340537224027265077027561910974345978666887073215575567268274007896274599\
73369886453711374624249667167908493069622800471286492225210885641379044604\
3880, 0.929715703331555548488596061745828246419965744807322127722781488266\
32573304569790032007527123593155842004016113084634171817052353489321504463\
22444251074867929704819460409918101257515824636055920265291434421212033201\
742509, 0.9393612838391069143518866718195622039679258883383831336963116391\
21215602686571518227060498903110162708264001117461713227399043165312451950\
74869282893173423143331341111971477676328238503869162722219539524624812678\
28884458, 0.94833778971938534159716203598369799315364985102599349259802978\
21821695248101388199112160661961606816854673783201461456466359336061365889\
63174337183746383196226230746222562366420468224973155189029560272085046609\
7825996857, 0.956631551285878827082366828075473739055726234155387367157051\
80230655996075122503462566275619796570274909321987592307739437134922049174\
81007546646888844257565044147698274382283996647521992312650402201452340331\
887218235808, 0.9642299385862228979765229795377265668961174944197910060274\
01504751569302586493191950132155374657768040980518643326321761775508204383\
37775382928617607132222430676635098583499905803949193554431090040460757546\
11667695534999, 0.97112138065493633737613300225000086753541698312098968174\
20196132708724908859818785154811837895967432250757755326676555100939673311\
30793127214127344593216775783258334685050050342689873118358457830445268179\
2780240332086932, 0.977295383171817452746740758510514754391823407352049600\
25967059414167082418936210676759454557426070071419556831424174019079654588\
36597499875572953162937478354953291690591944032661091348736409738985643014\
790868507192920075, 0.9827425445218996257261365778357274992514517429243056\
52236644745682294707023661514994118417484830892589995828434058512837220950\
73750106273494344490771789431047102297525542665053620221581790747721714894\
31075684928311544256, 0.98745457029286389669282261503456813812247985801815\
22384268507837278910227200555029920801993196391098748995811450042162141054\
29240912407907770170121547924124936305023893834489295488717059267370698617\
8711664981056179634934, 0.991424286369314535209144013854558236784347029407\
43991860376992898458792681473698735544119032705885169302897002492962467423\
28039356664140721170670353054775012175284131399005504936137563815561172838\
065116232954626866561007, 0.9946456512498777655132515835683156926410758889\
15450807268716938847202696068718497830285911531470408325233809197087438720\
46996777838766326386554592894540622283284167945358308130764799724499767693\
28416520856383745502888306, 0.99711377048284413894603175183245584904440430\
24773560179099716317942755749115337406933603441387321205689180432537204997\
93765344026488590900050205382638333059893966815825383249288579624966328345\
8186856506464773427536822301, 0.998824932199118844449747104091561492665537\
41719828181074094201480869965209396092113681259428890739885599816161026597\
29451960891365078332271292846986176785359784681034660447360031610621874473\
386314615271332077658945660660, 0.9997769113258153149400402495472835924958\
46778842925908908185609933660573417810910552455966965316082744421856142790\
83209419271384344317617576217826994481155798327284412762565867292784896842\
32987789829249251466083264493536]]):

numeric::_gldata160:=
                  hold([[0.00014402926426054152232724969892699182341711614\
74749917907106524158541757788881977789600690354747134755230562415371078371\
61563767869053298613234110990426063810553556085882220624484327729667512140\
3729396328926521549373972977, 0.000335219160076151640737653139377877549400\
70666874344252328766451330093974597674262378786296593660553976218357869052\
49874585075272517556914551339609987277241355422603527131605114906923071003\
151304759831103032332314031452855, 0.0005265638361068677441154594928326056\
93256339475650422180789825775138119206499029815746352138037690115416113486\
52114928401553113114427237589584367657353563465445133726566733822278973341\
93642653468262334026526305531728548627, 0.00071773137674159643149088714093\
90517092000606112676587834159713801214028451142911570285774836714157343760\
93275841292187632972068944677318656200458658563322693011827434926715096366\
6416217698974635538394885553314361223554709, 0.000908628879832592004544400\
78630477536959152370077543665664954210330592674168103567209694869118550038\
10172417151821615879524174538026384251679715479905703929819151537151745176\
362449526705761913658251904875364285599978182132, 0.0010991797483982443724\
28073493442406846768538755917899835297513334664778685265097116013975007920\
61393212030655313210969499016041645128945429288037664415269485152836852197\
8520278677701000873658379131059278329730270578151251, 0.001289310060918765\
96134288278198892961335423905724679555123802307426726543111201749971328902\
77158387782714760738336664310087878952569508531655385136541959728762763105\
66411774132188167362171074418020437809639851496770241078, 0.00147894666251\
97082874685481849089432192076740631683558108944050976173430511561535509461\
40979202122123443978834469696769383126750852816035941336584742859393944884\
259570575727208466559939903427597619240003942673910188566460, 0.0016680167\
73910433107000957173675107139536700489468082707067073104648118318015934457\
66591412346344911522464522774243004986063231113131460015588990987253572105\
6333440851589001469255263957062816133657048527220747218462019169, 0.001856\
44790129375785237689860342533353370320076179566762788349241983784134308186\
90978253453755809933340776924671266305047388381434456410292607144478025711\
54182374146114307857354141045574931228310928588064728104910175556740, 0.00\
20441678242839863389067299144840460117775789459411710862757583239200399930\
97381661636491819220389986671926668097925773766166829135502942997822278264\
145320927724354483225479847145455499272184295046760972806119721231098494, \
0.002231104608167486976347492561942761891616774492143301323043738784903773\
14790198197699545177883302735775063382844013556864654770034493535384104501\
43552576635272566167505182371702226371305242569263609966517097244499272510\
86, 0.00241718662485138690858234658787889260486365526474024238495474136954\
98019094468813867616762156321902361606847915342528454691465624892072601140\
40075837953815841500785007606300652609046545080855030429103054891061930868\
952890, 0.0026023425772236336651396967950115810889237900533891462848586394\
20926875928997411484596228292723127588674137115034297632481413964146613075\
01574373180957480770706753854978328387491414187134821934771925547186270125\
7518915890, 0.002786501524930452446756963319259732260881614018047653739721\
85962016535693868983220546919551352430957733758662893182211911367654195430\
16890848484929564850177213143568611744190880407437795029752642591596063856\
35197099494022, 0.00296959291074293877157003499008794117949242016921505330\
16255769333553935495953244693188965761067317689457567105256203352855458389\
80938236844830938364171594064726869094702575303139480398564195850055642741\
876422395475464069, 0.0031515465871390658624814321598298276816256995077437\
13075082856481188578460047195295474805922926023763410157338730390585186080\
66947487604072747514009855091500654566356378780212622335849597144272237118\
8510646432147654559581, 0.003332292842918906068764339658385913441084763324\
49857983789341484389981934976680309247028742429452876981401562554848310759\
70661390152696201205833729024643512989649635217718932696715749341748104069\
75697274633750423822395955, 0.00351176242975702245955778695057350742987147\
82878714769020694990972237767065200264271396100065912750755872068725425038\
34966469823266055218002567556411945552448581630045657856359210053063667163\
734536551655395585922293109807, 0.0036898865886369887599720508580693120948\
46114792327328551851530225278422124058165944849251175861593660401666904903\
64515138402565025154786555496952399153381607267974354099654595557163817864\
8745947591629371848474283801605400, 0.003866597076133493287710928674908786\
30727833663813819468899543796998494451947937100732331523602210755082674584\
22086504463737184521439234499453261820380942502790093963117737785534821760\
38470182122266903666749352393520524282, 0.00404182619051818993209803407754\
09047859034067342841346059137581833025194709023058766925438200142212603980\
28532165333740707760787846265388384041590886918572101140346158666301985234\
332259613346088402233208485863681673285767, 0.0042155067976712824189008198\
59843926063608336437929623127438959059478838149095373922615401611077171299\
68036346027678737653785801534987276161624849748262236420396696909615461767\
7134148449877528131953470085476550016693510176, 0.004387572356784118549801\
23449591952673484107823475466652291623265207789287990617274599638671005068\
14134943250515933120204786806622770558137058727734357074996252864175094566\
33477570223552746668653382238201843256135717093652, 0.00455795694584000245\
20198080390797889708988471571329556741347738593844371031256710769156824559\
38341773605551068856305806214635082048090592789621931695104328854898137200\
378872981922363365011144410034211906577721787560005209, 0.0047265952868616\
12503052854507235037127622415081478254235569352198027609855837463626520487\
06494982342525329231735879142661405242931197613747340403848667257579700856\
5140166635704010707887752545926085169251980833856527274190, 0.004893422770\
91416521783070568036045696080240579024498687363253213941505041513204536599\
36480488700758855473533292559498488677765690663887705566116771994630287741\
83777980298093255374327329288193935203703504041241186744596341, 0.00505837\
54828539697643184522790288370885316592194284256444139484497242958213431509\
99721469675684669524934080906007968568821838991682254602565892769776714839\
493407207653763423276668837056500835753546874170995731709525618557, 0.0052\
21390225812374980555463259915898740516871540203782937442667651090187011897\
85148437293176036529316515822543186006162797311736411862612097909467834265\
2034185426111291986368111779060361811774205700851595369237003615320525, 0.\
00538240454540537935017444906875253616615802066484473540565680939235608607\
30999567635709051225218449311432650956615893354621803041203150603150316189\
35404759821327027647955063350727335295267761001936090846668502153516867281\
, 0.0055413567536593890302637007116750630489646000469282897366390908538473\
76360584099967828236810593887758583803130659976188994039947694355193426156\
07838969078439437672523046823635789222555421395178432934866388372588130950\
3228, 0.005698185952643790671816524284353061373897410941865152940646201652\
91833385143079097815997494434990387695616946409497774226307373268310086424\
73571225200008825196274708004094338892380419090016648080068930356145347239\
81257537, 0.00585283205780116733075128184720944099901373821658645116091968\
14120112374167690036631795407021315802492865224352470971351851509744029474\
42168982782761594196561249726253426812408511739014456360052222712907828154\
582814357115, 0.0060052358209661353140107714997159665767102893429506952441\
41393260049563784858626186153258404313569199426909767863202652856931453215\
33237406618821589298557463316991566643105343129121502425864754032838691612\
9839203624675860, 0.006155338853063922474707460205369729644114143773454358\
18107096764990386406644333579227863278830080836771823249031693719312030855\
49989530560723565133760089716352121798631424455501771042242524856744208857\
40417417763751828174, 0.00630308364647994757002645844285031610687591124204\
64307181961825726410500945620945637919634542289978304064419887162933377765\
83776195563647252271653059715118688073737289943786960306613236123255252685\
474493057466189912060365, 0.0064484135970917979807822880736962031422438751\
80004667030962023939320792314828909550671018655141084304872281219853882360\
79864860534204796401507255634766411942484588849160805747159664517763766015\
7493317710224388437673100130, 0.006591273025955140779519089485118565385889\
36334052792519534677047339427551422708430081738590403851657801171923191104\
14697371993145074088127133884699977307491735301336967466216191789251723878\
65052026548044389330063373063963, 0.00673160720063524076302394915861230174\
52485400000854117355472118137096637630438002499833606222182701296966338794\
30710801247585340451684568874891973888264390642589083966043931200915499565\
892551337421261659726667606033542876, 0.0068693623561758992577939204415138\
44186932550124970193058999288547092925749396328805239470166638924845195713\
75507952676340267454147615965685417128925759335978169838201449386399774312\
4283955830800888055166589907081274817698, 0.007004485715697769675680027452\
25846724436091768017190557858912071153460952472433698622315924730299235338\
13939869411896203386662486481780911608061226192145555311262270552502812083\
16433807490817979467874194286509836693988347, 0.00713692551061815021189531\
35667788046400767764230530063019525964960301714782835103919443626466379723\
30884764370487613819488467509605677949372456181495239072119067171762723274\
541051592907614442857548846701691107698861185807, 0.0072666310004845009120\
12861423544440438418085349792821597101023516233967861260173528303261464628\
06006416042330484613615576101598791608685476476339843798214976400022890839\
3028638434341590704493685329876674544530717832675651, 0.007393552492414081\
69456054218834889663859765786004116112788136327191956901919392016818718420\
30239011866357600564510467561194690633466570036224337906872974651758414033\
28595871338384186692380419314114856223343429446463351744, 0.00751764136013\
22598604429056799602346234717101392247774833304854323004715302654555178424\
02834813846506333750854254907973042557125111832710860034826247321276531896\
675026418396162598386492976723147914377730675221629468964110, 0.0076388500\
62602190175537166614694883509780456182773386363415588309346568293659537633\
61374918070874529645157917854994766310746807006498418271943878648716342597\
2206018263481827255806301700213592069138015258141134771629542495, 0.007757\
13216223872778044290463765983129556893663297333656896244093222539279880649\
59365696741047958344666746601514998135700618381534750520079609032568738333\
92631984609685400358931096200038407195987848799702227558014872925130, 0.00\
78724423426995939462481565461465075111968864020193995516231188912522550270\
82077344998574216226004827088382366902408371923267026136244544216553698105\
108902731123311841534370116344800574192628527995896238677745959234763168, \
0.007984736426246977029331392501445048903900382838934525520041657056481078\
32953132062513477610914675025190116326249260290537114424421384415966533627\
55150774969884808559206252249612817166449365463329223924915075428985381441\
92, 0.00809397139067291584419668909831410942145251865279864876331964486751\
28254278394417976379908351934769894989611507684306774128307515681725776138\
05404363106193406658451091269873205598603642723751137502922911930070681773\
278636, 0.0082001053857819800265155632916292099786813833751212569934123552\
95456972528504032722842174054086317709836868520672098919546595431357465510\
86539702150091162223078261886303825141039336761064645749776653199544460202\
0169957505, 0.008303097749424931748789535610816689198622885608045757563153\
26805608648014456613426552986124843930082235869940209464075135908907956437\
64670342274516338575146215203038414407578924140838386959155627927325639582\
44581614545703, 0.00840290902307722532391123314249773084850255626155058495\
48544334004411028685672755720071881619982094729226613712149333373183323589\
38971567155242075129861800519942783511427321806849566215251992904125884665\
558309547800670392, 0.0084995009669563757295215701062234359259707389708969\
87383438107367800452738757602391931216634290122332693405483180402383207959\
00803238642785153366129865798136201291619794458434041473583103846913038709\
4454583334698453537037, 0.008592836574672403847795101254835180504726540744\
46472819737496132527548645712347844428814996604254374191518060542855389363\
21581681909037963258097500726135595348163351157613625644743389109994394379\
88250134823697842249310721, 0.00868288008740574517820669856382452166366940\
57502936357286805213056623695546863195973686998362170160051193071010027110\
59970194120234351332797431737862056944900956168053500981838654059844853579\
415860265358042076824817382163, 0.0087695970076071898800922910101675203836\
01916191729137603196059740013180591403416205518902779909342167669976698462\
99926801159032203362844859100863340379924179987766291357628287997811458528\
4270646091765323103771658555029984, 0.008852954112214605170927715687112613\
79614291046445935028262272451839493494724802247071098897830495244850566017\
32068725586296579143488814379445463541665917903694060788925357220781962276\
32010093713954310184254882170974564845, 0.00893291946538137627730438078983\
68841681761398765342840426252277208460057821183910436398219151437893363059\
55086576908908303456527020566862782383231210002362316740415979318959754674\
373748425392748173540281865693313206659463, 0.0090094624307116892394190733\
13850363965258460319854097790731921729946008383321352492829007365707014387\
15101823241414739490147731757982929175626979109408590218810017027238600706\
4052449201956307910649853285543439164386096569, 0.009082553682997967836243\
51392999110797211117756566431691414202794952919858296833950970011286999885\
21093937243658265052472736405717085092351797683128462326437777950122822229\
77456333728917141861896164335688669414354145971623, 0.00915216521945596765\
61181601013399458769170657349221112064709707944411300321240788404849515351\
13497027371997460182067064736987909208987311064061742532786630167673091010\
445908231936506268843350947484315690922459127502310902, 0.0092182703704532\
22814131730904057304539706680460480228759611571025182107729617567564843423\
33289160996374000064109883141478289208457796613146897110738765291517525914\
1363446823122212127033810284697611820951299137748747555374, 0.009280843809\
72673494027358990950358866838127819637975609923496495132921766089993626821\
03881357101516320648688159584239199942351843128491763542964927704944276997\
04205335550848372886142403831272593758043870079847656155074553, 0.00933986\
15640859897571801368251202059155962551704285218943626041934548582120889434\
96365171061569005123445379643299958328290293783756819628908763635800586996\
085848048370423095703201681159451511991864603454350739935217987835, 0.0093\
95301022597583758825102567434439503496952086309634063631719232840524733856\
58478017933987381977644331733702998252664399545483640602162657192655411123\
4421832947228295646701411003318371220296284172694893667220914852845648, 0.\
00944714094524794211655166608608683279505383285946471355966372819670366856\
43298572884253848824390730593845250470889474139039762063718270532200167917\
18040603683072589329364798954382570635548850728537343034804324463016812707\
, 0.0094953614710808089006210616773316221512796802514149063593405558291072\
23783549455544739305904855447560912807801298496867420555380005846870428939\
41738206576975578695865350008988470146601883976125639265716068226586711825\
1926, 0.009539944125806391937595286663551788093180912810435041543796716241\
66279534453024775376992093607501455344229828701146261135258998656010555771\
21087385044284575286256440141277712570179444949805232531789985951975386024\
45655446, 0.00958087182887924704948649785012709232910701781871508787497709\
71916903290863373619385685494268700565762390605088449774333689135447267000\
29921864131094256480628010012246335496608637010238765192365072399586430634\
670005363394, 0.0096181289000421899623051900395131547436755540872586079077\
24213514766916577837897017586640040707970073425407214153260735904047074653\
30930631781324214278465487074038862464432606154971231543804713113625918113\
1273818708466108, 0.009651701065333728751579059331162038092790700877743070\
81979036055875994329301676187829164336449649123580225439809172202779379179\
60729155876453727362911803001347084474400147634840016189718841317206505234\
18969770714842985922, 0.00968157546255671523232904786132882216832582178389\
07225913904874850801714271976942723848599240675795606105641816592801201209\
15339867317378625342124472604788345639586274381863056368946887042497922275\
540435969734161836651779, 0.0097077406462061201222260938288029916703673979\
05953093923409913136330475358414751091322003624422284573578267032632703758\
88766179499244164950765339616404212955467269010786099660657587825056490625\
0555609400586550866127348671, 0.009730186591854044030206133387232459041490\
91468758268127357301404032313743891922327176030525676546052433589517596133\
04241374860894797710137601685779598489094602058726830932141847675670880678\
23646359187983386345491564910419, 0.00974890469999028426936658805298103836\
07013379618954826576300827441399271782959068280272766091673242822310488800\
03167510333944381674554982622379623824314384596714943634963468335669675673\
992956972773871357700893060756979194, 0.0097638877993169860828924711499775\
14798370697598707068868249291316316494166607030367021305725474703689989622\
34473197867551155540485793027482592225759079020677125043128430569610958262\
3525364781967462724875594255981529091609, 0.009775130149496116025197242361\
19464247167515624477053442279641736272216937295336684180989661641792820971\
06512915904041110523607500977049315020370696629080812469289847576195963545\
49341584659102486829952965692712910376267364, 0.00978262744334870487732306\
23427780423660917017230611790797030858025867403899032011015179740747041786\
72339762176430565738732478686225487153985635211779890777927946698400601359\
602768401789909839478151563133699818102289514405, 0.0097863768085050175156\
47164266494264443030810154904696110999496439374374308734689565548417510465\
48874699273800992078183336649268291286521238592526106925043966666923820506\
2209771231996621144987901902292053055145144820125484, 0.009786376808505017\
51564716426649426444303081015490469611099949643937437430873468956554841751\
04654887469927380099207818333664926829128652123859252610692504396666692382\
05062209771231996621144987901902292053055145144820125484, 0.00978262744334\
87048773230623427780423660917017230611790797030858025867403899032011015179\
74074704178672339762176430565738732478686225487153985635211779890777927946\
698400601359602768401789909839478151563133699818102289514405, 0.0097751301\
49496116025197242361194642471675156244770534422796417362722169372953366841\
80989661641792820971065129159040411105236075009770493150203706966290808124\
6928984757619596354549341584659102486829952965692712910376267364, 0.009763\
88779931698608289247114997751479837069759870706886824929131631649416660703\
03670213057254747036899896223447319786755115554048579302748259222575907902\
06771250431284305696109582623525364781967462724875594255981529091609, 0.00\
97489046999902842693665880529810383607013379618954826576300827441399271782\
95906828027276609167324282231048880003167510333944381674554982622379623824\
314384596714943634963468335669675673992956972773871357700893060756979194, \
0.009730186591854044030206133387232459041490914687582681273573014040323137\
43891922327176030525676546052433589517596133042413748608947977101376016857\
79598489094602058726830932141847675670880678236463591879833863454915649104\
19, 0.00970774064620612012222609382880299167036739790595309392340991313633\
04753584147510913220036244222845735782670326327037588876617949924416495076\
53396164042129554672690107860996606575878250564906250555609400586550866127\
348671, 0.0096815754625567152323290478613288221683258217838907225913904874\
85080171427197694272384859924067579560610564181659280120120915339867317378\
62534212447260478834563958627438186305636894688704249792227554043596973416\
1836651779, 0.009651701065333728751579059331162038092790700877743070819790\
36055875994329301676187829164336449649123580225439809172202779379179607291\
55876453727362911803001347084474400147634840016189718841317206505234189697\
70714842985922, 0.00961812890004218996230519003951315474367555408725860790\
77242135147669165778378970175866400407079700734254072141532607359040470746\
53309306317813242142784654870740388624644326061549712315438047131136259181\
131273818708466108, 0.0095808718288792470494864978501270923291070178187150\
87874977097191690329086337361938568549426870056576239060508844977433368913\
54472670002992186413109425648062801001224633549660863701023876519236507239\
9586430634670005363394, 0.009539944125806391937595286663551788093180912810\
43504154379671624166279534453024775376992093607501455344229828701146261135\
25899865601055577121087385044284575286256440141277712570179444949805232531\
78998595197538602445655446, 0.00949536147108080890062106167733162215127968\
02514149063593405558291072237835494555447393059048554475609128078012984968\
67420555380005846870428939417382065769755786958653500089884701466018839761\
256392657160682265867118251926, 0.0094471409452479421165516660860868327950\
53832859464713559663728196703668564329857288425384882439073059384525047088\
94741390397620637182705322001679171804060368307258932936479895438257063554\
8850728537343034804324463016812707, 0.009395301022597583758825102567434439\
50349695208630963406363171923284052473385658478017933987381977644331733702\
99825266439954548364060216265719265541112344218329472282956467014110033183\
71220296284172694893667220914852845648, 0.00933986156408598975718013682512\
02059155962551704285218943626041934548582120889434963651710615690051234453\
79643299958328290293783756819628908763635800586996085848048370423095703201\
681159451511991864603454350739935217987835, 0.0092808438097267349402735899\
09503588668381278196379756099234964951329217660899936268210388135710151632\
06486881595842391999423518431284917635429649277049442769970420533555084837\
2886142403831272593758043870079847656155074553, 0.009218270370453222814131\
73090405730453970668046048022875961157102518210772961756756484342333289160\
99637400006410988314147828920845779661314689711073876529151752591413634468\
23122212127033810284697611820951299137748747555374, 0.00915216521945596765\
61181601013399458769170657349221112064709707944411300321240788404849515351\
13497027371997460182067064736987909208987311064061742532786630167673091010\
445908231936506268843350947484315690922459127502310902, 0.0090825536829979\
67836243513929991107972111177565664316914142027949529198582968339509700112\
86999885210939372436582650524727364057170850923517976831284623264377779501\
2282222977456333728917141861896164335688669414354145971623, 0.009009462430\
71168923941907331385036396525846031985409779073192172994600838332135249282\
90073657070143871510182324141473949014773175798292917562697910940859021881\
00170272386007064052449201956307910649853285543439164386096569, 0.00893291\
94653813762773043807898368841681761398765342840426252277208460057821183910\
43639821915143789336305955086576908908303456527020566862782383231210002362\
316740415979318959754674373748425392748173540281865693313206659463, 0.0088\
52954112214605170927715687112613796142910464459350282622724518394934947248\
02247071098897830495244850566017320687255862965791434888143794454635416659\
1790369406078892535722078196227632010093713954310184254882170974564845, 0.\
00876959700760718988009229101016752038360191619172913760319605974001318059\
14034162055189027799093421676699766984629992680115903220336284485910086334\
03799241799877662913576282879978114585284270646091765323103771658555029984\
, 0.0086828800874057451782066985638245216636694057502936357286805213056623\
69554686319597368699836217016005119307101002711059970194120234351332797431\
73786205694490095616805350098183865405984485357941586026535804207682481738\
2163, 0.008592836574672403847795101254835180504726540744464728197374961325\
27548645712347844428814996604254374191518060542855389363215816819090379632\
58097500726135595348163351157613625644743389109994394379882501348236978422\
49310721, 0.00849950096695637572952157010622343592597073897089698738343810\
73678004527387576023919312166342901223326934054831804023832079590080323864\
27851533661298657981362012916197944584340414735831038469130387094454583334\
698453537037, 0.0084029090230772253239112331424977308485025562615505849548\
54433400441102868567275572007188161998209472922661371214933337318332358938\
97156715524207512986180051994278351142732180684956621525199290412588466555\
8309547800670392, 0.008303097749424931748789535610816689198622885608045757\
56315326805608648014456613426552986124843930082235869940209464075135908907\
95643764670342274516338575146215203038414407578924140838386959155627927325\
63958244581614545703, 0.00820010538578198002651556329162920997868138337512\
12569934123552954569725285040327228421740540863177098368685206720989195465\
95431357465510865397021500911622230782618863038251410393367610646457497766\
531995444602020169957505, 0.0080939713906729158441966890983141094214525186\
52798648763319644867512825427839441797637990835193476989498961150768430677\
41283075156817257761380540436310619340665845109126987320559860364272375113\
7502922911930070681773278636, 0.007984736426246977029331392501445048903900\
38283893452552004165705648107832953132062513477610914675025190116326249260\
29053711442442138441596653362755150774969884808559206252249612817166449365\
46332922392491507542898538144192, 0.00787244234269959394624815654614650751\
11968864020193995516231188912522550270820773449985742162260048270883823669\
02408371923267026136244544216553698105108902731123311841534370116344800574\
192628527995896238677745959234763168, 0.0077571321622387277804429046376598\
31295568936632973336568962440932225392798806495936569674104795834466674660\
15149981357006183815347505200796090325687383339263198460968540035893109620\
0038407195987848799702227558014872925130, 0.007638850062602190175537166614\
69488350978045618277338636341558830934656829365953763361374918070874529645\
15791785499476631074680700649841827194387864871634259722060182634818272558\
06301700213592069138015258141134771629542495, 0.00751764136013225986044290\
56799602346234717101392247774833304854323004715302654555178424028348138465\
06333750854254907973042557125111832710860034826247321276531896675026418396\
162598386492976723147914377730675221629468964110, 0.0073935524924140816945\
60542188348896638597657860041161127881363271919569019193920168187184203023\
90118663576005645104675611946906334665700362243379068729746517584140332859\
5871338384186692380419314114856223343429446463351744, 0.007266631000484500\
91201286142354444043841808534979282159710102351623396786126017352830326146\
46280600641604233048461361557610159879160868547647633984379821497640002289\
08393028638434341590704493685329876674544530717832675651, 0.00713692551061\
81502118953135667788046400767764230530063019525964960301714782835103919443\
62646637972330884764370487613819488467509605677949372456181495239072119067\
171762723274541051592907614442857548846701691107698861185807, 0.0070044857\
15697769675680027452258467244360917680171905578589120711534609524724336986\
22315924730299235338139398694118962033866624864817809116080612261921455553\
1126227055250281208316433807490817979467874194286509836693988347, 0.006869\
36235617589925779392044151384418693255012497019305899928854709292574939632\
88052394701666389248451957137550795267634026745414761596568541712892575933\
59781698382014493863997743124283955830800888055166589907081274817698, 0.00\
67316072006352407630239491586123017452485400000854117355472118137096637630\
43800249983360622218270129696633879430710801247585340451684568874891973888\
264390642589083966043931200915499565892551337421261659726667606033542876, \
0.006591273025955140779519089485118565385889363340527925195346770473394275\
51422708430081738590403851657801171923191104146973719931450740881271338846\
99977307491735301336967466216191789251723878650520265480443893300633730639\
63, 0.00644841359709179798078228807369620314224387518000466703096202393932\
07923148289095506710186551410843048722812198538823607986486053420479640150\
72556347664119424845888491608057471596645177637660157493317710224388437673\
100130, 0.0063030836464799475700264584428503161068759112420464307181961825\
72641050094562094563791963454228997830406441988716293337776583776195563647\
25227165305971511868807373728994378696030661323612325525268547449305746618\
9912060365, 0.006155338853063922474707460205369729644114143773454358181070\
96764990386406644333579227863278830080836771823249031693719312030855499895\
30560723565133760089716352121798631424455501771042242524856744208857404174\
17763751828174, 0.00600523582096613531401077149971596657671028934295069524\
41413932600495637848586261861532584043135691994269097678632026528569314532\
15332374066188215892985574633169915666431053431291215024258647540328386916\
129839203624675860, 0.0058528320578011673307512818472094409990137382165864\
51160919681412011237416769003663179540702131580249286522435247097135185150\
97440294744216898278276159419656124972625342681240851173901445636005222271\
2907828154582814357115, 0.005698185952643790671816524284353061373897410941\
86515294064620165291833385143079097815997494434990387695616946409497774226\
30737326831008642473571225200008825196274708004094338892380419090016648080\
06893035614534723981257537, 0.00554135675365938903026370071167506304896460\
00469282897366390908538473763605840999678282368105938877585838031306599761\
88994039947694355193426156078389690784394376725230468236357892225554213951\
784329348663883725881309503228, 0.0053824045454053793501744490687525361661\
58020664844735405656809392356086073099956763570905122521844931143265095661\
58933546218030412031506031503161893540475982132702764795506335072733529526\
7761001936090846668502153516867281, 0.005221390225812374980555463259915898\
74051687154020378293744266765109018701189785148437293176036529316515822543\
18600616279731173641186261209790946783426520341854261112919863681117790603\
61811774205700851595369237003615320525, 0.00505837548285396976431845227902\
88370885316592194284256444139484497242958213431509997214696756846695249340\
80906007968568821838991682254602565892769776714839493407207653763423276668\
837056500835753546874170995731709525618557, 0.0048934227709141652178307056\
80360456960802405790244986873632532139415050415132045365993648048870075885\
54735332925594984886777656906638877055661167719946302877418377798029809325\
5374327329288193935203703504041241186744596341, 0.004726595286861612503052\
85450723503712762241508147825423556935219802760985583746362652048706494982\
34252532923173587914266140524293119761374734040384866725757970085651401666\
35704010707887752545926085169251980833856527274190, 0.00455795694584000245\
20198080390797889708988471571329556741347738593844371031256710769156824559\
38341773605551068856305806214635082048090592789621931695104328854898137200\
378872981922363365011144410034211906577721787560005209, 0.0043875723567841\
18549801234495919526734841078234754666522916232652077892879906172745996386\
71005068141349432505159331202047868066227705581370587277343570749962528641\
7509456633477570223552746668653382238201843256135717093652, 0.004215506797\
67128241890081985984392606360833643792962312743895905947883814909537392261\
54016110771712996803634602767873765378580153498727616162484974826223642039\
66969096154617677134148449877528131953470085476550016693510176, 0.00404182\
61905181899320980340775409047859034067342841346059137581833025194709023058\
76692543820014221260398028532165333740707760787846265388384041590886918572\
101140346158666301985234332259613346088402233208485863681673285767, 0.0038\
66597076133493287710928674908786307278336638138194688995437969984944519479\
37100732331523602210755082674584220865044637371845214392344994532618203809\
4250279009396311773778553482176038470182122266903666749352393520524282, 0.\
00368988658863698875997205085806931209484611479232732855185153022527842212\
40581659448492511758615936604016669049036451513840256502515478655549695239\
91533816072679743540996545955571638178648745947591629371848474283801605400\
, 0.0035117624297570224595577869505735074298714782878714769020694990972237\
76706520026427139610006591275075587206872542503834966469823266055218002567\
55641194555244858163004565785635921005306366716373453655165539558592229310\
9807, 0.003332292842918906068764339658385913441084763324498579837893414843\
89981934976680309247028742429452876981401562554848310759706613901526962012\
05833729024643512989649635217718932696715749341748104069756972746337504238\
22395955, 0.00315154658713906586248143215982982768162569950774371307508285\
64811885784600471952954748059229260237634101573387303905851860806694748760\
40727475140098550915006545663563787802126223358495971442722371188510646432\
147654559581, 0.0029695929107429387715700349900879411794924201692150533016\
25576933355393549595324469318896576106731768945756710525620335285545838980\
93823684483093836417159406472686909470257530313948039856419585005564274187\
6422395475464069, 0.002786501524930452446756963319259732260881614018047653\
73972185962016535693868983220546919551352430957733758662893182211911367654\
19543016890848484929564850177213143568611744190880407437795029752642591596\
06385635197099494022, 0.00260234257722363366513969679501158108892379005338\
91462848586394209268759289974114845962282927231275886741371150342976324814\
13964146613075015743731809574807707067538549783283874914141871348219347719\
255471862701257518915890, 0.0024171866248513869085823465878788926048636552\
64740242384954741369549801909446881386761676215632190236160684791534252845\
46914656248920726011404007583795381584150078500760630065260904654508085503\
0429103054891061930868952890, 0.002231104608167486976347492561942761891616\
77449214330132304373878490377314790198197699545177883302735775063382844013\
55686465477003449353538410450143552576635272566167505182371702226371305242\
56926360996651709724449927251086, 0.00204416782428398633890672991448404601\
17775789459411710862757583239200399930973816616364918192203899866719266680\
97925773766166829135502942997822278264145320927724354483225479847145455499\
272184295046760972806119721231098494, 0.0018564479012937578523768986034253\
33533703200761795667627883492419837841343081869097825345375580993334077692\
46712663050473883814344564102926071444780257115418237414611430785735414104\
5574931228310928588064728104910175556740, 0.001668016773910433107000957173\
67510713953670048946808270706707310464811831801593445766591412346344911522\
46452277424300498606323111313146001558899098725357210563334408515890014692\
55263957062816133657048527220747218462019169, 0.00147894666251970828746854\
81849089432192076740631683558108944050976173430511561535509461409792021221\
23443978834469696769383126750852816035941336584742859393944884259570575727\
208466559939903427597619240003942673910188566460, 0.0012893100609187659613\
42882781988929613354239057246795551238023074267265431112017499713289027715\
83877827147607383366643100878789525695085316553851365419597287627631056641\
1774132188167362171074418020437809639851496770241078, 0.001099179748398244\
37242807349344240684676853875591789983529751333466477868526509711601397500\
79206139321203065531321096949901604164512894542928803766441526948515283685\
21978520278677701000873658379131059278329730270578151251, 0.00090862887983\
25920045444007863047753695915237007754366566495421033059267416810356720969\
48691185500381017241715182161587952417453802638425167971547990570392981915\
1537151745176362449526705761913658251904875364285599978182132, 0.000717731\
37674159643149088714093905170920006061126765878341597138012140284511429115\
70285774836714157343760932758412921876329720689446773186562004586585633226\
930118274349267150963666416217698974635538394885553314361223554709, 0.0005\
26563836106867744115459492832605693256339475650422180789825775138119206499\
02981574635213803769011541611348652114928401553113114427237589584367657353\
56346544513372656673382227897334193642653468262334026526305531728548627, 0\
.0003352191600761516407376531393778775494007066687434425232876645133009397\
45976742623787862965936605539762183578690524987458507527251755691455133960\
99872772413554226035271316051149069230710031513047598311030323323140314528\
55, 0.00014402926426054152232724969892699182341711614747499179071065241585\
41757788881977789600690354747134755230562415371078371615637678690532986132\
34110990426063810553556085882220624484327729667512140372939632892652154937\
3972977], [0.0000561238639183805706846684887361865236220436383575597738894\
97450658137120857495698678996613336383370632529508463006876343107662297048\
61932601450010356462767687035998827115308531708174277614913120514231725545\
984824562275534, 0.0002956896793470990565012721884309851074712454621607073\
37457996834188427225142513782815940665259638111948045422924589285958993533\
38176749952783249149834182768776366904948461668533339198480569150818806963\
39796624302116316257, 0.00072659062424164792167432475854764433199594086908\
04130680484924370459674049522551678875751277828601930426788870514877684654\
85789318890268998930614193140759970580105016891809372432559808962151775434\
7630634629859385767883206, 0.001348757361360237708882206882977496710350300\
83682536565349103436785321218542041299268429696181728722028028367672039397\
83798613643849638190656795974548568318121913979530086667431210044523374776\
20322505658371141890071645977, 0.00216196325598743371749985617132144091996\
69845838967090340396873347103107049723257176398094624059787703982624324668\
89678603398078794456313816368088296494362564485695757043647129560108069429\
385725770224196540102661613567057, 0.0031658995539515856824333483172454437\
89900113057006080217457931563339892331642864623294392781247444984363647311\
26710045967729342488145896986448468389328958522495113809016390408619151898\
3938946159296797639077468554130121010, 0.004360182560176761305763395074832\
86748737367473205539785496083741944624536386788429828794187852592694960680\
38290056905799068873260621266485782551331874553284283262166626648089787976\
56912157600596612900822447608703011350322, 0.00574435510159639835154462289\
25853998421859116547295089336317919542486621582306311999355993609931359505\
87148542787692032364950312880475176219541344138762935301981480291098078800\
261566389927960087650344783886672365194935165, 0.0073178870520548279892052\
52956636814680110545634086693066243830599390200872600189724583772404857187\
71005128403071102056191677163597915595050979388604915003245246501885246754\
4278425236070757180855617390789132462110325735522, 0.009080175651736873313\
72764485183743740847965948783672188661006170085016307074434301906009433723\
17018668189041305591370966260338554244800760128796721988862183025103190908\
19120324792783360869460419749940507197153088946175175, 0.01103054578334704\
56662164643240350102932058553627557695170413877537688871723008505684031651\
21371847524795900893729074916391397507745440505080309556801655039511687316\
36730169152475825309896427236469593950010115022622642210, 0.01316825025053\
30122356475266603633532500914543599058874469242318316736596963209224069503\
40990976015632216611911031840862338892698804656680677489020424492358893549\
80469405553328764997429943417864329224038950093682473171623, 0.01549247007\
36062642738684507364438860554359931257034480585131826006066950267834215936\
25533942416555173389730743799385380154018386962670639125301096599260046336\
70437321108481203262591242242254855683703673483260339752267799, 0.01800231\
48081686204711209082991251931238638554199459686023975677169963022595495689\
26109787901579232026376331335981633589122011296377429563509339030615372897\
64408749934814714135894309609167991465039697072071488495002847830, 0.02069\
68228889083632825095837000041958253571276896040433826418726633253515992174\
17773744665576712290561028438141267492284691931462646627627173073647136460\
26409189736172129602152968400288615946069835700056667179230849082057, 0.02\
35749619995124774922842800315261993364281458242908576263067197116200380185\
78220454526595536058865739634786298833232104777509766877260743989965354775\
93637426621652187038705496330743995942805401937114325623275168067570554, 0\
.0266356294690740070836844669140472958433739721110160743146916409645422497\
46696130040671680736617702172598520755083787328266885618991527865853407850\
03619150718159950520282501968449667987962739759734897949625947916267797221\
, 0.0298776526951086372852174899035976103281101233173231699827296852892312\
21288103799766404398859028340662818267130011973803846769954928514543988775\
53208649286633426869678133993482183180403708720888173916514616193301973379\
893, 0.0332997895931609408716823721179947579788457766746066876619261775479\
97071903871755985217536948570781587988504291768159476921003304166221754869\
59899436798357248731143711646950707056295128711113399770003378923610431019\
851245, 0.0369007290729076077669545681224717986469948750742549869428064339\
11956167551899593453067069586830152172544208197806659864807731266844836261\
55079205918664399039697986701856756005281974290337539466488477437083600494\
232266281, 0.0406790915406213183648336778543028797205678114637653251065140\
94285312120292198494686375795932305152622808459309075397428850113451375432\
80554974236335960925351976854348056409774475805734178900947164098284696377\
083056985746, 0.0446334294278302865692100090099016672324046020064607598475\
42599413415498684737393739760602026614709590482911612030895640586332210486\
95889657608551872350330514581783602290957842182336711812735108617938235689\
539626562606010, 0.0487622277459878572068077716664250557351303976495641394\
43900072287628743077001821683865318441808040615891979885379015249696905974\
36645852345492733939575193687499887549784165889279094749212100917820372078\
420816347393868235, 0.0530639046669503242375571766367041682135982365419937\
46263097741968107171495278159007214578171909998244710231375602111128022721\
41235848372977289957329240537517194979945889610797407567871975332962780933\
413715974190941516121, 0.0575368121290474606633143004934950936238580619551\
05821356688366086026344534403057728601356770208680782895957831442254707382\
06397488652742914818378954801792240155178549583192352711379525149925759828\
764377235665469814381007, 0.0621792364685180950231755866280113948457128914\
23715708497943780253852964523421104459144312604092547081331832735390477040\
17435416315510911286788872382893641254761641514033724187655572970008210782\
894605018883623636249080356, 0.0669893990760718599090644403695039672844825\
20560886979376504739882518029019802003434120843648305899931073349968710463\
24799637329849017524737497979771319029624633504876877841547187532829288068\
763263675193014789998388761799, 0.0719654570783276460394458394264067197752\
03671415567889625639497035758653249312453335605307690986993913361519282059\
98410765049865491581688562860881906654822684636832409197491093063905381681\
241489999388386940872170079781701, 0.0771055040438691280518302437937329517\
19760158360124562415891718369490915729311806039502614323856255558523517550\
11969847028028670054783996218207286495082331740089425446631041201604719488\
623586164383070905627720298287419702, 0.0824075707136478700116962184008539\
66088742881539900669002509175143507863626760721941106352576840952801141700\
10072752581121058083074635940438737968466697885490838936130601257393022739\
893908724521870750211828551801329822345, 0.0878696257554548998770216872303\
63575028892980170646596619075241666987693253836946891079779992336305621272\
17620291827963264896650668391635645883486880816236440130517730961592274234\
633808006565473804869865133867441510729281, 0.0934895765421722201934289541\
76149378872975345283692226073180292491599409898058886999445517847843493810\
24313699317906753574055905879113599533402739212835962487603893737135911319\
159082291730273507862679924201443066574788755, 0.0992652699535064710953205\
51598700261999801443131179600912600128149615733004758132502312805071756608\
48143046942828251653625004636871987268198196258215046781631499216077262360\
429235142651387812922609586018419352108624980927, 0.1051944932008978655119\
81075760986275306303032774142078658354748893595353557536186568890300749718\
94564884349058719162133540624213966831954420526010473507351459576915933553\
91186022428874356370356488544861574487354617515564, 0.11127497467528856607\
04235823862768151336524411399788892607874430957308709695848928569042239500\
13103463109502876896804223637225249015356680063755772756270855992613465204\
0557284973171745484976893918870937866902443230728099, 0.117504384817425863\
44445747398606718786073697729202326388572350527961594357978618837450733073\
40901571573075588641775600857432319795905564230270170367188569339831722328\
430979226910169981983601942814802028232843131757322198, 0.1238803370103668\
44405659546861093817084680249367991779731651240347393563747479402846567308\
71963587692573993483191532469962835316901728673222626953337250397572799063\
15671621925110414219953612872262068478253075501606381706, 0.13040038849384\
27039296573474251149349268903053837590841756241896089765714775288870474657\
25573928506431432766239168403869167958526228755183790137544099728320213338\
0856490686040013844901578843691200125790492945339702199891, 0.137062041300\
13245990427840544490327021888643473593599609331056972065499434108238072525\
66374735147319722259510725245052557907048508634805627664486419290510736470\
669323938743684475812987772318367849817375912909347221361173, 0.1438627432\
11087572548234750893363165003455644357849463395020112140056642616636113975\
49943213265017036869610995965122983329137571733364075940934124139507959017\
97229350645329459739923592884299038506587663591708307107010429, 0.15079988\
87359408553641964786713517437162915320716166766226363578498553629622278076\
04238547400050307792414724802332148204335054594186938723422482017002006436\
9403933229509343512972436558220618362089988146792005563302445206, 0.157870\
82010952509243919003237324034392687104052542063917838285978805234475440402\
23414996298104488615385028708249042726302905486836566520098669513199714893\
401836963188046231906567084753618380728431928046772130348325839994, 0.1650\
72828310518950507774073957030140838330654671103012294631948602018707101092\
36221199783738453288538720297841149575590628322346760374092802354220551831\
92889690162098466230742179582821485407524931756328779184473819152852, 0.17\
24031540993300958865413357114240584549988913131566944124660157530694293799\
74157678128338601101145984957050724111423970400741929457568470409400738796\
4943908236946283366004916488449123149881873673429471201028060591094463, 0.\
17985898907521789873052436794241471127628237882583254659962754205791659531\
92913643355106141659526580597182421208296646890822414356149622740292107253\
463000690124373049930452089985948164444737395014736170954827220998590423, \
0.187437476752250732652939717436068968909528469397324578676328970246703753\
30614245548941203233781004761315874382072890436551187978754910603281491282\
43201909469939108844187691062012801093314119124616482165204672815536129973\
, 0.1951357136536856592025706346066761219789233527345801887145021395516411\
97368398359346088132086347805468998812074986210855304537475786005968238196\
21866950762954696109028292619584309146665746972754759424776880699285285352\
94, 0.20295075042435122661407170798047098604134438790306633759340315897909\
23950301122262611088245177260605854804426672569407531563901973022161535336\
73190023108296760445831615303519086127683007450138621218076432190782743168\
0337, 0.210879592960607213219749552779606091488294586146877648822177595736\
32822841585157565350227383239743623268284268361341811026387068669154479231\
81577030037839318861131655572670923483508826799993876077119258148451439578\
918155, 0.2189192035574484104881610987510129137998905389325874861807758127\
25933927740960143833538548529710275062715758400318329387150087024677395810\
41194087294739873524868126903689679901500058757946631093253089699628504459\
20106013, 0.22706650207231297134520498525366471211014122388704331929856245\
95672884097627277027353657279894459625711949653209814718105204105798454167\
45767983697877660132089966308979749179568676487426453103822414116671457065\
6720064732, 0.235318367105149448699796796794101767408659786351165919288661\
88370764468411234628108723792917494452174192073408101359662841104799653217\
07296243156616242663916465570151546810810536913226181330856415609512486169\
436382773624, 0.2436716371942904193488772628104483595979135560333636346450\
31073036117368451931506808930531435464918495568049269622920204010435878022\
80875289624217471886897676705046931591461830539932663291581993233242506832\
47110905008672, 0.25212311202767453202934904537674110419918971837771133876\
32453155675763759786629185753420995997654653336758398595750366483796412190\
12448523253763275186811763627283607379749368410014036836753225259242107294\
8256890525576383, 0.260669553668952937612216515547993955440919315722674247\
94456609423142335645307460108645619107623626468965215928925235335947890715\
05286796909792462756047388042182979293368062472798420489244097110797548009\
840165838043903342, 0.2693076877980103565295383813910261914624272312661438\
01205621126872011426416013183403633268890484316805818861525957437848749843\
87193513065658724130273940475793715283312029134380292621924475594078262174\
49647591273848808460, 0.27803420496542551565670484082035946989922528012144\
57286408318044568264801747695426492961994412892568621046330214294675170033\
01426432099378612835951496593401457766599162168594407882226340411304499593\
4144590151020012711802, 0.286845761860390346144240909153963488869820538045\
74760161297111626598373431874962749923888289350992518266648610910413284329\
21753932621969795195281478685590691118989573045155469907305281146868148139\
516289840356166732845211, 0.2957389825916021771408239937586232556920097414\
50281180032999604409279285098173967853483384561459527203188140436540674559\
70784225107535406945737007066001582712302686090032620363089742725966149757\
38571920947205096060586182, 0.30471045998063818993994844857670322542178997\
24972778365168679360141103325827236026408735398489236088844885670586132135\
44131472510542936462883909700270714082759838236552679469781133835093208869\
2819701510546432432232109203, 0.313756756867316614714407530261309288486424\
54792940548345618232042323210080320576444235502020454470286326302168797424\
31620180472604776094283374137732184748948511472312110743344743340073317283\
812770883444474004666984965740, 0.3228744074265445595024437016857806977075\
44441481416765689049099451537264880792873413470044755267400126161921430775\
68011998891957149716686289649522017116997664867784639735315091252809469284\
54201166610417999172848808723869, 0.33205991849614796023224868187397558574\
08841683782353239666650402817997050618181702103592822492713520170019194812\
89860640498233153055075575027433847684296257620213437647285273245185808151\
0164569592525602601461998895447331, 0.341309770915174933000065254003042315\
91247282613069936191292653195811559095746304623860262711204988708316017386\
97402969792100793578323968854314017870145319326638688674737219550664322387\
902000553726338238243416659467379653, 0.3506204208721597971606238709581331\
36768106908999115478304396046597895968994979995877851418485261371028848256\
61092168796426528355489565082419738135013371001860738211854629088896039146\
11485656846648544266417876244733084466, 0.35998830126283122158204729549425\
78003420498444563685123390826134265158821954621471643258100643745853029881\
20178127928414349845605856409137918720240882577912948223158588381799010620\
1795298940538781722155818886828436527753, 0.369409823056744328120836026816\
93662951533114599895130985317087446370934205515833857119374713259184824750\
72000606167459220307300037181654407223819597482479683975595815592406606633\
547809420920977504807937327080383645066089, 0.3788813766723131673707795146\
32513680452845033014070685870502651066898334128313508525755049731566975981\
36520354919185973763892097228154291083611467360405754840027255453308495918\
73797094403492494738415684528685836664036627, 0.38839933335971676334121655\
09658929311027511836211991521992932063156856610767033764251003145400298739\
57900764545015398552719176337287919338350853353869378148536621195474149411\
1473409131265937134858478564971090930742884152, 0.397960046591148907156954\
55265821218805660112738159647500611056470028742409194197774430317033650443\
27199783975628706007506362622551592669474366999182183245573280699591402834\
893009252803407325480866531131947129527343411757, 0.4075598534578790662991\
69550057118999945310509681956039082262316217701542682450075138352749499470\
68480578328269829992704425983934042625803716052944524601525174063116139441\
40603068763371976457226509085751692665389318103260, 0.41719507607358916640\
09455627547263341049477742813871152676886865618046855101303000159251622564\
64604848017342100613781287867845745648547176852625114357742828985087505513\
1358696259525603884402916527213729712570569667654290, 0.426862022983448598\
17191487534652160405795439986219508121849706759821478328922684991106669269\
40370130558895586307107103817697840293318364972880187717753465006590751959\
231149852865267671554897653904644556817830206926072630, 0.4365569905783876\
03574408954258384994021731174354559655865924029005993305530185612751774464\
55126647340855118376195912655111118176628983349224225309120255401994394959\
07919802475654613164588033779622003623185026136700063660, 0.44627626451402\
72037504711186440272214009020713432648460272788525085189042462489449187942\
25885188661455251336396405912589481984971034114967404584405200335749164864\
8096937498636252307146629749298240907427763034763055088122, 0.456016121133\
72204716769621081511920061922184477571668598835924546796663829787784103638\
02178685996170317425361612058786179137265723089033764159361792216779283000\
319652997878080972785875133806526475023235400392113098586349, 0.4657728288\
95170980695357413409187907165907996131038202458370127447420272968549090105\
71605944637028111799205088346245087789169166562071121630027602162927262890\
59913110860441337783050262344833728355771611292214296441446179, 0.47554264\
98000487794441215624702440224973480722654691477881777818588992576915786991\
47384687891674455047901481541436585320595742950314465186133036284652158634\
8764361056086943737356330223114822739208429092100582361554178525, 0.485321\
84082611131372632631980192864784998174525189294681400825674326044041187771\
86013848174666735948604199891210935629107596790610544564238968302843286320\
497197148896643785137356942176524153038085439456136095477416007029, 0.4951\
06655361225483862592466187928725572042856701280142506615120187105610275984\
67854271888456755278587496181267825574579296904142765819565661144724103919\
33525757648594407696442463389878537972652771128388492855250942471659, 0.50\
48933446387745161374075338120712744279571432987198574933848798128943897240\
15321457281115432447214125038187321744254207030958572341804343388552758960\
8066474242351405592303557536610121462027347228871611507144749057528340, 0.\
51467815917388868627367368019807135215001825474810705318599174325673955958\
81222813986151825333264051395800108789064370892403209389455435761031697156\
713679502802851103356214862643057823475846961914560543863904522583992970, \
0.524457350199951220555878437529755977502651927734530852211822218141100742\
30842130085261531210832554495209851845856341467940425704968553481386696371\
53478413651235638943913056262643669776885177260791570907899417638445821474\
, 0.5342271711048290193046425865908120928340920038689617975416298725525797\
27031450909894283940553629718882007949116537549122108308334379288783699723\
97837072737109400868891395586622169497376551662716442283887077857035585538\
20, 0.54398387886627795283230378918488079938077815522428331401164075453203\
33617021221589636197821314003829682574638387941213820862734276910966235840\
63820778322071699968034700212191902721412486619347352497676459960788690141\
3650, 0.553723735485972796249528881355972778599097928656735153972721147491\
48109575375105508120577411481133854474866360359408741051801502896588503259\
54155947996642508351351903062501363747692853370250701759092572236965236944\
911877, 0.5634430094216123964255910457416150059782688256454403441340759709\
94006694469814387248225535448733526591448816238040873448888818233710166507\
75774690879744598005605040920801975243453868354119662203779963768149738632\
99936339, 0.57313797701655140182808512465347839594204560013780491878150293\
24017852167107731500889333073059629869441104413692892896182302159706681635\
02711981228224653499340924804076885014713473232844510234609535544318216979\
3073927369, 0.582804923926410833599054437245273665895052225718612884732311\
31343819531448986969998407483774353539515198265789938621871213215425435145\
28231473748856422571710149124944868641303740474396115597083472786270287429\
430332345709, 0.5924401465421209337008304499428810000546894903180439609177\
37683782298457317549924861647250500529315194216717301700072955740160659573\
74196283947055475398474825936883860558593969312366280235427734909142483073\
34610681896739, 0.60203995340885109284304544734178781194339887261840352499\
38894352997125759080580222556968296634955672800216024371293992493637377448\
40733052563300081781675442671930040859716510699074719659267451913346886805\
2870472656588242, 0.611600666640283236658783449034107068897248816378800847\
80070679368431433892329662357489968545997012604209923545498460144728082366\
27120806616491466461306218514633788045258505888526590868734062865141521435\
028909069257115847, 0.6211186233276868326292204853674863195471549669859293\
14129497348933101665871686491474244950268433024018634796450808140262361079\
02771845708916388532639594245159972744546691504081262029055965075052615843\
15471314163335963372, 0.63059017694325567187916397318306337048466885400104\
86901468291255362906579448416614288062528674081517524927999393832540779692\
69996281834559277618040251752031602440418440759339336645219057907902249519\
2062672919616354933910, 0.640011698737168778417952704505742199657950155543\
63148766091738657348411780453785283567418993562541469701187982187207158565\
01543941435908620812797591174220870517768414116182009893798204701059461218\
277844181113171563472246, 0.6493795791278402028393761290418668632318930910\
00884521695603953402104031005020004122148581514738628971151743389078312035\
73471644510434917580261864986628998139261788145370911103960853885143431533\
51455733582123755266915533, 0.65869022908482506699993474599695768408752717\
38693006380870734680418844090425369537613973728879501129168398261302597030\
20789920642167603114568598212985468067336131132526278044933567761209799944\
6273661761756583340532620346, 0.667940081503852039767751318126024414259115\
83162176467603333495971820029493818182978964071775072864798299808051871013\
93595017668469449244249725661523157037423797865623527147267548141918489835\
430407474397398538001104552669, 0.6771255925734554404975562983142193022924\
55558518583234310950900548462735119207126586529955244732599873838078569224\
31988001108042850283313710350477982883002335132215360264684908747190530715\
45798833389582000827151191276130, 0.68624324313268338528559246973869071151\
35754520705945165438176795767678991967942355576449797954552971367369783120\
25756837981952739522390571662586226781525105148852768788925665525665992668\
2716187229116555525995333015034259, 0.695289540019361810060051551423296774\
57821002750272216348313206398588966741727639735912646015107639111551143294\
13867864558685274894570635371160902997292859172401617634473205302188661649\
067911307180298489453567567767890796, 0.7042610174083978228591760062413767\
44307990258549718819967000395590720714901826032146516615438540472796811859\
56345932544029215774892464593054262992933998417287697313909967379636910257\
27403385024261428079052794903939413817, 0.71315423813960965385575909084603\
65111301794619542523983870288837340162656812503725007611171064900748173335\
13890895867156707824606737803020480471852131440930888101042695484453009269\
4718853131851860483710159643833267154788, 0.721965795034574484343295159179\
64053010077471987855427135916819554317351982523045735070380055871074313789\
53669785705324829966985735679006213871640485034065985422334008378314055921\
177736595886955004065855409848979987288197, 0.7306923122019896434704616186\
08973808537572768733856198794378873127988573583986816596366731109515683194\
18113847404256215125015612806486934341275869726059524206284716687970865619\
70737807552440592173782550352408726151191540, 0.73933044633104706238778348\
44520060445590806842773257520554339057685766435469253989135438089237637353\
10347840710747646640521092849471320309020753724395261195781702070663193752\
7201579510755902889202451990159834161956096657, 0.747876887972325467970650\
95462325889580081028162228866123675468443242362402133708142465790040023453\
46663241601404249633516203587809875514767462367248131882363727163926202506\
315899859631632467747407578927051743109474423616, 0.7563283628057095806511\
22737189551640402086443966636365354968926963882631548068493191069468564535\
08150443195073037707979598956412197719124710375782528113102323294953068408\
53816946006733670841800676675749316752889094991327, 0.76468163289485055130\
02032032058982325913402136488340807113381162923553158876537189127620708250\
55478258079265918986403371588952003467829270375684338375733608353442984845\
3189189463086773818669143584390487513830563617226375, 0.772933497927687028\
65479501474633528788985877611295668070143754043271159023727229726463427201\
05540374288050346790185281894795894201545832542320163021223398679100336910\
202508204313235125735468961775858833285429343279935267, 0.7810807964425515\
89511838901248987086200109461067412513819224187274066072259039856166461451\
47028972493728424159968167061284991297532260418958805912705260126475131873\
09631032009849994124205336890674691030037149554079893987, 0.78912040703939\
27867802504472203939085117054138531223511778224042636717715841484243464977\
26167602563767317157316386581889736129313308455207681842296996216068113886\
8344427329076516491173200006123922880741851548560421081844, 0.797049249575\
64877338592829201952901395865561209693366240659684102090760496988777373889\
11754822739394145195573327430592468436098026977838464663268099768917032395\
541683846964809138723169925498613787819235678092172568319662, 0.8048642863\
46314340797429365393323878021076647265419811285497860448358802631601640653\
91186791365219453100118792501378914469546252421399403176180378133049237045\
30389097170738041569085333425302724524057522311930071471464705, 0.81256252\
32477492673470602825639310310904715306026754213236710297532962466938575445\
10587967662189952386841256179271095634488120212450893967185087175679809053\
0060891155812308937987198906685880875383517834795327184463870026, 0.820141\
01092478210126947563205758528872371762117416745340037245794208340468070863\
56644893858340473419402817578791703353109177585643850377259707892746536999\
309875626950069547910014051835555262604985263829045172779001409576, 0.8275\
96845900669904113458664288575941545001108686843305587533984246930570620025\
84232187166139889885401504294927588857602959925807054243152959059926120350\
56091763053716633995083511550876850118126326570528798971939408905537, 0.83\
49271716894810494922259260429698591616693453288969877053680513979812928989\
07637788002162615467114612797021588504244093716776532396259071976457794481\
6807110309837901533769257820417178514592475068243671220815526180847147, 0.\
84212917989047490756080996762675965607312895947457936082161714021194765524\
55959776585003701895511384614971291750957273697094513163433479901330486800\
285106598163036811953768093432915246381619271568071953227869651674160005, \
0.849200111264059144635803521328648256283708467928383323377363642150144637\
03777219239576145259994969220758527519766785179566494540581306127657751798\
29979935630596066770490656487027563441779381637910011853207994436697554793\
, 0.8561372567889124274517652491066368349965443556421505366049798878599433\
57383363886024500567867349829631303890040348770166708624282666359240590658\
75860492040982027706493546705402600764071157009614934123364082916928929895\
70, 0.86293795869986754009572159455509672978111356526406400390668943027934\
50056589176192747433625264852680277740489274754947442092951491365194372335\
51358070948926352933067606125631552418701222768163215018262408709065277863\
8826, 0.869599611506157296070342652574885065073109694616240915824375810391\
02342852247111295253427442607149356856723376083159613083204147377124481620\
98624559002716797866619143509313959986155098421156308799874209507054660297\
800108, 0.8761196629896331555943404531389061829153197506320082202683487596\
52606436252520597153432691280364123074260065168084675300371646830982713267\
77373046662749602427200936843283780748895857800463871277379315217469244983\
93618293, 0.88249561518257413655554252601393281213926302270797673611427649\
47203840564202138116254926692659098428426924411358224399142567680204094435\
76972982963281143066016827767156902077308983001801639805718519797176715686\
8242677801, 0.888725025324711433929576417613723184866347558860021110739212\
55690426912903041510714309577604998689653689049712310319577636277475098464\
33199362442272437291440073865347959442715026828254515023106081129062133097\
556769271901, 0.8948055067991021344880189242390137246936969672258579213416\
45251106404646442463813431109699250281054351156509412808378664593757860331\
68045579473989526492648540423084066446088139775711256436296435114551384255\
12645382484435, 0.90073473004649352890467944840129973800019855686882039908\
73998718503842669952418674976871949282433915185695305717174834637499536312\
80127318018037417849532183685007839227376395707648573486121870773904139815\
8064789137501907, 0.906510423457827779806571045823850621127024654716307773\
92681970750840059010194111300055448215215650618975686300682093246425944094\
12088640046659726078716403751239610626286408868084091770826972649213732007\
579855693342521124, 0.9121303742445451001229783127696364249711070198293534\
03380924758333012306746163053108920220007663694378727823797081720367351033\
49331608364354116513119183763559869482269038407725765366191993434526195130\
13486613255848927072, 0.91759242928635212998830378159914603391125711846009\
93309974908248564921363732392780588936474231590471988582998992724741887894\
19169253640595612620315333021145091610638693987426069772601060912754781292\
4978817144819867017765, 0.922894495956130871948169756206267048280239841639\
87543758410828163050908427068819396049738567614374444147648244988030152971\
97132994521600378179271350491766825991057455336895879839528051137641383561\
692909437227970171258029, 0.9280345429216723539605541605735932802247963285\
84432110374360502964241346750687546664394692309013006086638480717940015892\
34950134508418311437139118093345177315363167590802508906936094618318758510\
00061161305912782992021830, 0.93301060092392814009093555963049603271551747\
94391130206234952601174819709801979965658791563516941000689266500312895367\
52003626701509824752625020202286809703753664951231221584528124671707119312\
3673632480698521000161123820, 0.937820763531481904976824413371988605154287\
10857628429150205621974614703547657889554085568739590745291866816726460952\
29598256458368448908871321112761710635874523835848596627581234442702999178\
921710539498111637636375091964, 0.9424631878709525393366856995065049063761\
41938044894178643311633913973655465596942271398643229791319217104042168557\
74529261793602511347257085181621045198207759844821450416807647288620474850\
07424017123562276433453018561899, 0.94693609533304967576244282336329583178\
64017634580062537369022580318928285047218409927854218280900017552897686243\
97888871977278587641516270227100426707594624828050200541103892025924321280\
2466703721906658628402580905848388, 0.951237772254012142793192228333574944\
26486960235043586055609992771237125692299817831613468155819195938410802011\
46209847503030940256335414765450726606042480631250011245021583411072090525\
078789908217962792157918365260613176, 0.9553665705721697134307899909900983\
32767595397993539240152457400586584501315262606260239397973385290409517088\
38796910435941366778951304110342391448127649669485418216397709042157817663\
28818726489138206176431046037343739399, 0.95932090845937868163516632214569\
71202794321885362346748934859057146878797078015053136242040676948473771915\
40690924602571149886548624567194450257636640390746480231456519435902255241\
9426582109905283590171530362291694301425, 0.963099270927092392233045431877\
52820135300512492574501305719356608804383244810040654693293041316984782745\
57918021933401351922687331551637384492079408133560096030201329814324399471\
802570966246053351152256291639950576773372, 0.9667002104068390591283176278\
82005242021154223325393312338073822452002928096128244014782463051429218412\
01149570823184052307899669583377824513040100563201642751268856288353049292\
94370487128888660022999662107638956898014875, 0.97012234730489136271478251\
00964023896718898766826768300172703147107687787118962002335956011409716593\
37181732869988026196153230045071485456011224467913507133665731303218660065\
1781681959629127911182608348538380669802662010, 0.973364370530925992916315\
53308595270415662602788898392568530835903545775025330386995932831926338229\
78274014792449162126717331143810084721341465921499638084928184004947971749\
803155033201203726024026510205037405208373220278, 0.9764250380004875225077\
15719968473800663571854175709142373693280288379961981421779545473404463941\
13426036521370116676789522249023312273925601003464522406362573378347812961\
29450366925600405719459806288567437672483193242944, 0.97930317711109163671\
74904162999958041746428723103959566173581273366746484007825822262553344232\
87709438971561858732507715308068537353372372826926352863539735908102638278\
7039784703159971138405393016429994333282076915091794, 0.981997685191831379\
52887909170087480687613614458005403139760243228300369774045043107389021209\
84207679736236686640183664108779887036225704364906609693846271023559125006\
518528586410569039083200853496030292792851150499715217, 0.9845075299263937\
35726131549263556113944564006874296551941486817399393304973216578406374466\
05758344482661026925620061461984598161303732936087469890340073995366329562\
67889151879673740875775774514431629632651673966024773220, 0.98683174974946\
69877643524733396366467499085456400941125530757681683263403036790775930496\
59009023984367783388088968159137661107301195343319322510979575507641106450\
1953059444667123500257005658213567077596104990631752682837, 0.988969454216\
65295433378353567596498970679414463724423048295861224623111282769914943159\
68348786281524752040991062709250836086024922545594949196904431983449604883\
126836326983084752417469010357276353040604998988497737735779, 0.9909198243\
48263126686272355148162562591520340512163278113389938299149836929255656980\
93990566276829813318109586944086290337396614457551992398712032780111378169\
74896809091808796752072166391305395802500594928028469110538248, 0.99268211\
29479451720107947470433631853198894543659133069337561694006097991273998102\
75416227595142812289948715969288979438083228364020844049490206113950849967\
5475349811475324557215747639292428191443826092108675378896742644, 0.994255\
64489840360164845537710741460015781408834527049106636820804575133784176936\
88000644006390068640494128514572123079676350496871195248237804586558612370\
646980185197089019211997384336100720399123496552161133276348050648, 0.9956\
39817439823238694236604925167132512626325267944602145039162580553754636132\
11570171205812147407305039319617099430942009311267393787335142174486681254\
46715716737833373351910212023430878423994033870991775523912969886496, 0.99\
68341004460484143175666516827545562100998869429939197825420684366601076683\
57135376705607218752555015636352688732899540322706575118541030135515316106\
7104147750488619098360959138084810160610538407032023609225314458698790, 0.\
99783803674401256628250014382867855908003301541610329096596031266528968929\
50276742823601905375940212296017375675331103213966019212055436861836319117\
035056374355143042429563528704398919305706142742297758034598973383864329, \
0.998651242638639762291117793117022503289649699163174634346508965632146787\
81457958700731570303818271277971971632327960602162013863561503618093432040\
25451431681878086020469913332568789955476625223796774943416288581099283540\
, 0.9992734093757583520783256752414523556680040591309195869319515075629540\
32595047744832112424872217139806957321112948512231534514210681109731001069\
38580685924002941989498310819062756744019103784822456523693653701406142321\
17, 0.99970431032065290094349872781156901489252875453783929266254200316581\
15727748574862171840593347403618880519545770754107140410064666182325004721\
67508501658172312236330950515383314666608015194308491811930366020337569788\
3683, 0.999943876136081619429315331511263813476377956361642440226110502549\
34186287914250430132100338666361662936747049153699312365689233770295138067\
39854998964353723231296400117288469146829182572238508687948576827445401517\
543772]]):
