/*----0-----------------------------------------------------------
The utility numeric::_cubicSpline (used by numeric::cubicSpline and 
numeric::cubicSpline2d) computes the data [b, c, d] of the spline

S(x) = y[i] + b[i]*(x-x[i]) + c[i]*(x-x[i])^2 + d[i]*(x-x[i])^3

through the data table (x[i], y[i]), i = 0..n. 

It is a utility function without interface and documentation!

Call(s):   numeric::_cubicSpline(mode, n, h, y, yp0, ypn)

Parameters:
  mode -- either NotAKnot, Natural, Periodic or Complete
  n    -- the number of intervals, an integer >= 0 
          (there are n+1 nodes x[0],..,x[n])
  h    -- the node distances h[i] = x[i+1] - x[i] (i = 0.. n-1) 
          either a table or an = array(0..n-1, ...)
  y    -- a list [y[0], .., y[n]] of length n+1 or an
          array(0..n)
 yp0, ypn  -- the left/right 'Complete' boundary condition 
              only needed for mode = Complete

Return value: a list [b, c, d] with the three tables b, c, d 
              containing the spline data b[0],..,b[n-1],
              c[0],..,c[n-1], d[0],..,d[n-1]

Details: 
  -- apart from n, all data may be numeric or symbolic.
  -- As a utility, this function does not check its parameters!

--------------------------------------------------------------*/

numeric::_cubicSpline:= proc(mode, n, h, y, yp0, ypn)
local a, b, c, d, A, k;
begin
  if domtype(y) = DOM_LIST then
     y:= array(0..n, y):
  end_if;
  b:= table(); 
  c:= table(); 
  d:= table(); 
  case mode
  of NotAKnot do
     if n=0 then
        b[0]:=0;
        c[0]:=0;
        d[0]:=0;
     elif n=1 then
        b[0]:=(y[1]-y[0])/h[0];
        c[0]:=0;
        d[0]:=0;
     elif n=2 then 
        c[0]:=((y[1]-y[0])/h[0])/(h[0]+h[1]):
        c[1]:=((y[2]-y[1])/h[1])/(h[0]+h[1]):
        b[0]:=(2*h[0]+h[1])*c[0]-h[0]*c[1];
        b[1]:= h[1] *c[0]+h[0]*c[1];
        c[0]:= c[1]-c[0]; 
        c[1]:= c[0];
        d[0]:= 0; 
        d[1]:= 0:
     elif n=3 then
        (b[k]:= (y[k+1]-y[k])/h[k];) $ k=0..n-1;
        (c[k]:= b[k]-b[k-1];) $ k=1.. n-1;
        d[0]:=3*(h[1]+h[2])*(h[1]+h[0])*(h[2]+h[1]+h[0]);
        d[1]:=c[1]:
        c[1]:=((h[1]+h[2])*(2*h[1]+h[2])*d[1]+(h[0]^2-h[1]^2)*c[2])/d[0];
        c[2]:=((h[0]+h[1])*(2*h[1]+h[0])*c[2]+(h[2]^2-h[1]^2)*d[1])/d[0];
        c[0]:=c[1]*(1+h[0]/h[1])-c[2]*h[0]/h[1]:
        c[3]:=c[2]*(1+h[2]/h[1])-c[1]*h[2]/h[1];
        (b[k]:=b[k]-(2*c[k]+c[k+1])*h[k];) $ k=0..2;
        (d[k]:=(c[k+1]-c[k])/h[k];) $ k=0..2;
        (c[k]:=3*c[k];) $ k=0..2;
     else
        (b[k]:= (y[k+1]-y[k])/h[k];) $ k=0..n-1;
        (c[k]:= b[k]-b[k-1];) $ k=1.. n-1;
        d[1]:=(h[0]+h[1])*(2+h[0]/h[1]);
        d[2]:=2*(h[1]+h[2])-(h[1]^2-h[0]^2)/d[1];
        (d[k]:=2*(h[k-1]+h[k])-h[k-1]^2/d[k-1];) $ k=3..n-2;
        d[n-1]:=(h[n-2]+h[n-1])*(2+h[n-1]/h[n-2])-(h[n-2]^2-h[n-1]^2)/d[n-2];
        (c[k]:=c[k]-c[k-1]*h[k-1]/d[k-1];) $ k=2..n-2:
        c[n-1]:=(c[n-1]-c[n-2]*(h[n-2]-h[n-1]^2/h[n-2])/d[n-2])/d[n-1];
        for k from n-2 downto 2 do c[k]:=(c[k]-c[k+1]*h[k])/d[k]; end_for;
        c[1]:=(c[1]-c[2]*(h[1]-h[0]^2/h[1]))/d[1]:
        c[0]:=c[1]*(1+h[0]/h[1])-c[2]*h[0]/h[1]:
        c[n]:=c[n-1]*(1+h[n-1]/h[n-2])-c[n-2]*h[n-1]/h[n-2];
        (b[k]:=b[k]-(2*c[k]+c[k+1])*h[k];) $ k=0..n-1;
        (d[k]:=(c[k+1]-c[k])/h[k];) $ k=0..n-1;
        (c[k]:=3*c[k];) $ k=0..n-1;
     end_if;
     break;
  of Natural do
     if n=0 then 
        b[0]:=0;
        c[0]:=0;
        d[0]:=0;
     elif n=1 then 
        b[0]:=(y[1]-y[0])/h[0];
        c[0]:=0;
        d[0]:=0;
     elif n>1 then
        (b[k]:= (y[k+1]-y[k])/h[k];) $ k=0..n-1;
        (c[k]:= b[k]-b[k-1];) $ k=1.. n-1;
        d[1] :=2*(h[0]+h[1]);
        (d[k]:=2*(h[k-1]+h[k])-h[k-1]^2/d[k-1];) $ k=2..n-1;
        (c[k]:=c[k]-c[k-1]*h[k-1]/d[k-1];) $ k=2..n-1:
        c[n-1]:=c[n-1]/d[n-1]; c[n]:=0;
        for k from n-2 downto 1 do
            c[k]:=(c[k]-c[k+1]*h[k])/d[k]; 
        end_for;
        c[0]:=0;
        (b[k]:=b[k]-(2*c[k]+c[k+1])*h[k];) $ k=0..n-1;
        (d[k]:=(c[k+1]-c[k])/h[k];) $ k=0..n-1;
        (c[k]:=3*c[k];) $ k=0..n-1;
     end_if;
     break;
  of Periodic do
     if domtype(y[0]) = DOM_FLOAT and 
        domtype(y[n]) = DOM_FLOAT then
        if abs(y[n]-y[0]) > (1 + abs(y[0]))*10^(2-DIGITS) then 
           error("the input data are not periodic") 
        end_if;
     else 
        if y[n]<>y[0] then 
           error("input data not periodic") 
        end_if;
     end_if;
     if n=0 then
        b[0]:=0;
        c[0]:=0;
        d[0]:=0;
     elif n=1 then
        b[0]:=0;
        c[0]:=0;
        d[0]:=0;
     elif n=2 then 
        b[0]:=(y[1]-y[0])*(h[1]-h[0])/h[0]/h[1]; 
        b[1]:=b[0];
        c[0]:=3*(y[1]-y[0])/h[0]/h[1];
        c[1]:=-c[0];
        d[0]:=2*c[1]/3/h[0];
        d[1]:=2*c[0]/3/h[1];
     else
        (b[k]:= (y[k+1]-y[k])/h[k];) $ k=0..n-1;
        c[0]:= b[0]-b[n-1]; 
        (c[k]:= b[k]-b[k-1];) $ k=1.. n-1;
        //-----------------------------------------------------
/* solve

   (2*(h.(n-1)+h0)     h0                                              )(  x0   ) (  c0   )
   (      h0      2*(h0 + h1)      h1                                  )(  x1   ) (  c1   )
   (                   h1    2*(h1 + h2)      h2                       )(  x2   ) (  c2   )
   (                        ...         ...       ...                  )(  ..   ) (  ..   )
   (                             h.(n-1)      ...     h.(n-2)          )(  ..   ) (  ..   )
   (                                        h.(n-2) 2*(h.(n-2)+h.(n-1)))(x.(n-1)) (c.(n-1))

   via LR-decomposition:
   
   (   1     0                                  )(d0 h0            A0   )(  x0   ) (  c0   )
   ( h0/d0   1                                  )(   d1 h1         A1   )(  x1   ) (  c1   )
   (       h1/d1   1                            )(      d2 h2      A2   )(  x2   )=(  c2   )
   (              ...       ...                 )(         .. ..   ..   )(  ..   ) (  ..   )
   (                   h.(n-2)/d.(n-2)   1    0 )(            .. A.(n-2))(  ..   ) (  ..   )
   (   a0   a1             a.(n-3)    a.(n-2) 1 )(               d.(n-1))(x.(n-1)) (c.(n-1))
*/
        d[0]:=2*(h[n-1]+h[0]);
        (d[k]:=2*(h[k-1]+h[k])-h[k-1]^2/d[k-1];) $ k=1..n-2;
        a[0]:=h[n-1]/d[0]; 
        (a[k]:=-h[k-1]*a[k-1]/d[k];) $ k=1..n-3;
        a[n-2]:= (h[n-2]-h[n-3]*a[n-3])/d[n-2];
        A[0]:=h[n-1]: 
        (A[k]:=-h[k-1]*A[k-1]/d[k-1];) $ k=1..n-3;
        A[n-2]:= h[n-2]-h[n-3]*A[n-3]/d[n-3];
        d[n-1]:=2*(h[n-2]+h[n-1])-_plus( a[k]*A[k] $ k=0..n-2 );
        (c[k]:=c[k]-c[k-1]*h[k-1]/d[k-1];) $ k=1..n-2;
        c[n-1]:=c[n-1]-_plus(c[k]*a[k] $ k=0..n-2);
        c[n-1]:=c[n-1]/d[n-1]; 
        c[n-2]:=(c[n-2]-A[n-2]*c[n-1])/d[n-2];
        for k from n-3 downto 0 do
          c[k]:=(c[k]-c[k+1]*h[k]-c[n-1]*A[k])/d[k];
        end_for;
        c[n]:=c[0];
        (b[k]:=b[k]-(2*c[k]+c[k+1])*h[k];) $ k=0..n-1;
        b[n-1]:= b[n-1];
        (d[k]:=(c[k+1]-c[k])/h[k];) $ k=0..n-1;
        (c[k]:=3*c[k];) $ k=0..n-1;
     end_if;
     break;
  of Complete do
     if n=0 then
        if yp0<>ypn then
           error("boundary data ".
                  expr2text(Complete=[yp0,ypn]).
                 " make no sense for single point input");
        end_if;
        b[0]:=yp0;
        c[0]:=0;
        d[0]:=0;
    elif n=1 then
        b[0]:=yp0;
        c[0]:= 3*(y[1]-y[0])/h[0]^2-(2*yp0+ypn)/h[0];
        d[0]:=-2*(y[1]-y[0])/h[0]^3+(  yp0+ypn)/h[0]^2;
    else
        (b[k]:= (y[k+1]-y[k])/h[k];) $ k=0..n-1;
        c[0]:= b[0]-yp0;
        (c[k]:= b[k]-b[k-1];) $ k=1.. n-1;
        c[n]:= ypn-b[n-1];
        d[0]:=2*h[0]; 
        (d[k]:=2*(h[k-1]+h[k])-h[k-1]^2/d[k-1];) $ k=1..n-1;
        d[n]:=2*h[n-1]-h[n-1]^2/d[n-1];
        (c[k]:=c[k]-c[k-1]*h[k-1]/d[k-1];) $ k=1..n: 
        c[n]:=c[n]/d[n];
        for k from n-1 downto 0 do 
            c[k]:=(c[k]-c[k+1]*h[k])/d[k]; 
        end_for;
        (b[k]:=b[k]-(2*c[k]+c[k+1])*h[k];) $ k=0..n-1;
        (d[k]:=(c[k+1]-c[k])/h[k];) $ k=0..n-1;
        (c[k]:=3*c[k];) $ k=0..n-1;
     end_if;
     break;
  otherwise
     error("unknown mode");
  end_case;
  return([b, c, d]):
end_proc:
