
// ---------------  file butcher.mu 8.10.97 ---------------------------- 
/*  Help file:

 butcher -- Butcher parameters of Runge-Kutta schemes

 Call:

 numeric::butcher(method, digits)

 Parameter:

    method -- name of the numerical scheme, one of
              EULER1, RKF43, RK4, RKF34, RKF54a, RKF54b, DOPRI54, CK54,
              RKF45a, RKF45b, DOPRI45, CK45, DOPRI65, DOPRI56, BUTCHER6, 
              RKF87, RKF78, DOPRI78, DOPRI87,
              GAUSS = number_of_stages,
              GAUSS(number_of_stages)
    digits -- the number of significant decimal ditis with which
              the Butcher coefficients of the GAUSS(n) methods
              are computed

 Synopsis:

   An explicit s-stage Runge-Kutta method for the numerical integration
   of a dynamical system dy/dt = f(t,y) with step size h is a map 
   (t,y) -> (t+h,Y) defined by Y = y + h*b[1]*k1 + ... + h*b[s]*k.s
   with k1,...,k.s given by

      k1  = f(t,y) 
      k2  = f(t+c[2]*h, y + h*a[2,1]*k1)
         ...
      k.s = f(t+c[s]*h, y + h*a[s,1]*k1 + ... + h*a[s,s-1]*k.(s-1) )

   Various numerical schemes arise from different choices of the Butcher 
   parameters: the (s times s)-matrix a[i,j], the weights b=[b[1],..,b[s]]
   and the abscissae c=[0,c[2],...,c[s]].

   Embedded pairs of Runge-Kutta methods consist of two methods that share 
   the matrix a[i,j] and the abscissae c[i], but use different weights b[j].
   
   The call butcher(method) returns the list [s,c,a,b1,b2,order1,order2].
   These are the Butcher data of the method: s is the number of stages,
   c is the list of abscissae, a is the strictly lower Butcher matrix, 
   b1 and b2 are lists of weights. The integers order1 and order2 are the
   orders of the scheme when using the weights b1 or b2, respectively,
   in conjunction with the matrix a and the abscissae c.

   The methods EULER1, RK4 and BUTCHER6 are single methods with b1=b2 and
   order1=order2. All other methods are embedded pairs of Runge-Kutta-Fehlberg
   or Dormand-Prince or Cash-Karp type.

 Example:

>> ([s,c,a,b1,b2,order1,order2]) := numeric::butcher(RKF34):

>> s, c, a;

                               +-                                    -+
                               |   0,       0,        0,       0,   0 |
                               |                                      |
                               |  1/4,      0,        0,       0,   0 |
      +-                   -+  |                                      |
   5, | 0, 1/4, 4/9, 6/7, 1 |, |  4/81,   32/81,      0,       0,   0 |
      +-                   -+  |                                      |
                               | 57/98, -432/343, 1053/686,    0,   0 |
                               |                                      |
                               |  1/6,      0,      27/52,  49/156, 0 |
                               +-                                    -+

>> b1, b2;

   +-                        -+  +-                                  -+
   | 1/6, 0, 27/52, 49/156, 0 |, | 43/288, 0, 243/416, 343/1872, 1/12 |
   +-                        -+  +-                                  -+

>> order1, order2;

                                   3 , 4

 See also:  numeric::odesolve
     
 ------------------------------------------------------------------------*/

numeric::butcher:=proc(method, digits)
local s,order1,order2,C,A,B1,B2, i, j, tmp;
save DIGITS;
begin
     if args(0) < 1 then error("expecting at least 1 argument") end_if;
     if args(0) > 2 then error("expecting not more than 2 arguments") end_if;
     if has(method, GAUSS) then
         if op(method, 0) = GAUSS then
            // method is GAUSS(s)
            [method, s]:= [op(method, 0), op(method, 1)]:
         elif type(method)="_equal" then
            // method is GAUSS = s
            [method, s]:= [op(method, 1), op(method, 2)]:
         else
            error("specify Gauss method by 'GAUSS(number of stages)' or ".
                  " 'GAUSS = number of stages'");
         end_if:
         if method <> GAUSS then
            error("specify Gauss method by 'GAUSS(number of stages)' or ".
                  " 'GAUSS = number of stages'");
         end_if:
         if not testtype(s, Type::PosInt) then
            error("the number of stages must be a positive integer");
         end_if;
     end_if;
     case method 
     of EULER1 do
        s:=1;order1:=1;order2:=1;
        A:=array(1..s,1..s); A[1,1]:=0;
        B1:=array(1..s); B1[1]:=1;
        B2:=array(1..s); B2[1]:=1;
        C:=array(1..s); C[1]:=0;
        break;
     of RK4 do
        s:=4;order1:=4;order2:=4;
        A:=array(1..s,1..s,[
                           [ 0 , 0 , 0 , 0 ] ,
                           [1/2, 0 , 0 , 0 ] ,
                           [ 0 ,1/2, 0 , 0 ] ,
                           [ 0 , 0 , 1 , 0 ] ]);  
        B1:=array(1..s,[1/6,1/3,1/3,1/6]  );
        B2:=array(1..s,[1/6,1/3,1/3,1/6]  );
        C:=array(1..s,[0,1/2,1/2,1]);
        break;
     of RKF34 do
     of RKF43 do
        s:=5; order1:=3; order2:=4;
        A:=array(1..s,1..s,[
            [   0   ,    0   ,    0   ,   0    , 0  ],
            [  1/4  ,    0   ,    0   ,   0    , 0  ],
            [  4/81 ,  32/81 ,    0   ,   0    , 0  ], 
            [ 57/98 ,-432/343,1053/686,   0    , 0  ],
            [  1/6  ,    0   ,  27/52 , 49/156 , 0  ] ]);
        B1:=array(1..s,
            [  1/6   ,   0   ,  27/52 , 49/156 , 0  ] );
        B2:=array(1..s,
            [ 43/288 ,    0   ,243/416,343/1872,1/12] );
        C:=array(1..s,[0,1/4,4/9,6/7,1]);  
        if method = RKF43 then
           [order1, order2]:= [order2, order1]:
           [B1, B2]:= [B2, B1]:
        end_if;
        break;
     of RKF45a do
     of RKF54a do
        s:=6;order1:=4;order2:=5;
        A:=array(1..s,1..s,[
            [   0    ,    0    ,   0   ,  0   , 0   , 0  ],
            [  2/9   ,    0    ,   0   ,  0   , 0   , 0  ],
            [  1/12  ,   1/4   ,   0   ,  0   , 0   , 0  ], 
            [ 69/128 ,-243/128 ,135/64 ,  0   , 0   , 0  ],
            [-17/12  ,  27/4   ,-27/5  ,16/15 , 0   , 0  ],
            [ 65/432 ,  -5/16  , 13/16 , 4/27 ,5/144, 0  ] ]);
        B1:=array(1..s,
            [  1/9   ,    0    ,  9/20 , 16/45,1/12 , 0  ] );
        B2:=array(1..s,
            [ 47/450 ,    0    , 12/25, 32/225,1/30 ,6/25] );
        C:=array(1..s, [0,2/9,1/3,3/4,1,5/6]);  
        if method = RKF54a then
           [order1, order2]:= [order2, order1]:
           [B1, B2]:= [B2, B1]:
        end_if;
        break;
     of RKF45b do
     of RKF54b do
        s:=6;order1:=4;order2:=5;
        A:=array(1..s,1..s,[
            [    0    ,     0    ,     0    ,    0      ,   0  , 0  ],
            [   1/4   ,     0    ,     0    ,    0      ,   0  , 0  ],
            [   3/32  ,    9/32  ,     0    ,    0      ,   0  , 0  ], 
            [1932/2197,-7200/2197, 7296/2197,    0      ,   0  , 0  ],
            [ 439/216 ,    -8    , 3680/513 ,-845/4104  ,   0  , 0  ],
            [  -8/27  ,     2    ,-3544/2565,1859/4104  ,-11/40, 0  ] ]);
        B1:=array(1..s,
            [  25/216 ,     0    , 1408/2565, 2197/4104 , -1/5 , 0  ] );
        B2:=array(1..s,
            [  16/135 ,     0    ,6656/12825,28561/56430, -9/50,2/55] );
        C:=array(1..s,[0,1/4,3/8,12/13,1,1/2]);  
        if method = RKF54b then
           [order1, order2]:= [order2, order1]:
           [B1, B2]:= [B2, B1]:
        end_if;
        break;
     of CK45 do
     of CK54 do
        s:=6;order1:=4;order2:=5;
        A:=array(1..s,1..s,[
            [    0     ,    0  ,   0     ,     0      ,   0    , 0  ],
            [   1/5    ,    0  ,   0     ,     0      ,   0    , 0  ],
            [   3/40   ,   9/40,   0     ,     0      ,   0    , 0  ], 
            [   3/10   , - 9/10,  6/5    ,     0      ,   0    , 0  ],
            [ -11/54   ,   5/2 ,-70/27   ,   35/27    ,   0    , 0  ],
            [1631/55296,175/512,575/13824,44275/110592,253/4096, 0  ] ]);
        B1:=array(1..s,
            [2825/27648,  0  ,18575/48384,13525/55296,277/14336, 1/4] );
        B2:=array(1..s,
            [  37/378 ,   0  , 250/621,125/594, 0 ,512/1771] );
        C:=array(1..s,[0,1/5,3/10,3/5,1,7/8]);  
        if method = CK54 then
           [order1, order2]:= [order2, order1]:
           [B1, B2]:= [B2, B1]:
        end_if;
        break;
     of DOPRI45 do
     of DOPRI54 do
        s:=7;order1:=4;order2:=5;
        A:=array(1..s,1..s,[
        [     0    ,       0   ,     0   ,     0   ,     0     ,  0  , 0 ],
        [    1/5   ,       0   ,     0   ,     0   ,     0     ,  0  , 0 ],
        [    3/40  ,     9/40  ,     0   ,     0   ,     0     ,  0  , 0 ],
        [   44/45  ,   -56/15  ,   32/9  ,     0   ,     0     ,  0  , 0 ],
        [19372/6561,-25360/2187,64448/6561,-212/729,     0     ,  0  , 0 ],
        [ 9017/3168,  -355/33  ,46732/5247,  49/176,-5103/18656,  0  , 0 ],
        [   35/384 ,      0    ,  500/1113, 125/192,-2187/6784 ,11/84, 0 ]]);
        B1:=array(1..s,
        [ 5179/57600,     0    , 7571/16695,393/640,-92097/339200,
                                                                187/2100,1/40]);
        B2:=array(1..s,
        [   35/384 ,      0    ,  500/1113, 125/192,-2187/6784 ,11/84, 0 ]);
        C:=array(1..s,[0,1/5,3/10,4/5,8/9,1,1]);  
        if method = DOPRI54 then
           [order1, order2]:= [order2, order1]:
           [B1, B2]:= [B2, B1]:
        end_if;
        break;
     of DOPRI65 do
     of DOPRI56 do
        s:=8;order1:=5;order2:=6;
        A:=array(1..s,1..s,[
        [      0      ,      0   ,        0        ,      0         ,       0      ,     0  ,     0,0],
        [     1/10    ,      0   ,        0        ,      0         ,       0      ,     0       ,0,0],
        [    -2/81    ,   20/81  ,        0        ,      0         ,       0      ,     0       ,0,0],
        [   615/1372  , -270/343 ,    1053/1372    ,      0         ,       0      ,     0       ,0,0],
        [  3243/5500  ,  -54/55  ,   50949/71500   ,  4998/17875    ,       0      ,     0       ,0,0],
        [-26492/37125 ,   72/55  ,    2808/23375   , -24206/37125   ,    338/459   ,     0       ,0,0],
        [  5561/2376  ,  -35/11  ,  -24117/31603   ,  899983/200772 ,  -5225/1836  ,  3925/4056  ,0,0],
        [465467/266112,-2945/1232,-5610201/14158144,10513573/3212352,-424325/205632,376225/454272,0,0]]);
        B1:=array(1..s,
        [  821/10800,       0    ,   19683/71825   ,  175273/912600 ,  395/3672 , 785/2704, 3/50, 0]):
        B2:=array(1..s,
        [   61/864    ,     0    ,   98415/321776  ,   16807/146016 , 1375/7344 , 1375/5408,-37/1120,1/10]):
        C:=array(1..s,[0, 1/10, 2/9, 3/7, 3/5, 4/5, 1, 1]);  
        if method = DOPRI65 then
           [order1, order2]:= [order2, order1]:
           [B1, B2]:= [B2, B1]:
        end_if;
        break;
     of BUTCHER6 do
        s:=7;order1:=6;order2:=6;
        A:=array(1..s,1..s,[
        [   0   ,   0   ,   0   ,    0   ,   0    ,   0   ,  0 ],
        [  1/2  ,   0   ,   0   ,    0   ,   0    ,   0   ,  0 ],
        [  2/9  ,  4/9  ,   0   ,    0   ,   0    ,   0   ,  0 ],
        [  7/36 ,  2/9  , -1/12 ,    0   ,   0    ,   0   ,  0 ],
        [-35/144,-55/36 , 35/48 ,  15/8  ,   0    ,   0   ,  0 ],
        [ -1/360,-11/36 , -1/8  ,   1/2  ,  1/10  ,   0   ,  0 ],
        [-41/260, 22/13 , 43/156,-118/39 , 32/195 , 80/39 ,  0 ]]);
        B1:=array(1..s,
        [ 13/200,   0   , 11/40 ,  11/40 ,  4/25  ,  4/25 ,13/200]);
        B2:=array(1..s,
        [ 13/200,   0   , 11/40 ,  11/40 ,  4/25  ,  4/25 ,13/200]);
        C:=array(1..s,[0,1/2,2/3,1/3,5/6,1/6,1]);  
        break;
     of RKF78 do
     of RKF87 do
        s:=13;order1:=7;order2:=8;
        A:=array(1..s,1..s,[
[    0   , 0  ,   0  ,    0   ,    0    ,    0  ,    0    ,  0  ,  0   ,  0  ,0,0,0],
[   2/27 , 0  ,   0  ,    0   ,    0    ,    0  ,    0    ,  0  ,  0   ,  0  ,0,0,0],
[   1/36 ,1/12,   0  ,    0   ,    0    ,    0  ,    0    ,  0  ,  0   ,  0  ,0,0,0],
[   1/24 , 0  ,  1/8 ,    0   ,    0    ,    0  ,    0    ,  0  ,  0   ,  0  ,0,0,0],
[   5/12 , 0  ,-25/16,  25/16 ,    0    ,    0  ,    0    ,  0  ,  0   ,  0  ,0,0,0],
[   1/20 , 0  ,   0  ,   1/4  ,   1/5   ,    0  ,    0    ,  0  ,  0   ,  0  ,0,0,0],
[ -25/108, 0  ,   0  , 125/108, -65/27  , 125/54,    0    ,  0  ,  0   ,  0  ,0,0,0],
[  31/300, 0  ,   0  ,    0   ,  61/225 ,  -2/9 ,  13/900 ,  0  ,  0   ,  0  ,0,0,0],
[    2   , 0  ,   0  , -53/6  , 704/45  , -107/9,  67/90  ,  3  ,  0   ,  0  ,0,0,0],
[ -91/108, 0  ,   0  ,  23/108,-976/135 , 311/54, -19/60  ,17/6 ,-1/12 ,  0  ,0,0,0],
[ 2383/4100,0 ,   0  ,-341/164,4496/1025,-301/82,2133/4100,45/82,45/164,18/41,0,0,0],
[    3/205 ,0 ,   0  ,    0   ,    0    ,  -6/41,  -3/205 ,-3/41, 3/41 , 6/41,0,0,0],
[-1777/4100,0 ,   0  ,-341/164,4496/1025,-289/82,2193/4100,51/82,33/164,12/41,0,1,0]]);
        B1:=array(1..s,[41/840,0,0,0,0,34/105,9/35,9/35,9/280,9/280,41/840,0,0]);
        B2:=array(1..s,[0,0,0,0,0,34/105,9/35,9/35,9/280,9/280,0,41/840,41/840]);
        if method = RKF87 then
           [order1, order2]:= [order2, order1]:
           [B1, B2]:= [B2, B1]:
        end_if;
        C:=array(1..s,[0,2/27,1/9,1/6,5/12,1/2,5/6,1/6,2/3,1/3,1,0,1]);  
        break;
     of DOPRI78 do
     of DOPRI87 do
        s:=13;order1:=7;order2:=8;
        A:= array(1..s, 1..s, [
          [ 0 $ 13],
          [ 1/18, (0 $ 12)],
          [ 1/48,  1/16, (0 $ 11)],
          [ 1/32,  0 , 3/32, (0 $ 10)],
          [ 5/16,  0 ,-75/64, 75/64, (0 $ 9)],
          [ 3/80,  0 ,  0   ,  3/16, 3/20, (0 $ 8)],
          [ 215595617/4500000000, 0, 0,  
            202047683/1800000000,
           - 28693883/1125000000,
             23124283/1800000000, (0 $ 7)],
          [ 66016996153701/3902177326250000, 0, 0,
            61564180/158732637  ,  
            3271802639447/90940573281250, 
            237097478139549/1203722497250000,
           -161912169266623/937459080858300,
            (0 $ 6)],
          [  106908055840287011604724205082678731842228139668755470324033968 /
            1547244955627425267759249819773433340748072358454860433762892115, 0, 0,
           -3326566042747865071320272969068826562382429628196893707714304/
            5244898154669238195794067185672655392366346977813086216145397,
           -46500567565797490799187717931648015564552081051162294579336192/
            288469398506808100768673695211996046580149083779719741887996835,
            66175814452466937773382010800224266938162776139883519799876864/
            477285732074900675817260113896211640705337574980990845669231127,
            32384508860533350977596794216106263929469645114889689018224000000/
            34417604457401170956155757102071261646418231795851450982147889047,
            6992959981041103840944260661352231159203510904000000/
            33042342481018810238716485165383193327572243242031481, 
            (0 $ 5)],
          [ 260894151008480730084284470672139933/1421325013490221512603572052913600000,
            0,
            0,
            -47935887594224370081278093/19416926157935754782302500,
            -21209609968140082654499867/72813473092259080433634375,
            -478067943813838145397007307538340919/18058685393810181011919966733384957500,
            900441209031008890048771741642975/316184055204485046528981557673273,
            6560308981643238155096750/23314158982833116227901307,
            1586281686644478270321241459439899956623408540189275177/
            12818966182821619734532382093543907143647820508227904000,
            (0 $ 4)],
          [ -3537888137447543284197759883250565389108044263310165948474425159710636731595954854650655416497/
            2910824336323538173625211204193399073445136980728875486386847381423844207929413996365795372160,
            0,
            0,
            57734927296096213490195986510161785150655628847579913139677512558338/
            3462861058690909860560101355712198539460430780467999483590476152519,
            58136589819715213947998495417827491531516619738716928733057108804488/
            63485786076000014110268524854723639890107897641913323865825396129515,
           -8617504422972151277457893375438630236028886136463914517995894327509978540330649883853936191758/
            1422827356003955666286021739836568670733809618942592452678694399905211997194880986560236906623,
           -1143854143268998581243145802766242359853821552194138726866578505462056874876595750671356648375000/
            71474920057023097034992943423272256041447767784798367806522760503346281362652567699840050841071,
            9510127955267712012821291045236404187951471892919608779603631113757951383688676586000000/
            640442713035016068013243031310372054997403855609055497829738119092805353899259716758067,
            -37908090628225368214143361671686640954821290739015974021020970433581156876451744584836038740906161420874289698282876539354047/
            2834975576452033889133105141617259568923401003654816754771255279827907075030383528810130595931051142996414168702099771888768,
            469129681056375923592981992008509089747926851319355029313108452786682385500/
            91373781028750370317090992584700822484107581426538344206104513975450210153,
             (0 $ 3)],
          [ 1958922673428986634106103551173048441331074670619/
            7567471754262176741098437040463937193998864102936,
            0,
            0,
            -1001223049740744085493746/209702802505706151648867,
            -334547489245557624970856/768910275854255889379179,
            -8087733974918189113939656127805175187623401123642254/
             2652165332355584001500828800174615320092292418723869,
            1628632942831420348349707330117682145830647299628362875000/
            291978538804955341103817087640053599420722421654600306391,
            21948223698175038675046771052422421724880000000/
            3565436022376709312581724254049592008258089017,
            -54267627177215403544417295523591494649516849377556708992820366524561727721169438388562655/
             10720368622845193748575907743878360415014656180362451140903219099569360363252099035275896,
             866913270897846850517488900591750/395142407933000664398470768789029,
             631684024463416271494198334562971526842645715050553334167962432652660/
             4692070228733330664061258589083291440024327468270737777380103226200813,
             (0 $ 2)],
          [ 
            244243160072852833464007326668259361217813758065433278364253381707159614527/
            296978311748988019022003721877748371457004265602680170157831794727855418000,
            0,
            0,
           -12809892149221449009491294714486945224943574554/
             1098743559111682173770315288912293503657026025,
           -1632954932439519663277002063090004897445831675192/
            2155368615124083197546101825082949089673866052375,
            1061594416459040554925192830623056036316413650694835431789159007479820993374/
            1486881915611927741653907921925786654843992231079972487448527087733782237725,
            8785354897802660409108023192236406632837549386824049923257950524644427846365000/
            727518930034750783381806345268879439045273994465090063600364706611348924794039,
           -170118119270064144587518686687868360737147416105640544650180338000000/
             79955533363897856649649083446468505728231805471269635956237990645537,
            531609133548139280136406701096786614975700625797927817796331445186649025496856971895898968906431183529666985667/
            267117958120902962051806123109615325119667748294480822302761332374114010179481444484835226534388785372264881840,
           -230671649446981257985110549479769818838607621973433830/
            984570931162878942631021661431235925951528924940467141,
            2836554450042856982624224248925462225820273143650822979581005985197318752297307311296/
            16126079511299989627991220324765813568014382588342498535143038273525930451368573347585,
            0, 0]]):

        B1:= array(1..s, [ 
             92187119817654223053707/3119360243769907947063072,
             0,
             0,
             0,
             0,
             -99144223750529152532573312/119651789473247203725698145,
             293473521217734654456147200000000/942914382941105909089283754776277,
             2362128460019416086459950000000/957356298996457227336385950033,
            -10804236851885941993001039267877539509032738879344071846538733237026879249/
             4242043332273625347322648335546818327159493996342466384551548381952682848,
             1903508509022908065487658000/1318631413273668126310967289,
             3353611978178592070093075543861464823093159359897886234693513777549456/
             42228632058599975976551327301749622960218947214436639996420929035807317,
             2/45, 0 ]):

        B2:= array(1..s, [
            294557597468629925137199/7055695789479553689785520,
            0, 0, 0, 0,
           -11058283914655545819453952/199419649122078672876163575,
            6995196125892141900847360000000000/29230345871174283181767796398064587,
            24021869890604079322735000000000/34145707997540307774997765551177,
           -647437203430870449777090671544209815506960724527035214089453994024225192465188979781/
            852160593507118889455986864344194088444607925804676271618209346615496976451816295440,
            6097274141149745127093220000/9230419892915676884176771023,
            121073252867093784462008378171516248488154623800041236030793062447088131985234/
            765378214166507164250614881821645541711187136643008051609886228527683716897385,
           -533964851718922699640879/2242517685413402257156100,
            1/4]):

        C:=array(1..s,[0, 1/18, 1/12, 1/8, 5/16, 3/8, 59/400, 93/200,
                    5490023248/9719169821, 13/20, 1201146811/1299019798, 1, 1]):
        if method = DOPRI87 then
           [order1, order2]:= [order2, order1]:
           [B1, B2]:= [B2, B1]:
        end_if;
        break;
     of GAUSS do
        if args(0) = 2 then
           if domtype(digits) <> DOM_INT or digits < 1 then
               error("second argument: expecting a positive integer ".
                     "for the number of significant decimal digits. ".
                     "Got: ".expr2text(digits));
           else DIGITS:= digits + 10:
           end_if;
        else DIGITS:= DIGITS + 10:
        end_if;
        // s was set above
        order1:= 2*s: order2:= 2*s:
        [B1, C]:= numeric::gldata(s, DIGITS):
        B1:= array(1..s, float([op(B1)])):
        B2:= B1:
        C:= array(1..s, float([op(C)])):
        A:= array(1..s, 1..s):
        for i from 1 to s do
          tmp:= linalg::vandermondeSolve([C[j] $ j = 1..s], [C[i]^j/j $ j=1..s], Transposed);
          (A[i,j]:= tmp[j]) $ j = 1..s:
        end_for:
        break;
     otherwise return([NIL,NIL,NIL,NIL,NIL,NIL,NIL]);
     end_case;
     [s,C,A,B1,B2,order1,order2];
end_proc:

numeric::butcher:= prog::remember(numeric::butcher, (() -> DIGITS)):
