// 
//      
// 
// walter, 06/03/98 
/*--------------------------------------------------------------------------
Calls:
  numeric::cubicSpline([x0, y0], [x1, y1], .. <, Symbolic > <, BoundaryCondition > )
  numeric::cubicSpline([x0, x1, ..], [y0, y1, ..] <, Symbolic> <, BoundaryCondition>)
  numeric::cubicSpline([[x0, x1, ..], [y0, y1, ..]] <, Symbolic> <, BoundaryCondition>)

  Parameter:

    x0, x1, .., y0, .. -- numerical values
    Symbolic           -- option
    BoundaryCondition  -- option Natural or Periodic or NotAKnot or
                          Complete=[a,b], where a,b are numerical values

  Options: NoWarning

  Note: for 2 points, the call
            numeric::cubicSpline([x0, y0], [x1, y1])
        is interpreted as above, not as
            numeric::cubicSpline([x0, x1], [y0, y1])
        for backward compatibility with Release 2.0.
        //--------------------------------------------------
        //  Do not change this, otherwise some plot routines
        //  using cubicSpline might produce wrong results!
        //--------------------------------------------------

  Synopsis:

    S:=cubicSpline([x[0],y[0]],..,[x[n],y[n]],Option) is the cubic spline
    function interpolating the table [x[0],y[0]],..,[x[n],y[n]], i.e, 
    S(x[0])=y[0],..,S(x[n])=y[n]. The spline function is a piecewise 
    polynomial of degree <=3 on the intervals 
         (-infinity,x[1]], [x[1],x[2]], .. ,[x[n-1],infinity).
    S and its first two derivatives S',S'' are continuous at the points
    x[1],..,x[n-1].
    
    The 4 types of boundary conditions that may be specified are:
    *  Natural :  S''(x[0])=S''(x[n])=0.
    *  Periodic:  S(x[0])=S(x[n]), S'(x[0])=S'(x[n]), S''(x[0])=S''(x[n]).
       If y[0]<>y[n], then cubicSpline returns an error message.
    *  NotAKnot:  S''' is continuous at the points x[1] and x[n-1].
    *  Complete=[a,b] : S'(x[0])=a, S'(x[n])=b.
    The default is the NotAKnot boundary condition.

    The abscissae x[0] < .. < x[n] should be numerical real values 
    in ascending order. Otherwise cubicSpline will reorder the input
    pairs to achieve such an ordering. With the option Symbolic
    the ordering is not checked!!!!

    For y[0],..,y[n] both numerical as well as symbolic values are
    admissible.

    By default all input data are converted to floats. With the option 
    Symbolic this conversion may be suppressed. In this case symbolic 
    abscissae x[i] are admissible, which are assumed to be ordered.

    The spline function S returned by cubicSpline may be called in the 
    form S(z), where z is a numerical value. If S was generated with
    symbolic nodes (using the option Symbolic), then the spline function
    cannot be evaluated.

    Alternatively, S(z,i) returns the representation of the spline assuming
    x[i] <= z < x[i+1]. In this case S may contain symbolic nodes and
    z may be symbolic.

    Further, S(z, [derivative]) and S(z, i, [derivative]) are allowed,
    where derivative is a nonnegative Integer. In this case, S returns
    the value of the derivative-th derivative of the spline function.

  Examples:

  >> S:=numeric::cubicSpline([i, sin(i*PI/20)/(1+i^2/1000)] $ i=0..100):
  >> S(11/5);
                                  0.3371060912
    
  >> plotfunc(hold(S)(x),x=0..100); (plot the spline function)

  >> S:=numeric::cubicSpline([i, y.i] $ i=0..2, Symbolic, Complete=[a,5]):
  >> collect( S(x,0),x);

                 3 / 3 a   5 y0          3 y2       \
     y0 + a x + x  | --- + ---- - 2 y1 + ---- - 5/4 | +
                   \  4     4             4         /

         2 /        7 a   9 y0   3 y2       \
        x  | 3 y1 - --- - ---- - ---- + 5/4 |
           \         4     4      4         /

------------------------------------------------------------------------------*/

numeric::cubicSpline:=proc()
//option escape;
local boundaryCondition, noWarning,
      symbolic, yp0, ypn,
      n,nm1,x,y,h,hSet, a, b, c, d,
      i, j, tmp, indexSearch;
      option escape;
begin
   if args(0) < 1 then
      error("not enough arguments");
   end_if;
   boundaryCondition:= NotAKnot; // Default
   symbolic:= FALSE; // Default
   noWarning:= FALSE; // Default
   
   //--------------------------------
   // read the data x.i, y.i
   //--------------------------------

   // count, how many of the arguments are lists:
   for i from 1 to args(0) do
       if domtype(args(i)) <> DOM_LIST then
          break;
       end_if;
   end_for;
   j:= i - 1;
   // the first j arguments are lists.
   // Interpret the lists
   case j
   of 0 do
      error("expecting at least one list containing data");
   of 1 do // cubicSpline([[x0, x1, ..], [y0, y1, ..]]):
        if nops(args(1)) <> 2 then
           error("illegal specification of the data"):
        end_if;
        if domtype(args(1)[1]) <> DOM_LIST then
           // cubicSpline([x0, y0]):
           x[0]:= args(1)[1]:
           y[0]:= args(1)[2]:
           break;
        end_if;
        n:= nops(args(1)[1]);
        if n <> nops(args(1)[2]) then
           error("the number of x-nodes does not coincide with ".
                 "the number of y-values");
        end_if:
        for i from 1 to n do
          x[i-1]:=args(1)[1][i];
          y[i-1]:=args(1)[2][i];
        end_for:
        break;
   of 2 do //cubicSpline([x0, x1, ..], [y0, y1, ..]);
        n:= nops(args(1));
        if n <> nops(args(2)) then
           error("the number of x-nodes does not coincide ".
                 "with the number of y-values");
        end_if:
        if n <> 2 then
           // interpret cubicSpline([x0, y0], [x1, y1])
           // in the old way of Release 2.0
           for i from 1 to n do
             x[i-1]:= args(1)[i];
             y[i-1]:= args(2)[i];
           end_for:
           break;
        end_if;
        // treat n=1 and n=2 by the 'otherwise' branch
   otherwise
        n:= j;
        for i from 1 to n do
          if nops(args(j)) <> 2 then
             error("expecting a pair of data [x.i, y.i], received ".
                   expr2text(args(j))." as ".output::ordinal(j)." argument"):
          end_if;
          x[i-1]:=args(i)[1];
          y[i-1]:=args(i)[2];
        end_for:
        break;
   end_case;
   
   //-------------------------------------------------
   // the data are read. Now, search for the options:
   //-------------------------------------------------

   yp0:= 0: // initialize
   ypn:= 0: // initialize
   j:= j + 1:
   for i from j to args(0) do
     if domtype(args(i)) = DOM_LIST then
        error("should not happen"):
        /*
        if nops(args(i)) <> 2 then 
          error("input data must be of the form [x0,y0], [x1,y1], .. ");
        end_if;
        x[i-1]:=args(i)[1];
        y[i-1]:=args(i)[2];
        */
     elif args(i) = Symbolic then symbolic:=TRUE; 
     elif args(i) = Natural  then boundaryCondition:=Natural; 
     elif args(i) = Periodic then boundaryCondition:=Periodic;
     elif args(i) = NotAKnot then boundaryCondition:=NotAKnot;
     elif args(i) = NoWarning then noWarning:= TRUE;
     elif has(args(i),Complete) then 
            boundaryCondition:=Complete; 
            if testtype(args(i),"_equal")
               and domtype(op(args(i),2))=DOM_LIST
               and nops(op(args(i),2))=2 
            then yp0:=op(args(i),[2,1]); 
                 ypn:=op(args(i),[2,2]);
            else error("Use Complete = [a,b] to specify complete boundary ".
                       "data S'(x[0]) = a, S'(x[n]) = b for the spline S(x)")
            end_if
     else error("wrong type of input data or unknown option")
     end_if;
   end_for;
   n:=nops(x)-1:

   if n=0                    /* just to avoid warning about  */
      then h[0]:= float(0);  /* unitialized variable. This h */
   end_if;                   /* will never be used */
   if not symbolic
   then x:=map(x,float); 
        y:=map(y,float);
        (h[i]:= x[i+1]-x[i];) $ i=0..n-1;
        if boundaryCondition=Complete then
          yp0:=float(yp0); yp0:=subs(yp0,float(0)=0);
          ypn:=float(ypn); ypn:=subs(ypn,float(0)=0);
        end_if;
        hSet:= {h[i] $ i=0..n-1};
        if n>0 then
           if map(hSet,domtype)<>{DOM_FLOAT} 
              then error("use the option 'Symbolic' for symbolic nodes");
           end_if;
           if map(hSet, x-> bool(x>0) )<> {TRUE} then
              if not noWarning then
              warning("the abscissae are not in ascending order. Input data ".
                      "will be reordered automatically.");
              end_if;
              tmp:= sort([[x[i],y[i]] $ i=0..n], (a,b)-> bool(a[1]<b[1]));
              (x[i]:= tmp[i+1][1];) $ i=0..n;
              (y[i]:= tmp[i+1][2];) $ i=0..n;
              (h[i]:= x[i+1]-x[i];) $ i=0..n-1;
           end_if;
           if map({op(h)}, _not@iszero@op,2)<> {TRUE} then
              error("the abscissae must be distinct");
           end_if;
        end_if;
   else (h[i]:= x[i+1]-x[i];) $ i=0..n-1;
   end_if;

   [b, c, d]:= numeric::_cubicSpline(boundaryCondition, n, h, y, yp0, ypn):

   if n=0 then n:=1 end_if;
   nm1:= n-1;
   a:= y; // refer to the y-values as 'a' in the procedure below:
   return(proc(z)
     local i, h, indexknown, xderiv;
     option noDebug;
     begin
      if args(0)=0 then error("expecting at least one argument") end_if;
      xderiv := 0:
      indexknown:= FALSE;
      case args(0) 
      of 1 do
      of 2 do
         if args(0) = 2 then
            case domtype(args(2))
            of DOM_LIST do
               xderiv:= args(2):
               if nops(xderiv) <> 1 then
                  error("expecting a list with a single integer as second argument");
               end_if;
               xderiv:= op(xderiv);
               if domtype(xderiv) <> DOM_INT or xderiv < 0 then
                  error("expecting a specification [nonnegative integer] ".
                        " for the derivative, received ".expr2text(args(2)));
               end_if;
               break;
            of DOM_INT do
               i:= args(2):
               i:=max(0, min(i, nm1));
               indexknown:= TRUE:
               break;
            otherwise
              error("the second argument must be an integer or a list");
           end_case;
         end_if;
         if not indexknown then
           if domtype(float(z))<>DOM_FLOAT then
           // return the procedure call unevaluated.  The attempts which have
           // been commented out all fail, because they all replace the call
           // with a DOM_IDENT.  The last version returns the DOM_IDENT iff
           // the current procedure has been assigned to that ident.
              // return(procname(args()));
              // do not use procname, but 
              // (op(op(_act_proc_env(),1),6) (= procname)

              // return(op(op(_act_proc_env(),1),6)(args()));

              if eval( op(op(_act_proc_env(),1),6)) = op(_act_proc_env(),1) then
                return(procname(args()));
              else
                return(subsop(hold(ret)(args()),
                       0=op(_act_proc_env(),1), Unsimplified));
              end_if;
              //error("numerical real value expected");
           end_if;
   //--------------------------------------------------------
   // recursive utility for the returned spline procedure S: 
   // in  S(x), find the index i of the interval
   // node[i] <= x < node[i + 1] by a binary search with
   // runtime O(log(n)). It returns the index i with 0 <= i < n.
   //--------------------------------------------------------
   indexSearch:= proc(k, j, z)
   // search for i in k .. j such that x[i] <= k <
   local m;
   option noDebug;
   begin
      if j<=k+1 then return(k) end_if;
      m:=specfunc::round((k+j)/2);
      if domtype(float(z-x[m]))<>DOM_FLOAT then
         error("spline function contains symbolic nodes");
      end_if;
      if float(z-x[m])<0 then indexSearch(k, m, z)
                            else indexSearch(m, j, z)
      end_if;
   end_proc;
   //------------------------------------------------------

           if domtype(float(z-x[0]))<>DOM_FLOAT then
              error("the spline function contains symbolic nodes");
           end_if;
           if float(z-x[0])<0 then
                i:=0
           else if float(z-x[nm1])>=0 then 
                   i:= nm1
                else i:= indexSearch(0, nm1, z)
                end_if;
           end_if;
         end_if;
      end_case;
      if args(0) >= 3 then
           i:= args(2):
           if domtype(i) <> DOM_INT then
              error("expecting an integer as second argument, received ".expr2text(i)):
           end_if;
           xderiv:= op(args(3), 1):
           if domtype(xderiv) <> DOM_INT or xderiv < 0 then
                 error("expecting a specification [nonnegative integer] ".
                       "for the derivative, received ".expr2text(args(3)));
           end_if;
      end_if;
      i:=max(0, min(i, nm1));

      // use a Horner representation to compute the spline value
      // a[i] + b[i]*(z - x[i]) + c[i]*(z - x[i])^2 + d[i]*(z - x[i])^3
      h:= z - x[i]: 
      case xderiv 
      of 0 do return( context(((d[i]*h+c[i])*h+b[i])*h+a[i] )):
      of 1 do return( context( (3*d[i]*h+2*c[i])*h+b[i] )):
      of 2 do return( context(  6*d[i]*h+2*c[i] )):
      of 3 do return( context(  6*d[i] )):
      otherwise if symbolic then 
                     return(0);
                else return(float(0));
                end_if;
      end_case
    end_proc);
end_proc:
