//-----------------------------------------------------------------
// eigenvalues - numerical eigenvalues of a matrix 
//
// Call:
//    numeric::eigenvalues(A <, hard_soft>, <NoWarning>) 
//
// Parameter:
//
//         A - a square real or complex matrix (of domtype DOM_ARRAY or    
//                      category Cat::Matrix)                      
//  had_soft - either 'HardwareFloats' or 'SoftwareFloats'
//                                                                 
// return value: sorted list of numerical eigenvalues              
//                                                                 
// remark: Non-float numerical entries are allowed. For non-       
//         triangular matrices such entries are converted to       
//         floats, for triangular matrices such entries are        
//         processed symbolically.                                 
//         Non-numeric symbolic entries will lead to an error.     
//                                                                 
// The routines eigenvalues/vectors are MuPAD versions of the      
// standard algorithms from the Handbook of Automatic Computation  
// by Wilkinson and Reinsch. They were adapted to suit the case of 
// complex matrices.                                               
//                                                                 
// Example:                                                        
//                                                                 
//   numeric::eigenvalues(linalg::hilbert(2));                                         
//                   [1.267591879, 0.06574145409] 
//                                                                 
//-----------------------------------------------------------------

numeric::eigenvalues := proc(A)
local m, n, useHardwareFloats, HardwareFloatsRequested, B,
      hermitean, real, lowertriangular, uppertriangular,
      skew, tmp, i, j, _norm, dowarn,
      // d,e are lists generated by Hermitean2tridiagonal1 
      // and processed in tqli1()                          
      d, e , 
      // local subroutines 
      Hermitean2tridiag1, balance1, elmhess1, tqli1, hqr1;
save DIGITS;
begin

// ----------- define local subroutine --------------------------------
//                                                                     
// Hermitean2tridiag1(real)                                            
//         - reduction of a hermitean matrix to real tri-digonal form  
//                                                                     
// real - TRUE or FALSE                                                
//                                                                     
// transforms a globally defined hermitean matrix A                    
//                                                                     
// returns  d = list of real diagonal elements                         
//          e = list of real subdiagonal elements                      
// as global variables. The matrix A is replaced by a matrix           
// containing the Householder data of the transformation               
//                                                                     
// explanation: the unitary transformation B = Q A Q^T to              
//            [ d[1] e[1]              ]                               
//         B= [ e[1] d[2] e[2]         ]                               
//            [      e[2] d[3] e[3]    ]                               
//            [            ..   ..  .. ]                               
//         is computed via Householder transformations Q=.. H2 H1      
//         For complex hermitean matrix A complex phase factors is     
//         introduced to convert the resulting complex tridiagonal     
//         matrix to real tridiagonal form with positive offdiagonal   
//         elements. This amounts to e[i] -> abs(e[i]). For real       
//         symmetric matrices arbitrary signs may occur in the         
//         offdiagonal elements                                        
//---------------------------------------------------------------------

Hermitean2tridiag1 := proc(real) 
local j, t, i, tt, h, k;
begin  
   e:=[ 0 $ n ]:
   if real then // A is real and symmetric 
      for j from 1 to n-2 do
         t := _plus(specfunc::abs(A[i,j])^2 $ i=j+2..n);
         if iszero(t) then e[j]:=A[j+1,j]: // Householder transformation 
                       // is trivial only have to store subdiagonal in e 
         else tt:= t+specfunc::abs(A[j+1,j])^2 ; // Householder transformation 
              t:=specfunc::sqrt(tt):
              h := 1 / (tt + specfunc::abs(A[j+1,j]) * t);
              if not iszero(A[j+1,j]) then t:= sign(A[j+1,j]) * t; end_if:
              A[j+1,j] := t + A[j+1,j];    
              ( e[i] := h * ( _plus(A[i,k]*A[k,j] $ k=j+1..i) +
                              _plus(A[k,i]*A[k,j] $ k=i+1..n))
               ) $ i=j+1..n:
              h:= h/2 * _plus(A[k,j]*e[k] $ k=j+1..n);
              ( e[i] := e[i]-h*A[i,j] ) $ i=j+1..n:
              for i from j+1 to n do
                 (A[i,k]:=A[i,k]-e[i]*A[k,j]-A[i,j]*e[k])$ k=j+1..i-1: 
                 A[i,i]:=A[i,i]-2*e[i]*A[i,j]:
              end_for;
              e[j] := - t;
         end_if;       
      end_for;
      e[n-1]:=A[n,n-1];       // the last subdiagonal 
      d := [ A[j,j] $ j=1..n ]:
  else  // real = FALSE 
     for j from 1 to n-2 do
        t := _plus(specfunc::abs(A[i,j])^2 $ i=j+2..n);
        if iszero(t) then e[j]:=specfunc::abs(A[j+1,j]):
        else tt:= t+specfunc::abs(A[j+1,j])^2 ; // Householder transformation 
             t:=specfunc::sqrt(tt):
             h := 1 / (tt + specfunc::abs(A[j+1,j]) * t);
             if not iszero(A[j+1,j]) then t:= sign(A[j+1,j]) * t; end_if:
             A[j+1,j] := t + A[j+1,j];    
             ( e[i] := h * ( _plus(A[i,k]*A[k,j] $ k=j+1..i) +
                  _plus(conjugate(A[k,i])*A[k,j] $ k=i+1..n))
             ) $ i=j+1..n:
             h:= h/2 * _plus(conjugate(A[k,j])*e[k] $ k=j+1..n);
             ( e[i] := e[i]-h*A[i,j] ) $ i=j+1..n:
             for i from j+1 to n do
                (A[i,k]:= A[i,k]-e[i]*conjugate(A[k,j])-A[i,j]*conjugate(e[k])
                ) $ k=j+1..i-1 :
                // avoid imaginary numerical garbage on the diagonal: 
                A[i,i]:=A[i,i]-2*Re(e[i]*conjugate(A[i,j])):
             end_for;
             e[j] := specfunc::abs(t);
          end_if; 
     end_for;
     e[n-1]:=specfunc::abs(A[n,n-1]);
     d := [ A[j,j] $ j=1..n ]:
  end_if:
  d:=map(d, Re); e:=map(e,Re); 
  null():
end_proc:

// ----------- define local subroutine ---------------------------------
//                                                                      
// balance1()  - balance global matrix A to improve numerical stability 
//               of eigen computations                                  
//                                                                      
// uses - a global square real or complex matrix A                      
//                                                                      
// return value: A is replaced by D A D^{-1} with a diagonal            
//               matrix D such that columns and rows of A               
//               are of similar magnitude                               
//----------------------------------------------------------------------

balance1:=proc()
local base, base2, conv, i, j, c, r, f, s, g;
begin
 /*
 base:=float(2^16): // the PARI base is 2^32. Make sure 
 base2:=base^2:     // that division/multiplication with base2 
                    // is exact                                
 */
 base:= float(2)^(1/2);
 base2:= base^2:
 while TRUE do      
   conv:=TRUE;      
   for i from 1 to n do
     c:= _plus(specfunc::abs(A[j,i])$ j=1..i-1) +
         _plus(specfunc::abs(A[j,i])$ j=i+1..n);
     r:= _plus(specfunc::abs(A[i,j])$ j=1..i-1) +
         _plus(specfunc::abs(A[i,j])$ j=i+1..n);
     f:=1; s:=c+r;
     if iszero(c) or iszero(r) then next; end_if;
     g:=r/base: if c< g then f:=f*base;c:=c*base2: end_if:  // now              
     g:=r*base: if c>=g then f:=f/base;c:=c/base2: end_if:  // c/base<=r<base*c 
     if (c+r)<0.95*s*f then
       conv:= FALSE:
       for j from 1 to i-1 do A[i,j]:=A[i,j]/f: A[j,i]:=A[j,i]*f; end_for:
       for j from i+1 to n do A[i,j]:=A[i,j]/f: A[j,i]:=A[j,i]*f; end_for:
     end_if:
   end_for;
   if conv then break; end_if: 
 end_while;
 null():
end_proc:

// ----------- define local subroutine ---------------------------------
//                                                                      
// elmhess1()  - reduction of a complex matrix to Hessenberg form       
//                                                                      
// uses  - globally defined matrix A                                    
//                                                                      
// returns  Hessenberg form stored in A                                 
//                                                                      
// explanation: Hessenberg = L.(n-2) .. L.1 A L.1^(-1) .. L.(n-2)^(n-2) 
//          with elementary matrices L.1, L.2 .. from Gauss-Elimination 
//----------------------------------------------------------------------

elmhess1:=proc()
local m,i,pivot,j,tmp,y;
begin
  for m from 2 to n-1 do
    i:=m; pivot:=0:
    for j from m to n do                   // look for Pivot element 
       if specfunc::abs(A[j,m-1])>specfunc::abs(pivot) then pivot:=A[j,m-1];i:=j; end_if:
    end_for:
    if i<>m then 
                                                 // swap row i <-> m 
       for j from m-1 to n do tmp:=A[i,j]; A[i,j]:=A[m,j]: A[m,j]:=tmp:end_for:
                                             // swap columns i <-> m 
       for j from  1  to n do tmp:=A[j,i]; A[j,i]:=A[j,m]: A[j,m]:=tmp:end_for:
    end_if:
    if not iszero(pivot) then
     for i from m+1 to n do
         y:=A[i,m-1]:
         if not iszero(y) then
            y:=y/pivot; A[i,m-1]:=y;            // do the eliminations 
            for j from m to n do A[i,j]:=A[i,j]-y*A[m,j]:end_for:
            for j from 1 to n do A[j,m]:=A[j,m]+y*A[j,i]:end_for:
         end_if:
     end_for:
    end_if:
  end_for:
  null():
end_proc:

// ----------- define local subroutine ---------------------------
//                                                                
// tqli1() - compute eigenvalues of real tridiagonal matrix with  
//           the globally defined diagonal d=[d[1],d[2],..,d[n]]  
//           and subdiagonal e=[e[1],e[2],..,e[n-1]] using the QL 
//           method with implicit spectral shifts                 
//                                                                
// remark - small eigenvalues are computed well                   
//----------------------------------------------------------------

tqli1:=proc()
local macheps, maxiter, i, ip1, l, iter, m , g , r, s, c, f, b, p;
begin
  if n = 1 then return() end_if:
  e[n]:=0:
  macheps:=float(1/10^DIGITS)/2:
  maxiter:=max(100,3*DIGITS):
  for l from 1 to n do
    iter:=0:
    repeat //1
      for m from l to n-1 do 
        if specfunc::abs(e[m])<=(specfunc::abs(d[m])
                              +specfunc::abs(d[m+1]))*macheps
        then break end_if 
      end_for:
      if m <> l then 
          iter:=iter+1:
          if iter = maxiter then error("no convergence") end_if:
          g:=(d[l+1]-d[l])/(2*e[l]):
          r:=specfunc::sqrt(1.0+g^2):
          if 0<=g then g:=d[m]-d[l]+e[l]/(g+r)
                  else g:=d[m]-d[l]+e[l]/(g-r)
          end_if;
          s:=1:c:=1:p:=0;
          for i from m-1 downto l do
              ip1:=i+1:
              f:=s*e[i]:
              b:=c*e[i]:
              r:=specfunc::sqrt(f^2+g^2):
              e[ip1]:=r:
              s:=f/r:
              c:=g/r:
              g:=d[ip1]-p:
              r:=(d[i]-g)*s+2*c*b;
              p:=s*r:
              d[ip1]:=g+p:
              g:=c*r-b:
         end_for:
         d[l]:=d[l]-p:
         e[l]:=g:
         e[m]:=0:
      else break; //goto(1)
      end_if:
    until iter=maxiter end_repeat;
 end_for;
 null();
end_proc:

// ----------- define local subroutine --------------------------
//                                                               
// hqr1() - compute eigenvalues of complex Hessenberg matrix by  
//          QR iteration with explicit spectral shifts           
//                                                               
//---------------------------------------------------------------

hqr1:= proc()
local nn, i, its, maxits, j, k, l, m, 
      macheps, na, notlast, p, q, r, s, t, w, tmp, tmpp,
      x, y, z;
begin
    d:=[0$n]:                    // global list to store eigenvalues   
    macheps := float(1/10^DIGITS)/2; nn:=n; t:=0;
    maxits:=max(101,3*DIGITS):
    repeat //nextw
        its:=1; 
        na:=nn-1; 
        while TRUE do //nextit
            for l from nn downto 2 do //look for small subdiagonal elements
                if specfunc::abs(A[l,l-1]) 
                   <= macheps*(specfunc::abs(A[l-1,l-1])+specfunc::abs(A[l,l])) 
                   then break;
                end_if;
            end_for;
            x := A[nn,nn];
            if l<na then l:=1; end_if;
            if l = nn then        // one eigenvalue found 
               d[nn]:=x+t;
               nn:=na;
               break
            end_if;
            y := A[na,na]; w:=A[nn,na]*A[na,nn];
            if l = na then /*twow*/   //2 eigenvalues found 
                p := (y-x)/2; q:=p^2+w: x:=x+t;
                tmp:= p+specfunc::sqrt(q):
                tmpp:= p-specfunc::sqrt(q):
                if specfunc::abs(tmpp)>specfunc::abs(tmp) then tmp:= tmpp; end_if;
                if iszero(tmp) // tmp=0 -> double eigenvalue 0 + shift x
                   then d[na]:=x: d[nn]:=x: 
                   else d[na]:=x+tmp: d[nn]:=x-w/tmp:
                end_if;
                nn := nn-2; break
            end_if;
            if its = maxits then error("no convergence") end_if;
            // exceptional shifts 
            if its mod max(10,DIGITS) = 0 
            then t := x + t;
                 for i from 1 to nn do A[i,i]:=A[i,i]-x end_for;
                 s := specfunc::abs(A[nn,na]) + specfunc::abs(A[na,nn-2]);
                 x:=.75*s; y:=x; w:=-.4375*s^2
            end_if;
            its:=its+1;
            for m from nn-2 downto l do
                z := A[m,m]; r:=x-z; s:=y-z;
                p := (r*s-w)/A[m+1,m]+A[m,m+1];
                q := A[m+1,m+1]-z-r-s; r := A[m+2,m+1];
                s := specfunc::abs(p)+specfunc::abs(q)+specfunc::abs(r);
                p:=p/s; q:=q/s; r:=r/s;
                if (m = l)
                or (specfunc::abs(A[m,m-1])*(specfunc::abs(q)
                                            +specfunc::abs(r)) 
                 <= macheps*specfunc::abs(p)*(specfunc::abs(A[m-1,m-1])
                                             +specfunc::abs(z)
                                             +specfunc::abs(A[m+1,m+1])))
                   then break
                end_if;
            end_for;
            for i from m+2 to nn do A[i,i-2]:=0; end_for:
            for i from m+3 to nn do A[i,i-3]:=0; end_for:
            for k from m to na do
                notlast := bool(k <> na);
                if k <> m then
                    p := A[k,k-1]; q := A[k+1,k-1];
                    if notlast then r := A[k+2,k-1] else r := 0 end_if;
                    x := specfunc::abs(p)+specfunc::abs(q)+specfunc::abs(r);
                    if  iszero(x) then break end_if;
                    p := p/x; q := q/x; r := r/x;
                end_if;
                if iszero(p) then 
                     s:=specfunc::sqrt( specfunc::abs(q)^2
                                       +specfunc::abs(r)^2)
                else s :=sign(p)*specfunc::sqrt( 
                                         specfunc::abs(p)^2
                                        +specfunc::abs(q)^2
                                        +specfunc::abs(r)^2 ):
                end_if:
                if k <> m then A[k,k-1] := - s*x
                          else if l <> m then A[k,k-1] := -A[k,k-1] end_if:
                end_if;
                p := p+s; x := p/s; y := q/s; z := r/s;
                q :=conjugate(q/p); r := conjugate(r/p);
                for j from k to nn do          // row modifications 
                    p := A[k,j]+q*A[k+1,j];
                    if notlast then p := p+r*A[k+2,j]; 
                                    A[k+2,j] := A[k+2,j]-p*z
                    end_if;
                    A[k+1,j] := A[k+1,j]-p*y; A[k,j] := A[k,j]-p*x
                end_for;
                for i from l to min(k+3,nn) do // column modifications 
                    p := x*A[i,k]+y*A[i,k+1];
                    if notlast then 
                                    p:=p+z*A[i,k+2];
                                    A[i,k+2]:=A[i,k+2]-p*r
                    end_if;
                    A[i,k+1]:=A[i,k+1]-p*q; A[i,k]:=A[i,k]-p
                end_for:
            end_for:
        end_while;
    until nn < 1 end_repeat;
    null():
end_proc:

//-----------------------------------------------------------
//  the body of eigenvalues using the subroutines above:     
//-----------------------------------------------------------

  if args(0) < 1 then 
     error("expecting at least one argument")
  end_if;

  //------------------
  // Check the matrix
  //------------------
  if contains({DOM_ARRAY, DOM_HFARRAY}, domtype(args(1))) then
     if op(A, [0, 1]) <> 2 then
        error("first argument: expecting a 2-dimensional array");
     end_if;
     if op(A, [0, 2, 1]) <> 1 or op(A, [0, 3, 1]) <> 1 then
       if domtype(A) = DOM_ARRAY then
         A:= subsop(A, 0 = (2,  1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                                1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]))):
       else
         A:= hfarray(1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                     1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]),
                     [op(A)]);
       end_if:
     end_if;
     [n, m]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
  elif A::dom::hasProp(Cat::Matrix)=TRUE then
     [n, m]:= A::dom::matdim(A):
  else
     error("expecting an array, an hfarray, or a matrix of category 'Cat::Matrix'"):
  end_if;

  if m <> n then
     error("expecting a square matrix");
  end_if;

  //---------------------------------------------
  // Check the options
  //---------------------------------------------
  // set defaults
  if DIGITS <= 15 then
       useHardwareFloats:= TRUE;
  else useHardwareFloats:= FALSE;
  end_if;
  HardwareFloatsRequested:= FALSE;
  if domtype(A) = DOM_HFARRAY then
       useHardwareFloats:= TRUE;
       HardwareFloatsRequested:= TRUE; // implicit request
  end_if:
  dowarn:= TRUE;

  for i from 2 to args(0) do
      case args(i)
      of Hard do
      of HardwareFloats do
                     useHardwareFloats:= TRUE:
                     HardwareFloatsRequested:= TRUE;
                     break;
      of Soft do
      of SoftwareFloats do
                     useHardwareFloats:= FALSE:
                     HardwareFloatsRequested:= FALSE;
                     break;
      of NoWarning do
                     dowarn:= FALSE;
                     break;
      otherwise
         error("unknown option ".expr2text(args(i)));
      end_case;
  end_for;

  //--------------------------------
  // ----- use HardwareFloats ------
  //--------------------------------
  case useHardwareFloats // use case because we want to use break below
  of TRUE do
     if DIGITS > 15 then
        if HardwareFloatsRequested and dowarn then
           warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                   " cannot be achieved with the requested 'HardwareFloats'."):
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     if not numeric::startHardwareFloats("eigenvalues") then
        userinfo(1, "cannot start the hardware floats interface"):
        if HardwareFloatsRequested and dowarn then
           warning("cannot start the 'HardwareFloats' interface, ".
                   "using 'SoftwareFloats' instead");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     userinfo(1, "trying hardware floats"):

     /*-----------------------------------------------------------
     //  convert to hfarray
     -----------------------------------------------------------*/
     // The hfa module expects an hfarray as input.
     // If A contains symbols or numerical stuff that
     // cannot be represented by hardware floats, the
     // conversion to an hfarray will fail. Use traperror
     // to detect these cases and fall back to software floats
     if domtype(A) = DOM_HFARRAY then
        B:= A
     else
        if traperror((
             B:= hfarray(1..n, 1..m, [op(A)]);
           )) <> 0 then
           userinfo(1, "'HardwareFloats' failed"):
           useHardwareFloats:= FALSE;
           break;
        end_if;
     end_if:

     // normalize to avoid underflow/overflows in the hfa module
     _norm:= norm(B):
     if iszero(_norm) then
        return([float(0) $ n]);
     end_if:
     B:= B/_norm:

     /*-----------------------------------------------------------
     //  call the hfa module
     -----------------------------------------------------------*/
     if traperror((d:= hfa::eigenvalues(B))) = 0 then
        if has(d, RD_NAN) or
           has(d, RD_INF) or
           has(d, RD_NINF) then
           userinfo(1, "'HardwareFloats' failed"):
           useHardwareFloats:= FALSE;
        else
           // undo normalization and sort:
           return(numeric::sort(map(d, _mult, _norm)));
        end_if:
     else
        userinfo(1, "'HardwareFloats' failed"):
        useHardwareFloats:= FALSE;
     end_if:
  end_case; 
  //---------------------------------
  // end of useHardwareFloats = TRUE
  //---------------------------------

  //---------------------------------
  //---- use SoftwareFloats ---------
  //---------------------------------
  if HardwareFloatsRequested and dowarn then
     warning("'HardwareFloats' failed, using 'SoftwareFloats' instead"):
  end_if;

  userinfo(1, "using software floats"):
  // convert to domtype DOM_ARRAY for speed
  if domtype(A) = DOM_HFARRAY then
     assert(nops(A) = m*n):
     A:= array(1..n, 1..m, [op(A)]);
  elif A::dom::hasProp(Cat::Matrix)=TRUE then
     A:= expr(A);
  end_if;
  if domtype(A) <> DOM_ARRAY then
     error("the matrix must be an array, an hfarray, ".
           "or a matrix of category 'Cat::Matrix'");
  end_if:

  DIGITS := DIGITS + 2 + trunc(n/10);
  hermitean := TRUE;                // is matrix hermitean ?        
  skew      := TRUE;                // is matrix anti-hermitean ?   
  real      := TRUE;                // is matrix real ?             
  lowertriangular := TRUE:          // is matrix lower triangular ? 
  uppertriangular := TRUE:          // is matrix upper triangular ? 
  for i from 1 to n do              // find properties of matrix    
   for j from 1 to i-1 do           // (hermitean/real/triangular)  
     if  (not iszero(Im(A[i,j]))) or (not iszero(Im(A[j,i])))
                                  then     real := FALSE end_if:
     tmp:=conjugate(A[j,i]):
     if A[i,j]<> tmp       then        hermitean:= FALSE end_if:
     if A[i,j]<>-tmp       then             skew:= FALSE end_if:
     if not iszero(A[j,i]) then lowertriangular := FALSE end_if:
     if not iszero(A[i,j]) then uppertriangular := FALSE end_if:
   end_for:
   if (not iszero(Im(A[i,i]))) then 
       real:=FALSE; 
       hermitean:= FALSE
   end_if:
   if (not iszero(Re(A[i,i]))) then 
      skew:=FALSE 
   end_if;
  end_for:
  // switch to floats                                          
  A:=map(A,float):
  if lowertriangular or uppertriangular then    // triangular case 
      d:= numeric::sort([A[i,i] $ i=1..n]);
      return(d):
  end_if:
  if {op(map(A,domtype))} minus {DOM_FLOAT, DOM_COMPLEX} <> {}  then 
     error("non-numeric entries not handled"); 
  end_if: 
  A:=subs(A,float(0)=0):
  if skew then // -I*A is hermitean, same eigenvectors, -eigenvals*I 
     A:= map(A,_mult,-I);
     hermitean:= TRUE;
     real:= FALSE;
  end_if;
  if hermitean then           // hermitean case 
    Hermitean2tridiag1(real): // transform to tridiagonal form  
    tqli1():                  // QL-iteration without eigenvect 
  else                 // general case 
    balance1():        // balance matrix for numerical stability 
    elmhess1():        // transform to Hessenberg form 
    hqr1();            // compute eigenvalues of Hessenberg matrix 
  end_if:
  if skew then 
     d:=map(d,_mult,I); 
  end_if;
  numeric::sort(d);
end_proc:

//--------  end of procedure eigenvalues ------------------
