//-----------------------------------------------------------------
// eigenvectors(A) - numerical eigenvalues and vectors of a matrix 
//                                                                 
// Call:                                                           
//  numeric::eigenvectors(A 
//                        <, hard_soft> 
//                        <, NoResidues> 
//                        <, NoWarning> 
//                        <, ReturnType = d>) 
//                                                                 
// A - a square real or complex matrix 
//     (of domtype DOM_ARRAY or DOM_HFARRAY
//      or of category Cat::Matrix)                 
// hard_soft -  Soft, SoftwareFloats, Hard, HardwareFloats
//
// return value: list [ d, X , residues ]                          
//                                                                 
//         The sorted list d=[ d[1],d[2],... ] contains the        
//         eigenvalues.                                            
//                                                                 
//         X is the matrix of eigenvectors, i.e., the i.th         
//         column of X is a normalized numerical eigenvector       
//         corresponding to the eigenvalue d[i].                   
//         X is of domtype DOM_ARRAY.                              
//                                                                 
//         The list residues = [ res[1],res[2],... ] contains the  
//         residues res[i] = norm(A*x[i]-d[i]*x[i])/norm(x[i]),    
//         where x[i] is the i.th column of X. For hermitean       
//         matrices res[i] provides an estimate for the absolute   
//         error of the approximative eigenvalue d[i].             
//                                                                 
// remark: Non-float numerical entries are allowed. For non-       
//         triangular matrices such entries are converted to       
//         floats, for triangular matrices such entries are        
//         processed symbolically.                                 
//         Non-numeric symbolic entries will lead to an error.     
//         For matrices with multiple eigenvalues and an           
//         insufficient number of eigenvectors some of the         
//         returned eigenvectors may coincicde or may be zero !    
//                                                                 
// The routines eigenvalues/vectors are MuPAD versions of the      
// standard algorithms from the Handbook of Automatic Computation  
// by Wilkinson and Reinsch. They were adapted to suit the case of 
// complex matrices.                                               
//                                                                 
// Example:                                                        
//                                                                 
//   n:=2:                                                         
//   Hilbert:= array(1..n,1..n,                                    
//                [ [1/(i+j-1) $ i=1..n] $ j=1..n ]):  
//   data:=eigenvectors(Hilbert);                                  
// --                             +-                           -+  
// |                              | 0.4718579255, -0.8816745987 |  
// | [0.06574145409, 1.267591879],|                             |, 
// |                              |-0.8816745987, -0.4718579255 |  
// --                             +-                           -+  
//                                      --                         
//                                       |                         
//   [3.965706584e-20, 5.421010862e-20]  |                         
//                                       |                         
//                                      --                         
//   eigenvalues:=op(data,1);                                      
//                    [0.06574145409, 1.267591879]                 
//                                                                 
//   eigenvectors:=op(data,2);                                     
//                    +-                              -+           
//                    |   0.4718579255, -0.8816745987  |           
//                    |                                |           
//                    |  -0.8816745987, -0.4718579255  |           
//                    +-                              -+           
//                                                                 
//    residues:= op(data,3);                                         
//                    [3.965706584e-20, 5.421010862e-20]           
//                                                                 
//    first_eigenvector:=eigenvectors::col(eigenvectors,1);        
//                    +-               -+                          
//                    |   0.4718579255  |                          
//                    |                 |                          
//                    |  -0.8816745987  |                          
//                    +-               -+                          
//                                                                 
// The routines eigenvalues/vectors are MuPAD versions of the      
// standard algorithms from the Handbook of Automatic Computation  
// by Wilkinson and Reinsch. They were adapted to suit the case of 
// complex matrices.                                               
//-----------------------------------------------------------------

numeric::eigenvectors := proc(A)
local n, m, useHardwareFloats, HardwareFloatsRequested,
      nowarning, OK, _norm, reorder,
      result, returnType, oldA, B, tmp,
      hermitean, real, lowertriangular, uppertriangular, skew,
      i, j, d, e, phase, X , scale, perm, residues ,  
      // d,e,phase,scale,perm  are lists generated in   
      // balance/Hermitean2tridiagonal/elmhess. X is    
      // the eigenvector matrix generated in tqli2/hqr2.
      TriangularEigendata, Hermitean2tridiag2, tqli2,  // these are 
      QTXbyHouseholderData, balance2, elmhess2, hqr2,  // local     
      backtransform, sortEigendata,                    // subroutines      
      normalizeEigenvectors, 
      computeResidues, withResidues,
      colrangechanged, originalcolrange;
save DIGITS;
begin

// ----------- define local subroutine --------------------------- 
//TriangularEigendata(lower,upper) - numerical eigenvalues and     
//      eigenvectors of a globally defined triangular matrix A     
//                                                                 
//lower - TRUE or FALSE                                            
//upper - TRUE or FALSE                                            
//                                                                 
//return value: global list d = [d[1],..,d[n]] of eigenvalues      
//              and global matrix X.  The i.th column of X is a    
//              (numerical) eigenvector of A corresponding to the  
//              eigenvalue d[i]. For a multiple eigenvalue with    
//              an insufficient number of eigenvectors some of     
//              the returned eigenvectors are zero.                
//-----------------------------------------------------------------
TriangularEigendata:=proc(lower,upper)
local i,j,k,y,exists,t1,t2;
begin  
  d:=[ A[i,i] $ i=1..n ]:                      // eigenvalues 
  X:=array(1..n,1..n,[[float(0) $n]$n]):        // initialize eigenvectors 
  if lower and upper then                 // the matrix is diagonal 
     ( X[i,i]:=1 ) $ i=1..n;  return() end_if;
  if upper then                   // the matrix is upper triangular 
    for j from 1 to n do
      y[j]:=1; exists:=TRUE;
      for i from j-1 downto 1 do
        t1:=_plus( A[i,k]*y[k] $ k=i+1..j ):
        t2:=A[j,j]-A[i,i];
        if iszero(t2)        // multiple eigenvalue A[j,j] = A[i,i] 
        then if iszero(t1) 
               then y[i]:=0       // a further eigenvector is found 
               else exists:=FALSE; break; // no further eigenvector 
             end_if:                      // exists                 
        else y[i]:=t1/t2:
        end_if;
      end_for:
      if exists   // put j.th eigenvector (y[1],..,y[j],0,..,0) as  
                  // column j of eigenvectormatrix X, else return 0 
                  // vector if no further eigenvector was found for 
                  // multiple eigenvalue A[j,j]                     
      then (X[i,j] := y[i]) $ i=1..j; end_if:
    end_for:
    return();
 end_if:
 if lower then                    // the matrix is lower triangular 
    for j from n downto 1 do
      y[j]:=1; exists:=TRUE;
      for i from j+1 to n do
        t1:=_plus( A[i,k]*y[k] $ k=j..i-1 ):
        t2:=A[j,j]-A[i,i];
        if iszero(t2)        // multiple eigenvalue A[j,j] = A[i,i] 
        then if iszero(t1) 
               then y[i]:=0       // a further eigenvector is found 
               else exists:=FALSE; break; // no further eigenvector 
             end_if:                      // exists                 
        else y[i]:=t1/t2:
        end_if;
      end_for:
      if exists   // put j.th eigenvector (0,..,0,y[j],..,y[n]) as  
                  // column j of eigenvectormatrix X, else return 0 
                  // vector if no further eigenvector was found for 
                  // multiple eigenvalue A[j,j]                     
      then (X[i,j] := y[i]) $ i=j..n; end_if:
    end_for:
    return();
 end_if:
 null();
end_proc:

// ----------- define local subroutine ---------------------------- 
// Hermitean2tridiag2(real)  - reduction of a hermitean matrix      
//                             to real tri-digonal form             
//                                                                  
// real - TRUE or FALSE                                             
//                                                                  
// transforms a globally defined hermitean matrix A                 
//                                                                  
// returns  d = list of real diagonal elements                      
//          e = list of real subdiagonal elements                   
// as global variables. The matrix A is replaced by a matrix        
// containing the Householder data of the transformation            
//                                                                  
// explanation: the unitary transformation B = Q A Q^T to           
//           [ d[1] e[1]              ]                             
//        B= [ e[1] d[2] e[2]         ]                             
//           [      e[2] d[3] e[3]    ]                             
//           [            ..   ..  .. ]                             
//        is computed via Householder transformations Q=.. H2 H1    
//        For complex hermitean matrix A complex phase factors are  
//        introduced to convert the resulting complex tridiagonal   
//        matrix to real tridiagonal form with positive offdiagonal 
//        elements. For real symmetric matrices arbitrary signs     
//        may occur in the offdiagonal elements                     
//------------------------------------------------------------------

Hermitean2tridiag2 := proc(real) 
local i, j, k, t, tt, h;
begin  
   e:=[ 0 $ n ]:
   if real then // A is real and symmetric 
      for j from 1 to n-2 do
         t := _plus(specfunc::abs(A[i,j])^2 $ i=j+2..n);
         if iszero(t) then e[j]:=A[j+1,j]: // Householder transformation 
                       // is trivial only have to store subdiagonal in e 
         else tt:= t+specfunc::abs(A[j+1,j])^2 ; // Householder transformation 
              t:=specfunc::sqrt(tt):
              h := 1 / (tt + specfunc::abs(A[j+1,j]) * t);
              if not iszero(A[j+1,j]) then t:= sign(A[j+1,j]) * t; end_if:
              A[j+1,j] := t + A[j+1,j];    
              ( e[i] := h * ( _plus(A[i,k]*A[k,j] $ k=j+1..i) +
                              _plus(A[k,i]*A[k,j] $ k=i+1..n))
               ) $ i=j+1..n:
              h:= h/2 * _plus(A[k,j]*e[k] $ k=j+1..n);
              ( e[i] := e[i]-h*A[i,j] ) $ i=j+1..n:
              for i from j+1 to n do
                 (A[i,k]:=A[i,k]-e[i]*A[k,j]-A[i,j]*e[k])$ k=j+1..i-1: 
                 A[i,i]:=A[i,i]-2*e[i]*A[i,j]:
              end_for;
              e[j] := - t;
         end_if;       
      end_for;
      e[n-1]:=A[n,n-1];       // the last subdiagonal 
      d := [ A[j,j] $ j=1..n ]:
  else  // real = FALSE 
     for j from 1 to n-2 do
        t := _plus(specfunc::abs(A[i,j])^2 $ i=j+2..n);
        if iszero(t) then e[j]:=A[j+1,j]:
        else tt:= t+specfunc::abs(A[j+1,j])^2 ; // Householder transformation 
             t:=specfunc::sqrt(tt):
             h := 1 / (tt + specfunc::abs(A[j+1,j]) * t);
             if not iszero(A[j+1,j]) then t:= sign(A[j+1,j]) * t; end_if:
             A[j+1,j] := t + A[j+1,j];    
             ( e[i] := h * ( _plus(A[i,k]*A[k,j] $ k=j+1..i) +
                  _plus(conjugate(A[k,i])*A[k,j] $ k=i+1..n))
             ) $ i=j+1..n:
             h:= h/2 * _plus(conjugate(A[k,j])*e[k] $ k=j+1..n);
             ( e[i] := e[i]-h*A[i,j] ) $ i=j+1..n:
             for i from j+1 to n do
                (A[i,k]:= A[i,k]-e[i]*conjugate(A[k,j])-A[i,j]*conjugate(e[k])
                ) $ k=j+1..i-1 :
                // avoid imaginary numerical garbage on the diagonal: 
                A[i,i]:=A[i,i]-2*Re(e[i]*conjugate(A[i,j])):
             end_for;
             e[j] := - t;
          end_if; 
     end_for;
     e[n-1]:=A[n,n-1];
     d := [ A[j,j] $ j=1..n ]:

   /* now [d[1],..,d[n]] and [e[1],..,e[n-1]] are the diagonal and
     offdiagonal of a complex tridiagonal matrix B=QAQ^T. Apply
     further transformation with a phase matrix 
            P=diag(phase[1],..,phase[n-1],1)
     to obtain real tridiagonal form PQAQ^TP^T. */

     phase[n]:=1:
     for j from n-1 downto 1 do
        if iszero(e[j]) then phase[j]:=1
                        else phase[j]:=phase[j+1]*sign(e[j]):
        end_if: 
        e[j]:=specfunc::abs(e[j]):
     end_for:
  end_if:
  d:=map(d, Re); e:=map(e,Re);
  null():
end_proc:

// ----------- define local subroutine -------------------------
// tqli2() - compute eigenvalues and eigenvectors of real       
//           tridiagonal matrix with the globally defined       
//           diagonal d=[d[1],..,d[n]] and subdiagonal          
//           e=[e[1],..,e[n-1]] using the QL iteration with     
//           implicit spectral shifts. The eigenvectors are     
//           stored in a globally defined matrix X              
//                                                              
// return value - list of eigenvalues stored in global list d   
//               matrix X of eigenvectors                       
//--------------------------------------------------------------

tqli2:=proc()
local i, macheps, ip1, l, iter, maxiter, m, g, r, s, c, f, b,
      p, k, tmp;
begin
  if n=1 then X:=array(1..n,1..n):X[1,1]:=1:return():end_if:
  X:=array(1..n,1..n,[[float(0)$n]$n]): // initialize eigenvectors 
  (X[i,i]:=float(1)) $ i=1..n:    // as identity matrix      
  e[n]:=0:
  macheps:=float(1/10^DIGITS)/2:
  maxiter:=max(100,3*DIGITS):
  for l from 1 to n do
    iter:=0:
    repeat //1
      for m from l to n-1 do    // look for a single small subdiagonal 
                                // element to split the matrix         
        if specfunc::abs(e[m])
            <=(specfunc::abs(d[m])+specfunc::abs(d[m+1]))*macheps
        then break end_if 
      end_for:
      if m <> l then 
          iter:=iter+1:
          if iter = maxiter then error("no convergence") end_if:
          g:=(d[l+1]-d[l])/(2*e[l]):     // form implicit shift 
          r:=specfunc::sqrt(1.0+g^2):
          if 0<=g then g:=d[m]-d[l]+e[l]/(g+r)  // this is d[m]-shift 
                  else g:=d[m]-d[l]+e[l]/(g-r)  // this is d[m]-shift 
          end_if;
          s:=1:c:=1:p:=0;
          for i from m-1 downto l do
              ip1:=i+1;
              f:=s*e[i]:        // a plane rotation as in the orig.  
              b:=c*e[i]:        // QL, followed by Givens rotation   
              r:=specfunc::sqrt(f^2+g^2): // to restore tridiagonal form 
              e[ip1]:=r:
              s:=f/r:
              c:=g/r:
              g:=d[ip1]-p:
              r:=(d[i]-g)*s+2*c*b;
              p:=s*r:
              d[ip1]:=g+p:
              g:=c*r-b:
                                               // update eigenvectors 
              for k from 1 to n do 
                 tmp:=X[k,i]:
                 X[k,i  ]:=c*tmp-s*X[k,ip1]:
                 X[k,ip1]:=s*tmp+c*X[k,ip1]:
              end_for: 
         end_for:
         d[l]:=d[l]-p:
         e[l]:=g:
         e[m]:=0:
      else break; //goto(1)
      end_if:
    until iter=maxiter end_repeat;
  end_for;
  null();
end_proc:

// ----------- define local subroutine -----------------------------------
//                                                                        
// QTXbyHouseholderData(real)  - multiplies globally defined matrix X     
//           from the left with a unitary matrix Q^T stored in globally   
//           defined A by Householder data                                
//                                                                        
// real - TRUE or FALSE                                                   
//                                                                        
// uses - an arbitrary globally defined square matrix X                   
//       -a globally defined matrix A containing Householder data         
//                                                                        
// return value: Q^T X                                                    
//                                                                        
// explanation: after Hermitean2tridiag2() or General2Hessenberg(real)    
//              (transforming a matrix A=old_A to tridiagonal/Hessenberg  
//              form) the global matrix A contains the data of Householder
//              matrices H1,H2,..  The global list phase represents a     
//              diagonal phase matrix P is generated, such that           
//                P (.. H2 H1 oldA H1 H2 .. ) P^T =: B =                  
//                                       real tridiagonal or Hessenberg,  
//              i.e. oldA = H1 H2 .. P^T B P .. H2 H1                     
//                          |___ Q^T___|   |__ Q ___|                     
//              QTXbyHouseholderData(X,real) returns H1 H2 .. P^T X       
//              for an arbitrary matrix X.                                
//              If X represents the eigenvectors of B then this is the    
//              matrix of eigenvectors of oldA                            
//------------------------------------------------------------------------

QTXbyHouseholderData:=proc(real)
  local i,j,k,h,vX;
begin
   if hermitean and (not real) then
     for k from 1 to n-1 do
       (X[k,j]:=X[k,j]/phase[k]) $ j=1..n;
     end_for:
   end_if:
   for j from n-2 downto 1 do
      h:= _plus( specfunc::abs(A[k,j])^2 $ k=j+2.. n) :
      if not iszero(h) then
        h:= (h+specfunc::abs(A[j+1,j])^2)/2;
        if real then 
             for i from 1 to n do
                vX:=_plus(A[k,j]*X[k,i] $ k=j+1..n)/h:
                (X[k,i]:=X[k,i]-vX*A[k,j]) $ k=j+1..n;
             end_for:
        else
             for i from 1 to n do
               vX:=_plus(conjugate(A[k,j])*X[k,i] $ k=j+1..n)/h:
               (X[k,i]:=X[k,i]-vX*A[k,j]) $ k=j+1..n;
             end_for:
        end_if:
      end_if:
   end_for:
   null()
end_proc:

// ----------- define local subroutine -------------------------------
//                                                                    
// balance2()  - balance global matrix A                              
//                                                                    
// uses A - a square real or complex matrix                           
//                                                                    
// return value: A is replaced by D^{-1} A D with a diagonal          
//               matrix D such that columns and rows of A             
//               are of similar magnitude. The entries of             
//               D=diag(scale[1],..,scale[n]) are returned            
//               as global list scale.                                
//--------------------------------------------------------------------

balance2:=proc()
local base, base2, conv, i, j, c, r, f, s, g;
begin
 /*
 base:=float(2^16): 
 base2:=base^2:
        // the PARI base is 2^32. Make sure that division/multiplication 
        // with base2 is exact                                           
 */
 base:= float(2)^(1/2);
 base2:= 2;
 scale:=[ float(1) $n ]: // initialize list to store the balance factors 
 while TRUE do 
   conv:=TRUE;  
   for i from 1 to n do
     c:=_plus(specfunc::abs(A[j,i])$ j=1..i-1) +
        _plus(specfunc::abs(A[j,i])$ j=i+1..n);
     r:=_plus(specfunc::abs(A[i,j])$ j=1..i-1) +
        _plus(specfunc::abs(A[i,j])$ j=i+1..n);
     f:=1; s:=c+r;
     if iszero(c) or iszero(r) then next; end_if;
     g:=r/base: if c< g then f:=f*base;c:=c*base2: end_if:  // now              
     g:=r*base: if c>=g then f:=f/base;c:=c/base2: end_if:  // c/base<=r<base*c 
     if (c+r)<0.95*s*f then
       scale[i]:=scale[i]*f; conv:= FALSE:
       for j from 1 to i-1 do A[i,j]:=A[i,j]/f: A[j,i]:=A[j,i]*f; end_for:
       for j from i+1 to n do A[i,j]:=A[i,j]/f: A[j,i]:=A[j,i]*f; end_for:
     end_if:
   end_for;
   if conv then break; end_if: 
 end_while;
 null();
end_proc:

// ----------- define local subroutine -----------------------------------
//                                                                        
// elmhess2()  - reduction of a complex matrix to Hessenberg form         
//                                                                        
// uses  - globally defined matrix A                                      
//                                                                        
// returns  Hessenberg form stored in A, global list perm containing the  
//          permutations used in pivoting steps                           
//                                                                        
// explanation: Hessenberg = L.(n-2) .. L.1 A L.1^(-1) .. L.(n-2)^(n-2)   
//          with elementary matrices L.1, L.2 .. from Gauss-Elimination   
//------------------------------------------------------------------------

elmhess2:=proc()
local m,i,x,j,tmp,y;
begin 
  perm:=[i $ i=1..n]:
  for m from 2 to n-1 do
    i:=m; x:=0:
    for j from m to n do                   // look for Pivot element 
       if specfunc::abs(A[j,m-1])>specfunc::abs(x) then x:=A[j,m-1];i:=j; end_if:
    end_for:
    perm[m]:=i:
    if i<>m then 
                                                 // swap row i <-> m 
       for j from m-1 to n do tmp:=A[i,j]: A[i,j]:=A[m,j]: A[m,j]:=tmp:end_for:
                                              // swap column i <-> m 
       for j from  1  to n do tmp:=A[j,i]: A[j,i]:=A[j,m]: A[j,m]:=tmp:end_for:
    end_if:
    if not iszero(x) then
     for i from m+1 to n do
         y:=A[i,m-1]:
         if not iszero(y) then
            y:=y/x; A[i,m-1]:=y;                // do the eliminations 
            for j from m to n do A[i,j]:=A[i,j]-y*A[m,j]:end_for:
            for j from 1 to n do A[j,m]:=A[j,m]+y*A[j,i]:end_for:
         end_if:
     end_for:
    end_if:
  end_for:
  null():
end_proc:

// ----------- define local subroutine --------------------------
//                                                               
// hqr2() - compute eigenvalues and eigenvectors of complex      
//          Hessenberg matrix by QR iteration with explicit      
//          spectral shifts                                      
//                                                               
// return value - global list d = list of eigenvalues,           
//                global matrix X , i.th column is numerical     
//                eigenvectors corresponding to eigenvalue d[i]  
//---------------------------------------------------------------

hqr2:= proc()
local nn, i, its, maxits, j, k, l, m,  ssub, sssub ,
      macheps, na, notlast, p, q, r, t, w, tmp, tmpp,
      x, y, z, s, c, cs, b, QT, vec, exists;
begin
    d:=[0$n]:                         // list to store eigenvalues  
    ( ssub[j]:=A[j+2,j])$ j=1..n-2;   // save subdiagonals, they    
    (sssub[j]:=A[j+3,j])$ j=1..n-3;   // need to be restored later  
    QT:=array(1..n,1..n,[[float(0)$n]$n]): // initialize unitary transformation 
    (QT[i,i]:=float(1)) $ i=1..n:    // matrix QT as identity matrix      
    macheps := float(1/10^DIGITS)/2; nn:=n; t:=0;
    maxits:=max(101,3*DIGITS);
    repeat //nextw
        its:=1; na:=nn-1; 
        while TRUE do //nextit
            for l from nn downto 2 do //look for small subdiagonal elements
                if specfunc::abs(A[l,l-1])
                   <= macheps*(specfunc::abs(A[l-1,l-1])+specfunc::abs(A[l,l])) 
                   then break;
                end_if;
            end_for;
            x := A[nn,nn];
            if l <na then l:=1; end_if;
            if l = nn then                  // one eigenvalue found 
                 A[nn,nn]:=x+t;nn:=na; break end_if;
            y := A[na,na]; w:=A[nn,na]*A[na,nn];
            if (l = na) then           /*twow*/   //2 eigenvalues found 
                p := (y-x)/2; q:=p^2+w: x:=x+t;
                tmp:= p+specfunc::sqrt(q): tmpp:= p-specfunc::sqrt(q):
                if specfunc::abs(tmpp)>specfunc::abs(tmp) then tmp:=tmpp; end_if;
                b:=A[na,nn]:
                if iszero(b) 
                   then s:=1: c:=0;
                        A[na,na]:=x: A[nn,nn]:=y+t: A[na,nn]:=-A[nn,na]:
                   else if not iszero(tmp) 
                           then tmpp:=specfunc::sqrt( specfunc::abs(b)^2
                                                   +specfunc::abs(tmp)^2 );
                                s:=tmp/sign(b)/tmpp: c:=specfunc::abs(b)/tmpp:
                                A[nn,nn]:=x+tmp: A[na,na]:=x-w/tmp:
                                A[na,nn]:=A[na,nn]-A[nn,na]/sign(s)^2:
                           else s:=0: c:=1:
                        end_if;
                end_if;
                cs:=conjugate(s):
                for i from 1 to na-1 do 
                     tmp:=A[i,na];
                     A[i,na]:=  c*tmp - s*A[i,nn]:
                     A[i,nn]:= cs*tmp + c*A[i,nn]:
                end_for:   // columns na and nn=na+1 modified in A 
                for j from nn+1 to n do
                     tmp:=A[na,j]:
                     A[na,j] := c*tmp - cs*A[nn,j]:
                     A[nn,j] := s*tmp + c *A[nn,j]:
                end_for:      // rows na and nn=na+1 modified in A 
                for i from 1 to n do
                     tmp:=QT[i,na]:
                     QT[i,na]:=  c*tmp - s*QT[i,nn]:
                     QT[i,nn]:= cs*tmp + c*QT[i,nn]:
                end_for:  // columns na and nn=na+1 modified in QT 
                
     //----------------------------------------------------------------------
     // explanation of the previous steps:                                   
     // consider block  (A[na,na] A[na,nn] ) = ( y  b )                      
     //                 (A[ne,na] A[nn,nn] )   ( a  x )                      
     // Apply unitary Givens rotation to eliminate lower entry a:            
     //    ( c -s* )( y b )( c* s*) = (yy bb ) = ( c -s* )( yc*-bs  ys*+bc ) 
     //    ( s  c* )( a x )( -s c )   (aa xx)    ( s  c* )( ac*-xs  as*+xc ) 
     //                =  ( ycc* +xss* - bcs - ac*s* , bcc-as*s* +(y-x)cs* ) 
     //                   (-bss +ac*c* + (y-x) sc*  , yss*+xss* +bcs+ac*s* ) 
     // with  cc*+ss*=1. To eliminate aa, put c=c*=real, s= tc, i.e.         
     //         -b t^2 + (y-x) t + a = 0 , i.e.  (bt)^2 -(y-x)bt-ab = 0      
     // Take t= (stable) larger solution of this quadratic equation, then    
     //           c:= 1/sqrt(1+|t|^2) , s:= t/sqrt(1+|t|^2).                 
     // One obtains                                                          
     //  yy:=(y+x|t|^2-bt-at*)/(1+|t|^2) = x-a/t                             
     //  xx:=(y|t|^2+x+bt+at*)/(1+|t|^2) = x+b*t                             
     //  bb:=(b-a(t*)^2+(y-x)t*)/(1+|t|^2) = b-at*/t                         
     //  aa:=  0                                                             
     // or, after rescaling t -> T=bt, w=ab , p=(y-x)/2:                     
     //                                                                      
     //   solve    T^2-(y-x)T-w=0, i.e.  T= p +- sqrt(p^2+w)                 
     //   yy:= x - ((y-x)|b|^2+|b|^2T+wT*)/(|b|^2+|T|^2) = x - w/T           
     //   xx:= x + ((y-x)|b|^2+|b|^2T+wT*)/(|b|^2+|T|^2) = x + T             
     //   bb:= b-as*/s = b - a/sign(s)^2                                     
     //   c := |b|/sqrt(|b|^2+|T|^2)                                         
     //   s :=  T/sign(b)/sqrt(|b|^2+|T|^2)                                  
     //----------------------------------------------------------------------

                nn := nn-2; break
            end_if;
            if its = maxits then error("no convergence") end_if;
            // exceptional shifts 
            if its mod max(10, DIGITS) = 0 
            then t := x + t;
                 for i from 1 to nn do A[i,i]:=A[i,i]-x end_for;
                 s := specfunc::abs(A[nn,na]) + specfunc::abs(A[na,nn-2]);
                 x:=.75*s; y:=x; w:=-.4375*s^2
            end_if;
            its:=its+1;
            for m from nn-2 downto l do        // double QR steps 
                z := A[m,m]; r:=x-z; s:=y-z;
                p := (r*s-w)/A[m+1,m]+A[m,m+1];
                q := A[m+1,m+1]-z-r-s; r := A[m+2,m+1];
                s := specfunc::abs(p)+specfunc::abs(q)+specfunc::abs(r);
                p:=p/s; q:=q/s; r:=r/s;
                if (m = l) or 
                    (specfunc::abs(A[m,m-1])*(specfunc::abs(q)+specfunc::abs(r)) 
                 <= macheps*specfunc::abs(p)*(specfunc::abs(A[m-1,m-1])
                                           +specfunc::abs(z)
                                           +specfunc::abs(A[m+1,m+1])))
                   then break
                end_if;
            end_for;
            for i from m+2 to nn do A[i,i-2]:=0; end_for:
            for i from m+3 to nn do A[i,i-3]:=0; end_for:
            for k from m to na do
                notlast := bool(k <> na);
                if k <> m then
                    p := A[k,k-1]; q := A[k+1,k-1];
                    if notlast then r := A[k+2,k-1] else r := 0 end_if;
                    x := specfunc::abs(p)+specfunc::abs(q)+specfunc::abs(r);
                    if  iszero(x) then break end_if;
                    p := p/x; q := q/x; r := r/x
                end_if;
                if iszero(p) then s:=specfunc::sqrt(specfunc::abs(q)^2
                                                 +specfunc::abs(r)^2)
                else s :=sign(p)*specfunc::sqrt( specfunc::abs(p)^2
                                              +specfunc::abs(q)^2
                                              +specfunc::abs(r)^2 ):
                end_if:
                if k <> m then A[k,k-1] := - s*x
                          else if l <> m then A[k,k-1] := -A[k,k-1] end_if:
                end_if;
                p := p+s; x := p/s; y := q/s; z := r/s;
                q :=conjugate(q/p); r := conjugate(r/p);
                for j from k to n do          // row modifications 
                    p := A[k,j]+q*A[k+1,j];
                    if notlast then p := p+r*A[k+2,j]; 
                                    A[k+2,j] := A[k+2,j]-p*z
                    end_if;
                    A[k+1,j] := A[k+1,j]-p*y; A[k,j] := A[k,j]-p*x
                end_for;
                for i from l to min(k+3,nn) do // column modifications 
                    p := x*A[i,k]+y*A[i,k+1];
                    if notlast then p:=p+z*A[i,k+2];
                                    A[i,k+2]:=A[i,k+2]-p*r
                    end_if;
                    A[i,k+1]:=A[i,k+1]-p*q; A[i,k]:=A[i,k]-p
                end_for:
                for i from 1 to n  do // column modifications 
                    p := x*QT[i,k]+y*QT[i,k+1];
                    if notlast then p:=p+z*QT[i,k+2];
                                    QT[i,k+2]:=QT[i,k+2]-p*r
                    end_if;
                    QT[i,k+1]:=QT[i,k+1]-p*q; QT[i,k]:=QT[i,k]-p
                end_for:
            end_for:
        end_while;
    until nn < 1 end_repeat;

                // Now matrix A is upper triangular. Compute eigendata 
   d:=[ A[j,j] $ j=1..n ]:                        // eigenvalues 
   X:=array(1..n,1..n,[[float(0)$n]$n]); //initialize eigenvector matrix 
   for j from 1 to n do // compute eigenvector vec for eigenvalue d[j] 
       vec[j]:=1; exists:=TRUE;
       for i from j-1 downto 1 do
         tmp:=_plus(A[i,k]*vec[k] $ k=i+1..j ): tmpp:=d[j]-d[i];
         if iszero(tmpp)            // multiple eigenvalue d[j] = d[i] 
         then if iszero(tmp) 
                then vec[i]:=0       // a further eigenvector is found 
                else exists:=FALSE; break;   // no further eigenvector 
              end_if:                        // exists                 
         else vec[i]:=tmp/tmpp:
         end_if;
       end_for:
       if exists // transform j.th eigenvector (vec[1],..,vec[j],0,..) 
                 // with matrix QT and store as column j of eigenvector
                 // matrix X. If no further eigenvector was found for  
                 // multiple eigenvalue d[j] then j.th column in X     
                 // stays 0 as initialized                             
       then (X[i,j]:=_plus(QT[i,k]*vec[k]$ k=1..j))$ i=1..n;
       end_if:
   end_for:
   (A[j+2,j]:=ssub[j] )$ j=1..n-2; // restore the subdiagonals   
   (A[j+3,j]:=sssub[j])$ j=1..n-3; // containing the data of the 
                                        // transformation in elmhess2 
   null();
end_proc:

// ----------- define local subroutine --------------------------
//                                                               
// backtransform - undo the transformation                       
//                    A -> Hessenberg = L^{-1} A L               
//          performed by elmhess2(), i.e. multiply the matrix X  
//          of eigenvectors of Hessenberg  X -> LX.              
//          Then undo the scaling of rows and columns performed  
//          by balance2(), stored in the global list scale       
//                                                               
//explanation: the data of L= [ 1                    ]           
//                            [     1                ]           
//                            [   L[3,2]   1         ]           
//                            [   L[4,2] L[4,3]   1  ]           
//                            [     ..     ..    ..  ]           
//        are stored in A by elmhess2() in the form              
//            A = [ A[1,1]  ..                         ]         
//                [ A[2,1] A[2,2] ..                   ]         
//                [ L[3,2] A[3,2] A[3,2] ..            ]         
//                [ L[4,2] L[4,3] A[4,3] A[4,4] ..     ]         
//                [  ..     ..     ..      ..   ..  .. ]         
//---------------------------------------------------------------

backtransform:=proc()
local i,j,m,x,tmp;
begin
   for m from n-1 downto 2 do
      for i from m+1 to n do
        x:=A[i,m-1]; (X[i,j]:=X[i,j]+x*X[m,j] ) $ j=1..n;
      end_for:
      i:=perm[m];
      if i<>m then 
         (tmp:=X[i,j];X[i,j]:=X[m,j];X[m,j]:=tmp;)$ j=1..n;
      end_if:
   end_for:
   for i from 1 to n do
     (X[i,j]:=scale[i]*X[i,j]) $ j=1..n;
   end_for;
   null();
end_proc:

//---------------------------------------------------------------------
//---------------------------------------------------------------------
sortEigendata:=proc(real)
local i,j,position,lambda,Y;
begin 
  position:=[i$ i=1..n];     //brute force sorting of eigenvalues 
  if real
  then for j from 2 to n do             // search for sorted position  
          lambda:=d[j];                 // of orig. eigenvalue d[j]    
          for i from j downto 2 do      // which becomes d[position[j]]
             if d[i-1] >= lambda then break end_if:
             d[i]:=d[i-1]:
             position[i]:=position[i-1]:
          end_for;
          d[i]:=lambda;
          position[i]:=j:
       end_for:    // real eigenvalues are ordered from large to small now
  else for j from 2 to n do
         lambda:=d[j];   // search for sorted position of eigenvalue d[j] 
         // sorting criterion: from large real part to small real part.   
         // If tie, then from large abs(imaginary part) to small
         // abs(imaginary part). For complex conjugate pair take the one  
         // with positive imaginary part first                            
         for i from j downto 2 do
            if Re(d[i-1]) >= Re(lambda) then break end_if:
            if (Re(d[i-1])=Re(lambda)) then
               if specfunc::abs(Im(d[i-1])) > specfunc::abs(Im(lambda))
                                                  then break end_if:
               if specfunc::abs(Im(d[i-1]))=specfunc::abs(Im(lambda)) and 
                          (Im(d[i-1]) > Im(lambda)) then break end_if:
            end_if:
            d[i]:=d[i-1]:
            position[i]:=position[i-1]:
         end_for;
         d[i]:=lambda;
         position[i]:=j:
       end_for:    // eigenvalues are ordered from large to small now 
  end_if:
  Y:= X: 
  for j from 1 to n do
    (X[i,j]:= Y[i,position[j]])$ i=1..n;
  end_for:
  null();
end_proc:

// ----------- define local subroutine --------------------------------
//---------------------------------------------------------------------

// normalize all eigenvectors = columns of X such that
// they have Euclidean length 1 and the largest entry
// is real and positive

normalizeEigenvectors:=proc()
local j, k, i, nnorm, tmp, tmpp, done;
begin
  for j from 1 to n do
     // search for the largest entry A[k,j] 
     // in the j-th column. Make it real and positive:
     nnorm:= max(specfunc::abs(X[k,j]) $ k=1..n):
     if iszero(nnorm) then
        next;
     end_if:
     // make the largest entry of the eigenvector
     // real and positive
     done:= FALSE;
     for k from 1 to n do
         tmp:= specfunc::abs(X[k,j]);
         if (not done) and tmp > 0.999999*nnorm then 
            tmpp:= X[k,j]/tmp;
            (X[i,j]:= X[i,j]/tmpp;) $ i = 1 .. k-1;
            X[k,j]:= tmp;
            (X[i,j]:= X[i,j]/tmpp;) $ i = k+1 .. n;
            done:= TRUE;
         end_if;
         if tmp < 10^(-DIGITS)*nnorm then
            X[k,j]:= float(0);
         end_if;
     end_for;
     // normalize the eigenvector to the euclidean length 1
     if not iszero(nnorm) then
         nnorm:= specfunc::sqrt(_plus(specfunc::abs(X[k,j])^2 $ k=1..n )):
        (X[k,j]:= X[k,j]/nnorm;)$ k=1..n;
     end_if:
  end_for:
  null();
end_proc:

// ----------- define local subroutine --------------------------------
//                                                                     
// computeresidues( )  - computes the residues                                
//                         res[i] = norm( A x[i] - d[i] x[i] )         
//                where x[i] is the normalized ith column of the       
//                eigenvector matrix X and d[i] is the corresponding   
//                numerical eigenvalue                                 
// return value:  list [ res[1], res[2], .. ]                          
//---------------------------------------------------------------------

computeResidues:=proc() local i,j,k,res;       // residue = |Ax-lambda x|/|x| 
begin
  (res[j]:=specfunc::sqrt(_plus(specfunc::abs(
            _plus((oldA[i,k]*X[k,j])$ k=1..n) -d[j]*X[i,j] 
                          )^2 $ i=1..n))) $ j=1..n:
  [ res[j] $ j=1..n ]
end_proc:

//---------------------------------------------------------------------
// the body of eigenvectors using the subroutines above:               
//---------------------------------------------------------------------

  if args(0) < 1 then error("expecting at least one argument") end_if;
  if args(0) > 5 then error("expecting no more than five arguments") end_if;

  //----------------------------------------------
  // Check the matrix and set default return types
  //----------------------------------------------
  colrangechanged:= FALSE;
  if contains({DOM_ARRAY, DOM_HFARRAY}, domtype(args(1))) then
     if op(A, [0, 1]) <> 2 then
        error("first argument: expecting a 2-dimensional array or hfarray");
     end_if;
     if op(A, [0, 2, 1]) <> 1 or op(A, [0, 3, 1]) <> 1 then
       colrangechanged:= TRUE;
       originalcolrange := op(A, [0, 3]);
       if domtype(A) = DOM_ARRAY then
          A:= subsop(A, 0 = (2,  1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                                 1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]))):
       else
          A:= hfarray(1 .. op(A, [0, 2, 2]) + 1 - op(A, [0, 2, 1]),
                      1 .. op(A, [0, 3, 2]) + 1 - op(A, [0, 3, 1]), [op(A)]);
       end_if;
     end_if;
     [n, m]:= [op(A, [0, 2, 2]), op(A, [0, 3, 2])];
     returnType:= domtype(args(1));
  elif A::dom::hasProp(Cat::Matrix)=TRUE then
     [n, m]:= A::dom::matdim(A):
     returnType:= Dom::Matrix();
     if A::dom::constructor = Dom::DenseMatrix then
          returnType:= Dom::DenseMatrix();
     elif A::dom <> Dom::Matrix() then
          A:= Dom::Matrix()(A);
     end_if;
  else
     error("expecting an array, an hfarray, or a matrix of category 'Cat::Matrix'"):
  end_if;
  if not colrangechanged then
     originalcolrange:= 1..m;
  end_if;

  if m <> n then
     error("expecting a square matrix");
  end_if;

  //---------------------------------------------
  // Check the options
  //---------------------------------------------
  // set defaults
  if DIGITS <= 15 then
       useHardwareFloats:= TRUE;       // default
  else useHardwareFloats:= FALSE;      // default
  end_if;
  HardwareFloatsRequested:= FALSE;     // default
  if domtype(A) = DOM_HFARRAY then
       useHardwareFloats:= TRUE;
       HardwareFloatsRequested:= TRUE; // implicit request
  end_if:
  nowarning:= FALSE;                   // default
  withResidues:= TRUE;                 // default

  for i from 2 to args(0) do
      case args(i)
      of Hard do
      of HardwareFloats do
                     useHardwareFloats:= TRUE:
                     HardwareFloatsRequested:= TRUE;
                     break;
      of Soft do
      of SoftwareFloats do
                     useHardwareFloats:= FALSE:
                     HardwareFloatsRequested:= FALSE;
                     break;
      of NoErrors do
      of NoResidues do
                     withResidues:= FALSE;
                     break;
      of NoWarning do
                     nowarning:= TRUE;
                     break;
      otherwise
         if type(args(i)) = "_equal" and
            lhs(args(i)) = ReturnType then
              returnType:= rhs(args(i));
              if {returnType} minus {DOM_ARRAY, 
                                     DOM_HFARRAY, 
                                     Dom::DenseMatrix(), 
                                     Dom::Matrix()}
                 <> {} then
                 error("illegal return type ".expr2text(args(i)).
                       " specified. Choose between".
                       " DOM_ARRAY, DOM_HFARRAY,".
                       " Dom::Matrix(), or Dom::DenseMatrix()"):
              end_if;
         else error("unknown option ".expr2text(args(i)));
         end_if;
      end_case;
  end_for;

  //--------------------------------
  // ----- use HardwareFloats ------
  //--------------------------------
  case useHardwareFloats // use case because we want to use break below
  of TRUE do
     if DIGITS > 15 then
        if HardwareFloatsRequested and not nowarning then
           warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                   " cannot be achieved with the requested 'HardwareFloats'.".
                   " Using 'SoftwareFloats' instead.");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     if not numeric::startHardwareFloats("eigenvectors") then
        userinfo(1, "cannot start the hardware floats interface"):
        if HardwareFloatsRequested and not nowarning then
           userinfo(1, "cannot start the 'HardwareFloats' interface"):
           warning("cannot start the 'HardwareFloats' interface, ".
                   "using 'SoftwareFloats' instead");
        end_if;
        useHardwareFloats:= FALSE;
        break;
     end_if;

     userinfo(1, "trying hardware floats"):

     if domtype(A) <> DOM_HFARRAY then
         assert(nops(A) = m*n);
         OK:= traperror((
                         B:= hfarray(1..m, 1..n, [op(A)]);
                        )) = 0;
     else
        B:= A;
        OK:= TRUE:
     end_if:
     if (not OK) or
        has(B, RD_NAN) or
        has(B, RD_INF) or
        has(B, RD_NINF) then
          userinfo(1, "'HardwareFloats' failed"):
          useHardwareFloats:= FALSE;
          break;
     end_if:

     // normalize to avoid underflow/overflows in the hfa module
     _norm:= norm(B):

     // special case: nothing to be done for the 0 matrix
     if iszero(_norm) then
        d:= [float(0) $ n];
        X:= hfarray(1..n, 1..n) + 1;
        residues:= [float(0) $ n];
        break: // proceed to postprocessing
     end_if:
     B:= B/_norm:

     if traperror((
           result:= hfa::eigenvectors(B, withResidues)
        )) = 0 then
         if has(result, RD_NAN) or
            has(result, RD_INF) or
            has(result, RD_NINF) then
               userinfo(1, "'HardwareFloats' failed"):
               useHardwareFloats:= FALSE;
               break;
         end_if:
         //------------------------------
         // d is a List of length min(m, n)
         //------------------------------
         d:= map(result[1], _mult, _norm); // DOM_LIST
         //------------------------------
         // The matrix of eigenvectors
         //------------------------------
         X:= result[2]: // DOM_HFARRAY
         //------------------------------
         residues:= result[3];
         if residues <> NIL then
            residues:= map(residues, _mult, _norm); // DOM_LIST
         end_if:
         //-----------------------------------
         // sort eigenvalues and vectors
         // according to numeric::sort
         //-----------------------------------
         d:= numeric::sort(d, TRUE);
         reorder:= map(d, op, 2);
         d:= map(d, op, 1);

         // hfa::reorderColumns does a **referenced** change!!
         hfa::reorderColumns(X, reorder);

         if withResidues then
            if residues <> NIL then
               for i from 1 to n do
                   residues[i]:= residues[reorder[i]];
               end_for:
            end_if:
         end_if;

         break; // break for postprocessing

     else // proceed to the branch useHardwareFloats = FALSE
       userinfo(1, "'HardwareFloats' failed"):
       useHardwareFloats:= FALSE;
       break;
     end_if;

  end_case; 
  //--------------------------------
  // end of useHardwareFloats = TRUE
  //--------------------------------

  // use case instead of if useHardwareFloats = FALSE,
  // because we want to use break
  case useHardwareFloats
  of FALSE do
    //---------------------------------
    //---- use SoftwareFloats ---------
    //---------------------------------
    if HardwareFloatsRequested and not nowarning then
       warning("'HardwareFloats' failed, using 'SoftwareFloats' instead"):
    end_if;

    userinfo(1, "using software floats"):

    // convert to domtype DOM_ARRAY for speed 
    if A::dom::hasProp(Cat::Matrix)=TRUE then
       A:= expr(A);
    end_if;

    DIGITS := DIGITS + 2 + trunc(n/10);

    if domtype(A) <> DOM_HFARRAY then
       A:= map(A, float):
    end_if:

    if map({op(A)}, domtype) minus {DOM_FLOAT, DOM_COMPLEX} <> {}
       then error("non-numeric entries not handled");
    end_if:

    // Indexed reading and writing for arrays is
    // as fast as lists. Hfarrays, however, are
    // significantly slower. So: use list or array
    if domtype(A) = DOM_HFARRAY then
       assert(nops(A) = n^2);
       A:= array(1..n, 1..n, [op(A)]);
    end_if;

    if domtype(A) <> DOM_ARRAY then
       error("the matrix must be an array, an hfarray, ".
             "or a matrix of category 'Cat::Matrix'");
    end_if:

    // special case: nothing to be done for the 0 matrix
    if map({op(A)},iszero) = {TRUE} then
        d:= [float(0) $ n];
        X:= array(1..n, 1..n, [[0.0 $ i-1, 1.0, (0.0 $ n - i)] $ i = 1..n]):
        residues:= [float(0) $ n]:
        break: // proceed to postprocessing
    end_if;

    hermitean := TRUE;            // is matrix hermitean ?        
    skew      := TRUE;            // is matrix anti-hermitean ?   
    real      := TRUE;            // is matrix real ?             
    lowertriangular := TRUE:      // is matrix lower triangular ? 
    uppertriangular := TRUE:      // is matrix upper triangular ? 
 
    for i from 1 to n do          // find properties of matrix    
     for j from 1 to i-1 do       // (hermitean/real/triangular)  
       if  (not iszero(Im(A[i,j]))) or (not iszero(Im(A[j,i]))) 
                                        then real := FALSE end_if:
       tmp:=conjugate(A[j,i]):
       if A[i,j]<> tmp       then        hermitean:= FALSE end_if:
       if A[i,j]<>-tmp       then             skew:= FALSE end_if:
       if not iszero(A[j,i]) then lowertriangular := FALSE end_if:
       if not iszero(A[i,j]) then uppertriangular := FALSE end_if:
     end_for:
     if not iszero(Im(A[i,i]))then real:= FALSE; hermitean:= FALSE
     end_if;
     if not iszero(Re(A[i,i]))then skew:=FALSE end_if:
    end_for:
 
    oldA:=A;
    if lowertriangular or uppertriangular then // triangular case 
       TriangularEigendata(lowertriangular,uppertriangular):
       sortEigendata(real):
       normalizeEigenvectors():
       residues:=computeResidues():
       break;
    end_if;
    if skew then // -I*A is hermitean, same eigenvectors, -eigenvals*I 
       A:=map(A,_mult,-I);
       hermitean:=TRUE;
       real:=FALSE;
    end_if;
    oldA:=A:
    if hermitean then                            // hermitean case 
      Hermitean2tridiag2(real):  // transform to tri-diagonal form 
      tqli2();                   // QL-Iteration with eigenvectors 
      QTXbyHouseholderData(real):// transformation of eigenvectors 
      sortEigendata(TRUE):       // sort real eigenvalues/vectors  
    else                                           // general case 
      balance2();        // balance matrix for numerical stability 
      elmhess2();        // transform to Hessenberg form           
      hqr2():        // compute spectral data of Hessenberg matrix 
      backtransform():      // transform eigenvectors back through 
                            // elmhess2 and balance2               
      sortEigendata(FALSE): // sort complex (real=FALSE) data      
    end_if:
    normalizeEigenvectors(): 
    if withResidues then 
       residues:= computeResidues(): 
    end_if;
    if skew then 
       d:=map(d,_mult,I);
    end_if;
  end_case; // HardwareFloats

  //---------------
  // postprocessing
  //---------------
  if domtype(X) = returnType and 
     not colrangechanged then
         // nothing to be done
  else
     case returnType
     of DOM_ARRAY do
        X:= array(originalcolrange, 1..n, [op(X)]);
        break;
     of DOM_HFARRAY do
        X:= hfarray(originalcolrange, 1..n, [op(X)]);
        break;
     of Dom::DenseMatrix() do
        X:= (Dom::DenseMatrix())::create(X):
        break;
     of Dom::Matrix() do
        X:= Dom::Matrix()(X):
        break;
     otherwise
        error("illegal returnType"):
     end_case;
  end_if:

  if withResidues then 
       return([d, X, residues]):
  else return([d, X, NIL]):
  end_if;

end_proc:
//--------  end of procedure eigenvectors ---------------
