/*
   numeric::sum=sum::float - numerical evaluation of infinite sum(f(k),k=a..b)
                             using Euler-Maclaurin formula.

   Reference: Computational Recreations in Mathematica, Ilan Vardi, Addison Wesley,
	      1991, page 160, formula (8.16).

   /* algorithmisch von MacLaurin auf Levin umgestellt */

   Examples:

   >> freeze(sum)(1/k^2+1/k^3,k=1..infinity);

                         / 1    1                   \
                         | -- + --, k = 1..infinity |
                      sum|  2    3                  |
                         \ k    k                   /

   >> float(%);

                                2.84699097

   >> float(sum(ln(n)/n^(1.5-14.1*I),n=1..infinity));

                     - 0.3541793679 + 0.01416426969 I

   >>  numeric::sum(i[1], i[1]=0..100);

                                5050.0

   >>  numeric::sum(1/(1+i^2), i = -infinity..infinity);

                             3.153348094

   >> DIGITS:=40: numeric::maclaurin(ln(x)/x^2,x,2);

                0.9375482543158437537025740945678649778978

   >> numeric::sum(1/(sin(k[1]) + 2), k[1] = RootOf(X^10 - X - PI, X))

                          5.008218222

*/

numeric::sum:= proc(f,x) 
   local k, r, a, b, i;
begin
   if args(0)<>2 then error("expecting two arguments") end_if;
   if not contains({"_in", "_equal"}, type(x)) then 
       // this error message corresponds to the notation
       // used in the help page of numeric::sum. Don't
       // change this message without adapting the help page!
       error("second argument: expecting a range specification ".
             "such as 'i=a..b' or 'x in RootOf(...)'")
   end_if;
   k:= op(x,1);
   if domtype(k)<>DOM_IDENT and type(k)<>"_index" then 
       error("the summation index must be an identifier ".
             "or an indexed identifier")
   end_if;
   r:= op(x,2);
   if type(r)<>"_range" and 
      domtype(r) <> RootOf and 
      domtype(r) <> DOM_SET then 
       // this error message corresponds to the notation
       // used in the help page of numeric::sum. Don't
       // change this message without adapting the help page!
       error("second argument: expecting a range specification ".
             "such as 'i=a..b' or 'x in RootOf(...)'")
   end_if;
   a:=op(r,1); 
   if type(a) = DOM_FLOAT and iszero(frac(a)) then
      a:= round(a);
   end_if:
   b:=op(r,2);
   if type(b) = DOM_FLOAT and iszero(frac(b)) then
      b:= round(b);
   end_if:
   if type(r) = "_range" then
      if map({a,b},domtype) minus {DOM_INT,domtype(infinity)}<>{} then
      // error("integers or +/-infinity expected as bounds")
         return(hold(numeric::sum)(float(f), x)); 
      end_if;
      if b<a then 
         error("right bound is smaller than left bound") 
      end_if;
   end_if;

   // !! stuff like k! etc is popular in sums.
   // Note that k! -> fact(k), but fact(k) cannot be
   // evaluated for k <> DOM_INT. So use fact(k)=gamma(k+1):
   // Also for finite sums, so do it here:

   if has(f, fact) then f:= rewrite(f, gamma); end_if;

   // Now the work starts:

   if domtype(r) = DOM_SET then
     r:= float(r):
     a:= _plus( eval(subs(f, k = i)) $ i in r );
     return(numeric::complexRound(float(a)))
   end_if;

   if domtype(r) = RootOf then
      r:= float(r):
      // RootOf::float either returns
      // * symbolic RootOf (if there are symbolic parameters)
      // * a set with all roots (if this is a univariate polynomial)
      // * a set with a single root (for a non-polynomial univariate equation)
      if domtype(r) <> DOM_SET then
         return(hold(numeric::sum)(f, x));
      end_if;
      // r = RootOf(h(x), x); a = op(r, 1) = h(x), k = b = op(r, 2) = x
      // Now we know that a is a univariate expression in b, because
      // that is the only case when RootOf::float yields an explicit 
      // result (a set).
      // Is the RootOf equation a polynomial?
      if indets(float(a), PolyExpr) = {b} then
         a:= _plus( eval(subs(f, k = i)) $ i in r );
         //beautify by rewriting exp(I*x) = cos(x) + I*sin(x) etc
         if has(a, exp) then
            a:= subs(a, hold(exp) = proc(x) local i, y, re, im; begin
                            if type(x) = "_mult" and
                               has(x, I) then
                                 // exp(x) = exp(y*(u + I*v)) = exp(y*u)*(cos(y*v) + I*sin(y*v))
                                 for i from 1 to nops(x) do
                                   if testtype(op(x, i), Type::Numeric) and
                                      has(op(x, i), I) then
                                      y:= x/op(x, i);
                                      re:= exp(y*Re(op(x, i)));
                                      im:= y*Im(op(x, i));
                                      return(  re*cos(im)
                                            +I*re*sin(im));
                                   end_if;
                                 end_for;
                            end_if;
                            return(hold(exp)(x));
                          end_proc, EvalChanges):
         end_if;
         if nops(indets(a)) > 0 then
            a:= misc::maprec(a, {DOM_COMPLEX} = numeric::complexRound)
         else
            a:= numeric::complexRound(float(a)):
         end_if;
         return(expand(a, ArithmeticOnly)); 
      end_if;
      // Now we know that a is univariate non-polynomial
      // expression. Beware, r consists of at most one root!
      return(hold(sum)(f, x));
   end_if;

   if numeric::indets(f) minus {k} <> {} then
      return(hold(numeric::sum)(float(f), x)); 
      error("first argument may only contain ".expr2text(k). 
            " as symbolic parameter");
   end_if;

   // finite sum:
   if domtype(a)=DOM_INT and domtype(b) = DOM_INT then
      if b - a < 10^5  then 
         return(_plus(float(subs(f, k=i)) $ i=a..b));
      else
         return(hold(numeric::sum)(float(f), k = a..b));
      end_if:
   end_if;
   // infinite sum:
   if a=-infinity and b=infinity then
      r := float(eval(subs(f,k=0))) + numeric::sum_infinite(f + subs(f, k = -k), k);
      if r<>FAIL then
        return(r);
      end_if;
   elif domtype(a) = DOM_INT and b=infinity then
      r := numeric::sum_infinite(subs(f, k = k+a-1), k);
      if r<>FAIL then
        return(r);
      end_if;
   elif a = -infinity and domtype(b)=DOM_INT then
      r := numeric::sum_infinite(subs(f, k = -k+b+1), k);
      if r<>FAIL then
        return(r);
      end_if;
   end_if;
   // all cases should be covered above.
   // return symbolically if the summation failed
   hold(sum)(f, x)
end_proc:

numeric::sum:= funcenv(numeric::sum, NIL,
 table("print" = "numeric::sum",
       "info"  = "numeric::sum - numerical evaluation of sums",
       "type"  = "numeric::sum")
):


sum::float := numeric::sum:


/*
Infinite sum over f(k), where k=1..infinity

This is the driver function for infinite sums. It performs some analysis
on the series and hands the actual work over to helper functions.
*/
numeric::sum_infinite :=
proc(f,k)
  local
    Low,High,prec,
    a,N,b,e,
    j,soc,sgn,tail,tA,tB,
    isAlternating,isSameSign,isReal;
  save DIGITS;
begin

  // There is danger of evaluating gamma with very large
  // arguments symbolically, before float is applied
  // (e.g., gamma::float(k) returns a symbolic gamma(k),
  // which then gets k = integer and does its job
  // symbolically). Use a wrapper function to make sure
  // that gamma is only evaluated in float mode.
  // This gives a enormous speed up! Example:
  // sum(3^(10*k)*(k+3/4)!/(k!)^2/(k+10)!, k=1.. 1000)

  f:= subs(f, hold(gamma) = proc(x)
     local fx;
     name numeric::gamma_wrapper;
     begin
       fx:= float(x);
       if contains({DOM_FLOAT, DOM_COMPLEX}, domtype(fx)) then
          gamma::float(fx)
       else
          procname(args());
       end_if;
  end_proc);

  // Heuristic: How many steps of series acceleration should we try
  // before giving up?
  prec := DIGITS;
  Low := prec;
  High := 4*prec;
  
  a := array(1..5);
  
  // Heuristic: Find the start of convergence (soc)
  //
  // We consider convergence to have started after enough consecutive
  // summands have an absolute value strictly less than 1.0 and strictly greater
  // than summands further down the tail.
  //
  // Rationale:
  //  1) The Levin u transform does not necessarily detect when series are divergent
  //       (e.g. (-1)^k*0.5, (-1)^k*sqrt(k))
  //     In fact, it is often used when computing anti-limits of strongly diverging series.
  //     Testing for decreasing absolute values catches those cases.
  //     It also avoids time-consuming calculations for series that clearly cannot converge.
  //
  //  2) Some series begin with a huge initial sum before the final asymptotic
  //     behaviour becomes apparent.
  //     Therefore, wait until summands drop below an (admittedly arbitrary) bound.
  //
  //  3) Don't require strictly decreasing absolute values. At least Euler-MacLaurin can
  //     deal with some series where the absolute value of the summands is not stricly
  //     decreasing.
  //
  //  4) In extreme cases, up to 10^5 of the leading terms are checked for 
  //     'start of convergence'
  //
  // The choice of 5 consecutive summands in the inner loop is mostly
  // arbitrary. It should be larger than 3 because we use the summands
  // later to look at their signs.
  soc := 10; // soc = start of convergence
  for j from 1 to 5 do
    for N from 1 to 5 do
      a[N] := float(eval(subs(f,k=soc+N)));
      if not domtype(a[N]) in {DOM_FLOAT,DOM_COMPLEX} then
        userinfo(1, "Summand cannot be evaluated to float: ".a[N]);
        return(FAIL)
      end_if;
    end_for;
    
    if 1 > max(op(map(a, abs))) then
      b := [float(eval(subs(f,k=5*soc+N))) $ N = 1..3];
      e := max(op(map(b,abs)));
      if e < min(op(map(a, abs))) or iszero(e) then
        break;
      end_if;
    end_if;
    
    soc := soc * 10;
  end_for:
  
  // Looks like the absolute value of summands doesn't decrease
  // => the series diverges
  if j > 5 then
    userinfo(1, "Could not find start of convergence.");
    return(FAIL)
  end_if;
  
  userinfo(3, "Start of convergence: ".soc);
  
  // Analyse the behavior of the series sign
  // We 'know' how to deal with alternating and monotone cases, and we try
  // best effort on everything else.
  //
  // Look at five consecutive summands
  sgn := [ a[j]*a[j+1] $ j = 1..4 ];
  
  isReal := TRUE;
  if {map(op(sgn), domtype)} minus {DOM_FLOAT} <> {} then
    isAlternating := FALSE;
    isReal := FALSE;
    isSameSign := FALSE;
  elif sgn[1] >= 0 then
    isAlternating := FALSE;
    isSameSign := TRUE;
    for N from 2 to 4 do
      if sgn[N] < 0 then
        isSameSign := FALSE;
        break;
      end_if;
    end_for;
  else
    isSameSign := FALSE;
    isAlternating := TRUE;
    for N from 2 to 4 do
      if sgn[N] >= 0 then
        isAlternating := FALSE;
        break;
      end_if;
    end_for:
  end_if;

  userinfo(3, "isAlternating: ".isAlternating." isSameSign: ".isSameSign." isReal: ".isReal);
  
  // --- Begin the actual work. ---
  
  // Try the plain Levin u transform first in all cases, since it's very fast
  // and sufficient for many cases.
  tail := numeric::levinsum(subs(f, k = k+soc), k, Low,High,prec, a,N)[1];
  
  userinfo(3, "Tail calculated using Levin's u: ".tail);
  
  // If that didn't work, try different strategies "in parallel".
  // Unfortunately, neither strategy is strictly better than the other,
  // and test cases show up where either gives a slightly incorrect result.
  //
  // While it is not possible to completely avoid this for pathological cases,
  // we do our best to reduce the impact, especially since those test cases
  // were not specifically crafted to exploit weaknesses in either strategy.
  if tail = FAIL then
    tA := [FAIL, FAIL];
    if not isAlternating then
      // For non-alternating series, try the Euler-MacLaurin formula.
      // There are examples where estimating using numeric integration
      // yields a good result quickly and more reliably than van Wijngaarden.
      //  (e.g.: ln(k)/k^2 + 2000*(-1)^k/k^6 and similar series)
      //
      // Inside maclaurin, symbolic integration may hang or take an insane 
      // amount of time for a number of series that can easily be computed 
      // using van Wijngaarden, so skip the symbolic integration.
      tA[1] := numeric::maclaurin(f, k, soc + 1, "NoSymbolic");
    end_if;
    
    if isSameSign then
      // Try transforming into an alternating series.
      tA[2] := numeric::levin_wijngaarden(subs(f, k = k+soc), k, Low,High,prec)[1];
    end_if;
    
    if tA[1] <> FAIL then
      if tA[2] <> FAIL then
        if abs(tA[1]-tA[2])*10^prec >= min(abs(tA[1]),abs(tA[2])) then
          // Problem: At least one of the methods returned an incorrect result.
          userinfo(3, "Mismatching results. Retrying with increased precision.");
          DIGITS := 2*prec;
          
          tB[1] := numeric::maclaurin(subs(f, k = k+soc), k, 1, "NoSymbolic");
          tB[2] := numeric::levin_wijngaarden(subs(f, k = k+soc), k, 2*Low,2*High, 2*prec)[1];
          
          // Heuristic: If one method succeeds in computing a higher
          // precision result, and if that result agrees with the earlier
          // result, we trust that the result is correct.
          if tB[1] <> FAIL and abs(tA[1]-tB[1])*10^prec >= abs(tA[1]) then
            tB[1] := FAIL;
          end_if;
          if tB[2] <> FAIL and abs(tA[2]-tB[2])*10^prec >= abs(tA[2]) then
            tB[2] := FAIL;
          end_if;

          if (tB[1] = FAIL and tB[2] = FAIL) or
             (tB[1] <> FAIL and tB[2] <> FAIL) then
            // If both methods appear to be correct, something is
            // seriously wrong.
            // error("Mismatching results. Giving up.");
            return(FAIL);
          elif tB[1] <> FAIL then
            tail := tB[1];
          else
            tail := tB[2];
          end_if;
          
          DIGITS := prec;
        else
          // Both methods succeeded and produced (up to the desired precision) the same result
          tail := tA[1];
        end_if;
      else
        // Only Euler-MacLaurin succeeded
        tail := tA[1];
      end_if;
    elif tA[2] <> FAIL then
      // Only van Wijngaarden succeeded
      tail := tA[2];
    end_if;
  end_if;

  if tail = FAIL then
    return(FAIL);
  end_if;
  
  // Directly sum up the first part of the series plus the tail.
  // In order to avoid unnecessary computation when the above doesn't find 
  // a limit for the tail of the series, we do the explicit summation at the
  // end of the procedure
  return(_plus(float(eval(subs(f, k=j))) $ j = 1..soc) + tail)
end_proc:


/*

Sum f(i), i = 1 .. infinity up using Levin's u transform

Low,High - the minimum/maximum number of steps to perform
prec - the desired precision, in digits

References:
[1] Development of Non-Linear Transformations for Improving Convergence of Sequences
David Levin, Intern. J. Computer Math. 1973, Section B, Vol. 3, pp. 371-388

[2] HURRY: An Acceleration Algorithm for Scalar Sequences and Series
Theodore Fessler, William F. Ford, David A. Smith,
ACM Transactions on Mathematical Software, Vol 9, No 3, September 1983, Pages 346-354

*/
numeric::levinsum :=
proc(f,i, Low,High,prec)
  local
    a, A, k,
    Qp, Qq,
    invR, estErr,
    T, t;
  save DIGITS;
begin
  DIGITS := 4*(prec + length(prec) + 1);

  a := array(1 .. High+2);
  A := array(1 .. High+2);
  T := array(1 .. High);

  t := numeric::levin_next_summand(f,i, float(0));
  a[1] := t[1];
  f := subs(f, i = i+t[2]);
  A[1] := a[1];
  if a[1] = FAIL or iszero(a[1]) then
    return(A[1],NIL);
  end_if;

  t := numeric::levin_next_summand(f,i, A[1]);
  a[2] := t[1];
  f := subs(f, i = i+t[2]);
  A[2] := a[2] + A[1];
  if a[2] = FAIL or iszero(a[2]) then
    return(A[2],NIL);
  end_if;

  Qp := array(0..0);
  Qp[0] := A[1]/a[1];

  Qq := array(0..0);
  Qq[0] := 1/a[1];
  
  for k from 1 to High do
    // Calculate the step
    t := numeric::levin_next_summand(f,i, A[k+1]);
    a[k+2] := t[1];
    f := subs(f, i = i+t[2]);
    A[k+2] := a[k+2] + A[k+1];
    if a[k+2] = FAIL or iszero(a[k+2]) then
      return(A[k+2],T)
    end_if;

    // invR is the inverse of the remainder estimate,
    // premultiplied with 1/(k+1) (to simplify the recursive 
    // structure of the algorithm)
    invR := 1/((k+1)*(k+1)*a[k+1]); // u-transform
  //invR := 1/(k+1)/(a[k+1]*a[k+2])*(a[k+2]-a[k+1]); // v-transform
  //invR := 1/(k+1)/((k+1)*a[k+1] + 1/k); // ln(x)/x^2-transform
    Qp := numeric::step_levin(k+1, Qp, A[k+1]*invR);
    Qq := numeric::step_levin(k+1, Qq, invR);
    
    if iszero(Qq[k]) then
      T[k] := FAIL;
    else
      T[k] := Qp[k]/Qq[k];
    end_if;
    
    // Stopping condition: The approximation didn't change significantly 
    // in the last two steps.
    if k >= Low then
      if T[k-2] <> FAIL and T[k-1] <> FAIL and T[k] <> FAIL then
        estErr := abs(T[k] - T[k-1]) + abs(T[k] - T[k-2]);
        if estErr*10^prec < abs(T[k]) then
          return(T[k],T);
        end_if;
      end_if;
    end_if;
  end_for;
  
  userinfo(4, "No convergence:".
              " T[High-2] = ".(T[High-2]).
              " T[High-1] = ".(T[High-1]).
              " T[High] = ".(T[High]));
  
  return(FAIL,T)
end_proc:


// Calculate the next larger diagonal in the (numerator or denominator) triangle
numeric::step_levin :=
proc(K, oldQ, head)
  local Q, k, f, c;
begin
  Q := array(0..K-1);
  Q[0] := head;
  
  f := 1 / K;
  c:= (K-1)/K;
  
  for k from 1 to K-1 do
    Q[k] := Q[k-1] - (K-k)*f*oldQ[k-1];
    f := f * c;
  end_for;
  
  return(Q);
end_proc:


// Levin's transformation barfs when summands are zero.
// Therefore, we skip those. For reasonable cases, the main effect of this
// is to skip over zeros at the start of a series, and to allow summation
// of the zero series.
numeric::levin_next_summand :=
proc(f,i, S)
  local a,j;
begin
  a := 0;
  j := 0;
  
  repeat
    j := j + 1;
    a := a + float(eval(subs(f, i = j)));
    if not domtype(a) in {DOM_FLOAT,DOM_COMPLEX} then
      a := FAIL;
      break;
    end_if;
  until abs(a)*10^DIGITS > abs(S) or j > DIGITS end;
  
  if j > 1 then
    userinfo(5, "Working around tiny (or zero) summands. ".f.",".a.",".S);
  end_if;
  
  return([a,j]);
end_proc:


/* ---------------------------------------------
The Levin transformation works well for alternating real
sequences, since this leads to a numerically stable
evaluation of the accelerated sequence. For non-alternating
real sequences, there is danger of unstable numerical
evaluations. To cure this problem, rewrite the sum over
a non-alternating sequence as a sum over some alternating
sequence (the van Wijngaarden transform of the original
sequence).

van Wijngaarden's transformation:
Turn a monotone series into an alternating one.

The summands of the resulting series are monotone series which are
based on exponentially increasing subsequences of f(i).

--------------------------------------------- */
numeric::levin_wijngaarden :=
proc(f,i, Low,High,prec)
  local j,k,g,result;
begin
  j := genident();
  k := genident();
  
  g := (-1)^(k+1)*hold(numeric::levinsum)(2^(j-1)*subs(f, i=k*2^(j-1)),j, Low,High,prec+2)[1];
  
  // Exponential subsequences can easily cause overflow/underflow errors,
  // so trap errors.
  if traperror((result := numeric::levinsum(g, k, Low,High, prec+2))) = 0 then
    return(result);
  else
    return(FAIL,NIL);
  end_if;
end_proc:



/*  -----------------------
   Euler-MacLaurin formula:
   (old code going as far back as Paul Z.)
  
    sum(f(k), k=a..b) = (f(a)+f(b))/2 + int(f(k),k=a..b)
    + sum(bernoulli(2*m)/(2*m)!* ((D@@(2m-1))(f)(b)-(D@@(2m-1))(f)(a)], m=1..M)
    + Remainder(M)

----------------------- */

//numeric::maclaurin(f,k,a) = sum(f(k),k=a..infinity) 

numeric::maclaurin:= proc(f,k,a)
   local n,m,t,S,S0,macheps,noSymbolic;
begin
   f:= float(f);
   noSymbolic:=FALSE;
   if args(0)<>3 then
      if args(0)=4 and has(args(4), "NoSymbolic") then
         noSymbolic:=TRUE;
      else
         error("expecting three arguments")
      end_if;
   end_if;
   if domtype(a)<>DOM_INT then return(FAIL) end_if;

   // !! stuff like k! etc is popular in sums.
   // Note that k! -> fact(k), but fact(k) cannot be
   // evaluated for k <> DOM_INT. So use fact(k)=gamma(k+1):
   if has(f, fact) then f:= rewrite(f, gamma); end_if;

   // Heuristics: split off a finite sum to improve convergence
   // of Euler-MacLaurin for the infinite tail :
   //    sum(f(k),k=a..infinity) = 
   //              sum(f(k),k=a..n-1)       <- use _plus
   //            + sum(f(k),k=n..infinity)  <- use MacLaurin

   n:= a + 10*DIGITS;

   // S = first approximation via MacLaurin 
   //   = sum(f(k), k=a..n-1) + sum(f(k), k=n..infinity)
   //   = sum(f(k), k=a..n-1) + f(n)/2 + int(f(k), k=n..infinity)

   t:=float(eval(subs(f,k=n))); 
   if not contains({DOM_COMPLEX,DOM_FLOAT},domtype(t)) 
      then return(FAIL)
   end_if;

   // Try symbolic or numerical integration.
   // Use float(int(..)) instead of numeric::quadrature,
   // because int::float (=numeric::int) traps errors from
   // numeric::quadrature.

   if not noSymbolic then
      S0:= float(int(f, k=n..infinity));
   end_if;

   if noSymbolic or has(S0, limit) then
      // int gave a symbolic result, but limit was not 
      // strong enough and survived as symbolic limit.
      // Switch to purely numerical integration:
      S0:= int::float(f,k=n..infinity); 
   end_if;

   // give up, if integration fails:
   if not testtype(S0, Type::Numeric) then return(FAIL) end_if; 

   
   S:=float(eval(
         _plus(eval(subs(f,k=float(m)))$m=a..n-1)//   sum(f(k), k=a..n-1)
         + t/2                                   // + f(n)/2
         + S0                                    // + int(f(k), k=n..infinity)
            ));

   // give up, if something went wrong:
   if not testtype(S, Type::Numeric) then return(FAIL) end_if; 

   //Now add corrections - sum(bernoulli(2*m)/(2*m)!* (D@@(2m-1))(f)(a), m)
   //from Euler-MacLaurin formula until S becomes stationary:
   macheps:= float(10^(-DIGITS)):
   f:=diff(f,k); 
   m:=1; 
   repeat
      t:=float(eval(bernoulli(2*m)/fact(2*m)*subs(f,k=float(n))));
      if not testtype(t, Type::Numeric) then return(FAIL) end_if; 
      S:=S-t; 
      // stopping criterion: last step did not change S significantly:
      if specfunc::abs(t) < macheps * (macheps +specfunc::abs(S)) then break end_if;
      m:=m+1; 
      f:=diff(f,k,k);
   until FALSE end_repeat;

   // We should check in addition that f is (2m+4)-times differentiable with
   // diff(f,k$2*m+2) and diff(f,k$2*m+4) of one sign on 1..infinity !!!

   // Note that sum((-1)^k*f(k), k=a..infinity) 
   // leads to  int((cos(k*PI)+I*sin(k*PI))*f(k), k=a..infinity)
   // with imaginary round-off trash. Hence apply numeric::complexRound:

   return(numeric::complexRound(S, macheps))
end_proc:

// end of file
