/*
numeric::intsolve -- numerical solution of the indefinite 
                     integral int(f(x), x)
Call:

   numeric::intsolve( int(f(x),x) <, t0> <, opts> )

Parameter:

   t0   -- an optional real number (the initial value)
   opts -- all options of numeric::int can be used

Details:

  *) y := numeric::intsolve(int(f(x),x), t0, <,options>) 
     returns the function

     y :=  t ->  numeric::int(f(x), x=y0..t, <,options>)

     See help page of numeric::int for details on parameters
     and options.
  *) The returned function y can only be called with numerical 
     real arguments. The call y(t) then returns a numerical 
     approximation of the exact solution int(f(x), x=y0..t).
  *) y is defined with option remember: it remembers all 
     values it has ever computed. The next call y(t) takes 
     T0 from its remember table that is closest to t
     and the corresponding y(T0). Then it executes
            y(T0) + numeric::int(f(x), x=T0..t )
  *) Do not increase DIGITS, between calls of y.
     Because of option remember the results may be incorrect!
     Clear the remember table by y:= subsop(y, 5=NIL) first!

Examples:

>> y := numeric::intsolve( int(Ci(x),x) )

   proc y(t) ... end

>> y(2), y(3/4), y(-5.5)

   -0.06333576927, -0.5675160593, 0.07575088557 - 17.2787596 I

>> plotfunc2d( y(x) )

>> y := numeric::intsolve( int(Si(x),x), 1 ):
>> plotfunc2d( f(x), x=2..5 )

*/

numeric::intsolve  := proc(intdef, t0=0)
   local  f, n, opts, x;
begin
   n := args(0); 
   opts := null(); 
   if n < 1 then 
      error ("expecting at least 1 argument"); 
   end_if;
   if type(intdef) <> "int" or nops(intdef) <> 2 then
      error("expecting int(.,.) as first argument")
   end_if;
   f := op(intdef, 1);
   x := op(intdef, 2);
   if domtype(x) <> DOM_IDENT then
      error("expecting an indefinite integral int(..., x)")
   end_if;
   if n > 1 then 
      if domtype(float(t0)) = DOM_FLOAT then 
         opts := args(3..n);
      else
         opts := args(2..n);
         t0   := 0;
      end_if
   end_if;
   numeric::intsolve2(f, x, t0, opts)
end_proc:

/*
numeric::intsolve2 -- numerical solution of the indefinite 
                      integral int(f(x), x)
Call:

   numeric::intsolve2( f, x, t0 <, opts> )

Parameter:

   f(x) -- an expression in x
   x    -- an identifier
   t0   -- a real number (the initial value)
   opts -- all options of numeric::int can be used

Examples:

>> y := numeric::intsolve2(Ci(x), x, 0)
>> y := numeric::intsolve2(sin(x), x, 2)

Details: See numeric::intsolve
*/

numeric::intsolve2 := proc(f, x, t0)
   option escape;
   local  opts;
begin
   if args(0) < 3 then 
      error ("expecting at least 3 arguments"); 
   end_if;
   opts := args(4..args(0));

   proc(t)
   local y0, remembertable, n, dt, i, tt, dtt;
   option remember;
   begin
     if domtype(t) = DOM_IDENT then
        return(op(op(_act_proc_env(),1),6)(t))
     end_if;

     y0:= 0;
     // Search the remember table op(thisproc, 5) of this
     // function for precomputed values. Find the point t0 from
     // the remember table that is closest to t.

     // Use remember table = op(op(_act_proc_env(),1),5)
     remembertable:= op(op(_act_proc_env(),1),5);
     if domtype(remembertable) = DOM_TABLE then
       remembertable:= [op(remembertable)];
       n:= nops(remembertable);
       dt:= infinity;
       for i from 1 to n do // search for tt in remember table
                            // that is closest to t.
         tt:= float(op(remembertable[i],1));
	 if domtype(tt) <> DOM_FLOAT then
	    next
	 end_if;
         if (dtt:= specfunc::abs(float(t-tt)))<dt then
            t0:= tt;
            y0:= op(remembertable[i],2);
            dt:= dtt;
         end_if;
       end_for;
     end_if;
     // Integrate from t0 to t with initial value  
     // y0 = numeric::int(f, x=s..t0) ) from the
     // remember table:
     y0 + numeric::int(f, x=t0..t, opts)
     // numeric::int(f, x=s..t) =
     // numeric::int(f, x=s..t0) + numeric::int(f, x=t0..t) =
     // y0 + numeric::int(f, x=t0..t)
     end_proc

  end_proc:

