/*-------------------------------------------------------------------

numeric::isnonzero -- heuristic numerical test whether a (constant)
                      algebraic expression is non-zero

numeric::isless -- heuristic numerical test whether 2 expressions 
                   are real and one is smaller than the other

numeric::isreal -- heuristic numerical test whether a (constant)
                   algebraic expression is real

Call(s)     : numeric::isnonzero(x)
              numeric::isless(x, y)
              numeric::isreal(x)

Parameters  : x, y -- algebraic expressions
Return Value: TRUE, UNKNOWN

Details: numeric::isnonzero:
              TRUE is returned, if float(x) indicates that
              float(x) is unlikely to coincide with float(0),
              even if DIGITS is increased.

              FALSE is returned if x is an exact number
              0 (of type DOM_INT) or 0.0 (of type DOM_FLOAT)

              UNKNOWN is returned if x (or y) contain symbolic objects,
              or if the numerical test is not decisive, 
              because float evaluation is subject to severe 
              round-off problems.

           numeric::less:
              TRUE is returned if float(y - x) > 0 and
               if it is unlikely that float(y - x) <= 0
               even if DIGITS is increased
              FALSE is returned if float(y - x) <= 0 and
               if it is unlikely that float(y - x) < 0
               even if DIGITS is increased
              UNKNOWN is returned if x or y contain symbolic objects,
               or if the numerical test float(y - x) > 0 is not
               decisive, because float evaluation is subject to 
               severe round-off problems.

           numeric::isreal:
              TRUE is returned if float(x) is real and
               if it is unlikely that float(x) is non-real
               even if DIGITS is increased
              FALSE is returned if Im(float(x) <> 0 and
               if it is unlikely that Im(float(x)) = 0
               even if DIGITS is increased
              UNKNOWN is returned if x or y contain symbolic objects,
               or if the numerical test Im(float(x)) <> 0 is not
               decisive, because float evaluation is subject to 
               severe round-off problems.


Roughly speaking: 
         The answer TRUE/FALSE is reliable (with very high probability).
         If you get TRUE/FALSE, then you do not need to increase DIGITS
         to doublecheck.
  
         If you get UNKNOWN, then it is up to you to try again 
         with increased DIGITS, which then might yield a 
         decisive TRUE/FALSE.

Examples:

>> numeric::isnonzero(x)

                            UNKNOWN

>> numeric::isless(x, PI)
                            UNKNOWN

>> numeric::isnonzero(sqrt(24) - 3*sqrt(6))

                 TRUE     // this is a decisive answer !

>> numeric::isless(sqrt(24) , 3*sqrt(6))

                 TRUE     // this is a decisive answer !

>> numeric::isnonzero(sqrt(24) - 2*sqrt(6))

                 UNKNOWN

>> DIGITS:= 50: pi:= numeric::rationalize(float(PI)):
>> DIGITS:= 10:
>> float(PI - pi), 
   numeric::isnonzero(PI - pi),
   numeric::isless(PI , pi),
   numeric::isless(pi , PI);

                0.0, UNKNOWN, UNKNOWN, UNKNOWN

>> DIGITS:= 30:
>> float(PI - pi), 
   numeric::isnonzero(PI - pi),
   numeric::isless(PI , pi),
   numeric::isless(pi , PI);

                0.0, UNKNOWN, UNKNOWN, UNKNOWN

>> DIGITS:= 50:
>> float(PI - pi), 
   numeric::isnonzero(PI - pi),
   numeric::isless(PI , pi),
   numeric::isless(pi , PI);

-4.179....e-51, TRUE, TRUE, FALSE

-----------------------------------------------------------*/

numeric::isnonzero:= proc(x)
local x1, x2;
save DIGITS;
begin
   if has(x, infinity) then
      if type(x) = "stdlib::Infinity" then
         // x = symbol * infinity. 
         // This isnonzero, if symbol = op(x) isnonzero:
         return(numeric::isnonzero(op(x)));
      end_if;
      if indets(x) minus Type::ConstantIdents = {} then
         // enforce consistency with float
         if traperror((x := float(x)))<> 0 then 
           return(UNKNOWN);
         end_if;
         if iszero(x) then
           return(UNKNOWN);
         end_if;
      else
         return(UNKNOWN);
      end_if;
   end_if;

   case expr2text(domtype(x))
   of "DOM_INT" do
   of "DOM_RAT" do
   of "DOM_FLOAT" do
   of "DOM_COMPLEX" do
      if iszero(x) then
           return(FALSE)
      else return(TRUE)
      end_if
// of "stdlib::Infinity" do
   of "unit" do    
      return(TRUE)
   end_case;

   // float test
   if traperror((x1:= float(x))) <> 0 then
      return(UNKNOWN);
   end_if;
   if not has({DOM_FLOAT, DOM_COMPLEX}, domtype(x1)) then
      return(UNKNOWN);
   end_if:
   if iszero(x1) then
      // this could be total cancellation. Check with
      // higher precision:
      if traperror((x2:= float(x, 2*DIGITS))) <> 0 then
         return(UNKNOWN)
      end_if;
      if not iszero(x2) then
         // try again with higher DIGITS
	 DIGITS:= 2*DIGITS;
         return(numeric::isnonzero(x));
      else
         return(UNKNOWN) 
      end_if:
   end_if:
   if x1 = RD_NAN then 
      return(UNKNOWN) 
   end_if:
   if x1 = RD_INF or x1 = RD_NINF then
     return(TRUE)
   end_if;
   // Heuristics: increase DIGITS to verify
   // that float(x) is not complete round-off trash.
   // We need at least DIGITS + 10 to make sure,
   // that PARI adds a further block:

   // We do not want to treat real and non-real x separately,
   // so replace x by abs(x). The chances of x staying on
   // the same circle, when DIGITS is increased, is minimal,
   // i.e., checking the stability of abs(x) w.r.t. 
   // increasing DIGITS is heuristically ok.

   x1:= specfunc::abs(x1):
   DIGITS:= DIGITS + 10;
   if traperror((x2:= specfunc::abs(float(x)):)) <> 0 then
      return(UNKNOWN);
   end_if;
   DIGITS:= DIGITS - 10;
   if not has({DOM_FLOAT, DOM_COMPLEX}, domtype(x2)) or
      x2 = RD_NAN then
      return(UNKNOWN);
   end_if:
   // The leading digit of x1 is reliable, if 1/2 < x1/x2 < 2.
   // We do a stricter check: 1/1.1 < x1/x2 < 1.1:
   if (x1 < 1.1*x2 and x2 < 1.1*x1)
      then return(TRUE)
      else // no chance to decide the magnitude of
           // x via float with the present DIGITS.
           // Severe cancellation! 
           return(UNKNOWN)
   end_if;
end_proc:

//------------------------------------------------
numeric::isreal:= proc(x)
local x1, x2;
save DIGITS;
begin
   case expr2text(domtype(x))
   of "DOM_INT" do
   of "DOM_RAT" do
   of "DOM_FLOAT" do
   of "unit" do
      return(TRUE);
   of "DOM_COMPLEX" do
      return(FALSE)
   end_case;

   // float test
   if traperror((x1:= float(x))) <> 0 then
      return(UNKNOWN);
   end_if;
   if not has({DOM_FLOAT, DOM_COMPLEX}, domtype(x1)) then
      return(UNKNOWN);
   end_if:

   // Heuristics: increase DIGITS to verify
   // that float(x) is not complete round-off trash.
   // We need at least DIGITS + 10 to make sure,
   // that PARI adds a further block:

   // We do not want to treat real and non-real x separately,
   // so replace x by abs(x). The chances of x staying on
   // the same circle, when DIGITS is increased, is minimal,
   // i.e., checking the stability of abs(x) w.r.t. 
   // increasing DIGITS is heuristically ok.

   DIGITS:= DIGITS + 10;
   if traperror((x2:= float(x):)) <> 0 then
      return(UNKNOWN);
   end_if;
   DIGITS:= DIGITS - 10;

   if {domtype(x1), domtype(x2)} <> {DOM_COMPLEX} then
      return(UNKNOWN);
   end_if;
      
   // remaining case: both x1 and x2 have a non-zero
   // imaginary part. Is it trustworthy?
   x1:= specfunc::abs(op(x1, 2)):  // abs(Im(float(x)));
   x2:= specfunc::abs(op(x2, 2)):  // abs(Im(float(x)));

   // the leading digit of x1 is reliable, if 1/2 < x1/x2 < 2:
   if (x1<1.1*x2 and x2<1.1*x1) then
           return(FALSE)
      else // no chance to decide the magnitude of
           // x via float with the present DIGITS.
           // Severe cancellation! 
           return(UNKNOWN)
   end_if;
end_proc:

//------------------------------------------------
numeric::isless:= proc(x, y)
local fx, fy, z, fz, fzz, absfz, absfzz, xisreal, yisreal;
begin
      case domtype(x)
      of DOM_INT do
      of DOM_RAT do
      of DOM_FLOAT do
         case domtype(y)
         of DOM_INT do
         of DOM_RAT do
         of DOM_FLOAT do
            return(bool(x < y));
         of DOM_COMPLEX do
            return(FALSE)
         end_case;
         break;
      of DOM_COMPLEX do
         return(FALSE)
      end_case;

      if traperror((fy:= float(y))) <> 0 then
         return(UNKNOWN);
      end_if;
      if type(x) = unit and 
         type(fy) = DOM_FLOAT and
         fy <= 0 then 
            return(FALSE)
      end_if:
      if traperror((fx:= float(x))) <> 0 then
         return(UNKNOWN);
      end_if;
      if type(y) = unit and 
         type(fx) = DOM_FLOAT and
         fx <= 0 then 
            return(TRUE)
      end_if:

      //---------------------------------
      // safe float check that x is real:
      //---------------------------------
      xisreal:= numeric::isreal(x):
      if xisreal = FALSE then
         return(FALSE);
      end_if;
      //---------------------------------
      // safe float check that y is real:
      //---------------------------------
      yisreal:= numeric::isreal(y):
      if yisreal = FALSE then
         return(FALSE);
      end_if;

      //-------------------------------------
      // Here, we are reasonably sure that
      // both x and y are real.
      // Safe float check that z = y - x > 0:
      //-------------------------------------

      z:= y - x;
      if iszero(z) then 
         return(FALSE)
      end_if:

      //------------------------------------------
      // float test:
      // use traperror around float because it
      // tends to produce underflow/overflows
      //------------------------------------------

      if traperror((
         fz:= float(z);
         )) <> 0 then
         return(UNKNOWN);
      end_if;

      if not contains({DOM_FLOAT, DOM_COMPLEX}, domtype(fz)) then
         return(UNKNOWN);
      end_if;
                    
      case fz
        of RD_INF do
          return(TRUE)
        of RD_NINF do
          return(FALSE)
        of RD_NAN do
          return(FALSE)
      end_case;

      if traperror((
         fzz:= float(z, 2*DIGITS);
         )) <> 0 then
         return(UNKNOWN);
      end_if;

      if not contains({DOM_FLOAT, DOM_COMPLEX}, domtype(fzz)) then
         return(UNKNOWN);
      end_if;
                    
      case fzz
        of RD_INF do
          return(TRUE)
        of RD_NINF do
          return(FALSE)
        of RD_NAN do
          return(FALSE)
      end_case;

      if xisreal = TRUE and yisreal = TRUE then
         fz:= op(fz, 1); // throw away the complex part
         fzz:= op(fzz, 1); // throw away the complex part
      end_if;

      if {domtype(fz), domtype(fzz)} = {DOM_FLOAT} then 
        absfz:= specfunc::abs(fz);
        absfzz:= specfunc::abs(fzz);
        // the leading digit of absfz is reliable, if 1/2 < absfz/absfzz < 2:
        if absfz < 1.1*absfzz and absfzz < 1.1*absfz then
           // fzz is reliable
           if fzz > 0 then   // a decisive x < y
              return(TRUE)
           elif fzz < 0 then // a decisive x > y
              return(FALSE)
           end_if;
        else
           // no chance to decide the magnitude of
           // y - x via float with the present DIGITS.
           // Severe cancellation! 
           return(UNKNOWN)
        end_if;
      end_if:

      //--------------------------------------------------------------
      // If Im(x) or Im(y) consists of tiny round-off trash, we still
      // can decide x < y to be FALSE, if we can decide Re(x) > Re(y). 
      //--------------------------------------------------------------
      // at least one of the numbers has imaginary roundoff trash
      fz:= op(fz, 1);   // just consider fz = Re(z)
      fzz:= op(fzz, 1); // just consider fzz = Re(zz) 
      absfz:= specfunc::abs(fz);
      absfzz:= specfunc::abs(fzz);
      if absfz < 1.1*absfzz and absfzz < 1.1*absfz then
        // fzz is reliable
        if fzz < 0 then // a decisive x > y
           return(FALSE)
        elif fzz > 0 then // a decisive x < y
           return(UNKNOWN); // we were not sure whether x, y are real.
        end_if;
      end_if;

      return(UNKNOWN)
end_proc:
