// 
//       
// 
// W.Oevel, 06/04/98 
//------   file  linsolve.mu   6.4.98     ------------------------
//-------------------  W. Oevel  ---------------------------------
/*  Help file:

 linsolve - solve a system of linear equations numerically

 Call:

 numeric::linsolve( eqs  <, vars> 
                    <, hard_soft>
                    <, Symbolic> 
                    <, ShowAssumptions> 
                    <, NoWarning>
                    <, Normal = mynormal>
                    <, Iszero = myiszero>
                   )
                                                                   
 Parameter:

  eqs         --  a list or set of linear equations or linear expressions
                  (interpreted as equation expr = 0)
                  Also lists/sets of linear DOM_POLY polynomials are accepted.
  vars        --  a list or set of unknowns
  hard_soft   --  Soft, SoftwareFloats, Hard, HardwareFloats
  Symbolic    --  option, prevents conversion of coefficients to floats
  ShowAssumptions
              --  results in more detailed information about the
                  internal assumptions on symbolic parameters in the
                  equations (see below)
  NoWarning   --  no warning is given if linsolve switches to
                  the 'symbolic mode' due to symbolic parameters
  "dont_use_normal" -- undocumented option. Prevents the internal
                       use of normal. 
  mynormal    --  user defined procedure, accepting one expression as argument
                  and returning the 'normalized' version of this argument.
                  This procedure is used internally whereever a 'normal'
                 is used.
  myiszero    --  user defined procedure accepting one expression as argument
                  and returning TRUE or FALSE. This procedure is used internally
                  when searching for pivot elements. In the pivot search, the
                  first element with myiszero(thiselement) = FALSE is used as
                  the pivot element.
                                                                   
Synopsis:

  Let vars=[x[1],x[2],..] or vars={x[1],x[2],..} be the unknowns.

  numeric::linsolve(eqs,vars, <Symbolic>); returns  Solvedeqs

  numeric::linsolve(eqs,vars, <Symbolic>, ShowAssumptions); returns 
            [Solvedeqs, Constraints, PivotAssumptions]

  (*) Solvedeqs is a list of equations of the form x[1]=..,x[2]=..,..
      representing the solution of eqs. Those unknowns not subject
      to any constraint (free parameters) will not turn up
      on the left hand sides in Solvedeqs. E.g., if vars=[x[1],x[2]]
      and Solvedeqs=[x[2]=..], then x[1] is a free parameter spanning
      the kernel of eqs. Solvedeqs=[] corresponds to a trivial system
      eqs, where all unknowns are arbitrary.

      With assign(Solvedeqs) the solution may be assigned to the unknowns.

  (*) Without option ShowAssumptions:

     (*) Warning: if the equations contain symbolic parameters, then only 
         those solutions are returned which hold for arbitrary values of 
         these parameters.

         Symbolic denominators occuring in the final result or in
         intermediate results are implicitly assumed to be non-zero. 
         The fact that they may vanish for certain values of the 
         symbolic parameters is ignored.

  (*) With option ShowAssumptions:

     (*) PivotAssumptions is a list of inequalities involving the
         symbolic parameters of the coefficient matrix. The
         corresponding expressions appeared as symbolic pivot elements
         in the internal Gauss elimination. They are assumed to be
         nonzero and appear as denominators in the final result or in 
         itermediate results.

         If no symbolic pivot elements were used, then PivotAssumptions
         is the empty list. This will always be the case when
         there are no symbolic parameters in the coefficient
         matrix.

         The inequalities arising in PivotAssumptions are 
         determined by the algorithm during run time. The user
         has no control over these assumptions. In particular,
         changing the ordering of the equations in eqs or the
         ordering of the variables in vars may result in a
         different solution returned by linsolve.

     (*) Constraints is a list of equations involving the symbolic parameters
         in the equations.

         eqs is solvable if and only if the
         equations given by Constraints are satisfied.

         Such constraints arise only when the coefficient matrix 
         does not have full rank.

         If no such contraints arise, then 
         the returned value of Constraints is the empty list.

     (*) Under the assumptions PivotAssumptions and assuming solvability
         of eqs (i.e., assuming the equations given by Constraints),
         Solvedeqs represents the general solution of eqs.

  If the unknowns vars are not specified, then the symbolic parameters
  in eqs are assumed as unknowns, i.e., vars=indets(eqs, PolyExpr).

  Warning: linsolve does not check that the equations eqs are indeed
  linear in the unknowns vars. For non-linear systems linsolve may
  return wrong results!

  By default all coefficients in eqs are converted to floats. No 
  non-numeric symbolic parameters are allowed as coefficients of 
  the unknowns, unless the option Symbolic is used. In the symbolic 
  mode no conversion to floats will occur. We note that the presence 
  of non-numeric symbolic coefficients will slow down the computation
  drastically.

  Gaussian elimination with partial pivoting is used by linsolve.
  If no solution exits, then linsolve returns FAIL.

  Examples: 

>> numeric::linsolve({x=y-1,x+y=0},{x,y});

                            [x = -0.5, y = 0.5]

>> numeric::linsolve({x=y-1,x+y=0},{x,y}, ShowAssumptions);

                         [[x = -0.5, y = 0.5], [], []]

>> numeric::linsolve([x+y=1,y=z],[x,y,z],Symbolic);

                            [x = 1 - z, y = z]

>> numeric::linsolve([x+y=1,y=z],[x,y,z],Symbolic, ShowAssumptions);

                         [[x = 1 - z, y = z], [], []]

>> numeric::linsolve({x+y=1,x+y=a},{x,y},Symbolic);

                                   FAIL

>> numeric::linsolve({x+y=1,x+y=a},{x,y},Symbolic, ShowAssumptions);

                        [[x = 1 - y], [a - 1 = 0], []]

>> numeric::linsolve({x+y=1,x+y=a},{x,y,a});

                          [a = 1.0, x = 1.0 - y]

>> numeric::linsolve({x+y=1,x+y=a},{x,y,a}, ShowAssumptions);

                       [[a = 1.0, x = 1.0 - y], [], []]

>> numeric::linsolve({x=x-y+1,y=PI},Symbolic);

                             FAIL

>> numeric::linsolve({x=x-y+1,y=PI},Symbolic, ShowAssumptions);

                     [[y = 1], [PI - 1 = 0], []]

>> numeric::linsolve({x=x-y+1,y=PI},{x,y},Symbolic);

                                   FAIL

>> numeric::linsolve({x=x-y+1,y=PI},{x,y},Symbolic, ShowAssumptions);

                          [[y = 1], [PI - 1 = 0], []]

>> numeric::linsolve({x=x-y+1,y=PI},{x,y});         

                                   FAIL

>> numeric::linsolve({f(a)=sin(x),f(a)=1-sin(x)},{f(a),sin(x)});

                        [sin(x) = 0.5, f(a) = 0.5]


>> numeric::linsolve({f(a)=sin(x),f(a)=1-sin(x)},{f(a),sin(x)}, ShowAssumptions);

                   [[sin(x) = 0.5, f(a) = 0.5], [], []]


>> numeric::linsolve([a*x+b*y= A, c*x + d*y = B], [x]);

                                   FAIL

>> numeric::linsolve([a*x+b*y= A, c*x + d*y = B], [x], Symbolic, ShowAssumptions);
Warning: symbolic coefficient found, switching to symbolic mode

    -- --     A - b y --                                            --
    |  |  x = -------  |, [B a - A c - a d y + b c y = 0], [a <> 0]  |
    -- --        a    --                                            --

// Do it again with a=0: 

>> numeric::linsolve([b*y= A, c*x + d*y = B], [x], ShowAssumptions);


             -- --     B - d y --                          --
             |  |  x = -------  |, [A - b y = 0], [c <> 0]  |
             -- --        c    --                          --

// Do it again with a=0, c=0: 

>> numeric::linsolve([b*y= A, d*y = B], [x], Symbolic, ShowAssumptions);

                   [[], [b y - A = 0, d y - B = 0], []]

>> numeric::linsolve([a*x+b*y= A, c*x + d*y = B], [x, y]);

                     --     A d - B b      B a - A c --
                    |  x = ---------, y = ---------  |
                    --     a d - b c      a d - b c --

>> numeric::linsolve([a*x+b*y= A, c*x + d*y = B], [x, y], Symbolic, ShowAssumptions);

  -- --     A d - B b      B a - A c --                               --
  |  |  x = ---------, y = ---------  |, [], [a <> 0, a d - b c <> 0]  |
  -- --     a d - b c      a d - b c --                               --

>> numeric::linsolve([a*x+b*y= A, c*x + d*y = B, x = y], [x, y], Symbolic);
 
                         FAIL

>> numeric::linsolve([a*x+b*y= A, c*x + d*y = B, x = y], [x, y], Symbolic, ShowAssumptions);

-- --       B          B   --                                            --
|  |  x = -----, y = -----  |, [A c - B a - B b + A d = 0], [c + d <> 0]  |
-- --     c + d      c + d --                                            --

>> eqs:= [x[1]+eps*x[2]=1+eps, eps*x[1]+x[2]=1+eps, 2*x[1]+3*x[2]=5]:
>> numeric::linsolve(eqs, [x[1], x[2]], Symbolic);

                           [x[1] = 1, x[2] = 1]

>> numeric::linsolve(eqs, [x[1], x[2]], Symbolic, ShowAssumptions);

                                                   2
                [[x[1] = 1, x[2] = 1], [], [1 - eps  <> 0]]
                 
>> eqs:=[x+y=1,x+y=a]:
   ([Solution,Constraints, Assumptions]):=
   numeric::linsolve(eqs,{x,y}, ShowAssumptions);

                  [[x = 1.0 - y], [1.0 a - 1.0 = 0], []]

>> subs(eqs,Solution);

                            [1.0 = 1, 1.0 = a]

>> assign(Solution): eqs;

                            [1.0 = 1, 1.0 = a]


>> numeric::linsolve([a*x + b*y=c, b*x -a*y , a*b*x],[x,y,z],Symbolic,ShowAssumptions);

-- --       a c          b c   --      2                       2    2
|  |  x = -------, y = -------  |, [- a  b c = 0], [a <> 0, - a  - b  <> 0]
|  |       2    2       2    2  |
-- --     a  + b       a  + b  --

   --
    |
    |
   --

>> numeric::linsolve([b*x -a*y , a*x + b*y=c, a*b*x],[x,y,z],Symbolic,ShowAssumptions);

-- --       a c          b c   --      2                     2    2
|  |  x = -------, y = -------  |, [- a  b c = 0], [b <> 0, a  + b  <> 0]
|  |       2    2       2    2  |
-- --     a  + b       a  + b  --

   --
    |
    |
   --

>> numeric::linsolve([a*b*x, a*x + b*y=c, b*x -a*y ],[x,y,z],Symbolic,ShowAssumptions);

          -- --            c --                                --
          |  |  x = 0, y = -  |, [a c = 0], [a b <> 0, b <> 0]  |
          -- --            b --                                --

// the user may conclude from the last output: a*b<>0 -> a<>0,b<>0 -> c=0 
// These simplifications were not done automatically by linsolve

>> numeric::linsolve([a*b*x, b*x -a*y, a*x + b*y=c ],[x,y,z],Symbolic,ShowAssumptions);

              [[x = 0, y = 0], [c = 0], [a b <> 0, -a <> 0]]


See also:  linsolve, linalg::matlinsolve, numeric::matlinsolve
                                                                  */
//-----------------------------------------------------------------
numeric::linsolve:=proc(eqs, vars)
local useHardwareFloats, HardwareFloatsRequested, 
      A, uservars, extravars, kdim,
      newvars, newvarsSet, subseqs, neweqs,
      neqs, n, method, coefftypes, use_normal, 
      dont_use_normal_requested,
      i, j, tmp, B, b, triv, check,
      showassumptions, Arg, eqsSymbolicParameters, varsSet, dowarn,
      x, X, Xvars, nTerms, koeff, np2, np2mj, p,
      piv, pivot, pivindex, rownorm, constraints,
      pnorm, Bnorm,
      PivotAssumptions, lowerbands, jpl, minlength,
      result, kernindices,
      mynormal, myiszero;
begin
  if args(0)<1 then error("not enough arguments"); end_if;
  if args(0)>9 then error("expecting at most seven arguments");end_if; // 7 arguments are documented

  //------------------------------
  // set defaults
  //------------------------------
  method:= 2;  // default: numeric mode
  showassumptions:= FALSE; // default
  dowarn:= TRUE; // default
  uservars:= FALSE;
  if DIGITS <= 15 then
       useHardwareFloats:= TRUE;
  else useHardwareFloats:= FALSE;
  end_if;
  HardwareFloatsRequested:= FALSE;
  dont_use_normal_requested:= FALSE;
  mynormal:= x -> normal(x, Expand = FALSE);
//myiszero:= x -> contains({TRUE, UNKNOWN}, linalg::zeroTest(x));
  myiszero:= iszero;

  //------------------------------
  // check the optional parameters
  //------------------------------
  if args(0)>1 then
    for Arg in [args(2..args(0))] do
      case Arg
      of Hard do
      of HardwareFloats do
            HardwareFloatsRequested:= TRUE;
            if method = 1 and dowarn then
               warning("ignoring the option 'HardwareFloats' ".
                       "because of the option 'Symbolic'"):
               method:= 1; // symbolic method
               useHardwareFloats:= FALSE:
               HardwareFloatsRequested:= FALSE;
            else
               useHardwareFloats:= TRUE:
            end_if;
            break;
      of Soft do
      of SoftwareFloats do
            useHardwareFloats:= FALSE:
            HardwareFloatsRequested:= FALSE;
            break;
      of Symbolic do 
            method:=1;
            useHardwareFloats:= FALSE:
            HardwareFloatsRequested:= FALSE;
            break;
      of ShowAssumptions do
            showassumptions:= TRUE;
            useHardwareFloats:= FALSE:
            HardwareFloatsRequested:= FALSE;
            break;
      of NoWarning do 
            dowarn:= FALSE;
            break;
      of "dont_use_normal" do
            dont_use_normal_requested:= TRUE;
            break;
      otherwise
            if domtype(Arg)=DOM_LIST or 
               domtype(Arg)=DOM_SET then
              uservars:= TRUE; // the second argument is
                               // a list of indeterminats
              break;
            end_if;
            if type(Arg) = "_equal" then
               if op(Arg, 1) = Normal then
                 mynormal:= op(Arg, 2):
                 break;
               end_if:
               if op(Arg, 1) = Iszero then
                 myiszero:= op(Arg, 2):
                 break;
               end_if:
            end_if:
            error("unknown option");
      end_case;
    end_for:
  end_if;
     
  //-----------------------------------------
  // pre-processing
  //-----------------------------------------
  if eqs::dom::hasProp(Cat::Matrix)=TRUE then
     eqs:= expr(eqs):
  end_if:
  case domtype(eqs)
  of DOM_SET do
  of DOM_ARRAY do
     eqs:= [op(eqs)];
  end_case;

  if not contains({DOM_SET, DOM_LIST}, domtype(eqs)) then 
     error("expecting a set, a list, or a matrix of equations or expressions")
  end_if;
  
  // convert polynomials 
  (if testtype(eqs[i],DOM_POLY) 
      then eqs[i]:= expr(eqs[i]);
   end_if;) $ i=1.. nops(eqs);

  // convert equations a=b to expressions a-b 
  (if testtype(eqs[i],"_equal") then 
      eqs[i]:= op(eqs[i],1)-op(eqs[i],2);
  end_if;) $ i=1.. nops(eqs);
   // force full evaluation of the equations: 
// (eqs[i]:=eval(eqs[i]);) $ i=1..nops(eqs);

  if nops(eqs)=0 then
     if showassumptions then return([[],[],[]])
                        else return([])
     end_if
  end_if;

  //-----------------------------------------------------
  // convert a set of indeterminates into a *sorted* list
  // of indeterminates. Use DOM_SET::sort to sort the
  // set, such that on output the unknowns appear in the
  // same ordering as in the input set
  //-----------------------------------------------------
 
  if args(0)=1 or not uservars 
  then // get rid of Type::ConstantIdents (i.e., PI etc.).
       varsSet:= indets(eqs, PolyExpr) minus Type::ConstantIdents;
       // Note that indets(..,PolyExpr) enters things such as
       // 1/PI into varsSet. Eliminate this trash:
       extravars:= varsSet minus map(varsSet, float):
       // pick out those extravars that do not contain
       // symbolic variables
       extravars:= select(extravars, 
                          var ->  case domtype(float(var))
                                  of DOM_FLOAT do
                                  of DOM_COMPLEX do
                                     return(TRUE)
                                  otherwise
                                     return(FALSE)
                                  end_case):
       varsSet:= varsSet minus extravars;
       vars:=DOM_SET::sort(varsSet);
  else // linsolve is called with vars 
       if not contains({DOM_SET, DOM_LIST}, domtype(vars))
       then error("expecting a set or a list of unknowns")
       end_if;
       // force evaluation of vars ?
       // convert vars to List 
       if domtype(vars)<>DOM_LIST
          then //Variables are also needed as set 
               varsSet:= vars;
               vars:=DOM_SET::sort(vars);
          else varsSet:= {op(vars)};
       end_if; 
  end_if;

  n:=nops(vars);

  if n=0 then
     // eliminate trivial equations
     eqs:= map(eqs, proc(eq) begin 
                      if myiszero(eq)
                         then null()
                      else eq end_if
                    end_proc);
     if eqs =[] and showassumptions=FALSE  then return([]) end_if;
     if eqs =[] and showassumptions=TRUE   then return([[], [], []]) end_if;
     if eqs<>[] and showassumptions=FALSE  then return(FAIL) end_if;
     if eqs<>[] and showassumptions=TRUE   then
                if indets(eqs,PolyExpr)={} then return([FAIL,[], []])
                                           else return([[], eqs, []])
                end_if;
     end_if;
  end_if;

  neqs:=nops(eqs);

  //--------------------------------
  // ----- use HardwareFloats ------
  //--------------------------------
  case useHardwareFloats // use case because we want to use break below
  of TRUE do

     if DIGITS > 15 then
        if HardwareFloatsRequested and dowarn then
           warning("the current precision goal DIGITS = ".expr2text(DIGITS).
                   " cannot be achieved with the requested 'HardwareFloats'.".
                   " Using 'SoftwareFloats' instead.");
        end_if;
        break;
     end_if;

     if not numeric::startHardwareFloats("matlinsolve") then
        userinfo(1, "cannot start the hardware floats interface"):
        if HardwareFloatsRequested and dowarn then
           warning("cannot start the 'HardwareFloats' interface, ".
                   "using 'SoftwareFloats' instead");
        end_if;
        break;
     end_if;

     userinfo(1, "trying hardware floats"):

     //--------------------------------------
     // compute coefficient matrix
     //--------------------------------------

     // vars = a list of indeterminates provided by the user
     // These can be expressions. This is a problem because
     // of diff(x[i], x[j]) <> 0! Replace every indeterminate
     // by a freshly generated identifier:
     newvars:= map(vars, var -> if domtype(var) <> DOM_IDENT then
                                     return(genident("x")) 
                                else return(var);
                                end_if);
     newvarsSet:= {op(newvars)};

     // generate the equations to be passed to subs
     // subs(eqs, [x = x, f(y) = x1, z = z]).
     // Better, reduce this to
     // subs(eqs, [f(y) = x1]):
     subseqs:= zip(vars, newvars, 
                   (old, new) -> if old = new then 
                                      return(null())
                                 else return(old = new)
                                 end_if):

     // First substitute then float (otherwise, an
     // unknown such as f(PI) would be floated!)
     // We need a backup of the original eqs, if
     // HardwareFloats should FAIL, so copy the 
     // modified eqs into neweqs

     neweqs:= float(subs(eqs, subseqs));

     // convert the symbolic equations to an hfarray
     if traperror((

     // hfa::expr2HFA breaks a set of linear expressions
     // [a11*x1 + a12*x2 + ... + a.1.n * x.n + b1 (= 0),
     //  a21*x1 + a22*x2 + ... + a.2.n * x.n + b2 (= 0),
     //  ...]
     // in x1, x2, ... into the coefficient matrix A and
     // the (negative) right hand side b:

     [A, B]:= hfa::expr2HFA(neweqs, newvars);


/* -------------------------------------------------
        /* ------------------------------------------------
           given a list of expressions
           eqs = [x1*c11 + x2*c12 + .. x.n*c.1.n - b1,
                  x1*c21 + x2*c22 + .. x.n*c.2.n - b2, ...]
           with symbolic unknowns x1, x2, ..., x.n
           (stored in the list newvars) and floating
           point coefficients c.i.j, extract the c.i.j
           as a matrix (e.g., as hfarray):
        ---------------------------------------------------*/
        A:= hfarray(1..neqs, 1..n):
        B:= hfarray(1..neqs, 1..1):
        varindices:= table((newvars[j] = j) $ j = 1..n);
        for i from 1 to neqs do
          if type(neweqs[i]) = "_plus" then
             for tmp in neweqs[i] do
                  j:= varindices[op(tmp, 1)];
                  if domtype(j) = DOM_INT then
                     if op(tmp, 2) = FAIL then
                       // execeptional case: tmp = x[j] * 1
                       A[i, j]:= 1;
                     else
                       A[i, j]:= op(tmp, 2);
                    end_if;
                  else
                     B[i, 1]:= -tmp:
                  end_if:
             end_for;
          else //only one term neweqs[i] = x[j]*c.i.j
             j:= varindices[op(neweqs[i], 1)];
             if domtype(j) = DOM_INT then
                if op(eqs[i], 2) = FAIL then
                   // execeptional case: neweqs[i] = x.j * 1 
                   A[i, j]:= 1;
                else
                   A[i, j]:= op(neweqs[i], 2);
                end_if;
             else
                 B[i, 1]:= -neweqs[i];
             end_if;
          end_if;
        end_for;
------------------------------------------------------- */

     )) <> 0 then
        userinfo(1, "'HardwareFloats' failed"):
        if HardwareFloatsRequested and dowarn then
           warning("HardwareFloats failed, using SoftwareFloats"):
        end_if:
        useHardwareFloats:= FALSE;
        break;
     end_if;

     if has([A, B], RD_NAN) or
        has([A, B], RD_INF) or
        has([A, B], RD_NINF) then
          userinfo(1, "'HardwareFloats' failed"):
          if HardwareFloatsRequested and dowarn then
             warning("HardwareFloats failed, using SoftwareFloats"):
          end_if:
          useHardwareFloats:= FALSE;
          break;
     end_if:

     assert(domtype(A) = DOM_HFARRAY);
     assert(domtype(B) = DOM_HFARRAY);

     //-----------------------------------------------------
     // call the hfa module
     //-----------------------------------------------------

     if traperror((
           // call matlinsolve with 3rd parameter TRUE, so that
           // the return value is [special solution, kernel, kernelindices]
           result:= hfa::matlinsolve(A, -B, TRUE)
        )) = 0 then
         if has(result, RD_NAN) or
            has(result, RD_INF) or
            has(result, RD_NINF) then
               userinfo(1, "'HardwareFloats' failed"):
               useHardwareFloats:= FALSE;
               break;
         end_if:
     else // proceed to the branch useHardwareFloats = FALSE
       userinfo(1, "'HardwareFloats' failed"):
       if HardwareFloatsRequested and dowarn then
          warning("HardwareFloats failed, using SoftwareFloats"):
       end_if:
       useHardwareFloats:= FALSE;
       break;
     end_if;
     if has(result, FAIL) then
        if showassumptions then
           return([FAIL, [], []]);
        else
           return(FAIL);
        end_if:
     end_if:

     [A, B, kernindices]:= result:
     if B = 0 then
        kdim:= 0;
     else
        kdim:= op(B, [0, 3, 2]);
     end_if:

     tmp:= [0 $ n];
     for i from 1 to n do
       tmp[i]:= (vars[i] = A[i, 1] + 
                 _plus(vars[kernindices[j]]*B[i,j] $ j = 1..kdim));
     end_for;
     tmp:= map(tmp, proc()                    // eliminate trivial 
                    begin                     // equations. This   
                      if iszero(op(args(),1)  // version is faster 
                               -op(args(),2)) // than building up  
                      then null()             // the solution set  
                      else args(1)            // sequentially using
                      end_if                  // for i=1..n do     
                    end_proc );              

     //------------------------------
     //-------- return --------------
     //------------------------------
     userinfo(1, "hardware floats succeeded"):
     return(tmp);
  end_case; // of useHardwareFloats = TRUE

  //---------------------------------
  //---- use SoftwareFloats ---------
  //---------------------------------

  userinfo(1, "not using hardware floats"):

  np2:=n+2;
  x:=genident("x");   // generate free identifier for polynomials 
  // convert eqs to polynomial expressions 
  eqs:=subs(eqs,[(vars[j]=x^(np2-j)) $ j=1..n]);
  eqs:= map(eqs, poly, [x]);
  // pick out right hand side of equations 
  b:= map(eqs, coeff, 0);
  eqs:=subs(zip(eqs,b,proc() begin args(1)-poly(args(2), [x]) end_proc ));
  eqsSymbolicParameters:=(indets(map(eqs, expr),PolyExpr) minus {x}) minus varsSet;

  // Now convert polynomial expressions eqs to polynomials in x. Need 
  // storage p of type table, since neqs may be increased in the following. 
  // Similarly, store right hand side b of equations in table B. 

  p:= table(): // p must be a table! It may grow!
  (p[i]:=eqs[i]) $ i=1..neqs; 
  B:= table(); // B must be a table! It may grow!
  (B[i]:=-b[i];) $ i=1..neqs;

  (if p[i]=FAIL or degree(p[i])>n+1 then
      error("this system does not seem to be linear")
   end_if:) $ i=1..neqs;

  // convert to floats after conversion to polynomials! Otherwise 
  // float(x+2*y=b) -> x + 2.0*y = b, i.e., the integer coefficient 1 would survive! 
  if method=2 // numeric mode 
  then if map( float(eqsSymbolicParameters),domtype ) 
          minus {DOM_FLOAT,DOM_COMPLEX,DOM_INT,DOM_RAT} <> {}
       then method:=1;
            if dowarn then
               warning("symbolic coefficient found, switching to symbolic mode");
            end_if;
       else p:=map(p, mapcoeffs, float); 
            B:=map(B, float);
       end_if;
  end_if;

  use_normal := FALSE;
  if method=1 and not dont_use_normal_requested then
    coefftypes := {op(map([coeff(p[i])], domtype)) $ i=1..neqs};
    if coefftypes minus {DOM_FLOAT,DOM_COMPLEX,DOM_INT,DOM_RAT} <> {}
      then use_normal:=TRUE 
    end_if;
  end_if;

  userinfo(10, 
           if method = 2 then
                "using numerical mode"
           elif use_normal then
                "using symbolic mode with normalization"
           else "using symbolic mode without normalization"
           end_if
          );

  // count sudiagonal bands and store in lowerbands:  
  //  op(lterm(p[i]),[1,2]) = leading exponent    
  //  n+2-op(lterm(p[i]),[1,2]) = lowest index of Variables in p[i] 
  //  Watch out for poly(0,[x]) resulting in tmp=FAIL 

  lowerbands:=0: 
  for i from 1 to neqs do
      tmp:=op(lterm(p[i]),[1,2]);
      if tmp<>FAIL then lowerbands:=max(lowerbands,i-np2+tmp); end_if;
  end_for;

  (triv[j]:=FALSE;) $ j=1..max(n,neqs);
  constraints:={};

  check:=proc(b) // routine to check right hand side of equations 0=b 
  begin //b:=subs(b, float(0) = 0);
        if myiszero(b) then return(null()) end_if;
        if not hastype(b,{DOM_EXPR,DOM_IDENT,"_index"})
        then userinfo(1,"system has no solution [numeric::linsolve]"): 
             return(FAIL):
        end_if;
        if showassumptions=FALSE
        then userinfo(1,"no solution found [numeric::linsolve]"): 
             return(FAIL):
        end_if;
        return(b);
  end_proc;

  PivotAssumptions:= {};
  for j from 1 to n do
    np2mj:=np2-j;
    jpl:=min(j+lowerbands,neqs):
// ------------------- search for pivot element ------------------ 

    // ------- initialization: collect piv[j],..,piv[jpl] -------- 
    pivot:=0;pivindex:=j; minlength:=RD_INF;
    (piv[i]:=0;) $ i=j..max(neqs,n):
    /* store element j from p[i] = i.th row in piv[i] and 
      truncate p[i] to become the remainder of the row */
    for i from j to jpl do
        piv[i]:= coeff(p[i],np2mj); 
        // Make sure, vanishing piv[i] are identified as 0:
        if use_normal and j=1 and not myiszero(piv[i])
           then piv[i]:=mynormal(piv[i]); 
        end_if;
        /* Must cut piv[i] from p[i], even if myiszero(piv[i]). Note that
          piv[i]=float(0)<>0 might happen. */
        if piv[i]<>0 then p[i]:=lmonomial(p[i],Rem)[2]: end_if;
    end_for;

    // ------- search for best pivot element----------------- 
    // piv[j],..,piv[jpl] are candidates for the pivot elements.
     
    for i from j to jpl do

        if method=1 then  /*symbolic mode: take the "simplest" nonsymbolic
                           element<>0 as pivot */
           if (not myiszero(piv[i])) then
             // look for non-symbolic Pivot elements only 
             if indets(piv[i],PolyExpr)={} then 
               if myiszero(p[i])
               then /*remaining row is trivial. This piv[i] is the ideal
                     candidate for a pivot element: accept it! */
                    pivindex:=i; break;   //stop pivot search 
               else /*remaining row is nontrivial. Look for rows
                     with pivot element of smallest complexity */
                    length(piv[i]):
                    if last(1)<minlength 
                       then minlength:=last(1); pivindex:=i:
                    end_if;
               end_if;
             end_if;
           end_if;
        end_if;

        if method=2 then //numeric mode: do row pivoting 
           if not myiszero(piv[i]) then
              tmp:= [coeff(p[i])];
              if nops(tmp) = 0 or 
                 myiszero((
                   rownorm:=max(op(map([coeff(p[i])],specfunc::abs))):
                 )) then //remaining row is trivial 
                   pivindex:=i; break;  //stop pivot search 
              else specfunc::abs(piv[i])/rownorm:
                   if pivot<last(1) then pivot:=last(1);pivindex:=i;end_if;
              end_if;          
           end_if;
        end_if;

    end_for;

    /* so far we tried to avoid a symbolic pivot element. If no such
      pivot element exists, we have to start again. This time we must 
      accept symbolic elements. */
    if method=1 then
      if use_normal then
         piv[pivindex]:= mynormal(piv[pivindex]);
      end_if;
      if myiszero(piv[pivindex]) then
        pivot:=0; pivindex:=j; minlength:=RD_INF;
        for i from j to jpl do
          if use_normal then
             piv[i]:= mynormal(piv[i]);
          end_if;
          if not myiszero(piv[i]) then
            if myiszero(p[i]) then pivindex:=i; break;  
            else length(piv[i]):
                 if last(1)<minlength 
                    then minlength:=last(1); pivindex:=i:
                 end_if;
            end_if;
          end_if;
        end_for;
      end_if;
    end_if;

// ------- optimal Pivot element found, now exchange rows --------- 

    if j<>pivindex then
      ([p[j],p[pivindex]]):=[p[pivindex],p[j]]:
      ([B[j],B[pivindex]]):=[B[pivindex],B[j]]:
      ([piv[j],piv[pivindex]]):=[piv[pivindex],piv[j]]:
    end_if;


// ----- check if non-trivial pivot element was found ------- 
    if myiszero(piv[j]) then
       // Entzerrungsschritt: fuege eine weitere Gleichung ein 
       triv[j]:=TRUE; // mark x[j] as free parameter 
       // append equation as additional row 
       neqs:=neqs+1; piv[neqs]:=0; p[neqs]:=p[j]; B[neqs]:=B[j];
       // the number of lower bands was increased 
       lowerbands:=max(lowerbands,neqs-j-1);
       jpl:=min(j+lowerbands,neqs):
    else
       // if symbolic pivot must be used, this is assumed to be <>0.
       //  Store this assumption in PivotAssumptions: 
       if method=1 and showassumptions=TRUE then
         if numeric::isnonzero(piv[j]) = UNKNOWN then
            tmp:= numer(piv[j]);
            if indets(float(tmp),PolyExpr)<>{} or 
               numeric::isnonzero(tmp) <> TRUE then
               PivotAssumptions:= PivotAssumptions union {tmp};
            end_if;
         end_if;
       end_if;
    end_if:

// --- Now do the elimination. Column j is stored in piv[j],..,piv[neqs] -- 
    for i from j+1 to jpl do
        if not myiszero(piv[i])
           then tmp:=piv[i]/piv[j]:

                if method = 2 then
                   pnorm:= norm(p[i]);
                   // Beware: B[i] may contain symbols. We put Bnorm = 0
                   // to indicate this:
                   if contains({DOM_FLOAT, DOM_COMPLEX}, domtype(B[i])) then
                      Bnorm:= specfunc::abs(B[i]);
                   else
                      Bnorm:= 0;
                   end_if:
                end_if;

                case tmp
                of float(1) do
                of  1 do  p[i]:=p[i]-p[j];
                          B[i]:=B[i]-B[j];
                          break;
                of float(-1) do
                of -1 do  p[i]:=p[i]+p[j];
                          B[i]:=B[i]+B[j];
                          break;
                otherwise p[i]:=p[i]-mapcoeffs(p[j],_mult,tmp):
                          B[i]:=B[i]-tmp*B[j]:
                end_case;

                // Increase the chance of finding a kernel of the
                // matrix: if a row nearly vanishes, then regard it
                // as a zero row. This will generate a kernel vector.
                if method = 2 then
                   if norm(p[i]) <= 10^(-DIGITS)*pnorm then
                      p[i]:= poly(0, [x]);
                   end_if:

                   if (not myiszero(Bnorm)) and
                      // Beware: the original B[j] may have introduced symbols in B[i]. 
                      // We must double check that B[i] is still numeric
                      contains({DOM_FLOAT, DOM_COMPLEX}, domtype(B[i])) and
                      specfunc::abs(B[i]) <= 10^(-DIGITS)*Bnorm then
                      B[i]:= 0;
                   else
                      // keep the B[i] without cleaning it
                   end_if:

                end_if;

                if use_normal then 
                         p[i]:=mapcoeffs(p[i],mynormal):
                         B[i]:=mynormal(B[i]):
                end_if;
        end_if;
    end_for:
  end_for;

  //-------------------------------------------------
  // we now have neqs equations for n unknowns, where 
  // the first n equations are  upper triangular. 
  // Check that the remaining equations are trivial: 
  //-------------------------------------------------
  for j from n+1 to neqs do 
      check(B[j]):
      if last(1)=FAIL then 
         if showassumptions=FALSE then 
              return(FAIL)
         else return([FAIL, [], []])
         end_if;
      else 
         if last(1)<> null() and showassumptions=TRUE then
           constraints:=constraints union {numer(last(1))};
         end_if;
      end_if;
  end_for;
  delete check;
  constraints:=[op(constraints)];

  // -- Now system is in upper triangular form with nonzero diagonal elements 
  // piv[i]. The equations with zero pivot are marked by triv[i]=TRUE      -- 

  // convert univariate polynomials to expressions for fast sparse            
  // backsubstition: coeff*x^(n+2-i) <-> coeff*X[i]. Use expressions to avoid 
  // costly substitutions in the backsubstitution: the kernel simplfies the   
  // known terms automatically (very fast)  

  if triv[n] then 
       X[n]:=vars[n]
  else X[n]:=B[n]/piv[n];
       if use_normal then X[n]:=mynormal(X[n]): end_if:
  end_if;
  for i from n-1 downto 1 do
     if triv[i] then X[i]:=vars[i]
     else
        nTerms:=nterms(p[i]):  // pick out non-trivial coefficients of the poly 
        Xvars:=[X[np2-op(nthterm(p[i],j),[1,2])]$ j=1..nTerms]:
        koeff:=[coeff(p[i])]:
        p[i]:=_plus( koeff[j]*Xvars[j] $ j=1..nTerms):
        if piv[i]=1 
          then X[i]:= (B[i]-p[i]):
          else X[i]:= (B[i]-p[i])/piv[i]:
        end_if;
        if use_normal then X[i]:=mynormal(X[i]): end_if:
     end_if;
  end_for;
  tmp:=[ (vars[i]=X[i]) $ i=1..n ];
  tmp:= map(tmp, proc()                         // eliminate trivial 
                 begin                          // equations. This   
                   if op(args(),1)=op(args(),2) // version is faster 
                   then null()                  // than building up  
                   else args(1)                 // the solution set  
                   end_if                       // sequentially using
                 end_proc );                    // for i=1..n do     
                                                //   if triv[i] ..   
  constraints:=map(constraints,
                   proc(x) begin 
                    if myiszero(x) then null() else x end_if
                   end_proc);
  if showassumptions=FALSE then return(tmp)
  else return([
       tmp, 
        map([op(constraints)],proc() begin args(1)=0 end_proc ),
        map([op(PivotAssumptions)], proc() begin args(1)<>0 end_proc)
              ]);
  end_if;
end_proc:
