// ---------------  file odesolv2.mu, 11.11.99, by Walter Oevel ------------ 
/*  Help file:

 numeric::odesolve2 -- numerical solution of the initial value problem
                       of a dynamical system d/dt y(t) = f(t,y),  y(t0)=y0

 Call:    numeric::odesolve2(f, t0, y0 
                             <, RememberLast>
                             <, method >
                             <, Stepsize = h >
                             <, MaxStepsize = hmax >
                             <, RelativeError = rtol>,
                             <, AbsoluteError = atol>)

 Parameter:

    f      -- a function or procedure of 2 variables t and y representing
              the vector field of the dynamial system dy/dt = f(t,y)
    t0     -- numerical value for the initial time
    y0     -- a list or vector of initial conditions y(t0)=y0
    method -- name of the numerical scheme, one of
              EULER1, RKF43, RK4, RKF34, RKF54a, RKF54b, DOPRI54, CK54,
              RKF45a, RKF45b, DOPRI45, CK45, 
              DOPRI65, xDOPRIT65, DOPRI56, xDOPRI56,
              BUTCHER6, RKF87, RKF78, xRKF43, xRKF34, xRKF54a, 
              xRKF54b, xDOPRI54, xCK54, xRKF45a,
              xRKF45b, xDOPRI45, xCK45, xRKF87, xRKF78,
              DOPRI78, xDOPRI78, DOPRI87, xDOPRI87 
    h      -- numerical value for the stepsize
    hmax   -- numerical value for the maximal stepsize
    rtol   -- a positive real value >= 10^(-DIGITS)
    atol   -- a positive real value 


 Details:
  *) y:= numeric::odesolve2(f, t0, y0 <,options>) returns the function

            y:=  t ->  numeric::odesolve(t0..t, f, y0 <,options>)

     See help page of numeric::odesolve for details on parameters
     and options.
  *) The options Symbolic and Alldata=n for numeric::odesolve
     are accepted by numeric::odesolve2, but have no effect:
     they are not passed to numeric::odesolve.
  *) The returned function y can only be called with numerical real
     arguments. The call y(t) then returns a numerical approximation
     of the exact solution y(t) of the initial value problem.
  *) y is defined with option remember: it remembers all values
     it has ever computed. The next call y(t) takes the
     time T0 from its remember table that is closest to t
     and the corresponding y(T0). Then it executes
            numeric::odesolve(T0..t, f, y(T0) )
  *) With RememberLast, option remember is not used, but only
     the preceding call is remembered.
     - This option is recommended if many (= hundreds) of
       calls will occur with monotonically increasing
       or decreasing time values (e.g., when plotting).
     - This option avoids the search inside the remember table.
       This search may dominate the costs (without RememberLast)
       after several hundreds of calls.
  *) Do not increase DIGITS, between calls of y.
     Because of option remember the results may be incorrect!
     Clear the remember table by y:= subsop(y, 5=NIL) first!

 Example 1: // solve y'(t) = y(t), y(0) = 1:
   >> f:= (t,Y) -> Y: t0:= 0: Y0:= [1]:
   >> y:= numeric::odesolve2(f, t0, Y0);

                       proc Y(t) ... end

   >> y(0), y(1), y(2);

              [1.0], [2.718281828], [7.389056099]

   // plot the numerical solution:
   >> plotfunc(hold(y)(t)[1], t=0..1);
              
 Example 2: // solve y''(t) + t*y(t) = sin(t); y(0)=0, y'(0)=0
   >> f:= (t,Y) -> [Y[2], sin(t) - t*Y[1]]: t0:= 0: Y0:= [0,0]:
   >> Y:= numeric::odesolve2(f, t0, Y0, RelativeError = 10^(-3)):
   >> Y(0), Y(1), Y(2);

      [0.0, 0.0], 
       [0.1531947386, 0.4282080969],
        [0.8057230535, 0.6414397614]

   // plot y(t):
   >> plotfunc2d(hold(Y)(t)[1], t=0..1);
   // plot y'(t):
   >> plotfunc2d(hold(Y)(t)[2], t=0..1);

---------------------------------------------------------------------- */

numeric::odesolve2:= proc(f, tt0, yy0)
    option escape;
    local last_t, last_y, Opts;
begin
   if args(0) < 3 then error ("expecting at least 3 arguments"); end_if;
   if not testtype(float(tt0), DOM_FLOAT) then
      error("second argument: the initial time must be a real ".
            "numerical value, got: ".expr2text(tt0));
   end_if;
   Opts:= op(map([args(4..args(0))], 
                 opt -> (if has(opt, Alldata) or has(opt, Symbolic)
                    then null() else opt end_if)));
   if has([Opts], RememberLast) then
      Opts:= subs(Opts, RememberLast = null());
      last_t:= float(tt0):
      last_y:= float(yy0):
      //---------------------------------------------------
      // With RememberLast, return the following procedure:
      //---------------------------------------------------
      return(
        proc(t)
        local pname;
        begin
           if domtype(float(t))<>DOM_FLOAT then
             // return(procname(args()));
             // do not use procname, but
             // (op(op(_act_proc_env(),1),6) (= procname)
             pname := op(op(_act_proc_env(),1),6);
             if pname <> NIL and
                eval(pname) = op(_act_proc_env(), 1) then
                  return(pname(args()));
             else return(hold(numeric::odesolve2)(f, tt0, yy0, Opts, RememberLast)(t));
             end_if;
           end_if;
           userinfo(1, "integrating from t0=".expr2text(last_t).
                       " to t="              .expr2text(t).
                       " using Y(t0)="       .expr2text(last_y));
           [last_t, last_y]:= 
                [t, numeric::odesolve(last_t..t, f, last_y, Opts, "CalledByOdesolve2")];
           return(last_y);
        end_proc);
   end_if;

   //------------------------------------------------------
   // Without RememberLast, return the following procedure:
   //------------------------------------------------------
   proc(t)
   local pname, t0, y0, remembertable, n, dt, i, tt, dtt;
   option remember;
   begin
     if domtype(float(t))<>DOM_FLOAT then
        // return(procname(args()));
        // do not use procname, but
        // (op(op(_act_proc_env(),1),6) (= procname)
        pname := op(op(_act_proc_env(),1),6);
        if pname <> NIL and
           eval(pname) = op(_act_proc_env(), 1) then
               return(pname(args()));
          else return(hold(numeric::odesolve2)(f, tt0, yy0, Opts)(t));
        end_if;
     end_if;
     t0:= tt0; y0:= yy0;
     // Search the remember table op(thisproc, 5) of this
     // function for precomputed values. Find the point t0 from
     // the remember table that is closest to t.

     // Use remember table = op(op(_act_proc_env(),1),5) , not 
     // remember table = op(Y, 5), where Y would be the name of this proc.
     remembertable:= op(op(_act_proc_env(),1),5);
     if domtype(remembertable) = DOM_TABLE then
       remembertable:= [op(remembertable)];
       n:= nops(remembertable);
       dt:= specfunc::abs(float(t-tt0)):
       for i from 1 to n do // search for time tt in remember table
                            // that is closest to t.
         tt:= float(op(remembertable[i],1));
         if domtype(tt) <> DOM_FLOAT then // the remember table
            next;                         // may have symbolic
         end_if:                          // entries
         if (dtt:= specfunc::abs(float(t-tt)))<dt then
            t0:= tt;
            y0:= op(remembertable[i],2);
            dt:= dtt;
         end_if;
       end_for;
     end_if;
     // Integrate from t0 to t with initial value y0 from
     // the remember table:
     userinfo(1, "integrating from t0=".expr2text(t0).
                 " to t="              .expr2text(t).
                 " using Y(t0)="       .expr2text(y0));
     numeric::odesolve(t0..t, f, y0, Opts, "CalledByOdesolve2");
  end_proc
end_proc:

/* --- end of file ---- */
