/*----------------------------------------------------------------------------*/
/* Das Besondere an diesem Differentialgleichungsloeser fuer                  */
/* Systeme vom Typ                                                            */
/*    dY/dt = f(t,Y)*Y  (mit Matrix-wertigem f und Y = Vektor oder Matrix)    */
/* ist, dass gewisse Erhaltungsgroessen auch numerisch erhalten bleiben.      */
/* Genauer:                                                                   */
/* ist f(t,Y) aus einer Lie-Unteralgebra der n x n Matrizen und G die         */
/* entsprechende Matrix-Lie-Gruppe, dann bleibt die numerische Loesung        */
/* auf der "homogenen Mannigfaltigkeit" {g*Y(0); g aus G}.                    */
/*                                                                            */
/* Beispiel: ist f(t,Y) schiefsymmetrisch fuer alle t und Y, so ist           */
/* die entsprechende Lie-Gruppe G die Menge aller orthogonalen Matrizen,      */
/* d.h., sowohl die exakte als auch die numerische Loesung bleibt auf         */
/* der homogenen Mannigfaltigkeit                                             */
/*                                                                            */
/*    {g*Y(0); g orthogonal} = {Y; <Y,Y> = <Y(0),Y(0)>} = Kugel               */
/*                                                                            */
/* Mit anderen Worten: der Erhaltungssatz <Y,Y>=const ist auch                */
/* numerisch gueltig.                                                         */
/* Speziell koennen auch integrable Lax-Gleichungen                           */
/*                                                                            */
/*       dY/dt =  f(t,Y)*Y - Y*f(t,Y)                                         */
/*                                                                            */
/* so geloest werden, dass die Eigenwerte der Matrix Y (diese sind            */
/* Erhaltungssaetze) auch numerisch erhalten bleiben.                         */
/*                                                                            */
/* So weit, so gut. Allerdings: der Algo ist recht laanngggsssaaaammmm....    */
/*----------------------------------------------------------------------------*/
/* Based on diploma thesis                                                    */
/*   "Numerik dynamischer Systeme auf Lie-Gruppen, Renate Wagner, 1999.       */
/* Code rewritten and optimized by W. Oevel, 6.1.1999                         */
/*----------------------------------------------------------------------------*/
/* numeric::odesolveGeometric - numerische Loesung dynamischer Systeme auf    */
/*  Matrix-Lie-Gruppen und in Matrizen-Raeumen eingebetteten homogenen        */
/*  Mannigfaltigkeiten                                                        */
/*                                                                            */
/* numeric::odesolveGeometric(t0..t, f, Y0) berechnet numerisch die Loesung   */
/* Y(t) des Anfangswertproblems                                               */
/*                                                                            */
/*      dY/dt = f(t,Y)*Y, Y(t0)=Y0,  t aus R, Y = (komplexe) n x p-Matrix,    */
/*                                       f(t,Y) = (komplexe) n x n-Matrix     */
/*                                                                            */
/* Aufruf:                                                                    */
/* numeric::odesolveGeometric(t0..t, f, Y0, <, LieGroupAction = LAMBDA>       */
/*                            <, RelativeError=tol> <, Stepsize = h>          */
/*                            <, method> <, Alldata = n>                      */
/* oder                                                                       */
/* numeric::odesolveGeometric(f, t0..t, Y0, <, LieGroupAction = LAMBDA>       */
/*                            <, RelativeError=tol> <, Stepsize = h>          */
/*                            <, method> <, Alldata = n>                      */
/* Parameter:                                                                 */
/*                                                                            */
/* t0     : reeller numerischer Wert fuer den Anfangszeitpunkt                */
/* t      : reeller numerischer Wert fuer den Endzeitpunkt                    */
/* f      : eine Matrix-wertige Prozedur zweier Parameter t und Y             */
/* Y0     : Liste, Vektor (=array(1..n)) oder n x p-Matrix vom Typ DOM_ARRAY  */
/*          oder Cat::Matrix. Y0 legt den Einbettungsraum C^n=C^(n x 1)       */
/*          bzw. C^(n x p) der homogenen Mannigfaltigkeit fest. Listen und    */
/*          Vektoren entsprechen p=1, intern wird auch in diesem Fall mit     */
/*          Matrizen vom Typ array(1..n,1..1) operiert.                       */
/* tol    : positiver numerischer Wert >= 10^(-DIGITS) (relativer lokaler     */
/*          Diskretisierungsfehler                                            */
/* LAMBDA : ein Prozedur zweier Parameter G,Y, die die Wirkung des Elementes  */
/*          G der Lie-Gruppe auf den Punkt Y der homogenen Mannigfaltigkeit   */
/*          beschreibt (G= n x n-Matrix, Y = n x p-Matrix)                    */
/* h      : numerischer Wert fuer die Schrittweite                            */
/* method : Name eines der folgenden Runge-Kutta-Verfahren:                   */
/*          EULER1, RK4, BUTCHER6, RKF43, RKF34, RKF54a, RKF45a,              */
/*          RKF54b, RKF45b, DOPRI45, DOPRI54, RKF87, RKF78,                   */
/*          xRKF43, xRKF34, xRKF54a, xRKF45a,                                 */
/*          xRKF54b, xRKF45b, xDOPRI45, xDOPRI54, xRKF87, xRKF78              */
/*          DOPRI65, xDOPRI65, DOPRI56, xDOPRI56,                             */
/*          DOPRI87, xDOPRI87, DOPRI78, xDOPRI78.                             */
/* n      : ganze Zahl, die die Groesse der Output-Liste kontrolliert         */
/*                                                                            */
/* Rueckgabewert:                                                             */
/*    Loesung Y(t) als Liste, array(1..n) oder array(1..n,1..p), je nach Typ  */
/*    des Eingabeparameters Y0                                                */
/*                                                                            */
/* Beschreibung:                                                              */
/*                                                                            */
/* --Die Funktion f(t,Y) wird intern aufgerufen mit t= reeller numerischer    */
/*   Wert und domtype(Y)= domtype(Y0). Eingabedaten dieses Typs muessen ver-  */
/*   arbeitet werden koennen. Die Ausgabe von f(t,Y) muss eine n x n-Matrix   */
/*   vom Typ array(1..n,1..n) oder Cat::Matrix sein.                          */
/*                                                                            */
/* --Intern wird zu Loesung von                                               */
/*                    dY/dt = f(t,Y)*Y,  Y(t0)=Y0            (0)              */
/*   schrittweise die Matrix-Differentialgleichung                            */
/*                    dG/dt = f(t,G*Y0)*G,  G(t0) = 1        (1)              */
/*   auf den n x n-Matrizen geloest, die Loesung von (0) ergibt sich          */
/*   schrittweise durch                                                       */
/*                    Y(t0+h)=G(t0+h)*Y0.                                     */
/*   Nimmt f(t,G*Y0) Werte auf einer Unter-Lie-Algebra der n x n-Matrizen an, */
/*   so ergeben sich auch numerisch Werte fuer G(t), welche bis auf           */
/*   Rundungsfehler in der entsprechenden Matrix-Lie-Gruppe liegen.           */
/*                                                                            */
/* --Zur Loesung von (1) wird nach Munthe-Kaas mit dem Ansatz G(t)=exp(u(t))  */
/*   die entsprechende Differentialgleichung                                  */
/*             du/dt = dexpinv(u,f) =  f - 1/2 [u,f] + 1/12*[u,[u,v]] + ...   */
/*             f= f(t, exp(u(t))*Y0),  u(t0) = 0                              */
/*   auf der Lie-Algebra der n x n-Matrizen durch ein klassisches Verfahren   */
/*   geloest, welches durch die Option <method> bestimmt wird. Das Standard-  */
/*   verfahren ist das eingebettete Paar DOPRI78 der                          */
/*   Ordnungen 7 und 8.                                                       */
/*                                                                            */
/* --Mit der Option LieGroupAction=LAMBDA kann die Standard-Gruppen-Aktion    */
/*   LAMBDA(G,Y)= G*Y der nxn-Matrizen G auf die in dem Raum der nxp-Matrizen */
/*   Y eingebettete homogene Mannigfaltigkeit geaendert werden. Es wird die   */
/*   Matrix-Diffgleichung                                                     */
/*                dG/dt = f(t, LAMBDA(G,Y0))*G, G(t0) = 1                     */
/*   auf den n x n Matrizen geloest und odesolveGeometric liefert             */
/*                Y(t) = LAMBDA(G(t),Y0).                                     */
/*                                                                            */
/*   Beispiel: sei Y eine n x n-Matrix, sei LAMBDA(G,Y)= G*Y*G^(-1).          */
/*   Die Differentialgleichung, die nun fuer Y(t)= LAMBDA(G(t),Y0) geloest    */
/*   wird, ist die Lax-Gleichung                                              */
/*       dY/dt = d/dt LAMBDA(G(t),Y0)                                         */
/*             = dG/dt*G^(-1)*Y(t) - Y(t)*dG/dt*G^(-1)                        */
/*             = f(t,Y)*Y - Y*f(t,Y),                                         */
/*   also die isospektrale Lax-Gleichung                                      */
/*                                                                            */
/*                  dY/dt = [ f(t,Y), Y].                                     */
/*                                                                            */
/*   Die Funktion LAMBDA(G,Y) wird intern aufgerufen mit Y vom selben         */
/*   Domain-Typ wie die Anfangsbedingung Y0. Das Gruppenelement G wird        */
/*   als array G=array(1..n,1..n) uebergeben. Falls Y vom Typ densematrix()   */
/*   ist, wird G als densematrix() uebergeben.                                */
/*   Eingabedaten dieses Typs muessen verarbeitet werden koennen.             */
/*   Die Ausgabe der Funktion LAMBDA sollte ein                               */
/*   array(1..n,1..p) oder eine Cat::Matrix-Matrix entsprechender             */
/*   Dimension sein. Fuer Y0 = Liste bzw. array(1..n) (d.h., die homogene     */
/*   Mannigfaltigkeit ist der C^n) werden auch Ausgabedaten des Typs          */
/*   Liste bzw. array(1..n) akzeptiert.                                       */
/*                                                                            */
/* --Alle Algorithmen koennen mit konstanter Schrittweite oder adaptiv        */
/*   durchgefuehrt werden.Standardmaessig sind sie adaptiv und es wird das    */
/*   Munthe-Kaas Verfahren basierend auf dem klassischen Paar DOPRI78         */
/*   benutzt.                                                                 */
/*                                                                            */
/* --Mit der Option <Stepsize=h> wird die automatische Schrittweitensteuerung */
/*   abgeschaltet und mit konstanter Schrittweite gearbeitet. Dabei wird      */
/*   eventuell ein letzter Schritt mit einer kleineren Schrittweite durchge-  */
/*   fuehrt, um genau bis zum Ende des Integrationsintervalls zu integrieren. */
/*                                                                            */
/* --Die Option Alldata=n ist identisch mit der bei numeric::odesolve, siehe  */
/*   dort.                                                                    */
/*                                                                            */
/*   Examples:                                                                */
/*                                                                            */
/*  -- >> f:= proc(t,Y) begin array(1..2,1..2,[[0,1],[t,0]]) end_proc:        */
/*                                                                            */
/*     Die Gleichung dY/dt = f(t,Y)*Y entspricht mit Y=(y1,y2)^T der          */
/*     Airy-Differentialgleichung                                             */
/*           dy1/dt = y2,  dy2/dt = t*y1,  also diff(y1,t,t)=t*y1.            */
/*     Die Anfangsbedingung Y0 = [1,0] entspricht y1(0)=1, y2(0)=0.           */
/*     Die Anfangsbedingung als Liste mit 2 Elementen uebergibt               */
/*     odesolveGeometric die Information, eine im C^2 eingebettete            */
/*     Mannigfaltigkeit mit den 2x2-Matrizen als Gruppe zu benutzen. Der      */
/*     Rueckgabewert entspricht der Startbedingung Y0, ist also eine Liste:   */
/*                                                                            */
/*     >> numeric::odesolveGeometric(0..1,f,[1,0]);                           */
/*                                                                            */
/*                      [1.17229997, 0.5340348342]                            */
/*                                                                            */
/*  -- Im folgenden Beispiel wird die Matrix-Differentialgleichung            */
/*              dY/dt = t*Y^2 + 1  = (t*Y+Y^(-1))*Y                           */
/*     auf den 2x2-Matrizen geloest. Zur bequemen Berechnung von Y^(-1) wird  */
/*     in der Funktion f der Eingabeparameter Y (intern als array(1..2,1..2)  */
/*     an f uebergeben) in eine Matrix vom Typ Dom::Matrix(Dom::Float)        */
/*     gewandelt:                                                             */
/*                                                                            */
/*     >> f:= proc(t,Y) begin                                                 */
/*              Y:= Dom::Matrix(Dom::Float)(Y);                               */
/*              t*Y + Y^(-1);                                                 */
/*            end_proc:                                                       */
/*                                                                            */
/*     >> Y0:= array(1..2,1..2,[[1,2],[2,1]]):                                */
/*                                                                            */
/*     >> numeric::odesolveGeometric(0..0.1,f,Y0);                            */
/*                    +-                          -+                          */
/*                    |  1.126045926, 2.021708127  |                          */
/*                    |                            |                          */
/*                    |  2.021708127, 1.126045926  |                          */
/*                    +-                          -+                          */
/*                                                                            */
/*     Anmerkung: ist f(t,Y) lokal analytisch in Y, so laesst sich die        */
/*                Matrix-DGL dY/dt= f(t,Y)*Y besser per Diagonalisierung      */
/*                Y(0) = T*diag(L0(0),L1(0),..)*T^(-1) durch                  */
/*                Y(t) = T*diag(L0(t),L1(t),..)*T^(-1) loesen, wobei          */
/*                die Eigenwerte  d L.i /dt = f(t, L.i)*L.i erfuellen.        */
/*                                                                            */
/*  -- Im folgenden Beispiel wird die Matrix-Differentialgleichung            */
/*                 dY/dt = A*Y - Y*A                                          */
/*     mit einer festen Matrix A geloest. Die Mannigfaltigkeit der            */
/*     2x2-Matrizen wird dazu mit der Gruppenwirkung LAMBDA(G,Y)=G*Y*G^(-1)   */
/*     ausgestattet, sodass die Differentialgleichung                         */
/*                     dY/dt = f(t,Y)*Y - Y*f(t,Y)                            */
/*     geloest wird. Die Funktion f liefert die globale Matrix A zurueck:     */
/*                                                                            */
/*     >> A:=array(1..2,1..2,[[1,2],[3,4]]):                                  */
/*     >> f:= proc(t,Y) begin A; end_proc:                                    */
/*                                                                            */
/*     Die Gruppenwirkung LAMBDA erhaelt intern als Argumente Matrizen vom    */
/*     Typ DOM_ARRAY. Diese werden in Dom::Matrix verwandelt, um die Matrizen */
/*     bequem multiplizieren und invertieren zu koennen:                      */
/*                                                                            */
/*     >> LAMBDA:= proc(G,y)                                                  */
/*                 begin G:= Dom::Matrix(Dom::Float)(G);                      */
/*                       y:= Dom::Matrix(Dom::Float)(y);                      */
/*                       return(G*y*G^(-1));                                  */
/*                 end_proc:                                                  */
/*                                                                            */
/*     Die Anfangsbedingung Y0 legt die Dimension der Matrizen fest:          */
/*                                                                            */
/*     >> Y0:= array(1..2,1..2,[[1,2],[2,1]]):                                */
/*                                                                            */
/*     >> numeric::odesolveGeometric(0..1,f,Y0, LieGroupAction=LAMBDA);       */
/*                                                                            */
/*                    +-                           -+                         */
/*                    |  87.73544231, -38.87501205  |                         */
/*                    |                             |                         */
/*                    |  193.4156815, -85.73544231  |                         */
/*                    +-                           -+                         */
/*                                                                            */
/*     Mit Y(t) = LAMBDA(G(t),Y0) =  G(t)*Y0*G(t)^(-1) bleiben die Eigenwerte */
/*     erhalten:                                                              */
/*                                                                            */
/*     >> numeric::eigenvalues(%) = numeric::eigenvalues(Y0);                 */
/*                                                                            */
/*                       [-1.0, 3.0] = [-1.0, 3.0]                            */
/*                                                                            */
/*----------------------------------------------------------------------------*/

numeric::odesolveGeometric:= proc(diffequ, x0_xn, Y0)
option escape;
local x0, xn, n, m, nn, tmp, i, X, DomMatrixOutput, init_type, 
      output_convert, method, adaptive, DIFFEQU,
      alldata, datacount, nextentry, datatable, append_to_datatable, 
      h, tol, tol1, defaulttol, 
   // extrapol1, extrapol2, 
      lo_order, hi_order, 
      data, getButcherData, stages, C, A, B, Bb, startingstepsize,
      convert, lambda, LAMBDA, LAMBDAdefined,
      array_add, poly_add, dexpinv,
      MKStep, MKPairStep, adaptiveMKStep, adaptiveMKPairStep,
      stepsize_info, localerr_info, stepcount,
      localerror, array_norm, poly_norm, commutator,
      compare,
      result; 
save  DIGITS;
begin
  if args(0) < 3 then error("not enough arguments"); end_if;

   // the call odesolve(x0_xn, diffequ, y0) is also allowed
   // for backward compatibility with MuPAD 2.0 and earlier.   
   
   if domtype(x0_xn) = DOM_PROC and
      type(diffequ) = "_range" then
      [x0_xn, diffequ] := [diffequ, x0_xn];
   end_if;

  if type(x0_xn) <> "_range" then
    error("wrong specification of integration interval"); 
  end_if;

  x0:=op(x0_xn,1);
  xn:=op(x0_xn,2);

  if domtype(diffequ)<>DOM_PROC then
      error("vector field must be presented by a procedure");
  end_if;

  if iszero(float(xn-x0)) then return(Y0) end_if;

  /* check output of second argument. Do this before
     conversion of Y0 to internal standard format 
  */

  tmp:=diffequ(float(x0), map(Y0, float));
  if tmp::dom::hasProp(Cat::Matrix)=TRUE
     then tmp:= tmp::dom::expr(tmp);
          DomMatrixOutput:= TRUE;
     else DomMatrixOutput:= FALSE;
  end_if;

  /* The user defined callback function diffeq(t, Y)
     should be fed with data Y of the same type as
     the initial value Y0.

    init_type = 0 :  Y = an array(1..n, 1..m)
    init_type = 1 :  Y = a list [..]
    init_type = 2 :  Y = an array(1..n)
    init_type = 3 :  Y = a matrix(..)

  */

  /* 
     convert Y0 to internal standard form array(1..n,1..m), but
     remember its original type
  */

  init_type:= NIL; // initialize

  if Y0::dom::hasProp(Cat::Matrix)=TRUE
    then Y0:= Y0::dom::expr(Y0);
         init_type:= 3:
  end_if;

  case domtype(Y0)
  of DOM_LIST do  n := nops(Y0);
                  m:=1; 
                  if init_type = NIL then
                     init_type:= 1; /* return list */
                  end_if;
                  Y0:= array(1..n,1..m, [[Y0[i]] $ i=1..n]);
                  break;
  of DOM_ARRAY do case op(Y0, [0,1]) 
                  of 1 do n := op(Y0,[0,2,2]); 
                          m := 1;
                          if init_type = NIL then
                            init_type:= 2; /* return array(1..n,[..]) */
                          end_if;
                          Y0:= array(1..n,1..m, [[Y0[i]] $ i=1..n]);
                          break;
                  of 2 do n := op(Y0,[0,2,2]);
                          m := op(Y0,[0,3,2]);
                          if init_type = NIL then
                            init_type:= 0;  /* return array(1..n,1..m,..) */
                          end_if;
                          break;
                  otherwise error("wrong type of initial condition");
                  end_case;
                  break;
  otherwise error("initial condition must be a vector or a matrix");
  end_case;

  /* continue to check output of second argument. We need dimension n: */
  if domtype(tmp)<>DOM_ARRAY
     or op(tmp,[0,1])<>2 
     or op(tmp,[0,2,2])<>n
     then error("output of 2nd argument not compatible with initial condition");
  end_if;

  /* wrap diffequ: convert the output matrix to list of polynomial columns */
  X:= genident("X");
  if DomMatrixOutput
     then DIFFEQU:= proc(x,y) local A, i, j;
                    begin // convert input y for user function diffequ
                          // to the type specified by initial condition.
                          // This is to allow intutitive implementations
                          // of the user-defined matrix field diffeq!
                          case init_type
                           of 0 do break;
                           of 1 do y:= [y[i,1] $ i=1..n]: break;
                           of 2 do y:= array(1..n, [y[i,1] $ i=1..n]); break;
                           of 3 do y:= matrix(y); break;
                          end_case;
                          A:= map(diffequ(x,y),float); 
                          A:= A::dom::expr(A); 
                          A:= subs(A, float(0)=0);
                          [poly([[A[i,j],i] $ i=1..n],[X]) $ j=1..n];
                    end_proc;
     else DIFFEQU:= proc(x,y) local A, i, j;
                    begin // convert input y for user function diffequ
                          // to the type specified by initial condition.
                          // This is to allow intutitive implementations
                          // of the user-defined matrix field diffeq!
                          case init_type
                           of 0 do break;
                           of 1 do y:= [y[i,1] $ i=1..n]: break;
                           of 2 do y:= array(1..n, [y[i,1] $ i=1..n]); break;
                           of 3 do y:= matrix(y); break;
                          end_case;
                          A:=map(diffequ(x,y),float);
                          A:= subs(A, float(0)=0);
                          [poly([[A[i,j],i] $ i=1..n],[X])$j=1..n];
                    end_proc;
  end_if;

   /* ---- subroutine getButcherData ----- */
   getButcherData:=proc(method)
   local i,b;
   begin 
     ([stages,C,A,b,B,lo_order,hi_order]):=numeric::butcher(method, DIGITS);
     if stages=1 then Bb:=array(1..stages);Bb[1]:=B[1]-b[1]
     else Bb:=array(1..stages,[ (B[i]-b[i]) $ i=1..stages ])
     end_if;
   end_proc;

  adaptive := TRUE;                  /* default */
  alldata  := FALSE;                 /* default */
  method:=1; getButcherData(DOPRI78);/* default */
  defaulttol := float(10^(-DIGITS)); /* default */
  tol:= defaulttol;                  /* default */
  LAMBDAdefined:= FALSE;             /* default */

  if args(0) > 3 then            /* get options */
     (case op(args(i),1)
        /* method = 1: embedded pairs via MKPairStep */
        of  RKF43       do method:=1;getButcherData(RKF43);break
        of  RKF34       do method:=1;getButcherData(RKF34);break
        of  RKF54a      do method:=1;getButcherData(RKF54a);break
        of  RKF45a      do method:=1;getButcherData(RKF45a);break
        of  RKF54b      do method:=1;getButcherData(RKF54b);break
        of  RKF45b      do method:=1;getButcherData(RKF45b);break
        of  DOPRI54     do method:=1;getButcherData(DOPRI54);break
        of  DOPRI45     do method:=1;getButcherData(DOPRI45);break
        of  CK54        do method:=1;getButcherData(CK54); break
        of  CK45        do method:=1;getButcherData(CK45); break
        of  DOPRI65     do method:=1;getButcherData(DOPRI65);break
        of  DOPRI56     do method:=1;getButcherData(DOPRI56);break
        of  RKF87       do method:=1;getButcherData(RKF87);break
        of  RKF78       do method:=1;getButcherData(RKF78);break
        of  DOPRI87     do method:=1;getButcherData(DOPRI87);break
        of  DOPRI78     do method:=1;getButcherData(DOPRI78);break
        /* method = 2: single methods/subprocesses of pairs via MKStep */
        of  EULER1      do method:=2;getButcherData(EULER1);break
        of  RK4         do method:=2;getButcherData(RK4);break
        of  BUTCHER6    do method:=2;getButcherData(BUTCHER6);break
        of xRKF43       do method:=2;getButcherData(RKF43);break
        of xRKF34       do method:=2;getButcherData(RKF34);break
        of xRKF54a      do method:=2;getButcherData(RKF54a);break
        of xRKF45a      do method:=2;getButcherData(RKF45a);break
        of xRKF54b      do method:=2;getButcherData(RKF54b);break
        of xRKF45b      do method:=2;getButcherData(RKF45b);break
        of xDOPRI54     do method:=2;getButcherData(DOPRI54);break
        of xDOPRI45     do method:=2;getButcherData(DOPRI45);break
        of xCK54        do method:=2;getButcherData(CK54); break
        of xCK45        do method:=2;getButcherData(CK45); break
        of xRKF78       do method:=2;getButcherData(RKF78);break
        of xRKF87       do method:=2;getButcherData(RKF87);break
        of xDOPRI65     do method:=2;getButcherData(DOPRI65);break
        of xDOPRI56     do method:=2;getButcherData(DOPRI56);break
        of xDOPRI78     do method:=2;getButcherData(DOPRI78);break
        of xDOPRI87     do method:=2;getButcherData(DOPRI87);break

        of Stepsize do adaptive:=FALSE;
           if type(args(i)) = "_equal"
              then h:=op(args(i),2);
              else error("specify step size by Stepsize=h");
           end_if;
           break
         of RelativeError do
              if type(args(i)) = "_equal"
                 then tol:= float(op(args(i),2));
                 else error("specify relative truncation error by RelativeError=tol");
              end_if;
              break
        of Alldata do alldata :=TRUE;
           if type(args(i)) = "_equal" 
           then nn:=op(args(i),2) else nn:=1; end_if;
           break
        of LieGroupAction do 
           if type(args(i)) = "_equal" 
           then LAMBDAdefined:= TRUE;
                LAMBDA:=op(args(i),2);
                if domtype(LAMBDA)<>DOM_PROC then
                   error("Lie group action must be defined by a procedure");
                end_if;
           else error("specify the group action by 'LieGroupAction = procedure'");
           end_if;
           break
        otherwise error("unknown option");
      end_case;
    ) $ i=4..args(0);
  end_if;

  if adaptive then
      if domtype(tol)<>DOM_FLOAT or tol<=0 then
         error("the tolerance for the relative error must be ".
               "a positive numerical value");
      end_if;
      if tol<0.99*defaulttol then
         error("the tolerance for the relative error must be larger ".
               "than the working precision 10^(-DIGITS)")
      end_if;
  end_if;

  // For tiny integration intervals there is danger, that
  // adaptiveRKPairStep etc. think that stepsize is too small.
  // Avoid the resulting error by doing 1 non-adaptive step with
  // the tiny stepsize:
  if adaptive then
     h:= float(xn-x0);
     if specfunc::abs(h) <= defaulttol*
                            max(specfunc::abs(float(x0)),
                                specfunc::abs(float(xn))) then
         adaptive:= FALSE;
      end_if;
  end_if;

  if method=1 then tol1 := tol/2^(lo_order+1)/2;
              else tol1 := tol/2^(hi_order+2)/2;
  end_if;

  DIGITS:=DIGITS+2+trunc(ln(n+stages+2^(hi_order+2))/ln(10));
  userinfo(1,"internal working precision set to DIGITS = ".expr2text(DIGITS));

  x0:=float(x0);
  if domtype(x0)<>DOM_FLOAT then
     error("non-numerical initial time not admissible"); 
  end_if;

  xn:=float(xn);
  if domtype(xn)<>DOM_FLOAT then
     error("non-numerical time not admissible");
  end_if;

  if not adaptive then
    h:= float(h);
    userinfo(2, "using constant step size h = ".expr2text(h));
    if domtype(h)<>DOM_FLOAT 
       then error("non-numerical step size not admissible"); 
    end_if;
    if h*(xn-x0)<0 
       then error("sign of step size not compatible with range of integration");
    end_if;
  end_if;

  Y0:=map(Y0,float);
  C:=map(C,float);
  A:=map(A,float);
  B:=map(B,float);
  Bb:=map(Bb,float);

  /*------------------------------------------------------------------------*/
  /* local subroutine: standard Lie algebra action is lambda(v,u)= exp(v)*u.*/
  /* This may be redefined by LieGroupAction=LAMBDA. In this case,          */
  /* lambda(v,y)=LAMBDA(exp(v),y) is used.                                  */
  /*------------------------------------------------------------------------*/
  convert:= proc(y) local i;
  begin 
        if y::dom::hasProp(Cat::Matrix)=TRUE
           then y:= y::dom::expr(y);
        end_if;
        y:= map(y, float);
        case domtype(y)
        of DOM_LIST do  if n<>nops(y) then break; end_if;
                        return(array(1..n,1..m,[[y[i]] $ i=1..n]));
        of DOM_ARRAY do case op(y, [0,1]) 
                        of 1 do if n<>op(y,[0,2,2]) then break; end_if;
                                return(array(1..n,1..m, [[y[i]] $ i=1..n]));
                        of 2 do if n<>op(y,[0,2,2]) or m<>op(y,[0,3,2])
                                then break; end_if;
                                return(y);
                        end_case;
        end_case;
        error("wrong output from Lie group action");
  end_proc;

  /* set the Lie algebra action */
  if LAMBDAdefined <> TRUE
  then lambda:=proc(v,y) local k,j; begin /* default */
                  v:= array(1..n,1..n,[[coeff(v[j],k)$j=1..n]$k=1..n]);
                  numeric::expMatrix(v,y);
               end_proc;
  else // make sure that the user defined procedure
       // LAMBDA(G, Y) is fed with data G = array(1..n,1..n)
       // and Y of the same domain type as the initial value Y0: 
       case init_type 
       of 0 do // Y = an array(1..n, 1..m)
           lambda:=proc(v,y) local k,j; begin 
              v:= array(1..n,1..n,[[coeff(v[j],k)$j=1..n]$k=1..n]);
              convert(LAMBDA(numeric::expMatrix(v),y));
           end_proc;
           break;
       of 1 do // Y = a list [..]
           lambda:=proc(v,y) local k,j; begin 
              v:= array(1..n,1..n,[[coeff(v[j],k)$j=1..n]$k=1..n]);
              y:= [y[j, 1] $ j = 1.. n];
              convert(LAMBDA(numeric::expMatrix(v),y));
           end_proc;
           break;
       of 2 do // Y = an array(1..n)
           lambda:=proc(v,y) local k,j; begin 
              v:= array(1..n,1..n,[[coeff(v[j],k)$j=1..n]$k=1..n]);
              y:= array(1..n, [y[j, 1] $ j = 1.. n]);
              convert(LAMBDA(numeric::expMatrix(v),y));
           end_proc;
           break;
       of 3 do // Y = a matrix()
           lambda:=proc(v,y) local k,j; begin 
              v:= array(1..n,1..n,[[coeff(v[j],k)$j=1..n]$k=1..n]);
              convert(LAMBDA(matrix(numeric::expMatrix(v)),
                                    matrix(y)));
           end_proc;
           break;
       end_case;
  end_if;

  /* convert output data from array(1..n,1..m,[..]) */
  /* back to the input type of Y0                   */
  output_convert:=proc(y)
  local i, tmp;
  begin case init_type
        of 1 do return([y[i,1] $ i=1..n]);
        of 2 do if n=1 then
                   tmp:= array(1..n): tmp[1]:= y[1,1];
                   return(tmp);
                end_if;
                return(array(1..n,[y[i,1] $ i=1..n]));
        of 3 do return(matrix(y));
        otherwise return(y);
        end_case;
  end_proc;

  if alldata then
    if domtype(nn)<>DOM_INT then
      error("use integer n in option Alldata = n");
    end_if;
    datatable[1]:= [x0,output_convert(Y0)];
    datacount:=1; nextentry:=2;
    /*------------  define local subroutine -------------*/
    append_to_datatable:=proc(x,y)
    begin
        if datacount=nn then datacount:=0; end_if;
        if (datacount=0 or specfunc::abs(x-xn)<defaulttol )
           and /*prevent last point from being inserted twice: */
          (specfunc::abs(x-datatable[nextentry-1][1])>defaulttol)
        then datatable[nextentry]:=[x,output_convert(y)];
             nextentry:=nextentry+1;
        end_if;
        datacount:=datacount+1;
    end_proc;
    /*---------------------------------------------------*/
  end_if;

//extrapol2:= 1/( 2^hi_order -1 ); /* factors for extrapolating results  */
//extrapol1:= 2^hi_order;          /* when comparing stepsizes h and h/2 */

  /*-----------------------------------------------------------*/
  /*--  define local subroutine: matrix addition of 2 arrays --*/
  /*-----------------------------------------------------------*/
  array_add:=proc(A,B) local n,m,i,j;
  begin n:=op(A,[0,2,2]); m:=op(A,[0,3,2]);
        array(1..n,1..m,[[(A[i,j]+B[i,j])$j=1..m]$i=1..n]);
  end_proc;

  /*-------------------------------------------------*/
  /*--  define local subroutine: matrix commutator --*/
  /*-------------------------------------------------*/
  commutator:=proc(a,b) local j,k,A12,A21,B12,B21,C11,aa,bb,c;   
    begin 
     //   if n=1 then return([poly(0,[X])]) end_if;/* won't arise, see dexpinv*/
          if n=2 then            /* don't use 2*8 multiplications, 6 suffice: */
             A12:= coeff(a[2],1); A21:= coeff(a[1],2);
             B12:= coeff(b[2],1); B21:= coeff(b[1],2);
             C11:= A12*B21-A21*B12;
             aa := coeff(a[1],1) - coeff(a[2],2);
             bb := coeff(b[1],1) - coeff(b[2],2);
             return([poly([[C11,1], [A21*bb - B21*aa, 2]], [X]),
                     poly([[B12*aa - A12*bb,1], [-C11,2]], [X])]);
          else c:= [0$n]; /* initialize list as container */
               (c[j]:=_plus(mapcoeffs(a[k],_mult,coeff(b[j],k))
                           -mapcoeffs(b[k],_mult,coeff(a[j],k)) $ k=1..n);
               ) $ j=1..n;
               return(c);
          end_if;
  end_proc;

  /*-----------------------------------------------------------*/
  /*- define local subroutine: matrix addition of 2 poly lists */
  /*-----------------------------------------------------------*/
  poly_add:= proc(a,b) local j;
             begin
               (b[j]:=a[j]+b[j];) $ j=1.. n; b; 
             end_proc;

  /*-------------------------------------------------------------------------*/
  /*--  define local subroutine dexpinv(u,v,q) =                             */
  /*  v-1/2*[u,v]+1/12*[u,[u,v]]+..+bernoulli(q-2)/(q-2)!*[u,[u,..,[u,v]..]] */
  /*-------------------------------------------------------------------------*/
  dexpinv:= proc(u,v,q) local erg, j, ad;
  begin if n=1 or q<=2 then return(v); end_if;
        ad:= commutator(u,v);
        erg:= poly_add(v,map(ad,mapcoeffs,_mult,-1/2)); 
        if q>3 then 
          ad:= commutator(u,ad);
          erg:= poly_add(erg,map(ad,mapcoeffs,_mult,1/12));
          if q>5 then 
             for j from 4 to q-2 step 2 do
                ad:= commutator(u,commutator(u,ad));
                erg:= poly_add(erg,map(ad,mapcoeffs,_mult,float(bernoulli(j))/j!));
              /* for the adaptive error control it would be advisable  */
              /* to make sure that dexpinv is sufficiently precise.    */
              /* j should run to infinity and break according to:      */
              /* if poly_norm(ad)<poly_norm(erg)*tol then break; end_if; */
             end_for;
           end_if;
        end_if;
        return(erg);
  end_proc;

  /*--------------------------------------------------*/
  /*--  define local subroutine: norm of a matrix-- */
  /*--------------------------------------------------*/
  array_norm:= proc(A) local i,j; begin
      max(_plus(specfunc::abs(A[i,j]) $ j=1..op(A,[0,3,2])) $ i=1..n);
  end_proc;

  poly_norm:= proc(u) local i,j; begin
      max(_plus(specfunc::abs(coeff(u[j],i)) $j=1..n) $ i=1..n);
  end_proc;

  /*--------------------------------------------------------------*/
  /* --------   define local subroutine   ----------------------- */
  /* first choice of stepsize following Hairer, Noersett, Wanner, */
  /* Solving ODEs I, chapter II.5.1 : " Starting step size"       */
  /*--------------------------------------------------------------*/
  if adaptive then
    startingstepsize:=proc(x0,Y0)
    local vz, k1, k1norm, i, den, Y_norm, effectiveOrder, dummy, y, Y, h0, h1;
    begin if iszero(xn-x0) then return(xn-x0); end_if;
          if float(xn-x0) >= 0 then vz:=1 else vz:=-1; end_if;
          k1:= DIFFEQU(x0,Y0);
          k1norm:= poly_norm(k1);
          den:=float(1/max(abs(x0),abs(xn))^(hi_order+1)+k1norm^(hi_order+1));
          Y_norm:=array_norm(Y0);
          /* if Y=0 then switch from relative error to absolute error */
          if iszero(Y_norm) then Y_norm:= 1.0 end_if;
          h0:=float( (Y_norm*tol/den)^(1/(hi_order+1)) );
          /* one Euler step to get away from Y0 ,because this often leads */
          /* to untypical starting vectors k1 with vanishing components   */
          Y0:=lambda(map(k1,mapcoeffs,_mult,vz*h0),Y0) ;
          k1:=DIFFEQU(x0+vz*h0,Y0);
          k1norm:= poly_norm(k1);
          den:=float(1/max(abs(x0),abs(xn))^(hi_order+1)+k1norm^(hi_order+1));
          Y_norm:=array_norm(Y0);
          if iszero(Y_norm) then Y_norm:= 1.0 end_if;
          h1:=float( (Y_norm*tol/den)^(1/(hi_order+1)) );
          h:= vz*min(h0,h1);
          if specfunc::abs(h)>=specfunc::abs(xn-x0) then return(xn-x0); end_if;
          /* Improve this starting stepsize. Try it and estimate the local error: */
          if method=1 then
             effectiveOrder:= min(lo_order,hi_order);
             repeat /* reduce stepsize, if overflow occurs */
               if traperror((([dummy, Y, localerror]):= [MKPairStep(h,x0,Y0)];))<>20
                 then Y_norm:=array_norm(Y); break;
                 else h:=h/2;
               end_if;
             until TRUE end_repeat;
          end_if;
          if method=2 then
             effectiveOrder:= hi_order; /* hi_order+1 bei lokaler Extrapolation*/
             k1:=DIFFEQU(x0,Y0);
             repeat /* reduce stepsize, if overflow occurs */
               if traperror(( y:=(MKStep(h,x0,Y0,k1))[2];
                              Y:=(MKStep(h/2,MKStep(h/2,x0,Y0,k1)))[2];))<>20
                 then Y_norm:=array_norm(Y);
                      /* keine lokale Extrapolation! */
//                    Y:=map(array_add(map(Y,_mult,extrapol1),map(y,_mult,-1)),
//                           _mult,extrapol2);
                      localerror:= array_norm(array_add(Y,map(y,_mult,-1)));
                 else h:= h/2;
               end_if;
             until TRUE end_repeat;
          end_if;
          /* is Y=0 then switch to absolute error: */
          if iszero(Y_norm) then Y_norm:=1.0; end_if;
          /* estimate stepsize leading to a relative local error = tol */
          h:= specfunc::abs(h);
          if Y_norm*tol*h^(effectiveOrder+1) >=
             localerror*specfunc::abs(xn-x0)^(effectiveOrder+1)
             then return(xn-x0);
             else return(vz*h*(Y_norm*tol/localerror)^(1/(effectiveOrder+1)));
          end_if;
    end_proc: /*--------------------------------*/
  end_if;

  /*--------------------------------------------------*/
  /*--  define the numerical Munthe-Kaas steps     -- */
  /*------------------------------------------------------------------------- */
  /* method 1 is the Munthe-Kaas scheme based on an embedded Runge-Kutta pair */
  /*------------------------------------------------------------------------- */
  if method=1 then

    MKPairStep:=proc(h,x0,Y0)
    local i,j,k,finalstep,ks,u,y,localerror;
    begin if specfunc::abs(h)>=specfunc::abs(xn-x0) 
             then finalstep:=TRUE; h:=xn-x0; else finalstep:=FALSE;
          end_if;
          if args(0)=4 then ks[1]:=args(4); else ks[1]:=DIFFEQU(x0,Y0); end_if;
          for i from 2 to stages do   /* RK Zwischenstufen berechnen */
            u:= [0$n]; /* storage for u = sum(A[i,j]*ks[j],j=1..i-1) */
            (u[k]:= _plus(mapcoeffs(ks[j][k],_mult,A[i,j]) $j=1..i-1);) $k=1..n;
            u:=map(u,mapcoeffs,_mult,h);
            ks[i]:=dexpinv(u,DIFFEQU(x0+C[i]*h,lambda(u,Y0)),hi_order);
          end_for;
          u:= [0$n]; /* storage for u = sum(B[j]*ks[j],j=1..stages) */
          (u[k]:= _plus(mapcoeffs(ks[j][k],_mult,B[j]) $j=1..stages);) $k=1..n;
          u:=map(u,mapcoeffs,_mult,h);
          y:=lambda(u,Y0);
          u:= [0$n]; /* storage for u = sum(Bb[j]*ks[j],j=1..stages) */
          (u[k]:= _plus(mapcoeffs(ks[j][k],_mult,Bb[j]) $j=1..stages);) $k=1..n;
          u:=map(u,mapcoeffs,_mult,h);
          localerror:=array_norm(array_add(lambda(u,y),map(y,_mult,-1)));
          if finalstep
             then return(xn,y,localerror);
             else return(x0+h,y,localerror);
          end_if;
    end_proc:
    /* ---------------------------------------*/
    adaptiveMKPairStep:=proc(h,x0,Y0)
    local finalstep,ks1,dummy,y,y_norm,i,j,localerror;
    begin 
      if specfunc::abs(h)<defaulttol*specfunc::abs(x0) then
         warning("adaptive step size h=".expr2text(h)." at time t=".
                 expr2text(x0)." violates abs(h)>=abs(t)/10^DIGITS. ".
                 " Increase DIGITS and/or use option RelativeError=tol.");
         error("adaptive step size too small");
      end_if;
      if specfunc::abs(h)>=specfunc::abs(xn-x0)
         then finalstep:=TRUE; h:=xn-x0;
         else finalstep:=FALSE;
      end_if;
      ks1:=DIFFEQU(x0,Y0);
      while TRUE do
        ([dummy,y,localerror]):=[MKPairStep(h,x0,Y0,ks1)];
        y_norm:=array_norm(y);
        if localerror<=y_norm*tol1
        then stepsize_info:="adjusting step size to          h = ".expr2text(h);
             localerr_info:="estimated absolute local error: ".expr2text(localerror);
             if finalstep
                then return(h,xn,y);
                else return(2*h,x0+h,y);
             end_if;
        end_if;
        if localerror>y_norm*tol then
           finalstep:=FALSE; h:=h/2; next;
        end_if;
        break;
      end_while;
      stepsize_info:="adjusting step size to          h = ".expr2text(h);
      localerr_info:="estimated absolute local error: ".expr2text(localerror);
      h,x0+h,y;
    end_proc;
  end_if;

 /*------------------------------------------------------------------------- */
 /* method 2 is the Munthe-Kaas scheme based on a simple Runge-Kutta scheme  */
 /*------------------------------------------------------------------------- */
  if method=2 then

    MKStep:=proc(h,x0,Y0)
    local i,j,k,finalstep,ks,u,y;
    begin if specfunc::abs(h)>=specfunc::abs(xn-x0) 
             then finalstep:=TRUE; h:=xn-x0; else finalstep:=FALSE;
          end_if;
          if args(0)=4
             then ks[1]:=args(4); else ks[1]:= DIFFEQU(x0,Y0);
          end_if;
          for i from 2 to stages do
            u:= [0$n]; /* storage for u = sum(A[i,j]*ks[j],j=1..i-1) */
            (u[k]:= _plus(mapcoeffs(ks[j][k],_mult,A[i,j]) $j=1..i-1);) $k=1..n;
            u:=map(u,mapcoeffs,_mult,h);
            ks[i]:=dexpinv(u,DIFFEQU(x0+C[i]*h,lambda(u,Y0)), hi_order);
          end_for;
          u:= [0$n]; /* storage for u = sum(B[j]*ks[j],j=1..stages) */
          (u[k]:= _plus(mapcoeffs(ks[j][k],_mult,B[j]) $j=1..stages);) $k=1..n;
          u:= map(u,mapcoeffs,_mult,h);
          y:=lambda(u,Y0);
          if finalstep then return(xn,y) else return(x0+h,y); end_if;
    end_proc;
    /* -----------------------------*/
    adaptiveMKStep:=proc(h,x0,Y0)
    local finalstep,ks1,y,data,Y,i,j,Y_norm,dy_norm;
    begin 
      if specfunc::abs(h)<defaulttol*specfunc::abs(x0) then
        warning("adaptive step size h=".expr2text(h)." at time t=".
                 expr2text(x0)." violates abs(h)>=abs(t)/10^DIGITS. ".
                 " Increase DIGITS and/or use option RelativeError=tol.");
         error("adaptive step size too small");
      end_if;
      if specfunc::abs(h)>=specfunc::abs(xn-x0) 
         then finalstep:=TRUE; h:=xn-x0;
         else finalstep:=FALSE;
      end_if;
      ks1:=DIFFEQU(x0,Y0);
      y:=(MKStep(h,x0,Y0,ks1))[2];
      data:=(MKStep(h/2,x0,Y0,ks1));
      while TRUE do
        Y:=(MKStep(h/2,data))[2];
        Y_norm:=array_norm(y);
        /* hier keine lokale Extrapolation */
//      Y:=map(array_add(map(Y,_mult,extrapol1),map(y,_mult,-1)),_mult,extrapol2);
        dy_norm:=array_norm(array_add(Y,map(y,_mult,-1)));
        if dy_norm<=Y_norm*tol1
        then stepsize_info:="adjusting step size to          h = ".expr2text(h);
             localerr_info:="estimated absolute local error: ".expr2text(dy_norm);
             if finalstep
             then return(h,xn,Y);
             else return(2*h,x0+h,Y);
             end_if;
        end_if;
        if dy_norm>Y_norm*tol then
          h:=h/2; y:=data[2]; data:=MKStep(h/2,x0,Y0,ks1);
          finalstep:=FALSE; next;
        end_if;
        break;
      end_while;
      stepsize_info:="adjusting step size to          h = ".expr2text(h);
      localerr_info:="estimated absolute local error: ".expr2text(dy_norm);
      h,x0+h,Y;
    end_proc;

  end_if;

// stopping criterion for the integration with running time x
// is:  .. until compare(x, xn)
compare:= if x0 <= xn then
             (a, b) -> bool(a >= b);
          else
             (a, b) -> bool(a <= b);
          end_if:


  /*------------------*/
  /*   main program   */
  /*------------------*/
  stepcount:= 0:
  if adaptive then
    data:=startingstepsize(x0,Y0),x0,Y0;
    userinfo(2, "start:  time = ".expr2text(data[2]));
    userinfo(3, "           Y = ".expr2text(data[3]));
    case method
      of 1 do repeat
                userinfo(3, "trying next step with step size h = ".expr2text(data[1]));
                if traperror((data:=adaptiveMKPairStep(data))) <> 0 
                then userinfo(1,"overflow/underflow occurred, reducing step size"); 
                     data[1]:= data[1]/2;
                else stepcount:= stepcount+1;
                     userinfo(3, stepsize_info);
                     userinfo(3, localerr_info);
                     userinfo(2, "accept: time = ".expr2text(data[2]));
                     userinfo(3, "           Y = ".expr2text(data[3]));
                     if alldata then append_to_datatable(data[2],data[3]); end_if;
                end_if;
              until compare(data[2], xn) end_repeat;
              break
      of 2 do repeat
                userinfo(3, "trying next step with step size h = ".expr2text(data[1]));
                if traperror((data:=adaptiveMKStep(data))) <> 0 
                then userinfo(1,"overflow/underflow occurred, reducing step size"); 
                     data[1]:= data[1]/2;
                else stepcount:= stepcount+1;
                     userinfo(3, stepsize_info);
                     userinfo(3, localerr_info);
                     userinfo(2, "accept: time = ".expr2text(data[2]));
                     userinfo(3, "           Y = ".expr2text(data[3]));
                     if alldata then append_to_datatable(data[2],data[3]); end_if;
                end_if;
              until compare(data[2], xn) end_repeat;
              break
    end_case;
    userinfo(1, "solution after ".expr2text(stepcount)." steps");
    if alldata then result:=[datatable[i] $ i=1..nops(datatable)]
               else result:=output_convert(data[3]);
    end_if;
  end_if;

  if not adaptive then
    // Make sure that tiny intervals are handled by just
    // returning the initial data. Above, this situation
    // lead to adaptive:= FALSE, so we have to handle this
    // here:
    if specfunc::abs(h)<=defaulttol*max(specfunc::abs(x0),
                                        specfunc::abs(xn)) then
         method:= 0: // flag to indicate that nothing has to be done
    end_if;
    data:=x0,Y0;
    userinfo(2,"time = ".expr2text(data[1]));
    userinfo(3,"   Y = ".expr2text(data[2]));
    case method
      of 0 do // Step size is too small to increment x.
              // Just return the initial data.
              break
      of 1 do repeat
                stepcount:= stepcount+1;
                data:=MKPairStep(h,data[1],data[2]);
                userinfo(2, "accept: time = ".expr2text(data[1]));
                userinfo(3, "           Y = ".expr2text(data[2]));
                if alldata then append_to_datatable(data[1],data[2]); end_if;
              until compare(data[1], xn) end_repeat;
              break
      of 2 do repeat
                stepcount:= stepcount+1;
                data:=MKStep(h,data);
                userinfo(2, "accept: time = ".expr2text(data[1]));
                userinfo(3, "           Y = ".expr2text(data[2]));
                if alldata then append_to_datatable(data[1],data[2]); end_if;
              until compare(data[1], xn) end_repeat;
              break
    end_case;
    userinfo(1, "solution after ".expr2text(stepcount)." steps");
    if alldata then result:=[datatable[i] $ i=1..nops(datatable)]
               else result:=output_convert(data[2]);
    end_if;
  end_if;
  result;
end_proc:

