// ---------------  file odesolve.mu 10.10.97 ------------------------ 
/*
ToDo: besseres Interface, z.B. wie im Dubbel-calc_init:

>> dsol(Ai''(x)=x*Ai(x),([Ai(0)=0, Ai'(0)=1], [1]))

            {Ai(1.0) = 1.085339648, D(Ai)(1.0) = 1.347444527}

dsol := proc(X, anfBed)
          option hold;
          local fields, Indets, x2, sol;
        begin
          X := level(eval(context(X)));
          if contains({DOM_SET, DOM_LIST}, domtype(X)) then
            X := op(X);
          end_if;
          [anfBed, x2] := [context(anfBed)];
          anfBed := map(op(anfBed),float@level@eval);
          if contains({DOM_SET, DOM_LIST}, domtype(anfBed)) then
            anfBed := op(anfBed);
          end_if;
          x2 := float(level(eval(op(x2))));
          if x2::dom <> DOM_FLOAT then
            error("Die DGL kann numerisch nur an einer festen, ".
                  "reellen Stelle ausgwertet werden.");
          end_if;
          Indets := op(indets({X}) minus Type::ConstantIdents);
          if domtype(Indets) <> DOM_IDENT then
            error("Die DGL darf keine freien Parameter enthalten")
          end_if;
          fields := sort(map([op(misc::subExpressions({X}, "D"))],
                          a -> op(a)(Indets)));
          sol := numeric::odesolve2(
                   numeric::ode2vectorfield({X, anfBed}, fields))(x2);
          {op(zip(subs(fields, Indets=x2), sol, `=`))};
        end_proc

*/
/*  Help file:

 odesolve -- numerical solution of the initial value problem
             of a dynamical system

 Call:

 numeric::odesolve( t0 .. t, f, y0 <, method > <, Stepsize = h >
           <, RelativeError = rtol> 
           <, AbsoluteError = atol> 
           <, MaxStepsize = hmax> 
           <, Alldata = n > 
           <, Symbolic> )
 numeric::odesolve( f, t0..t, y0 <, method > <, Stepsize = h >
           <, RelativeError = rtol> 
           <, AbsoluteError = atol> 
           <, MaxStepsize = hmax> 
           <, Alldata = n >
           <, Symbolic> )

 Parameter:

    t0     -- numerical value for the initial time
    t      -- numerical value for the final time
    f      -- a function or procedure representing the vector field 
              of the dynamial system dy/dt = f(t,y)
    y0     -- a list or vector of initial conditions y(t0)=y0
    method -- name of the numerical scheme, one of
              EULER1, RKF43, RK4, RKF34, RKF54a, RKF54b, DOPRI54, CK54,
              RKF45a, RKF45b, DOPRI45, CK45, DOPRI65, xDOPR65, 
              DOPRI56, xDOPRI56, BUTCHER6, RKF87, RKF78, 
              xRKF43, xRKF34, xRKF54a, xRKF54b, xDOPRI54, xCK54, xRKF45a,
              xRKF45b, xDOPRI45, xCK45, xRKF87, xRKF78,
              DOPRI78, xDOPRI78, DOPRI87, xDOPRI87,
              GAUSS(NumberOfStages) or 
              GAUSS = NumberOfStages
    h      -- numerical value for the step size
    n      -- an integer controlling the size of the output list
  rtol,atol-- positive real values >= 10^(-DIGITS)

 Synopsis:

  numeric::odesolve( t0..t, f, y0 <, Options> ) returns the solution
  y(t) of the initial value problem dy/dt= f(t,y), y(t0)=y0, where 
  y0, y(t) are real or complex vectors.

  The vector field f=f(t,y) must be represented by a function or a
  procedure with two input parameters: the scalar t and the vector y.
  It has to return the vector f(t,y) either as a list or as an array.
  Also for scalar equations the field y has to be represented as a
  1-dimensional list or array. For instance, the input data for
  the scalar initial value problem dy/dt = t*sin(y), y(0)=1, have
  to be of the form

    f := proc(t,y) begin [t*sin(y[1])] end_proc; y0 := [ 1 ];

  The result of numeric::odesolve is a vector of the same type as
  the initial condition y0, i.e., either a list or an array.

  As default an adaptive mechanism is used to control the step size
  such that accepted steps are accurate within the relative precision
  given by 10^(-DIGITS). If no method is specified, then the embedded
  pair DOPRI78 of order 8 is used as default method.

  All methods presently implemented are Runge-Kutta type single step
  schemes. The methods RKF43, RKF34, RKF54a, RKF54b, DOPRI54, CK54, RKF45a, 
  RKF45b, DOPRI45, DOPRI65, DIPRI56, DOPRI87, DOPRI78, CK45, RKF87, RKF78 are embedded pairs
  of Runge-Kutta-Fehlberg, Dormand-Prince and Cash-Karp type, respectively.
  For them estimates of the local truncation error 
  are obtained in the usual way by comparing the results of the two 
  submethods of the pair. The names indicate the orders of the subprocesses. 
  For instance, RKF34 and RKF43 denote the same embedded Runge-Kutta-Fehlberg
  pair of order 3 and 4. However, in RKF34 the result of the fourth order 
  submethod is accepted, whereas RKF43 advances using the result of the 
  third order submethod. In both cases the truncation error of the lower 
  order subprocess is estimated and controlled.

  For single methods such as EULER1 (the first order Euler method),
  RK4 (the classical fourth order Runge-Kutta method) or BUTCHER6 (a 
  Runge-Kutta method of order 6) the relative local error is controlled
  by comparing steps with different step sizes. By local extrapolation 
  the effective order of the method is increased by one. Local 
  extrapolation is also available for the implemented embedded pairs. 
  For instance, the method xRKF78 uses extrapolation of the 8th order 
  subprocess of RKF78, yielding a method of effective order 9. However, 
  the cheap error estimate based on the embedded pair is not used in this 
  case implying some loss of efficiency when comparing RKF78 and xRKF78.

  The Butcher data of the implemented methods are available via 
  numeric::butcher(method), where method is one of EULER1, RKF43, RK4,
  RKF34, RKF54a, RKF54b, DOPRI54, CK54, RKF45a, RKF45b, DOPRI45, CK45,
  DOPRI65, DOPRI56, BUTCHER6, RKF87, RKF78, DOPRI87, DOPRI78.

  The option Stepsize = h switches off the adaptive mechanism and
  uses a fixed step size h in the Runge-Kutta iteration. A final step 
  with smaller step size is used to match the end t of the integration
  interval t0 .. t .

  The call odesolve( t0..t0+h, f, y0, Symbolic, <,method> ) returns
  a list/array of expressions representing a single step of the numerical 
  scheme with step size h. In this mode symbolic values for t0, h, and 
  the components of y0 are admissible.

  With the option Alldata = n numeric::odesolve returns the list
  [ [t0,y0], [t1,y1] , .. ] , where [ t.i, y.i ] are accepted
  intermediate points generated by the Runge-Kutta iteration. This
  list may be useful, e.g., for further graphical processing of the
  output. The integer n controls the size of the returned list.
  For n=1 all accepted data are returned. This case may also be
  invoked by entering the simplified option Alldata, which is
  equivalent to Alldata = 1. For n > 1 only each n.th accepted 
  data point is stored in the list. The initial point [t0,y0] and
  the final point [t,y(t)] are always present in the output list. 
  For n<=0 only the initial and the final data [ [t0,y0], [t,y(t)] ] 
  are returned. 

  WARNING: only local errors are controlled by the adaptive mechanism.
  No control of the global error is granted.

  WARNING: the run time of the numeric integration with a method of
  order p grows like O(10^(DIGITS/(p+1)), when DIGITS increases.
  Computations with high precision are very expensive.

  WARNING: presently only explicit single step methods of Runge-Kutta
  type are implemented. Stiff problems cannot be handled efficiently
  with these methods.

  Example: 

>>  f := proc(t,y) begin  [ t*sin(y[1]) ] end_proc:

>>  numeric::odesolve( 0..10, f, [1]);

                               [3.141592653]

>>  numeric::odesolve( t0..t0+h, f, [y0], Symbolic, EULER1 );

                             [y0 + h t0 sin(y0)]

>>  f := proc(t,y) begin array(1..2,[ y[1]*y[2], y[1]/y[2] ]) end_proc:

>>  DIGITS:=20:                                      

>>  numeric::odesolve( 0..PI/3 , f, [ 1 , 1 ] , xRKF78 );  

               [5.4131223316245629744, 2.4238006323952340528]

>>  DIGITS:=4:

>>  numeric::odesolve( 0.. 1 , f, [1,1] , Stepsize=0.2 ,  Alldata = 2 );

     [ [0.0, [1.0, 1.0]], [0.4, [1.619, 1.419]], [0.8, [3.16, 1.955]],

       [1.0, [4.839, 2.322]] ]

>> "plot the numerical solution y(t) of dy/dt = t*sin(y)-y, y(0)=1";

>>  DIGITS:=10:

>>  f:= proc(t,y) begin [t*sin(y[1])-y[1]] end_proc;

>>  data := numeric::odesolve( 0..5 , f , [1] , Alldata ):

>>  plotpoints := [ point(data[i][1],data[i][2][1]) $ i=1..nops(data) ]:

>>  plot2d( [ Mode=List , plotpoints ] );


  See also:  numeric::butcher 
     
 ------------------------------------------------------------------------*/

numeric::odesolve := proc(diffequ, x0_xn, y0)
local x0, xn, n, nn, tmp, i, method, adaptive, symbolic, alldata, datacount,
      ignoreSignOfStepsize,
      nextentry, datatable, append_to_datatable, h, hmax, 
      defaulttol, rtol, rtol1, atol, atol1,
      extrapol1, extrapol2, lo_order, hi_order, data, getButcherData,
      stages, C, A, B, Bb, eps,
      RKPairStep,     adaptiveRKPairStep,
      RKStep,         adaptiveRKStep,
      RK4Step,        adaptiveRK4Step,
      ImplicitRKStep, adaptiveImplicitRKStep,
      symbolicRKStep,
      startingstepsize,
      stepsize_error,
      stepsize_info, localerr_info, stepcount, // dowarn,
      compare,
      result;
save  DIGITS;
begin 

   if args(0) < 3 then error ("not enough arguments"); end_if;

   // the call odesolve(x0_xn, diffequ, y0) is also allowed
   // for backward compatibility with MuPAD 2.0 and earlier. 

   if /*domtype(x0_xn) = DOM_PROC and*/
      type(diffequ) = "_range" then
      [x0_xn, diffequ] := [diffequ, x0_xn];
   end_if;

   if type(x0_xn) <> "_range" then
      error("wrong specification of the integration interval");
   end_if;
   x0:=op(x0_xn,1);
   xn:=op(x0_xn,2);

   eps:= 10.0^(-DIGITS); // relative precision for Newton iteration

   // The differential equation can be a procedure 
   // or a function expression
   if domtype(diffequ) <> DOM_PROC and 
      domtype(diffequ) <> DOM_EXPR then
      error("the vector field must be presented by a function or a procedure");
   end_if;

   if not has({DOM_ARRAY, DOM_HFARRAY, DOM_LIST}, domtype(y0)) then
      if y0::dom::hasProp(Cat::Matrix)=TRUE then
         y0:= y0::dom::convert_to(y0, matrix);
      end_if;
   end_if:

   case domtype(y0) 
      of DOM_ARRAY do 
      of DOM_HFARRAY do 
           n:= op(y0,[0,2,2]); 
           break;
      of matrix do 
      of DOM_LIST do  
            n:= nops(y0); 
      break;
      otherwise error("the initial condition must be a vector (list or array)");
   end_case;

   tmp:= diffequ(x0,y0);
   case domtype( tmp ) 
      of DOM_ARRAY do 
      of DOM_HFARRAY do 
           nn := op(tmp,[0,2,2]); 
           break
      of matrix do 
      of DOM_LIST do  
           nn := nops(tmp); 
           break;
      otherwise error("the vector field yields wrong type of output");
   end_case;
   if n<>nn 
    then error("size of vector field and initial vector do not match");
   end_if;

   // --------   define local subroutine   ---------- 
   getButcherData:=proc(method)
   local i,b;
   begin ([stages,C,A,b,B,lo_order,hi_order]):=numeric::butcher(method, DIGITS);
         if stages=1 then Bb:=array(1..stages);Bb[1]:=B[1]-b[1]
         else Bb:=array(1..stages,[ (B[i]-b[i]) $ i=1..stages ])
         end_if;
   end_proc;
   // ---------------------------------------------- 

   adaptive := TRUE;                 // default 
   symbolic := FALSE;                // default 
   alldata  := FALSE;                // default 
// dowarn:= FALSE;                   // default
   method:=1; getButcherData(DOPRI78); // default 
   defaulttol := float(10^(-DIGITS));// default 
   rtol:= defaulttol;                // default 
   atol:= float(defaulttol^10);      // default
   hmax:= RD_INF;                    // default
   ignoreSignOfStepsize:= FALSE;     // default
   if args(0) > 3 then               // get options 
     for i from 4 to args(0) do
       if has(args(i), GAUSS) then
           method:=4: // implicit RKStep
           getButcherData(args(i));
           next;
       end_if;
       case op(args(i),1)
         of  EULER1   do method:=2;getButcherData(EULER1);break
         of  RK4      do method:=3;getButcherData(RK4);   break
         of  RKF34    do method:=1;getButcherData(RKF34); break
         of xRKF34    do method:=2;getButcherData(RKF34);break
         of  RKF43    do method:=1;getButcherData(RKF43); break
         of xRKF43    do method:=2;getButcherData(RKF43);break
         of  RKF45a   do method:=1;getButcherData(RKF45a);break
         of xRKF45a   do method:=2;getButcherData(RKF45a);break
         of  RKF54a   do method:=1;getButcherData(RKF54a);break
         of xRKF54a   do method:=2;getButcherData(RKF54a);break
         of  RKF45b   do method:=1;getButcherData(RKF45b);break
         of xRKF45b   do method:=2;getButcherData(RKF45b);break
         of  RKF54b   do method:=1;getButcherData(RKF54b);break
         of xRKF54b   do method:=2;getButcherData(RKF54b);break
         of  DOPRI45  do method:=1;getButcherData(DOPRI45);break
         of xDOPRI45  do method:=2;getButcherData(DOPRI45);break
         of  DOPRI54  do method:=1;getButcherData(DOPRI54);break
         of xDOPRI54  do method:=2;getButcherData(DOPRI54);break
         of   CK54    do method:=1;getButcherData(CK54)   ;break
         of  xCK54    do method:=2;getButcherData(CK54)   ;break
         of   CK45    do method:=1;getButcherData(CK45)   ;break
         of  xCK45    do method:=2;getButcherData(CK45)   ;break
         of  DOPRI65  do method:=1;getButcherData(DOPRI65);break
         of xDOPRI65  do method:=2;getButcherData(DOPRI65);break
         of  DOPRI56  do method:=1;getButcherData(DOPRI56);break
         of xDOPRI56  do method:=2;getButcherData(DOPRI56);break
         of  BUTCHER6 do method:=2;getButcherData(BUTCHER6);break
         of  RKF78    do method:=1;getButcherData(RKF78); break
         of xRKF78    do method:=2;getButcherData(RKF78); break
         of  RKF87    do method:=1;getButcherData(RKF87); break 
         of xRKF87    do method:=2;getButcherData(RKF87); break
         of  DOPRI78  do method:=1;getButcherData(DOPRI78);break
         of xDOPRI78  do method:=2;getButcherData(DOPRI78);break
         of  DOPRI87  do method:=1;getButcherData(DOPRI87);break
         of xDOPRI87  do method:=2;getButcherData(DOPRI87);break
         of GAUSS     do method:=4:getButcherData(GAUSS(stages)); break;
         of Stepsize do adaptive:=FALSE;
                        if type(args(i)) = "_equal" 
                           then h:=op(args(i),2);
                           else error("step size specification must be ".
                                      "of the form Stepsize = h ");
                        end_if;
                        break
         of RelativeError do
                        if type(args(i)) = "_equal"
                           then rtol:= float(op(args(i),2));
                           else error("specification of relative error must be ".
                                      "of the form RelativeError = rtol ");
                        end_if;
                        break
         of AbsoluteError do
                        if type(args(i)) = "_equal"
                           then atol:= float(op(args(i),2));
                           else error("specification of absolute error must be ".
                                      "of the form AbsoluteError = atol ");
                        end_if;
                        break
         of MaxStepsize do
                        if type(args(i)) = "_equal"
                           then hmax:= float(op(args(i),2));
                           else error("specification of maximal stepsize must be ".
                                      "of the form MaxStepsize = hmax ");
                        end_if;
                        break
         of Symbolic do symbolic:=TRUE; break
         of Alldata  do alldata :=TRUE;
                        if type(args(i)) = "_equal" 
                           then nn:=op(args(i),2)
                           else nn:=1;
                        end_if;
                        break
         of "CalledByOdesolve2" do
            ignoreSignOfStepsize:= TRUE;
            break;
         otherwise error("unknown option");
       end_case;
     end_for;       
   end_if;

   if not testtype(hmax, DOM_FLOAT) or hmax < 0 then
      error("expecting a real numerical value for the maximal ".
            "step size, got: ".expr2text(hmax));
   end_if;

   if symbolic then
     if not adaptive then
       warning("ignoring Stepsize=h, using step size t-t0 = ".expr2text(xn-x0));
     end_if;
     // --------   define local subroutine   ---------- 
     symbolicRKStep:=proc(h, x0, y0)
     local i,j,tmp,ii,Y,k;
     begin k[1]:=diffequ(x0,y0);
           for i from 2 to stages do
               Y:=y0; // initialize storage for intermediate stages 
               for j from 1 to i-1 do
                 tmp:=A[i,j]*h; (Y[ii]:=Y[ii]+tmp*k[j][ii]) $ ii=1..n;
               end_for;
               k[i]:= diffequ(x0+C[i]*h,Y);
           end_for;
           Y:=y0;  // initialize storage for result 
           for j from 1 to stages do
               tmp:=B[j]*h; (Y[ii]:=Y[ii]+tmp*k[j][ii]) $ ii=1..n;
           end_for;
           return(Y)
     end_proc:    // ---------------------------------- 
     result:= symbolicRKStep(xn-x0,x0,y0);
     return(result);
   end_if;

   if adaptive then
      if domtype(rtol)<>DOM_FLOAT or rtol < 0 then
         error("the relative error tolerance must be a nonnegative numerical value");
      end_if;
      if domtype(atol)<>DOM_FLOAT or atol < 0 then
         error("the absolute error tolerance must be a nonnegative numerical value");
      end_if;
      if rtol < 0.99*defaulttol then
         error("the relative error tolerance must be larger than ".
               "the working precision 10^(-DIGITS).");
      end_if;
   end_if;

   // For tiny integration intervals there is danger, that
   // adaptiveRKPairStep etc. think that step size is too small.
   // Avoid the resulting error by doing 1 non-adaptive step with
   // the tiny step size:
   if adaptive then
      h:= float(xn-x0);
      if specfunc::abs(h) <= defaulttol*
                             max(specfunc::abs(float(x0)),
                                 specfunc::abs(float(xn))) then
         adaptive:= FALSE;
      end_if;
    end_if;

   if method=1 then rtol1:= 0.5*rtol/2^(lo_order+1);
                    atol1:= 0.5*atol/2^(lo_order+1):
               else rtol1:= 0.5*rtol/2^(hi_order+1); 
                    atol1:= 0.5*atol/2^(hi_order+1):
   end_if;
    
   DIGITS:=DIGITS+2+trunc(ln(n+stages+2^(hi_order+2))/ln(10));
   userinfo(1,"internal working precision set to DIGITS = ".expr2text(DIGITS));

   x0:=float(x0);
   if domtype(x0)<>DOM_FLOAT then error("non-numerical initial time not admissible");end_if;

   xn:=float(xn);
   if domtype(xn)<>DOM_FLOAT then error("non-numerical time not admissible"); end_if;

   if not adaptive then
      h:=float(h);
      userinfo(2, "using constant step size h = ".expr2text(h));
      if domtype(h)<>DOM_FLOAT then error("non-numerical step size not admissible"); end_if;
     
      if h*(xn-x0)<0 then 
         if ignoreSignOfStepsize then
            h:= -h;
         else
            error("sign of step size not compatible with range of integration");
         end_if:
      end_if;
   end_if;

   y0:=map(y0,float);
   C:=map(C,float);
   A:=map(A,float);
   B:=map(B,float);
   Bb:=map(Bb,float);

   if alldata then
      if domtype(nn)<>DOM_INT then
         error("use integer n in option Alldata = n");
      end_if;
      datatable[1]:= [x0,y0];
      datacount:=1; nextentry:=2;
      //------------  define local subroutine -------------
      append_to_datatable:=proc(x,y)
      begin  
          if datacount=nn then datacount:=0; end_if;
          if (datacount=0 or specfunc::abs(x-xn)<defaulttol )
             and //prevent last point from being inserted twice: 
            (specfunc::abs(x-datatable[nextentry-1][1])>defaulttol)
          then datatable[nextentry]:=[x,y];
               nextentry:=nextentry+1;
          end_if;
          datacount:=datacount+1;
      end_proc;
      //---------------------------------------------------
   end_if;

   extrapol2:= 1/( 2^hi_order -1 ); // factors for extrapolating results  
   extrapol1:= 2^hi_order;          // when comparing step sizes h and h/2 

if adaptive then

    stepsize_error:= (h, x0) ->
         error("adaptive step size h = ".expr2text(h)." at time t = ".
               expr2text(x0)." too small. ".
               "It violates abs(h) >= abs(t)/10^DIGITS. ".
               "Increase DIGITS and/or use option RelativeError = rtol,".
               "AbsoluteError = atol.");

    // --------   define local subroutine   ----------------------- 
    // first choice of step size following Hairer, Noersett, Wanner, 
    // Solving ODEs I, chapter II.5.1 : " Starting step size"       


    startingstepsize:=proc(x0,y0)
    local vz,k1,k1norm,ii,den, Y_norm, effectiveOrder,
          dummy, y, Y, h0, h1, localerror;
    begin if iszero(xn-x0) then return(xn-x0) end_if;
          if float(xn-x0) >= 0 then vz:=1 else vz:=-1; end_if;
          k1:=map(diffequ(x0,y0),float); 
          k1norm:= _plus( specfunc::abs(k1[ii])^2 $ ii=1..n )^(1/2);
          den:=float(   1/max(abs(x0),abs(xn))^(hi_order+1) 
                      +                 k1norm^(hi_order+1)  );
          Y_norm:= _plus( specfunc::abs(y0[ii])^2 $ ii=1..n )^(1/2);
          /* if Y=0 then switch from relative error to absolute error */
          if iszero(Y_norm) then Y_norm:= 1.0 end_if;
          h0:=float( (max(Y_norm*rtol,atol)/den)^(1/(hi_order+1)) );
          // one Euler step to get away from y0 ,because this often leads 
          // to untypical starting vectors k1 with vanishing components   
          (y0[ii]:=y0[ii]+vz*h0*k1[ii] ) $ ii=1..n;
          k1:=map(diffequ(x0+vz*h0,y0),float); 
          k1norm:= _plus( specfunc::abs(k1[ii])^2 $ ii=1..n )^(1/2);
          den:=float(   1/max(abs(x0),abs(xn))^(hi_order+1) 
                      +                 k1norm^(hi_order+1)  );
          Y_norm:= _plus( specfunc::abs(y0[ii])^2 $ ii=1..n )^(1/2);
          h1:=float( (max(Y_norm*rtol,atol)/den)^(1/(hi_order+1)) );
          h:= vz*min(h0,h1);
          if h > hmax then return(hmax); end_if;
          if specfunc::abs(h)>=specfunc::abs(xn-x0) then return(xn-x0); end_if;
          /* Improve this starting step size. Try it and estimate the local error: */
          if method=1 then
             effectiveOrder:= min(lo_order,hi_order);
             ([dummy, Y, localerror]):= [RKPairStep(h,x0,y0)];
             Y_norm:= _plus( specfunc::abs(Y[ii])^2 $ ii=1..n )^(1/2);
          end_if;
          if method=2 then
             effectiveOrder:= hi_order+1;
             k1:=map(diffequ(x0,y0),float);
             y:=(RKStep(h,x0,y0,k1))[2];
             Y:=(RKStep(h/2,RKStep(h/2,x0,y0,k1)))[2];
             Y_norm:= _plus( specfunc::abs(Y[ii])^2 $ ii=1..n )^(1/2);
             (Y[ii]:=(extrapol1*Y[ii]-y[ii])*extrapol2) $ ii=1..n ;
             localerror:= (_plus(specfunc::abs(Y[ii]-y[ii])^2 $ ii=1..n ))^(1/2);
          end_if;
          if method=3 then
             effectiveOrder:= hi_order+1;
             k1:=map(diffequ(x0,y0),float);
             y:=(RK4Step(h,x0,y0,k1))[2];
             Y:=(RK4Step(h/2,RK4Step(h/2,x0,y0,k1)))[2];
             Y_norm:= _plus( specfunc::abs(Y[ii])^2 $ ii=1..n )^(1/2);
             (Y[ii]:=(extrapol1*Y[ii]-y[ii])*extrapol2) $ ii=1..n ;
             localerror:= (_plus(specfunc::abs(Y[ii]-y[ii])^2 $ ii=1..n ))^(1/2);
          end_if;
          if method=4 then
             effectiveOrder:= hi_order+1;
             k1:=map(diffequ(x0,y0),float);
             while TRUE do
                y:=(ImplicitRKStep(h,x0,y0,k1))[2];
                if y = FAIL then h:= h/2; next; end_if;
                Y:=(ImplicitRKStep(h/2,x0,y0,k1))[2];
                if Y = FAIL then h:= h/2; next; end_if;
                Y:=(ImplicitRKStep(h/2, x0 + h/2, Y))[2];
                if Y = FAIL then h:= h/2; next; end_if;
                break;
             end_while:
             Y_norm:= _plus( specfunc::abs(Y[ii])^2 $ ii=1..n )^(1/2);
             (Y[ii]:=(extrapol1*Y[ii]-y[ii])*extrapol2) $ ii=1..n ;
             localerror:= (_plus(specfunc::abs(Y[ii]-y[ii])^2 $ ii=1..n ))^(1/2);
          end_if;
          /* is Y=0 then switch to absolute error: */
          if iszero(Y_norm) then Y_norm:=1.0; end_if;
          /* estimate step size leading to a relative local error = tol */
          h:= specfunc::abs(h);
          // If the estimate is very bad, the adaptive mechanism may
          // produce an error 'step size too small'.
          if h <defaulttol*specfunc::abs(x0) then 
             h := defaulttol*specfunc::abs(x0)
          end_if;
          if h > hmax then return(hmax) end_if;
          if max(Y_norm*rtol,atol)*h^(effectiveOrder+1) >=
             localerror*specfunc::abs(xn-x0)^(effectiveOrder+1)
             then return(xn-x0);
             else return(vz*h*(max(Y_norm*rtol,atol)/localerror)^(1/(effectiveOrder+1)));
          end_if;
    end_proc: //--------------------------------

end_if;

if method=1 then

// --------   define local subroutine   ---------- 
RKPairStep:=proc(h, x0, y0)
local i,j,tmp,ii,Y,k,finalstep,localerror;
begin 
      if args(0)=4 then k[1]:=args(4);
                   else k[1]:=map(diffequ(x0,y0),float);
      end_if;
      if h > hmax then h:= hmax; end_if;
      if specfunc::abs(h) >= specfunc::abs(xn-x0)
         then finalstep:=TRUE; h:=xn-x0;
         else finalstep:=FALSE;
      end_if;
      for i from 2 to stages do
          Y:=y0; // initialize storage for intermediate stages 
          for j from 1 to i-1 do
             if not iszero(A[i,j]) then
                 tmp:=A[i,j]*h; (Y[ii]:=Y[ii]+tmp*k[j][ii]) $ ii=1..n;
             end_if;
          end_for;
          k[i]:= map(diffequ(x0+C[i]*h,Y),float);
      end_for;
      Y:=y0;  // initialize storage for result 
      for j from 1 to stages do
         if not iszero(B[j]) then
            tmp:=B[j]*h; (Y[ii]:=Y[ii]+tmp*k[j][ii]) $ ii=1..n;
         end_if;
      end_for;
      localerror:=[ 0 $ n ];
      for j from 1 to stages do
         if not iszero(Bb[j]) then
             (localerror[ii]:=localerror[ii]+Bb[j]*k[j][ii]) $ ii=1..n;
         end_if;
      end_for;
      localerror:= specfunc::abs(h)*
           _plus(specfunc::abs(localerror[ii])^2 $ ii=1..n )^(1/2);
      if finalstep
         then return(xn  ,Y,localerror)
         else return(x0+h,Y,localerror)
      end_if;
end_proc:
// --------   define local subroutine   ---------- 
adaptiveRKPairStep:=proc(h,x0,y0)
local k1,finalstep,dummy,Y,localerror,Y_norm,ii;
begin 
      if h > hmax then h:= hmax; end_if;
      if specfunc::abs(h)<=defaulttol*specfunc::abs(x0) then 
         stepsize_error(h, x0);
      end_if;

      if specfunc::abs(h) >= specfunc::abs(xn-x0)
         then finalstep:=TRUE; h:=xn-x0; 
         else finalstep:=FALSE;
      end_if;
      k1:=map(diffequ(x0,y0),float);
      while TRUE do
        ([dummy,Y,localerror]):=[RKPairStep(h,x0,y0,k1)];
        Y_norm:=(_plus(specfunc::abs(Y[ii])^2 $ ii=1..n ))^(1/2);
        if localerror<=max(Y_norm*rtol1, atol1)
           then stepsize_info:="adjusting step size to          h = ".expr2text(h);
                localerr_info:="estimated absolute local error: ".expr2text(localerror);
                if finalstep
                   then return(h,xn,Y);     // final step: make sure x0+h=xn  
                   else return(2*h,x0+h,Y); // accept, but suggest 2h as next 
                end_if;                     // step size                       
        end_if;
        if localerror> max(Y_norm*rtol, atol) then      // try again with smaller step size 
             finalstep:=FALSE; h:=h/2; next; 
        end_if;
        break;
      end_while;
      stepsize_info:="adjusting step size to          h = ".expr2text(h);
      localerr_info:="estimated absolute local error: ".expr2text(localerror);
      h,x0+h,Y;  // accept step, suggest h as step size for next step 
end_proc:

end_if;

if method=2 then
// --------   define local subroutine   ---------- 

RKStep:=proc(h, x0, y0)
local i,j,tmp,ii,Y,k,finalstep;
begin 
      if h > hmax then h:= hmax; end_if;
      if args(0)=4 then k[1]:=args(4);
                   else k[1]:=map(diffequ(x0,y0),float);
      end_if;
      if specfunc::abs(h) >= specfunc::abs(xn-x0)
         then finalstep:=TRUE; h:=xn-x0;
         else finalstep:=FALSE;
      end_if;
      for i from 2 to stages do
          Y:=y0; // initialize storage for intermediate stages 
          for j from 1 to i-1 do
             if not iszero(A[i,j]) then
                 tmp:=A[i,j]*h; (Y[ii]:=Y[ii]+tmp*k[j][ii]) $ ii=1..n;
             end_if;
          end_for;
          k[i]:= map(diffequ(x0+C[i]*h,Y),float);
      end_for;
      Y:=y0;  // initialize storage for result 
      for j from 1 to stages do
         if not iszero(B[j]) then
            tmp:=B[j]*h; (Y[ii]:=Y[ii]+tmp*k[j][ii]) $ ii=1..n;
         end_if;
      end_for;
      if finalstep then return(xn,Y) else return(x0+h,Y); end_if;
end_proc:
// --------   define local subroutine   ---------- 

adaptiveRKStep:=proc(h,x0,y0)
local k1,finalstep,y,data,Y,Y_norm,ii,dy_norm;
begin 
      if h > hmax then h:= hmax; end_if;
      if specfunc::abs(h)<=defaulttol*specfunc::abs(x0) then 
         stepsize_error(h, x0):
      end_if;
      if specfunc::abs(h) >= specfunc::abs(xn-x0)
         then finalstep:=TRUE; h:=xn-x0; 
         else finalstep:=FALSE;
      end_if;
      k1:=map(diffequ(x0,y0),float);
      y:=(RKStep(h,x0,y0,k1))[2];
      data:=RKStep(h/2,x0,y0,k1);
      while TRUE do
        Y:=(RKStep(h/2,data))[2];
        Y_norm:=(_plus(specfunc::abs(Y[ii])^2 $ ii=1..n ))^(1/2);
        (Y[ii]:=(extrapol1*Y[ii]-y[ii])*extrapol2) $ ii=1..n ;
        dy_norm:= (_plus(specfunc::abs(Y[ii]-y[ii])^2 $ ii=1..n ))^(1/2);
        if dy_norm<=max(Y_norm*rtol1, atol1)
           then stepsize_info:="adjusting step size to          h = ".expr2text(h);
                localerr_info:="estimated absolute local error: ".expr2text(dy_norm);
                if finalstep
                   then return(h,xn,Y);     // final step: make sure x0+h=xn  
                   else return(2*h,x0+h,Y); // accept, but suggest 2h as next 
                end_if;                     // step size                       
        end_if;
        if dy_norm>max(Y_norm*rtol, atol) then         // try again with smaller step size 
           h:=h/2; y:=data[2]; data:=RKStep(h/2,x0,y0,k1);
           finalstep:=FALSE; next; 
        end_if;
        break;
      end_while;
      stepsize_info:="adjusting step size to          h = ".expr2text(h);
      localerr_info:="estimated absolute local error: ".expr2text(dy_norm);
      h,x0+h,Y;  // accept step, suggest h as step size for next step 
end_proc:

end_if;

if method=3 then
// --------   define local subroutine   ---------- 

RK4Step:=proc(h, x0, y0)
local finalstep,i,Y,k1,k2,k3,k4;
begin 
      if h > hmax then h:= hmax; end_if;
      if specfunc::abs(h) >= specfunc::abs(xn-x0)
         then finalstep:=TRUE; h:=xn-x0;
         else finalstep:=FALSE;
      end_if;
      Y:=y0; // initialize storage of same type as y0 
      if args(0)=4 then k1:=args(4) else k1:=map(diffequ(x0,y0),float) end_if;
      (Y[i]:=y0[i]+h/2*k1[i])$ i=1..n;k2:=map(diffequ(x0+h/2,Y),float);
      (Y[i]:=y0[i]+h/2*k2[i])$ i=1..n;k3:=map(diffequ(x0+h/2,Y),float);
      (Y[i]:=y0[i]+h  *k3[i])$ i=1..n;k4:=map(diffequ(x0+h  ,Y),float);
      (Y[i]:=y0[i]+h/6*(k1[i]+2*(k2[i]+k3[i])+k4[i]))$ i=1..n;
      if finalstep then return(xn,Y) else return(x0+h,Y); end_if;
end_proc:
// --------   define local subroutine   ---------- 

adaptiveRK4Step:=proc(h,x0,y0)
local k1,finalstep,y,data,Y,Y_norm,ii,dy_norm;
begin 
      if h > hmax then h:= hmax; end_if;
      if specfunc::abs(h)<=defaulttol*specfunc::abs(x0) then 
         stepsize_error(h, x0):
      end_if;
      if specfunc::abs(h) >= specfunc::abs(xn-x0)
         then finalstep:=TRUE; h:=xn-x0; 
         else finalstep:=FALSE;
      end_if;
      k1:=map(diffequ(x0,y0),float);
      y:=(RK4Step(h,x0,y0,k1))[2];
      data:=RK4Step(h/2,x0,y0,k1);
      while TRUE do
        Y:=(RK4Step(h/2,data))[2];
        Y_norm:=(_plus(specfunc::abs(Y[ii])^2 $ ii=1..n ))^(1/2);
        (Y[ii]:=(extrapol1*Y[ii]-y[ii])*extrapol2) $ ii=1..n ;
        dy_norm:= (_plus(specfunc::abs(Y[ii]-y[ii])^2 $ ii=1..n ))^(1/2);
        if dy_norm<=max(Y_norm*rtol1, atol1)
           then stepsize_info:="adjusting step size to          h = ".expr2text(h);
                localerr_info:="estimated absolute local error: ".expr2text(dy_norm);
                if finalstep
                   then return(h,xn,Y);     // final step: make sure x0+h=xn  
                   else return(2*h,x0+h,Y); // accept, but suggest 2h as next 
                end_if;                     // step size                       
        end_if;
        if dy_norm>max(Y_norm*rtol, atol) then         // try again with smaller step size 
           h:=h/2; y:=data[2]; data:=RK4Step(h/2,x0,y0,k1);
           finalstep:=FALSE; next;
        end_if;
        break;
      end_while;
      stepsize_info:="adjusting step size to          h = ".expr2text(h);
      localerr_info:="estimated absolute local error: ".expr2text(dy_norm);
      h,x0+h,Y;  // accept step, suggest h as step size for next step 
end_proc:
end_if;

if method=4 then

// --------   define local subroutine   ---------- 
ImplicitRKStep:=proc(h, x0, y0)
local iter, maxiter, i, ii, j, jj, tmp, tmp1, tmp2, 
      y, dy, Y, k1, k, finalstep, J, f, dk, normk, normdk;
begin 
      if h > hmax then h:= hmax; end_if;
      maxiter:= 20; // max number of newton steps
      if specfunc::abs(h) >= specfunc::abs(xn-x0)
         then finalstep:=TRUE; h:=xn-x0; 
         else finalstep:=FALSE;
      end_if;

      if args(0)=4 then 
           k1:=args(4);
      else k1:=map(diffequ(x0,y0),float);
      end_if;

      if specfunc::abs(h) >= specfunc::abs(xn-x0)
         then finalstep:=TRUE; h:=xn-x0;
         else finalstep:=FALSE;
      end_if;

      //---------------------------------------
      // choose starting vector (k[1],..,k[s]) for the Newton search
      //---------------------------------------
/* ---------------------------
      for i from 1 to stages do
          Y:=y0; // initialize storage for starting vectors 
          tmp:= h*C[i];
          (Y[ii]:= Y[ii] + tmp*k1[ii]) $ ii=1..n;
          k[i]:= map(diffequ(x0+C[i]*h,Y),float);
      end_for:
--------------------------- */
      for i from 1 to stages do
          k[i]:= k1;
      end_for:

      //---------------------------------------
      // Y(i, k) is the i-th intermediate stage
      //---------------------------------------
      Y:= proc(i, k) local ii; begin
             [y0[ii] + _plus(h*A[i,j]*k[j][ii] $ j = 1..stages) $ ii=1..n];
          end_proc;

      //---------------------------------------
      // f(k) = [ k[i] - f(y0 + h*sum(a[i,j]*k[j] $ j=1..stages)) $ i = 1..stages]:
      //---------------------------------------
      f:= proc(k) 
          local r, i, ii, tmp, normk;
          begin
            r:= [0 $ n*stages]:
            normk:= 0:
            for i from 1 to stages do
              tmp:= map(diffequ(x0 + C[i]*h, Y(i, k)), float);
              for ii from 1 to n do
                r[(i-1)*n + ii]:= k[i][ii] - tmp[ii];
              end_for:
              normk:= max(normk, _plus(specfunc::abs(tmp[ii])^2 $ ii = 1..n)^(1/2));
            end_for:
            return([r, normk]):
       end_proc:

      //-------------------------------------------
      // Newton search for the solution k[i] of
      //    k[i] = f(y0 + h*sum(a[i,j]*k[j] $ j=1..stages))
      // with i = 1..stages
      //-------------------------------------------
      // Initialize the Jacobian
      //-------------------------------------------
      J:= array(1..n*stages, 1..n*stages):
      dy:= max(specfunc::abs(h)/10.0^5, 10.0^(4 - DIGITS)):

/* The Jacobian at the point y0
      tmp1:= k1;
      for jj from 1 to n do
        tmp2:= map(diffequ(x0, [y0[ii] $ ii = 1..jj-1, y0[jj] + dy, y0[ii] $ ii = jj+1..n]), float);
        for ii from 1 to n do // do not use zip, since tmp1/2 may be arrays
          tmp2[ii]:= (tmp2[ii] - tmp1[ii])/dy;
        end_for:
        for i from 1 to stages do
         for ii from 1 to n do
          for j from 1 to stages do
            J[(i-1)*n + ii, (j-1)*n + jj]:= -h*A[i,j]*tmp2[ii];
          end_for:
        end_for:
       end_for:
      end_for:
*/
      for iter from  1 to maxiter do
         //--------------------------------------------------
         // compute the Jacobian via numerical differentiaton
         //--------------------------------------------------
          for i from 1 to stages do
           y:= Y(i, k):
           tmp1:= map(diffequ(x0, y), float);
           for jj from 1 to n do
             tmp2:= map(diffequ(x0, [y[ii] $ ii = 1..jj-1, y[jj] + dy, y[ii] $ ii = jj+1..n]), float);
             for ii from 1 to n do // do not use zip, since tmp1/2 may be arrays
               tmp2[ii]:= (tmp2[ii] - tmp1[ii])/dy;
             end_for:
             for ii from 1 to n do
               for j from 1 to stages do
                 J[(i-1)*n + ii, (j-1)*n + jj]:= -h*A[i,j]*tmp2[ii];
               end_for:
             end_for:
           end_for:
          end_for:

          for i from 1 to n*stages do 
              J[i, i]:= J[i, i] + float(1); 
          end_for:
          //--------------------------------------------------
          // the Jacobian is computed
          //--------------------------------------------------
          //--------------------------------------------------
          // do a Newton step
          //--------------------------------------------------
          [tmp, normk]:= f(k):
          dk:= numeric::matlinsolve(J, tmp, ReturnType = DOM_ARRAY, SoftwareFloats):
          if has(dk, FAIL) then
             if adaptive then 
                return(FAIL, FAIL);
             else
                error("cannot solve the equations for the step at ".
                      "time ".expr2text(x0). " with step size ".
                       expr2text(h). ". Try a smaller step size.");
             end_if;
          end_if;
          dk:= dk[1]: // forget the kernel
          for i from 1 to stages do
            for ii from 1 to n do
               k[i][ii] := k[i][ii] - dk[(i-1)*n + ii, 1];
            end_for:
          end_for:
          //--------------------------------------------------
          // Newton step done
          //--------------------------------------------------
          // precisiongoal reached?
          //--------------------------------------------------
          normdk:= _plus(dk[i, 1]^2 $ i = 1..n*stages)^(0.5);
          if normdk <= eps*normk then
             break;
          end_if;
      end_for:
      //------------------------------------------------
      // if no convergence of the Newton iteration:
      //------------------------------------------------
      if iter > maxiter then
         if adaptive then       // adaptiveImplicitRKStep will
            return(FAIL, FAIL); // try again with smaller step size
         else
            error("cannot solve the equations for the step at ".
                  "time ".expr2text(x0). " with step size ".
                  expr2text(h). ". Try a smaller step size.");
         end_if:
      end_if;

      //--------------------------------------------------
      // the intermediate stages k[1],..,k[s] are computed
      //--------------------------------------------------
      Y:=y0;  // initialize storage for result 
      for j from 1 to stages do
         if not iszero(B[j]) then
            tmp:=B[j]*h; (Y[ii]:=Y[ii]+tmp*k[j][ii]) $ ii=1..n;
         end_if;
      end_for;
      if finalstep then return(xn,Y) else return(x0+h,Y); end_if;
end_proc:

adaptiveImplicitRKStep:=proc(h,x0,y0)
local k1,finalstep,y,data,Y,Y_norm,ii,dy_norm;
begin 
      if h > hmax then h:= hmax; end_if;
      k1:=map(diffequ(x0,y0),float);
      while TRUE do // adapt step sizes in this while loop
        if specfunc::abs(h)<=defaulttol*specfunc::abs(x0) then 
           stepsize_error(h, x0);
        end_if;

        if specfunc::abs(h) >= specfunc::abs(xn-x0)
           then finalstep:=TRUE; h:=xn-x0; 
           else finalstep:=FALSE;
        end_if;

        y:=ImplicitRKStep(h,x0,y0,k1)[2];
        if has(y, FAIL) then h:= h/2; next; end_if;
        data:=ImplicitRKStep(h/2,x0,y0,k1);
        if has([data], FAIL) then h:= h/2; next; end_if;
        Y:=(ImplicitRKStep(h/2,data))[2];
        if has(Y, FAIL) then h:= h/2; next; end_if;
        Y_norm:=(_plus(specfunc::abs(Y[ii])^2 $ ii=1..n ))^(1/2);
        (Y[ii]:=(extrapol1*Y[ii]-y[ii])*extrapol2) $ ii=1..n ;
        dy_norm:= (_plus(specfunc::abs(Y[ii]-y[ii])^2 $ ii=1..n ))^(1/2);
        if dy_norm<=max(Y_norm*rtol1, atol1)
           then stepsize_info:="adjusting step size to          h = ".expr2text(h);
                localerr_info:="estimated absolute local error: ".expr2text(dy_norm);
                if finalstep
                   then return(h,xn,Y);     // final step: make sure x0+h=xn  
                   else return(2*h,x0+h,Y); // accept, but suggest 2h as next 
                end_if;                     // step size                       
        end_if;
        if dy_norm>max(Y_norm*rtol, atol) then         // try again with smaller step size 
           h:=h/2; 
           next; 
        end_if;
        break;
      end_while;
      stepsize_info:="adjusting step size to          h = ".expr2text(h);
      localerr_info:="estimated absolute local error: ".expr2text(dy_norm);
      h,x0+h,Y;  // accept step, suggest h as step size for next step 
end_proc:

end_if;

// stopping criterion for the integration with running time x
// is:  .. until compare(x, xn)
compare:= if x0 <= xn then
             (a, b) -> bool(a >= b);
          else
             (a, b) -> bool(a <= b);
          end_if:

// --------   main program  ---------- 

   stepcount:= 0;
   if adaptive then
      data:=startingstepsize(x0,y0),x0,y0;
      userinfo(2, "start:  time = ".expr2text(data[2]));
      userinfo(3, "           Y = ".expr2text(data[3]));
      case method 
      of 1 do repeat
                userinfo(3, "trying next step with step size h = ".expr2text(data[1]));
                data:=adaptiveRKPairStep(data);
                stepcount:= stepcount+1;
                userinfo(3, stepsize_info);
                userinfo(3, localerr_info);
                userinfo(2, "accept: time = ".expr2text(data[2]));
                userinfo(3, "           Y = ".expr2text(data[3]));
                if alldata then append_to_datatable(data[2],data[3]); end_if;
              until compare(data[2], xn) end_repeat;
              break
      of 2 do repeat
                userinfo(3, "trying next step with step size h = ".expr2text(data[1]));
                data:=adaptiveRKStep(data);
                stepcount:= stepcount+1;
                userinfo(3, stepsize_info);
                userinfo(3, localerr_info);
                userinfo(2, "accept: time = ".expr2text(data[2]));
                userinfo(3, "           Y = ".expr2text(data[3]));
                if alldata then append_to_datatable(data[2],data[3]); end_if;
              until compare(data[2], xn) end_repeat;
              break
      of 3 do repeat
                userinfo(3, "trying next step with step size h = ".expr2text(data[1]));
                data:=adaptiveRK4Step(data);
                stepcount:= stepcount+1;
                userinfo(3, stepsize_info);
                userinfo(3, localerr_info);
                userinfo(2, "accept: time = ".expr2text(data[2]));
                userinfo(3, "           Y = ".expr2text(data[3]));
                if alldata then append_to_datatable(data[2],data[3]); end_if;
              until compare(data[2], xn) end_repeat;
              break
      of 4 do repeat
                userinfo(3, "trying next step with step size h = ".expr2text(data[1]));
                data:=adaptiveImplicitRKStep(data);
                stepcount:= stepcount+1;
                userinfo(3, stepsize_info);
                userinfo(3, localerr_info);
                userinfo(2, "accept: time = ".expr2text(data[2]));
                userinfo(3, "           Y = ".expr2text(data[3]));
                if alldata then append_to_datatable(data[2],data[3]); end_if;
              until compare(data[2], xn) end_repeat;
              break
      end_case;
      userinfo(1, "solution after ".expr2text(stepcount)." steps");
      if alldata then result:= [datatable[i] $ i=1..nops(datatable)]
                 else result:= data[3]; 
      end_if;
   end_if;

   if not adaptive then
      // Make sure that tiny intervals are handled by just
      // returning the initial data. Above, this situation
      // lead to adaptive:= FALSE, so we have to handle this
      // here:
      if specfunc::abs(h)<=defaulttol*max(specfunc::abs(x0),
                                          specfunc::abs(xn)) then 
           method:= 0: // flag to indicate that nothing has to be done
      end_if;
      data:=x0,y0;
      userinfo(2, "time = ".expr2text(data[1]));
      userinfo(3, "   Y = ".expr2text(data[2]));
      case method
      of 0 do // Step size is too small to increment x.
              // Just return the initial data.
              break 
      of 1 do repeat 
                data:=RKPairStep(h,data[1],data[2]);
                userinfo(2, "time = ".expr2text(data[1]));
                userinfo(3, "   Y = ".expr2text(data[2]));
                if alldata then append_to_datatable(data[1],data[2]); end_if;
              until compare(data[1], xn) end_repeat;
              break 
      of 2 do repeat 
                data:=RKStep(h,data);
                userinfo(2, "time = ".expr2text(data[1]));
                userinfo(3, "   Y = ".expr2text(data[2]));
                if alldata then append_to_datatable(data[1],data[2]); end_if;
              until compare(data[1], xn) end_repeat;
              break 
      of 3 do repeat
                data:=RK4Step(h,data);
                userinfo(2, "time = ".expr2text(data[1]));
                userinfo(3, "   Y = ".expr2text(data[2]));
                if alldata then append_to_datatable(data[1],data[2]); end_if;
              until compare(data[1], xn) end_repeat;
              break 
      of 4 do repeat 
                data:=ImplicitRKStep(h,data);
                userinfo(2, "time = ".expr2text(data[1]));
                userinfo(3, "   Y = ".expr2text(data[2]));
                if alldata then append_to_datatable(data[1],data[2]); end_if;
              until compare(data[1], xn) end_repeat;
              break 
      end_case;
      if alldata then result:= [datatable[i] $ i=1..nops(datatable)]
                 else result:= data[2]; 
      end_if;
  end_if;
/*
  if dowarn then
    warning("some steps could not be computed with the precision goal of ".
            expr2text(round(-log(10,rtol)) )." digits. The result may be less precise.");
  end_if;
*/
  result;
end_proc:

