/*
numeric::partfrac(f) -- partial fraction expansion
    of a univariate expression using a numerical
    factorization of the denominator

Call:  numeric::partfrac(f, <x>)

Parameters: 
      f -- a univariate arithemetical expression.
           If f is mulitvariate, only a symbolic
           partfrac is attempted.
      x -- an identifier or indexed identifier.
           If not specified, it is searched for
           internally

Details:

   If f is a univariate rational expression f(x) = n(x)/d(x),
   then the denominator polynomial is factorized as
   d(x) = c * (x - r1)*(x - r2) * ..., where r1, r2 etc.
   are numerical roots of d(x). This factorization is
   used to compute a partial fraction expansion (using 
   the symbolic partfrac after applying numeric::rationalize).
   Finally, float is applied.

   For more general f, symbolic factors that are not rational 
   expressions in x are split off, before the numerical partfrac 
   is done on the rational part of a factorization. Thus, also 
   expressions such as exp(x)/(x^3 - 2) can be expanded.

   Multivariate expressions are processed by the symbolic
   partfrac without additional numerical processing.
-------------------------------------------------- */

numeric::partfrac:= proc(f)
local f_orig, x, num, den, den_others, i,
      numfactors_symbolic,
      numfactors_numeric,
      numpolys, c, roots, r, simp;
begin
  f_orig:= f;
  if not testtype(f, Type::Arithmetical) then
     return(f_orig);
  end_if;
  if stdlib::hasfloat(f) then
     f:= numeric::rationalize(f);
  end_if:
  if args(0) = 2 then
     x:= args(2);
     if numeric::indets(f) <> {x} then
        return(float(partfrac(f, x)));
     end_if;
  else
     x:= op(numeric::indets(f));
  end_if:
  if nops([x]) <> 1 then
     return(float(partfrac(f)));
  end_if;

   //---------------------
  // The work starts here:
   //---------------------

  [num, den]:= [numer(f), denom(f)]; // normal(f, List);

  if type(den) = "_mult" then
     [den, den_others]:= split(den, testtype, Type::PolyExpr(x))[1..2];
  else
      den_others:= 1:
  end_if;

  num:= Factored::convert_to(factor(num), DOM_LIST);

  // split non-polynomial factors from the numerator:
  numfactors_symbolic:= 1:
  numfactors_numeric:= num[1]:
  numpolys:= 1:
  for i from 2 to nops(num) step 2 do
      if not has(num[i], x) then
         numfactors_numeric:= numfactors_numeric*num[i]^num[i+1];
      elif testtype(num[i], Type::PolyExpr(x)) then
         numpolys:= numpolys*num[i]^num[i+1];
      else
         numfactors_symbolic:= numfactors_symbolic*num[i]^num[i+1];
      end_if;
  end_for:
  //---------------------------------------------------------------
  // now: num = numfactors_symbolic * numfactors_numeric * numpolys
  // Further down we will stick only the polynomial 
  // part numpolys into the symbolic partfrac, so 
  // that a factor such as exp(x) is returned as a 
  // factor of the partial fraction expansion
  //---------------------------------------------------------------

  //--------------------------------------------
  // numerical factorization of the denominator:
  //--------------------------------------------
  c:= coeff(den, x, degree(den, [x])); 
  roots:= numeric::polyroots(den):
  roots:= map(roots, numeric::rationalize);
  den:= c*_mult((x - r) $ r in roots);
  f:= partfrac(numpolys/den, x):

  //------------------------------------------
  // partfrac produces awkward stuff such as
  // 1.234*10^20/(1.3*10^19*x + 2.5*10^20).
  // Simplify numerator and denominator by
  // making the denominators monic.
  // Further, multiply the numerator polynomials
  // with the numeric factors (producing an
  // *expanded* polynomial), then do a separate
  // multiplication with the symbolic factors
  // (otherwise you might get stuff such as
  //  5.3333-44 exp(x) (6.25e42 x - 1.25e43)
  // in the numeractors!)
  //------------------------------------------
  simp:= proc(h)
         local num, den, c;
         begin 
            [num, den]:= [numer(h), denom(h)];
          //[num, den]:= normal(h, List);
            c:= coeff(den, x, degree(den, [x])); 
            float(numfactors_symbolic/den_others) *
            float(numfactors_numeric*num/c) / float(den/c)
         end_proc:
  if type(f) = "_plus" then
     f:= map(f, simp);
  elif type(f) = "_mult" then
     f:= simp(f);
  else
     f:= float(numfactors_symbolic/den_others)*float(numfactors_numeric*f);
  end_if:

  return(f);
end_proc:
