//----------------------------------------------------------------------------------
// numeric::polyrootbound(p) -- upper bounds for the roots of a univariate polynomial
//
// Call:       numeric::polyrootbound(p)
//
// Parameters: p - a univariate polynomial of type DOM_POLY or
//             a univariate polynomial expression with numerical
//             coefficients
//
// Return Value: a positive real floating point number b.
//               All roots z of p satisfy |z| <= b
//
// Details:
//       *)  Consider the roots of the polynomial
//              p(x) = x^n + a[n-1]*x^(n-1) + ... + a[1]*x + a[0].
//           Consider
//              q(x) = x^n - abs(a[n-1])*x^(n-1) - ... - abs(a[1])*x - abs(a[0]).
//           For a[0] <> 0, the polynomial q(x) has exactly one positive real root
//           b, say.
//
//           Theorem: all (complex) roots z of p satisfy |z| <= b
//
//       *)  numeric::polyrootbound(p) returns the bound b described above,
//           approximated to about 3 leading correct digits.
//       *)  For a constant polynomial, infinity is returned.
//----------------------------------------------------------------------------------

numeric::polyrootbound:= proc(p)
local X, n, lc, i, 
      rootbound1, rootbound2, rootbound,
      pp, t, A, B, C, rtbd;
save DIGITS;
begin
   if args(0) <> 1 then 
      error("expecting exactly one argument (a univariate polynomial)");
   end_if;
   if domtype(p) <> DOM_POLY then
      // p is a polynomial expression
      X:= indets(p) minus Type::ConstantIdents;
      if nops(X) > 1 then
         error("only univariate polynomials are considered")
      end_if;
      if nops(X) = 0 then
         // p is a constant:
         return(infinity);
      end_if;
      X:= op(X);
      p:= poly(p, [X]);
      if p = FAIL then
         error("the argument does not seem to be a polynomial");
      end_if;
   else // domtype(p) = DOM_POLY 
      X := op(p, 2);
      if nops(X) > 1 then
         error("only univariate polynomials are considered")
      end_if;
      if degree(p) = 0 then
         // p is a constant <> 0
         return(infinity);
      end_if;
      X:= op(X);
   end_if;

   // reduce p(X) = c[n]*X^n + c[n-1]*X^(n-1) + .. + c[0]
   // to p(X) = X^n + a[n-1]*X^(n-1) + ... + a[0]
   n:= degree(p):
   p:= mapcoeffs(p, float):
   lc:= lcoeff(p);  // the leading coefficient
   p:= lmonomial(p, Rem)[2];
   // the polynomial was  p_original(X) = lc* X^n + p(X),

   if iszero(p) then
      // the original polynomial was a monomial lc*x^n.
      // The only root is 0 --> the rootbound is 0.
      return(return(float(0)));
   end_if;

   // Factor out zero solutions:
   i:=0; 
   while iszero(coeff(p,i)) do 
         i:= i+1; 
   end_while;

   if i > 0 then 
        p:= p/poly([[lc, i]],[X]); // p := p/(lc*X^i);
        n:= n - i;
   else p:= multcoeffs(p, 1/lc);
   end_if;

   p:= mapcoeffs(p, specfunc::abs);
   // Now, p has positive real coefficients
   // and numerical evaluation at positive
   // real points is numerically stable.
   // Since the relative precision goal is 1/1000,
   // a small numerical precision suffices:

   DIGITS:= 8:
   if map({coeff(p)}, domtype) <> {DOM_FLOAT} then
      error("polynomials with symbolic coefficients are not handled");
   end_if;

   // introduce P(X) = X^n - p(X), where p(X) has positive
   // coefficients. This polynomial X^n - p(X) has exactly
   // one positive real root b that we have to search for:

   // pp:= poly2list(p);
   // rootbound:= max((1 + pp[i][1]) $ i=1..nops(p));
   // A standard root bound is given by norm(p) =
   // = max(|coefficients|).

   rootbound1:= 1 + norm(p);
   pp:= poly2list(p);
   rootbound2:= 2*max( op(t, 1)^(1/(n - op(t, 2))) $ t in pp);
   rootbound:= min(rootbound1, rootbound2);

   //-------------------------------------------
   // Search for the root b with b^n - p(b) = 0.
   // Decrease rootbound by a factor of 1/8 until
   // rootbound < b
   //-------------------------------------------
   repeat
         rootbound:= rootbound/8;
   until rootbound^n <= evalp(p, X = rootbound) 
   end_repeat;
   //-------------------------------------------
   // Now,  rootbound <= b < 8*rootbound.
   // Increase by a factor of 2 until rootbound > b
   // This while loop takes at most 3 steps:
   //-------------------------------------------
   repeat
         rootbound:= 2*rootbound;
   until rootbound^n >= evalp(p, X = rootbound)
   end_repeat;
   //-------------------------------------------
   // now,  rootbound/2 < b <= rootbound
   //-------------------------------------------

   //--------------------------------------------
   // Standard bisectioning for bracketing
   // b by A*rootbound < b <= B*rootbound
   // With a relative precision goal of 
   // B - A <= 1/1000 at most 10 bisectioning
   // step will suffice:
   //--------------------------------------------
   [A, B]:= [float(1/2), float(1)]:
   while B - A > 1/1000 do
      C:= (A + B)/2:
      rtbd:= C*rootbound;
      if (rtbd)^n > evalp(p, X = rtbd) then
           // A*rootbound <= b <= C*rootbound
           B:= C;
      else // C*rootbound <= b <= B*rootbound
           A:= C;
      end_if;
   end_while;
   //--------------------------------------------
   // Now,  A*rootbound <= b <= B*rootbound
   //--------------------------------------------
   // Increase B to make sure that the rootbound
   // is not too small due to roundoff:
   (B + 10.0^(-DIGITS))*rootbound;
end_proc:
