// 
//    
// 
// W. Oevel,  20/10/99 

/*
numeric::polysysroots -- numerical roots of system of polynomial equations

Call: numeric::polysysroots(eqs, <vars>, <NoWarning>);

Parameters: eqs  -- univariate equation (may have symbolic parameter)
                    or set or list of multivariate equations.
                    Also expressions or DOM_POLY polynomials are
                    accepted and interpreted as homogeneous equations.

Options:    vars -- indeterminate or a set or list of indeterminates
            NoWarning -- By default, the solutions are checked for
                         numerical stability. If solutions seem to 
                         be marred by numeric instability, a warning
                         is issued. With the option 'NoWarning', this
                         check is suppressed and no warnings are issued.

Return value: a set of lists {[x1=..,x2=..,..], 
                              [x1=..,x2=..,..],
                              ..}
              of numerical roots
              If symbolic parameters are involved in the polynomial
              system of equations, some curios stuff computed by
              the symbolic solver is returned.

Details: 
        This routine calls the symbolic solver solve, which computes 
        a Groebner basis in triangular form. 
        The last univariate RootOf object of the Groebner basis
        is solved numerically, the results are substituted into the
        other elements of the Groebner basis.

Examples:

>> numeric::polysysroots(x^2=PI),
   numeric::polysysroots(x^2=PI, x),
   numeric::polysysroots(x^2-PI),
   numeric::polysysroots(x^2-PI, x),
   numeric::polysysroots(poly(x^2-PI, [x])),
   numeric::polysysroots(poly(x^2-PI, [x]), x);

{[x = -1.77245385], [x = 1.77245385]},
   {[x = -1.77245385], [x = 1.77245385]},
   {[x = -1.77245385], [x = 1.77245385]},
   {[x = -1.77245385], [x = 1.77245385]},
   {[x = -1.77245385], [x = 1.77245385]},
   {[x = -1.77245385], [x = 1.77245385]}

>> numeric::polysysroots(x^2=exp(y), x),
   numeric::polysysroots(x^2-exp(y), x),
   numeric::polysysroots(poly(x^2-exp(y), [x]), x),
   numeric::polysysroots(x^2=exp(y), [x]),
   numeric::polysysroots(x^2-exp(y), [x]),
   numeric::polysysroots(poly(x^2-exp(y), [x]), [x]);

            1/2                1/2
{[x = exp(y)   ], [x = - exp(y)   ]},
               1/2                1/2
   {[x = exp(y)   ], [x = - exp(y)   ]},
               1/2                1/2
   {[x = exp(y)   ], [x = - exp(y)   ]},
               1/2                1/2
   {[x = exp(y)   ], [x = - exp(y)   ]},
               1/2                1/2
   {[x = exp(y)   ], [x = - exp(y)   ]},
               1/2                1/2
   {[x = exp(y)   ], [x = - exp(y)   ]}

>> numeric::polysysroots(x^2=exp(c), [x,y]),
   numeric::polysysroots(x^2-exp(c), [x,y]),
   numeric::polysysroots(poly(x^2-exp(c), [x]), [x,y]),
   numeric::polysysroots(poly(x^2-exp(c), [x]), [x,y]);

            1/2                1/2
{[x = exp(c)   ], [x = - exp(c)   ]},
               1/2                1/2
   {[x = exp(c)   ], [x = - exp(c)   ]},
               1/2                1/2
   {[x = exp(c)   ], [x = - exp(c)   ]},
               1/2                1/2
   {[x = exp(c)   ], [x = - exp(c)   ]}


>> DIGITS:=3:
   numeric::polysysroots({x^2+y+1, y^2+x+1=2}),
   numeric::polysysroots({x^2+y+1, y^2+x+1=2}, {x,y}),
   numeric::polysysroots({x^2+y+1, y^2+x+1=2}, [y,x]),
   numeric::polysysroots({x^2+y+1, y^2+x+1=2}, [x,y]),
   numeric::polysysroots({x^2+y+1, poly(y^2+x-1)}, [x,y]);

{[y = -1.2, x = -0.453],
   [y = -1.0, x = 0.0],
   [y = 1.1 - 0.665 I, x = 0.226 + 1.46 I],
   [y = 1.1 + 0.665 I, x = 0.226 - 1.46 I]},
{[y = -1.2, x = -0.453],
   [y = -1.0, x = 0.0], [y = 1.1 - 0.665 I, x = 0.226 + 1.46 I],
   [y = 1.1 + 0.665 I, x = 0.226 - 1.46 I]},
{[y = -1.2, x = -0.453],
   [y = -1.0, x = 0.0], [y = 1.1 - 0.665 I, x = 0.226 + 1.46 I],
   [y = 1.1 + 0.665 I, x = 0.226 - 1.46 I]},
{[x = 0.0, y = -1.0],
   [x = - 0.453 + 0.0 I, y = - 1.2 + 1.16e-10 I],
   [x = 0.226 + 1.46 I, y = 1.1 - 0.665 I],
   [x = 0.226 - 1.46 I, y = 1.1 + 0.665 I]},
{[x = 0.0, y = -1.0],
   [x = - 0.453 + 0.0 I, y = - 1.2 + 1.16e-10 I],
   [x = 0.226 + 1.46 I, y = 1.1 - 0.665 I],
   [x = 0.226 - 1.46 I, y = 1.1 + 0.665 I]}

>> DIGITS:=2:
   numeric::polysysroots({x^2+y+exp(z), y+x+1=exp(z)}, {x,y}),
   numeric::polysysroots({x^2+y+exp(z), y+x+1=exp(z)}, [y,x]),
   numeric::polysysroots({x^2+y+exp(z), y+x+1=exp(z)}, [x,y]),
   numeric::polysysroots({x^2+y+exp(z), poly(y+x+1-exp(z))}, [x,y]);

{[y = exp(z) - 0.5 (5.0 - 8.0 exp(z))    - 1.5,
                             1/2
   x = 0.5 (5.0 - 8.0 exp(z))    + 0.5],
                                      1/2
 [ y = exp(z) + 0.5 (5.0 - 8.0 exp(z))    - 1.5,
                                   1/2
   x = 0.5 - 0.5 (5.0 - 8.0 exp(z))   ]}, ...
}

>> DIGITS:=2: z:=1.0:
   numeric::polysysroots({x^2+y+exp(z), y+x+1=exp(z)}, {x,y}),
   numeric::polysysroots({x^2+y+exp(z), y+x+1=exp(z)}, [y,x]),
   numeric::polysysroots({x^2+y+exp(z), y+x+1=exp(z)}, [x,y]),
   numeric::polysysroots({x^2+y+exp(z), poly(y+x+1-exp(z))}, [x,y]);

  {[y = 1.2 - 2.0 I, x = 0.5 + 2.0 I],
   [y = 1.2 + 2.0 I, x = 0.5 - 2.0 I]},
  {[y = 1.2 - 2.0 I, x = 0.5 + 2.0 I],
   [y = 1.2 + 2.0 I, x = 0.5 - 2.0 I]},
  {[x = 0.5 - 2.0 I, y = 1.2 + 2.0 I],
   [x = 0.5 + 2.0 I, y = 1.2 - 2.0 I]},
  {[x = 0.5 - 2.0 I, y = 1.2 + 2.0 I],
   [x = 0.5 + 2.0 I, y = 1.2 - 2.0 I]}

*/

numeric::polysysroots:= proc(eqs)
   local s, doWarn, vars, uvars, orig_vars, abseqs, 
         absvalue, i, checkstability, s0, sol, // var,
         solution, param, paramSet, p;
 begin
   userinfo(10, "numeric::polysysroots called with args ".expr2text(args()));
   if args(0)<1 then error("expecting at least one argument") end_if;
   if args(0)>3 then error("expecting at most three arguments") end_if;
   s:= eqs; // copy of original form of equations
   // convert equation(s) to list of equation(s)

   if eqs::dom::hasProp(Cat::Matrix)=TRUE then
     eqs:= expr(eqs):
   end_if:
   case domtype(eqs)
   of DOM_SET do
   of DOM_ARRAY do
       eqs:= [op(eqs)];
   end_case;
   if domtype(eqs)<>DOM_LIST then eqs:= [eqs]; end_if;
   // convert equations to expressions:
   eqs:= subs(eqs, float(0)=0);
   eqs:= map(eqs, proc(eq) begin if type(eq)= "_equal" then 
                  op(eq,1)-op(eq,2) else eq end_if; end_proc);
   // eliminate trivial equations
   eqs:= select(eqs, _not@iszero);

   doWarn:= TRUE; // default
   uvars:= null():
   orig_vars:= [op(numeric::indets(eqs))];
   if args(0) = 1 then // no vars specified, no 'NoWarning' given
     // vars:= [op(numeric::indets(eqs))];
        vars:= orig_vars;
   end_if;
   if args(0) = 2 then
      if args(2) <> NoWarning then 
        // vars specified, no 'NoWarning' specified
        // convert unknowns to list of unknowns:
        vars:= args(2):
        uvars:= vars;
      else 
        // no vars specified, 'NoWarning' given
        doWarn:= FALSE;
        vars:= [op(numeric::indets(eqs))];
      end_if;
   end_if;
   if args(0) = 3 then // vars specified and 'NoWarning' given
        vars:= args(2):
        uvars:= vars;
        if args(3) <> NoWarning then
           error("3rd argument: expecting the option 'NoWarning'. ".
                 "Received: ".expr2text(args(3)));
        end_if;
        doWarn:= FALSE;
   end_if;
   if domtype(vars)=DOM_SET   then vars:= [op(vars)]; end_if;
   if domtype(vars)<>DOM_LIST then vars:= [vars]; end_if;

   if nops(eqs)=0 then 
      // return({[(var = var) $ var in vars]}):
      return({[]});
      ////handle trivial equations in the same way as solve:
      //return(solve(s, uvars)); 
   end_if;

   if vars=[] then error("no indeterminate(s)"); end_if;

   //check that system is polynomial w.r.t. vars
   if has(map(eqs,testtype,Type::PolyExpr(vars)), FALSE) then
     if args(0)=2
       then error("equation(s) are not polynomial in the specified indeterminates")
       else error("equation(s) are not polynomial")
      end_if;
   end_if;

   //---------------------------------------------------
   // Given the polynomial eqs[i] = a0 + a1*x + a2*x^2 + ...,
   // compute the polynomials abseqs[i]:= |a0| + |a1|*x + |a2|*x^2 + ...
   // This only needs to be done when the automatic check for 
   // stability is switched on (triggered by doWarn = TRUE)
   //---------------------------------------------------
   if doWarn then
      abseqs:= eqs;
      for i from 1 to nops(eqs) do
         if domtype(eqs[i]) <> DOM_POLY then
              abseqs[i]:= expr(mapcoeffs(poly(eqs[i], vars), abs));
         else abseqs[i]:= expr(mapcoeffs(eqs[i], abs));
              eqs[i]:= expr(eqs[i]):
         end_if;
      end_for:
   end_if;

   //---------------------------------------------------
   // Utility to check for numerical instabilities.
   // It issues a warning if a solution looks fishy
   // (one warning per solution). If activated, it
   // requires the list abseqs computed above.
   // A solution x is acceptable if  
   // |a0+a1*x+a2*x^2+...| 
   //     <= 10^(-DIGITS)*(|a0|+|a1|*|x|+|a2|*|x|^2+..)
   //---------------------------------------------------
   checkstability:= proc(result)
   name numeric::polysysroots::checkstability;
   local s, ss, dobreak, i, residue;
   begin
      if not doWarn then 
         return();
      end_if;
      if not domtype(result) = DOM_SET then
         return();
      end_if;
      for s in result do 
        // s is a list [var1 = value1, var2 = value2, ..]
        //    or       [var1 in RootOf(..), ..]
        ss:= map(s, op, 2); // ss = [value1, value2, ..]
        dobreak:= FALSE;
        for i from 1 to nops(ss) do
          if not testtype(ss[i], Type::Numeric) then
             dobreak:= TRUE;
             break;
          end_if;
        end_for:
        if dobreak then 
           break; 
        end_if;
        for i from 1 to nops(eqs) do
          residue:= subs(eqs[i], s, EvalChanges);
          if not testtype(residue, Type::Numeric) then
             next;
          end_if;
          absvalue:= subs(abseqs[i], map(s, eq -> lhs(eq) = abs(rhs(eq))), EvalChanges); 
          if not testtype(absvalue, Type::Numeric) then
             next;
          end_if;
          if specfunc::abs(residue) > 10^(3 - DIGITS)*absvalue then
             warning("the solution ".expr2text(s)." seems to be marred ".
                     "by some numerical instability. Insertion into the ".
                     "equation ".expr2text(eqs[i] = 0). " produces the ".
                     "residue ".expr2text(residue));
             break; // one warning per solution suffices
          end_if;
        end_for:
      end_for:
   end_proc:

   // -------------------------------------
   // Do the work:
   // Only need to call solvelib::allvalues:
   // -------------------------------------
// s:= solvelib::allvalues(solve(eqs, vars, IgnoreSpecialCases, BackSubstitution = FALSE));
// s:= solvelib::allvalues(solve(eqs, vars, IgnoreSpecialCases, VectorFormat)): 
// Do not use VectorFormat: 
// solve(x^2 = y, [x, y]) --> {[x = - z^(1/2), y = z], [x = z^(1/2), y = z]}
// solve(x^2 = y, [x, y], VectorFormat) 
//  -->   { +-      -+ |         }       { +-        -+ |         }
//        { |   1/2  | |         }       { |     1/2  | |         }
//        { |  z     | |         }       { |  - z     | |         }
//        { |        | | z in C_ } union { |          | | z in C_ }
//        { |    z   | |         }       { |     z    | |         }
//        { +-      -+ |         }       { +-        -+ |         }
   s:= solvelib::allvalues(solve(eqs, vars, IgnoreSpecialCases));

   // float, resolve RootOfs that can be handled numerically
   s:= eval(float(s));
   s0:= s;

   //---------------------------------------------------
   // postprocessing: convert [x, y] in SomeSet to the
   // old form {[x = value, y = value], ...}
   //---------------------------------------------------
   if type(s) = "_in" then
      if [op(op(s, 1))] <> vars then
         error("unexpected variables"):
      end_if;
      s:= op(s, 2);
   end_if;

   if type(s) = solvelib::cartesianPower and op(s, 2) = 1 then
      s:= op(s, 1);
   end_if;

   if type(s) = RootOf and nops(vars) < 2 then
      // univariate RootOfs are returned directly
      return(s);
   end_if;

   if type(s) = solvelib::VectorImageSet then
// if op(s, 0) = hold(solvelib::Union) then
      solution:= op(s, 1);
      if solution::dom::hasProp(Cat::Matrix) = TRUE then
         if nops(solution) <> nops(vars) then
            error("unexpected number of matrix entries");
         end_if;
         solution:= [op(solution, i) $ i = 1..nops(vars)];
      end_if;
      if domtype(solution) <> DOM_LIST then
         error("unexpected solution type");
      end_if;
       
      param:= op(s, 2):
      paramSet:= op(s, 3):

      // handle trivial equations
      for i from 1 to nops(param) do
          if (paramSet[i] = C_ or paramSet[i] = R_) then
             param[i]:= NIL; // ignore this parameter
          end_if;
      end_for:

      // handle RootOfs that were resolved by float
      for i from 1 to nops(param) do
          if type(paramSet[i]) = DOM_SET then
             solution:= subs(solution, param[i] = p) $ p in paramSet[i];
             param[i]:= NIL;
          elif param[i] <> NIL then
             break;
          end_if;
      end_for:
      if {op(param)} = {NIL} then
         // all internal parameters were resolved
         s:= {solution};
      end_if;
    end_if;

    if type(s) <> DOM_SET then
       // The solution is not a finite set.
       // We cannot handle this numerically.
       return(s0);
    end_if;

    //--------------------------------------------
    // At this point, the solution is a finite set.
    // Turn solution values into lists of equations:
    //--------------------------------------------
    // turn matrices into lists:
    //--------------------------------------------
    s:= [op(s)]; // turn set s to a list s to avoid
                 // rearrangement after subsop
    for i from 1 to nops(s) do
      sol:= op(s, i);
      if sol::dom::hasProp(Cat::Matrix) = TRUE then
          if nops(sol) <> nops(vars) then
            error("unexpected number of matrix entries");
          end_if;
          s:= subsop(s, i = [op(sol, i) $ i = 1..nops(vars)]);
      end_if;
    end_for:
    // turn s back into a set
    s:= {op(s)};

   if map(s, domtype) <> {DOM_LIST} then
      return(s0);
   end_if;
       
   //-----------------------------------------------------
   // convert [val1, val2, ..] to [var1 = val1, var2 = val2, ...]
   //-----------------------------------------------------
   s:= map(s, proc(sol) begin
                 for i from 1 to nops(sol) do
                   if not type(op(sol,i)) = "_equal" then
                      sol[i]:= (vars[i] = op(sol, i));
                   end_if;
                 end_for:
                 return(sol);
              end_proc):

   //-----------------------------------------------------
   // convert solutions of underdetermined equations from
   // [x = new_ident, y = f(new_ident), ...] into the old
   // form [x = x, y = f(x), ...]
   //-----------------------------------------------------
   if map(s, domtype) = {DOM_LIST} then
      s:= map(s, sol -> (for i from 1 to nops(sol) do
                           if type(op(sol, i)) = "_equal" and
                              domtype(op(sol, [i, 2])) = DOM_IDENT and
                              not has(orig_vars, op(sol, [i, 2])) then
                                  sol:= subs(sol, op(sol, [i, 2]) = op(sol, [i, 1]),EvalChanges);
                           end_if;
                         end_for;
                         sol;)):

      // delete trivial equations x = x etc:
      s:= map(s, map, sol -> if op(sol, 1) = op(sol, 2) then
                                null();
                             else
                                sol;
                             end_if);

      // clean round-off trash
      s:= map(s, map, map, numeric::complexRound);
      checkstability(s);
      return(s);
   end_if;
   return(s0);
end_proc:

// end of file 
