/* this file contains
-- numeric::realroot
-- numeric::BrentFindRoot  = auxiliary routine for numeric::realroot
-- numeric::BrentFindMinimum  = auxiliary routine for numeric::realroot
-- numeric::findbracket  = auxiliary routine for numeric::realroot

 -------------------------------------------------------------- 

numeric::realroot -- find a real root of a real-valued 
                   univariate function

Calls: numeric::realroot(f(x), x=a..b 
                         <, AcceptSingularity = v>  
                         <, SearchLevel = n>)

Parameters:   f -- any procedure accepting 1 numerical argument
                   and returning a numerical value
              x -- the unknown: identifier or indexed identifier
            a,b -- finite real numerical values (a search interval)

Options:      v -- TRUE or FALSE. Default: TRUE
              n -- nonnegative integer. Default: 1.

Return Value: a real floating point value or FAIL.
              If AcceptSingularity = TRUE, then this value is any
              type of "zero-crossing": the function f changes its
              sign in a neighbourhood of this point. This may be
              a singularity.
              If AcceptSingularity = FALSE, then this value is a 
              proper root or a zero-crossing jump discontinuity
              with finite height.

              FAIL is returned, if no root was found.

Details: (when specifying a search interval a..b)
 *) The internal search never touches points outside
    the interval a..b. The returned value is guaranteed
    to lie in this interval.
 *) For any x from the interval a..b the procedure f must
    produce values that can be converted to **real** floating 
    point numbers via float(f(x)).
 *) If f is continuous, then the zero-crossing is definitely a root.
 *) With AcceptSingularity = FALSE singularities may be found
    internally, but are not accepted as roots and are not returned.
 *) In the following: "root" is synonymous to "zero-crossing",
    i.e., "root" is either a proper root, a jump discontinuity,
    or a singularity (if AcceptSingularity = TRUE).
 *) The absolute precision goal dr for the numerical root r is

         abs(dr) <= 10^(-DIGITS)*max(abs(r),10^(-DIGITS))

    I.e., large roots are approximated with a relative
    precision 10^(-DIGITS), small roots with an absolute
    precision of 10^(-2*DIGITS).

    This is the precision which you would get, if no floating 
    point round-off were present in the numerical evaluation 
    of the procedure f.  Round-off will usually increase the 
    numerical error of r.
 *) Singularities in f are allowed, but numeric::realroot is not
    guaranteed to work: the internal search may accidentally
    hit a singularity, leading to an error.
    Note that the search might converge to a singular
    point x0 for zero-crossing singularities of the type
    1/(x-x0)^n with odd n, so there is an increased chance
    of a crash in this case. 

    However, if the singularities are not at the borders nor 
    at the midpoint of the interval, then there is a very good
    chance not to hit a singularity!
 *) Internally, a modified Brent-algorithm combining
    bisectioning, secant steps and quadratic interpolation
    is used. 
 *) If a..b brackets a root of f, i.e., if
              sign(f(a))<>sign(f(b)),
    then numeric::realroot is guaranteed to find a root.
 *) If a..b does not bracket a root of f, i.e., if
              sign(f(a)) = sign(f(b)),
    then the internal search drifts towards points
    with small values of abs(f).
    
    If a bracket of a root is found on the way, then f 
    succeeds in finding the root to the full precision as 
    specified above. 
 *) If no bracket is found in the course of the 
    algorithm, then FAIL is returned.
 *) The optional parameter n in SearchLevel=n
    controls the refinement of the search. It is
    meant for balancing the chances of finding a root 
    versus the time you have to wait until FAIL
    is returned, if no root exists (or no root could be found)
    
    The default value n = 1 seems to be high enough to
    find roots for most cases. 

    Increasing n by 1 quadruples the time before a FAIL
    is returned for functions without roots. !!!

    Do not use n >> 1, or you may have to wait for 
    hours in those cases, where no root exists!

----------------------------------------------------------
Examples:

>> numeric::realroot(x - PI, x = -PI..3*sqrt(2)); 
                          3.141592653

>> numeric::realroot(x^2 = PI^2, x = 0..3*sqrt(2)); 
                          3.141592653

// the root is not in the search interval:
>> numeric::realroot(x - PI, x = 4..5); 
                             FAIL

>> numeric::realroot(exp(-x) - sin(x), x = 1000..1010);
                          1002.168056

>> f:= 0.1*x + sin(x) - 2*cos(3*x) - exp(-x*sin(x)):
>> numeric::realroot(f, x = -10..10);
                               0.5518530559

// a discontinuous function
>> f:= proc(x)
         begin
             if x < 2
                then -0.00001*x -1 
                else exp(10*PI*x)
             end_if
         end_proc:
>> numeric::realroot(hold(f)(x), x = -10000..10000); 
                          1.999999999

>> numeric::realroot((x-2)^2, x = 0..5); 
                              2.0

>> numeric::realroot((x-2)^3, x = 0..4);
                              2.0

>> numeric::realroot((x-2)^3, x = 0..5);
                          1.999999999

>> numeric::realroot((x-2)^100, x = 0..5);
                              2.0
 
>> numeric::realroot((x-2)^101, x = 0..5);
                              2.0

--------------------------------------------*/

numeric::realroot:= proc(F, range)
local A, B, opt, opts, f,
      acceptSingularity,
      recursionLevel,
      round_to_boundary,
      criticalPoints,
      check,
      macheps, tol,
      safe,
      a, fa, b, fb, x, fx, xx, X,
      x1, x2, x3, x4, ffp, i, j;
begin

  acceptSingularity:= FALSE; // Default
  recursionLevel:= 1;       // Default

  //-------------------------
  //-----  check input ------
  //-------------------------
   
  if args(0)<2 then error("expecting at least 2 arguments"); end_if;
  if args(0)>4 then error("expecting at most 4 arguments"); end_if;

  if F::dom::hasProp(Cat::Matrix)=TRUE then
     F:= expr(F):
  end_if:
  case domtype(F)
   of DOM_SET do
   of DOM_ARRAY do
      if nops(F) <> 1 then
         error("expecting one equation or expression");
      else
         F:= op(F, 1);
      end_if:
  end_case;

  if type(F) = "_equal" and 
     testtype(op(F,1), Type::Arithmetical) and
     testtype(op(F,2), Type::Arithmetical)
  then F:= op(F,1)-op(F,2)
  end_if;

  if not testtype(F, Type::Arithmetical) then
     error("first argument must be an arithmetical expression");
  end_if:

  if type(range) <> "_equal" or type(op(range,2)) <> "_range" then
     error("second argument must be of the form x=a..b");
  end_if:

  X:= op(range, 1):
  range:= op(range, 2);

  if numeric::indets(F) minus {X} <> {} then
     // double check: if Y = numeric::odesolve2(..), then
     // we want to use numeric::realroot(Y(t)[1], t = a..b).
     // However, numeric::indets finds {Y(t)[1]}. 
     // Check heuristically, whether everything is ok:
     case domtype(float(subs(F, 
            X=op(range,1) +0.12345* (op(range,2) - op(range,1)),EvalChanges)))
     of DOM_FLOAT do
     of DOM_COMPLEX do break;
     otherwise 
       error("the function seems to contain symbolic objects");
     end_case:
  end_if;

  f:= proc(x) begin
                float(subs(F, X=x, EvalChanges))
      end_proc;

  // search optional arguments
  opts:= args(3..args(0)):
  for opt in [opts] do
      if has(opt, hold(AcceptSingularity)) 
      then if type(opt)="_equal" and 
              domtype(op(opt,2)) = DOM_BOOL
           then acceptSingularity:= op(opt,2);
           else error("specification should be AcceptSingularity = TRUE/FALSE");
           end_if;
           next; // go to next option
      end_if;
      if has(opt, hold(SearchLevel))
      then if type(opt)="_equal"
           then recursionLevel:= op(opt,2);
                if domtype(recursionLevel)<>DOM_INT or recursionLevel<0
                then error("specification should be SearchLevel = nonnegative integer")
                end_if;
           end_if;
           next; // go to next option
      end_if;
      // should never arrive here
      error("unknown option");
  end_for;
  userinfo(1, "numeric::realroot called with the following options: - ".expr2text(opts));

  // clear SearchLevel from the options:
  opts:= op(map([opts], proc() begin 
                        if has(args(1), SearchLevel) 
                           then null() 
                           else args(1)
                        end_if; end_proc));

  A:= op(range,1);
  a:= float(A);
  if domtype(a) <> DOM_FLOAT then
     error("cannot convert left boundary to a real float");
  end_if;
  B:= op(range,2);
  b:= float(B);
  if domtype(b) <> DOM_FLOAT then
     error("cannot convert right boundary to a real float");
  end_if;
  //--------------------------------------------------
  // Symbolic analysis: if the exact 0 is a root, then
  // this is the preferred solution. Return it as 0.0:
  //--------------------------------------------------
  if a <= 0 and 0 <= b then
     if traperror((fx := f(0))) = 0 and iszero(fx) then
        return(float(0));
     end_if;
  end_if;
  if a <= 1 and 1 <= b then
     if traperror((fx := f(1))) = 0 and iszero(fx) then
        return(float(1));
     end_if;
  end_if;
  if a <= -1 and -1 <= b then
     if traperror((fx := f(-1))) = 0 and iszero(fx) then
        return(float(-1));
     end_if;
  end_if;


  //------ Check left boundary --------------------------
  if traperror( (fa:=f(A)) )<>0 then
     // f(A) may exist, but f'(A) not (e.g., f=sqrt, A=0).
     // Do not use error when investigating f' (recursionLevel=0)!
     if recursionLevel>0 
       then error("cannot evaluate function at left boundary");
       else return(FAIL);
     end_if;
  end_if;
  if domtype((fa:= float(fa)))<> DOM_FLOAT then
     error("cannot convert function(left boundary) to a real float")
  end_if;
  if iszero(fa) then return(a) end_if;

  //------ Check right boundary --------------------------
  if traperror( (fb:=f(B)) )<>0 then
     // f(B) may exist, but f'(B) not (e.g., f(x)=sqrt(1-x), B=1).
     // Do not use error when investigating f' (recursionLevel=0)!
     if recursionLevel>0
       then error("cannot evaluate function at right bound");
       else return(FAIL);
     end_if;
  end_if;
  if domtype((fb:= float(fb)))<> DOM_FLOAT then
     error("cannot convert function(right bound) to a real float")
  end_if;
  if iszero(fb) then return(b) end_if;

  //-----------------------------------------------------
  // Now f(A) <> 0, f(B) <> 0. If A=B<>root, then FAIL:
  //-----------------------------------------------------

  if iszero(float(A-B)) then return(FAIL) end_if;

  /* ---- check: procedure to verify potential roots ---------
  input safe=TRUE --> check whether x is a bracked root
                      or a bracketed singularity
  input safe=FALSE --> check whether x is a root up to
                       roundoff
  ------*/
  check:= proc(x, safe)
  local fx, ffxx, dx, dfx, xx, i;
  begin
     if x=FAIL then return([FAIL, FALSE]) end_if;
     if safe and acceptSingularity then return([x, TRUE]) end_if;
     if traperror( ( fx:= float(f(x)) ) ) <> 0
     then return([x, FALSE])
     end_if;
     if iszero(fx) then return([x, TRUE]) end_if;
     if safe and not acceptSingularity
     then if specfunc::abs(fx) <= 1000*(specfunc::abs(fa)+specfunc::abs(fb)) 
             then return([x, TRUE]);  // bracketed proper root
             else return([x, FALSE]); // bracketed singularity
          end_if;
     end_if;
     if not safe then
       // unsafe x may be a local minimum. Check, whether it is
       // a root up to round-off. First, estimate round-off
       // in fx = fx_exact + round-off  by increasing DIGITS.
       // Compute round-off dfx = fx_highDIGITS -fx
       DIGITS:= DIGITS + 20:
       if traperror( (ffxx:= float(f(x))) ) <> 0
       then return([x, FALSE])
       end_if;
       dfx:= specfunc::abs(ffxx - fx); //dfx = estimated round-off
       DIGITS:= DIGITS - 20:
       // accept fx as zero, if |fx| <= round-off:
       if specfunc::abs(fx)<= 1.1*dfx then return([x, TRUE]); end_if;
       // Second try: change last decimal in x and evaluate f for
       // 20 such points. If round-off produces f-values with 
       // different signs, then accept x as root:
       dx:= macheps*max(specfunc::abs(x),macheps);
       for i from 1 to 10 do
         xx:= x + i*dx;
         if xx<=b then
            if traperror((ffxx:= float(f(xx)))) <> 0
               then return([x, FALSE])
            end_if;
            if ffxx*fa<=0 then return([xx,TRUE]) end_if;
         end_if:
         xx:= x - i*dx;
         if xx>=a then
           if traperror((ffxx:= float(f(xx)))) <> 0
             then return([x, FALSE])
           end_if;
           if ffxx*fa<=0 then return([xx,TRUE]) end_if;
         end_if:
       end_for;
     end_if;
     [x, FALSE];
  end_proc:

  //-----------------------------------------------------
  //----------- start the numerical search --------------
  //-----------------------------------------------------

  // switch to pure float arithmetic in the following, i.e.,
  // use a=float(A), b=float(B) instead of exact A, B.

  if a>b then ([a,b]):= [b,a] end_if;

  // Make sure that f is positive at the boundaries.
  // Change sign, if fa=f(a) < 0:
  if fa<0 then f:= _negate@f; fa:= -fa; end_if;

  // machine precision used in stopping criteria:
  macheps:= float(10^(-DIGITS));
  tol:= float(10^(-DIGITS/2));

  //-----------------------------------
  // call numeric::BrentFindRoot to search for the root:
  //-----------------------------------
  x1:= FAIL; // initialize
  userinfo(1, "-calling numeric::BrentFindRoot(f,".expr2text(a..b).")");
  // numeric::BrentFindRoot returns [root, safe] with safe=TRUE/FALSE
  ([x1, safe]):= numeric::BrentFindRoot(f, a..b);
  userinfo(1, "-result: ".expr2text([x1, safe]));
  ([x1, safe]):= check(x1, safe);
  userinfo(1, "-accept: ". expr2text(safe));
  if safe then return(x1) end_if;

  //---------------------------------------------
  // call numeric::BrentFindMin to search for a minimum:
  //---------------------------------------------
  x2:= FAIL; // initialize
  // Search for minimum:
  userinfo(1, "-calling numeric::BrentFindMin(f,".expr2text(a,x1,b).")");
  // numeric::BrentFindMin returns [root, safe] with safe=TRUE/FALSE
  if x1 = FAIL 
    then ([x2, safe]):= numeric::BrentFindMin(f, a, (a+b)/2, b);
    else ([x2, safe]):= numeric::BrentFindMin(f, a, x1, b);
  end_if:
  userinfo(1, "-result: ".expr2text([x2, safe]));
  userinfo(1, "-polish by numeric::BrentFindRoot(f,".expr2text(a..x2).")");
  // use numeric::BrentFindRoot to polish a point x2 with f(x2)<0 found by
  // numeric::BrentFindMin. Such points are marked by safe=TRUE:
  if safe then ([x2, safe]):= numeric::BrentFindRoot(f, a..x2); end_if;
  userinfo(1, "-result: ".expr2text([x2, safe]));
  // check for singularities and double zeroes
  ([x2, safe]):= check(x2, safe);
  userinfo(1, "-accept: ". expr2text(safe));
  if safe then return(x2) end_if;

  //---------------------------------------------
  // Call numeric::findbracket to search for a bracket:
  //---------------------------------------------
  x3:= FAIL; // initialize
  userinfo(1, "-calling numeric::findbracket(f,".expr2text(a..b).")");
  // numeric::findbracket returns aa..bb (safe) or FAIL.
  x3:= numeric::findbracket(f, a..b, 5);
  userinfo(1, "-result: ".expr2text(x3));
  if x3 <> FAIL then
    userinfo(1, "-polish by numeric::BrentFindRoot(f,".expr2text(x3).")");
    ([x3, safe]):= numeric::BrentFindRoot(f, x3)
  end_if;
  // check for singularities and double zeroes
  ([x3, safe]):= check(x3, safe);
  userinfo(1, "-accept: ". expr2text(safe));
  if safe then return(x3) end_if;

  //---------------------------------------------
  // Search for extrema = root of the derivative:
  //---------------------------------------------
  x4:= FAIL;
  if recursionLevel>0 then
     ffp:= diff(F,X);
     if (not has(ffp, hold(diff))) and
        (not has(ffp, hold(D)))
     then
          userinfo(1, "calling realroot with f' = ".expr2text(ffp));
          x4:= numeric::realroot(ffp, X=a..b, opts, SearchLevel = 0);
          userinfo(1, "exiting realroot with f' = ".expr2text(ffp));
          userinfo(1, "with result: ".expr2text(x4));
          // x4 might be a degenerate root:
          userinfo(1, "checking result");
          if x4<> FAIL 
          and // Newton criterion satisfied
             specfunc::abs(f(x4)) <= 
             macheps*max(macheps,specfunc::abs(x4))*
             specfunc::abs(float(subs(ffp,X=x4,EvalChanges)))
          and // not a singularity
              specfunc::abs(f(x4)) <= 
              1000*(specfunc::abs(fa)+specfunc::abs(fb))
          then 
               return(x4)
          end_if;
          ([x4, safe]):= check(x4, FALSE);
          userinfo(1, "accept: ". expr2text(safe));
          if safe then return(x4) end_if;
      end_if;
  end_if;

  //--------------------------------------------------------------
  // eliminate minima x1, x2, x3 and extremum x4 from the search 
  // interval and search each half. Round these values towards the
  // other boundary to make sure that these points are really 
  // eliminated from the search.  
  //--------------------------------------------------------------

  // If x is to close to a or b, then FAIL is returned.
  // Otherwise return [x-dx, x+dx]:
  round_to_boundary:= proc(x, a, b) 
  local tmp;
  begin tmp:= specfunc::abs(x)*tol;
        if specfunc::abs(x-a)<tmp or specfunc::abs(x-b)<tmp
        then return(FAIL)
        else if x>0
                then return([x*(1-tol),x*(1+tol)])
                else return([x*(1+tol),x*(1-tol)])
             end_if;
        end_if;
  end_proc;

  //--------------------------------------------------------------
  // split search on a..b to search on a..xx and xx..b, where
  // xx is either a minimum x1,x2,x3 or an extremum x4:
  //--------------------------------------------------------------
  if recursionLevel>0 then

     // eliminates FAILs
     criticalPoints:= select([x4,x3,x2,x1], _not@has, FAIL);

     // eliminate double entries to avoid
     // unnecessary costly recursion:
     for i from 1 to nops(criticalPoints)-1 do
        xx:= criticalPoints[i];
        j:= i+1;
        while j <= nops(criticalPoints) do
            if abs(criticalPoints[j]-xx) <= tol*(abs(xx)+macheps)
              then delete criticalPoints[j];
              else j:= j+1;
            end_if;
        end_while;
        if i>= nops(criticalPoints) then break; end_if;
     end_for;

     // eliminate points too close to a:
     criticalPoints:= map(criticalPoints,
           proc(x) begin
             if abs(x-a) <= tol*(abs(a)+tol)
             then null() else x end_if;
           end_proc);
     // eliminate points too close to b:
     criticalPoints:= map(criticalPoints,
           proc(x) begin
             if abs(x-b) <= tol*(abs(b)+tol)
             then null() else x end_if;
           end_proc);

     //recursive search of both half intervals a..xx and xx..b:
     for xx in criticalPoints do
       xx:= round_to_boundary(xx, a, b);
       if xx = FAIL then next end_if;
       x:= numeric::realroot(F,X=a..xx[1],opts, SearchLevel=recursionLevel-1);
       if x <> FAIL then return(x) end_if;
       x:= numeric::realroot(F,X=xx[2]..b,opts, SearchLevel=recursionLevel-1);
       if x <> FAIL then return(x) end_if;
     end_for;
  end_if;
  // give up:
  FAIL
end_proc:

//----  end of numeric::realroot -----

/* --------------------------------------------------------------------
This is an auxiliary routine used by numeric::realroot. It is not
meant to be called by the user, so it should not be documented!

numeric::BrentFindRoot -- ParabolicFit/Bisection search for
                          the root of a function

Call:  numeric::BrentFindRoot(f, a..b)

Parameters:  f  -- procedure to be called with one argument
                   x from the interval a..b, producing real
                   numerical values.
            a,b -- real numbers.

Return value:  [x, TRUE/FALSE] with x from the interval a..b.
            TRUE indicates that x is a zero-crossing point
            (i.e., a root, if f is continuous) 
            FALSE indicates that x is a heuristic minimum
            of abs(f(x)).

Details:
    * The routine searches for a root of f on the interval a..b.
    * If the input data a..b bracket a root, i.e., if
               f(a)*f(b) <= 0,
      then BrentFindRoot is guaranteed to find a sign changing
      point in a..b. The return value is [x, TRUE], with TRUE
      indicating that x is a bracketed sign changing point.

      This is a root, if f is continuous.
      It may also be a zero crossing jump dicontinuity or
      a zero crossing singularity such as 1/(x-root).
    * If the input data a..b do not bracket a minimum, i.e., if
               f(a)*f(b) > 0,
      then the abscissa x with the smallest value of abs(f)
      encountered in the internal search is returned.

      This may be a bracketed zero-crossing point of f,
      if a bracket was found during the search. The return
      value is [x, TRUE].

      If no bracket was found during the search, then
      [x, FALSE] is returned. x is a heuristic minimum of abs(f).

Background: 
    This is a MuPAD version of "zbrent" (chapter 9.3, pages 361-362)
    from W.H.Press, S.A.Teukolsky, W.T.Vetterling, B.P. Flannery:
    Numerical Recipes in C, 2nd edition, Cambridge University Press, 
    1992.

    This routine was modified to accept input data that do not 
    necessarily bracket a zero-crossing, leading to a heuristic
    minimum of abs(f).
      
Examples:
// root x = PI not found due to roundoff, heuristic minimum returned:
numeric::BrentFindRoot( x -> (1-exp(-10*(x-PI)^2)), -1..4);
                           [3.141592653, FALSE]

// root bracketed:
numeric::BrentFindRoot( x -> (0.9-exp(-10*(x-PI)^2)), -1..4);
                            [3.038947383, TRUE]

// root not found due to roundoff, heuristic minimum returned:
numeric::BrentFindRoot( x -> ((x-1/3)^2), -10..1000);
                           [0.3333333333, FALSE]

// root bracketed:
numeric::BrentFindRoot( x -> ((x-1/3)^3), -10..1000);
                           [0.3333333333, TRUE]

// no root on the search interval, minimum at left border:
numeric::BrentFindRoot( x -> ((x-1/3)^4), 1..1000);
                               [1.0, FALSE]

// no root on the search interval, minimum at right border:
numeric::BrentFindRoot( x -> ((x-1/3)^4), -10..-1);
                              [-1.0, FALSE]
----------------------------------------------------------------------*/

numeric::BrentFindRoot:= proc(f, range)
local A, a, B, b, macheps, itmax, safe,
      iter, c, fa, fb, fc, d,
      e, p, q, r, s, tol1, xm;
save DIGITS;
begin
  if type(range)<> "_range" then
     error("second argument must be a range");
  end_if:
  ([A, B]):= [op(range)]:
  macheps:= float(10^(-DIGITS));
  DIGITS:= DIGITS + 2:

  a:= float(A): fa:= float(f(a)):
  b:= float(B): fb:= float(f(b)):
  c:= b;        fc:= fb;
  if fa*fb > 0
     then safe:= FALSE;
     else safe:= TRUE;
  end_if;

  //---------------------------------------------------------
  // Maximal number itmax of bisection steps:
  // In safe mode you get linear convergence with a factor 1/2
  // by 3 steps, if the root is multiple (arbitrary multiplicity).
  // This seems to be the worst case.
  // So itmax needs to satisfy 
  //    (B-A)/2^(itmax/3) <= absolute precision goal
  //                       = macheps*max(abs(x),macheps) ,
  // where x is the solution (in A..B)
  // --> ln(2)*itmax/3 >= ln(B-A)+ln(10)*DIGITS-ln(max(abs(x),macheps))
  //                    = ln(B-A)+ln(10)*DIGITS+min(-ln(abs(x)),ln(10)*DIGITS)
  // Choose:
  //         ln(2)*itmax/3 >=     ln(1+|B-A|) + 2*ln(10)*DIGITS
  // sufficient:   itmax   >= 4.2*ln(1+|B-A|) + 19.9*DIGITS 
  //---------------------------------------------------------

  itmax:= 10 + round(5*specfunc::ln(1+specfunc::abs(b-a))) + 22*DIGITS;
  
  //-------------------------------------
  //-----  start Brent iteration --------
  //-------------------------------------
  // We have three points (a,fa), (b,fb), (c,fc).
  // If b,c do not bracket a root, then replace
  // (c,fc) by (a,fa). Last stepsize is e = d = b-a

  for iter from 1 to itmax  do
     if fb*fc > 0 then
      c:= a; fc:= fa; d:= b-a: e:= d;
     end_if;
     // We have moved (a,fa) -> (c,fc).
     // Check again:
     if not safe then
        if fb*fc <= 0 then safe:= TRUE end_if;
     end_if;
     // In safe mode the root is bracketed between b and c.
     // b should be the point with smaller value of f.
     // Swap b,c, if this is not the case:  
     if specfunc::abs(fc) < specfunc::abs(fb) then
        a:=  b;  b:=  c;  c:=  a:
       fa:= fb; fb:= fc; fc:= fa:
     end_if:

     // check, if root b is approximated to required precision:
     // I replace the following line from Numerical Recipes:
     // tol1:= 2*macheps*specfunc::abs(fb) + abstol/2;
     // by the following precision goal  (i.e., 
     //  goal = relative machine-precision for large b,
     //  goal = absolute precision macheps^2 for small b)

     tol1:= max(macheps*specfunc::abs(b)/2, macheps^2);
     xm:= (c-b)/2;

     // convergence criterion
     if specfunc::abs(xm) <= tol1 or iszero(fb) then 
       return([b,safe]);
     end_if:

     // Next step of Brent iteration:
     if specfunc::abs(e) >= tol1 and specfunc::abs(fa) > specfunc::abs(fb)
     then // do interpolation
          s:= fb/fa;
          if a=c
          then // only 2 points (b,fb),(c,fc) availabe, 
               // try secant step:
               p:= 2*xm*s; 
               q:= 1-s;
          else // three points (a,fa),(b,fb),(c,fc) available, 
               // try quadratic interpolation:
               q:= fa/fc; 
               r:= fb/fc;
               p:= s*( 2*xm*q*(q-r) - (b-a)*(r-1) );
               q:= (q-1)*(r-1)*(s-1)
          end_if;
          // (p,q) -> (abs(p), sign(p)*q);
          if p>0 then q:= -q else p:= -p end_if;
          // check whether b + d = b + p/q would leave bracket:
          if 2*p < min(3*xm*q - specfunc::abs(tol1*q), specfunc::abs(e*q))
            then e:= d; d:= p/q;  // accept interpolation, next b = b+d
                                  // will be inside bracket
            else d:= xm; e:=d;    // interpolation failed, use bisection
          end_if;
     else // bounds are decreasing too slowly, use bisection
          d:= xm; e:= d;
     end_if;
     // store last best guess in a.
     a := b; fa:= fb;   
     // replace b by next approximation of the root:
     if specfunc::abs(d) > tol1
     then b:= b+d;
     else if xm > 0 
             then b:= b + tol1;
             else b:= b - tol1; 
          end_if;
     end_if;
     fb:= float(f(b));
   end_for;
   //warning("reached maximal count in numeric::BrentFindRoot");
   return([FAIL, FALSE]);
end_proc:

// --- end of numeric::BrentFindRoot ---

/* --------------------------------------------------------------------
This is an auxiliary routine used by numeric::realroot.
It is not meant to be called by the user, so it should
not be documented!

numeric::BrentFindMin -- ParabolicFit/GoldenSection search for 
                         the minimum of a function

Call:  numeric::BrentFindMin(f, a, b, c)

Parameters:  f  -- procedure to be called with one argument
                   x from the interval a..c,  producing real 
                   numerical values.
          a,b,c -- real numbers satisfying a < b < c or
                   c < b < a.

Return value:  [x, TRUE/FALSE] with x from the interval a..b.

Details:  
    * The routine searches for a minimum of f on the interval a..c.
      Simultaneously it searches for values x on a..c with f(x) <= 0.
    * If f(a) <= 0 then [a, TRUE] is returned.
      If f(b) <= 0 then [b, TRUE] is returned.
      If f(c) <= 0 then [c, TRUE] is returned.
    * Any result of the form [x, TRUE] implies that f(x)<= 0.
    * If the input data a,c bracket a minimum, i.e., if
               f(b) < min(f(a),f(c)),
      then x is a local minimum of f (return value [x, FALSE])
      (assuming  f to be continuous)
      or x satisfies f(x)<= 0  (return value [x, TRUE]).
    * If the input data do not bracket a minimum, i.e., if
               f(b) >= min(f(a),f(c)),
      then the abscissa x with the smallest value of f 
      encountered in the internal search is returned.
      In this sense x is a heuristic minimum of f.

Background: 
    This is a MuPAD version of "brent" (chapter 10, pages 404-405)
    from W.H.Press, S.A.Teukolsky, W.T.Vetterling, B.P. Flannery:
    Numerical Recipes in C, 2nd edition, Cambridge University Press, 
    1992.

    This routine was modified to accept input data that do not 
    necessarily bracket a minimum and to exit prematurely whenever
    x with f(x)<=0 is encountered.

Examples:
  numeric::BrentFindMin( x -> (1-exp(-10*(x-PI)^2)), -1, 0, 4); 
                           [3.141592361, FALSE]
  numeric::BrentFindMin( x -> (0.9-exp(-10*(x-PI)^2)), -1, 0, 4); 
                            [3.045248729, TRUE]
  numeric::BrentFindMin( x -> ((x-1/3)^2), -10, 2, 1000); 
                           [0.3333333333, FALSE]
  numeric::BrentFindMin( x -> ((x-1/3)^4), -10, 2, 1000); 
                           [0.3333354225, FALSE]
  numeric::BrentFindMin( x -> ((x-1/3)^4), 1, 2, 1000); 
                               [1.0, FALSE]
  numeric::BrentFindMin( x -> ((x-1/3)^4), -10, -2, -1); 
                              [-1.0, FALSE]
  numeric::BrentFindMin( x -> (10-x^2), -3, -2, 1); 
                              [-3.0, FALSE]
----------------------------------------------------------------------*/

numeric::BrentFindMin:= proc(f,a,b,c)
local itmax, CGOLD, iter,
      macheps, tol, tol1, tol2,
      u, v, w, x, 
      fu, fv, fw, fx,
      d, e, old_e, 
      p, q, r, xm; 
begin
  if (fx:= float(f(a))) <= 0 then return([a, TRUE]) end_if;
  x:= a;
  if (fu:= float(f(b))) <= 0 then return([b, TRUE]) end_if;
  if fu<fx then x:= b; fx:= fu; end_if;
  if (fu:= float(f(c))) <= 0 then return([c, TRUE]) end_if;
  if fu<fx then x:= c; fx:= fu; end_if;

  // switch to float arithmetic
  ([a, b, c, x]):= [float(a), float(b), float(c), float(x)]:
  if domtype(a) <> DOM_FLOAT or
     domtype(b) <> DOM_FLOAT or
     domtype(c) <> DOM_FLOAT
  then error("abscissae must be real numerical values")
  end_if;

  // check input
  if iszero(specfunc::abs(c-a)) then return(FAIL) end_if;
  if (c-b)*(b-a) <= 0 then
   //error("1st and 3rd argument must bracket 2nd argument")
     b:= (a+c)/2;
     fu:= float(f(b));
     if fu<fx then x:= b; fx:= fu; end_if;
  end_if;

  // see BrentFindRoot for rationale of the following itmax:
  itmax:= 10 + round(5*specfunc::ln(1+specfunc::abs(c-a))) + 22*DIGITS;

  CGOLD:= float((3-5^(1/2))/2);
  tol:= float(10^(-DIGITS/2));
  macheps:= float(10^(-DIGITS));

  // interpretation:
  // a, c = bracket of minimum
  // e = the distance moved on the step before last
  // x = point with least function value found so far
  // w = point with second least function value found so far
  // v = previous value of w
  // u = last point where f was evaluated

  ([w, v]):= [x $ 2]:    // x=w=v=point with minimal f-value so far
  ([fv, fw]) := [fx $ 2];// minimal f-value so far
  e:= float(0);
  ([a, c]):= [min(a,c), max(a,c)]:  // need a<c in the following

  for iter from 1 to itmax do
     xm:= (a+c)/2;
     tol1:= tol*(specfunc::abs(x)+macheps);
     tol2:= 2*tol1;

     // stopping criterion
     if specfunc::abs(x-xm) <= (tol2 - (c-a)/2) then
        return([x, FALSE]);
     end_if;

     if specfunc::abs(e)>tol1 // construct a trial parabolic fit
     then ([r, q]):= [(x-w)*(fx-fv), (x-v)*(fx-fw)];
          p:= (x-v)*q - (x-w)*r;
          q:= 2*(q-r);
          if q>0 then p:=-p else q:= -q; end_if;
          old_e:= e:
          e:= d;
          // is parabolic step acceptable?
          if specfunc::abs(p) >= specfunc::abs(q*old_e/2)
          or p <= q*(a-x) or p >= q*(c-x)
          then // do golden section step
               if x>= xm then e:= a-x; else e:= c-x; end_if;
               d:= CGOLD*e;
          else // do parabolic step
               d:= p/q;
               u:= x+d;
               if u-a < tol2 or c-u < tol2 then
                  if xm>x then d:=  tol1; else d:= -tol1; end_if;
               end_if;
          end_if;
     else if x>=xm then e:= a-x; else e:= c-x; end_if;
          d:= CGOLD*e;
     end_if; 

     // next point:
     if specfunc::abs(d)>=tol1
       then u:= x+d;
       else if d>=0 then u:= x+tol1 else u:= x-tol1 end_if;
     end_if;
     // The new function evaluation:
     fu:= float(f(u));
     if fu<=0 then
        return([u, TRUE]);
     end_if;

     //Housekeeping: prepare for the next step:
     if fu <=fx
     then if u>=x then a:=x; else c:=x; end_if;
          ([v, w, x, fv, fw, fx]):= [w, x, u, fw, fx, fu]:
     else if u<x then a:=u else c:=u end_if;
          if (fu<=fw or w=x)
            then ([v, w, fv, fw]):= [w, u, fw, fu];
            else if fu=fv or v=x or v=w 
                   then ([v, fv]):= [u, fu];
                 end_if;
          end_if;
     end_if;
  end_for;
  FAIL;
end_proc:
// ---------------------------------------------------

/* ----   Tests for numeric::BrentFindMin --------
print("-------------- minimum exists: --------------");
f:= x-> ( x^2):
t:=time():   numeric::BrentFindMin(f, -2, 1, 2),(time()-t)*msec;
f:= x-> ( x^2-2*x+2):
t:=time():   numeric::BrentFindMin(f, -2, 1, 2),(time()-t)*msec;
f:= x-> ( x^2+2*x+2):
t:=time():   numeric::BrentFindMin(f, -2, 1, 2),(time()-t)*msec;
f:= x-> ( (x-PI)^2 + 1):
t:=time():   numeric::BrentFindMin(f,-200, 100, 300),(time()-t)*msec;
f:= x-> ( 1-cos(x)):
t:=time():   numeric::BrentFindMin(f, -2, 1, 2),(time()-t)*msec;
t:=time():   numeric::BrentFindMin(f, PI, 6, 3*PI),(time()-t)*msec;
t:=time():   numeric::BrentFindMin(f, PI, 2*PI-0.1, 3*PI),(time()-t)*msec;
f:= x-> ( 2-cos(x)):
t:=time():   numeric::BrentFindMin(f, -2, 1, 2),(time()-t)*msec;
t:=time():   numeric::BrentFindMin(f, PI, 6, 3*PI),(time()-t)*msec;
f:= x -> ( 1- 0.5*exp(-(x+PI)^2) - exp(-100*(x-PI)^2) ):
t:=time():   numeric::BrentFindMin(f, -10, -5, 100),(time()-t)*msec;
t:=time():   numeric::BrentFindMin(f, -10, -5, 100),(time()-t)*msec;
t:=time():   numeric::BrentFindMin(f, -PI, -3, 100),(time()-t)*msec;
t:=time():   numeric::BrentFindMin(f, -PI, -3, 100),(time()-t)*msec;

print("------------  no minimum exists: --------------");
f:= x -> (x):
t:=time():   numeric::BrentFindMin(f, -100, -99, 1000),(time()-t)*msec;
f:= x -> (-x + 10000):
t:=time():   numeric::BrentFindMin(f, -100, -99, 1000),(time()-t)*msec;
t:=time():   numeric::BrentFindMin(f, -100, -99, 1000),(time()-t)*msec;
f:= x -> ( 1+ 0.5*exp(-(x+PI)^2) + exp(-100*(x-PI)^2) ):
t:=time():   numeric::BrentFindMin(f, -10, -5, 100),(time()-t)*msec;
t:=time():   numeric::BrentFindMin(f, -10, -5, 100),(time()-t)*msec;
t:=time():   numeric::BrentFindMin(f, -PI, -3, 100),(time()-t)*msec;
t:=time():   numeric::BrentFindMin(f, -PI, -3, 100),(time()-t)*msec;
f:= x-> (-x^2 +10000):
t:=time():   numeric::BrentFindMin(f, -2, 1, 2),(time()-t)*msec;
f:= x-> ( (x-PI)^2 + 1):
t:=time():   numeric::BrentFindMin(f, 100, 200, 300),(time()-t)*msec;

print("-------- premature exit: bracket found: -----------");
f:= x-> (-1+x^3 +x^4):
t:=time():   numeric::BrentFindMin(f, -2, 1, 2),(time()-t)*msec;
f:= x-> ( (x-PI)^2 -1):
t:=time():   numeric::BrentFindMin(f, 100, 200, 300),(time()-t)*msec;
t:=time():   numeric::BrentFindMin(f,-200, 100, 300),(time()-t)*msec;
--- end of tests --- */

/* --------------------------------------------------------------------
This is an auxiliary routine used by numeric::realroot. It is not
meant to be called by the user, so it should not be documented!

numeric::findbracket -- find a subinterval bracketing a root of
                        a function by a brute force deterministic
                        search

Call:         numeric::findbracket(f, A..B <, nmax>)
Parameters:   f -- procedure (function in one variable)
              A -- real numerical value
              B -- real numerical value
              nmax -- integer >= 0 (default: nmax = 10)

Return Value: a range a..b representing a subinterval 
              of A..B. It satisfies f(a)*f(b) <= 0,
              i.e., a...b brackets a zero crossing of f.
              If no such subinterval was found, then
              FAIL is returned. 
Details:

 *) Searches for a point x satisfying f(x)*f(A) <= 0
 *) The search points are of the form 
       x = A + i*dx,  dx = (B-A)/(2^n), i=0..2^n, n <= nmax
 *) Note that in the worst case 1+2^nmax function calls
    are evaluated, before FAIL is returned.
    nmax = 10 implies at most 1025 calls, this is reasonable.
    Increasing nmax by 1 doubles the time before a FAIL
    is accepted! Do not use nmax > 20, or you may have to
    wait for hours!
 *) If x is a root
       then a..b with a=b=x is returned.
       else a..b with a=x-dx, b=x is returned,
    where dx = (B-A)/(2^n), with some n<= nmax
 *) The returned result a..b is guaranteed to satisfy
       f(a)*f(A)>=0, f(b)*f(A)<=0, f(a)*f(b)<= 0.
 *) If no x in A..B with f(x)*f(A)<=0 is found, 
    then FAIL is returned.
--------------------------------------------*/

numeric::findbracket:= proc(f, range)
local a, b, fa, fx, x, dx, N, n, nmax, i;
begin
  if type(range) <> "_range" then
     error("second argument must be a range");
  end_if:
  ([a, b]):= [op(range)]:

  // is a a root ?
  fa:= float(f(a)):
  if iszero(fa) then return(a..a); end_if;

  // is b a root or sign(f(a))<> sign(f(b)) ?
  fx:= float(f(b)):
  if fx*fa < 0  then return(a..b) end_if;
  if iszero(fx) then return(b..b) end_if;
 
  if iszero(a-b) then return(FAIL) end_if;
 
  // Now switch to pure float modus:
  a:= float(a):
  b:= float(b):
 
  nmax:= 10: // default, worst case: 2^(nmax)=1024 function calls before FAIL.
  if args(0) = 3 then nmax:= args(3) end_if;
  if domtype(nmax)<>DOM_INT or nmax < 0 then
     error("third argument must be a nonnegative integer")
  end_if;
 
  // Systematic search for subintervals:
  dx:= b - a: N:= 1:
  for n from 1 to nmax do
      // N:= 2^n, dx = (b-a)/N; 
      // investigate the points 
      //  x = a+dx/2, a+3*dx/2, a+5*dx/2, ..  ,b - dx/2
      // Note that x = a, a+dx, a+2*dx, .. , b were
      // investigated in previous loops
      x:= a + dx/2:        // dx = 1/N
      for i from 1 to N do // N=2^n
          fx:= float(f(x)):
          if iszero(fx) then return(x..x) end_if;
          if fx*fa < 0  then return((x-dx/2)..x) end_if; 
          x:= x + dx; // 
      end_for;
      N:= 2*N;
      dx:= dx/2;
  end_for:
  FAIL;
end_proc:

//------------------------------------------------------
// Code for a random search for a bracket. It seems
// somewhat slower and less successful than the
// deterministic search above, so do not use the
// random version. 

/* -----------------------------------------------------
numeric::findbracket2 -- find a subinterval bracketing a root
                         of a function by a brute force random
                         search

Call:         numeric::findbracket2(f, A..B)
Parameters:   f -- procedure (function in one variable)
              A -- real numerical value
              B -- real numerical value

Return Value: a range a..b representing a subinterval
              of A..B. It satisfies f(a)*f(b) <= 0,
              i.e. a..b brackets a zerocrossing of f.
              If no such subinterval was found, then
              FAIL is returned.

Details:
 *) The search points are chosen randomly on A..B.
--------------------------------------------*/
/* -------- do not use the following code ---------------

numeric::findbracket2:= proc(f, range)
local a, fa, b, fx, x, xList, r, 
      cachesize, maxfcalls,
      i, j, jj, dist;
save SEED;
begin
 if type(range) <> "_range" then
    error("second argument must be a range");
 end_if:
 [a, b]:= [op(range)]:
 // worst case: maxfcalls function calls before FAIL.
 maxfcalls:= 1023: 
 cachesize:= 100:
 fa:= float(f(a)):
 if iszero(fa) then return(a..a) end_if;
 fx:= float(f(b)):
 if fx*fa < 0 then return(a..b) end_if;
 if iszero(fx) then return(b..b) end_if;

 // random generator for uniform floats on [a,b]:
 SEED:= 1; 
 // Do not use DIGITS for generating the random numbers,
 // because different results might follow for different
 // user settings of DIGITS.
 // r:= float(a)+float(b-a)*float@random(0..10^DIGITS)/10^DIGITS;
 r:= float(a)+float(b-a)*float@random(0..10^10)/10^10;

 // Cache for investigated points with sign(f(x))=sign(f(a))
 // Make sure that the cache is not too big:
 xList:= [0 $ cachesize]:

 for i from 1 to maxfcalls do
     x:= r();
     fx:= float(f(x));
     if float(fx)*fa < 0
     then // bracket found, we are done!
          // Do not return a..x, but search for the point
          // in xList closest to x:
          dist:= specfunc::abs(x-a);
          for j from 1 to min(i,cachesize) do
              if (b:= specfunc::abs(xList[j]-x)) < dist then
                 jj:= j; dist:= b;
              end_if;
          end_for;
          a:= xList[jj]:
          if a<x 
            then return(a..x)
            else return(x..a)
          end_if;
     end_if;
     if iszero(fx) then return(x..x) end_if;
     if i<=cachesize then xList[i]:= x: end_if:
 end_for:
 FAIL;
end_proc:

do not use the numeric::findbracket2 code, 
use numeric::findbracket ! 
----------------------------------------------------------------------------*/

// ---- end of file ----
