// 
//       
// 

/*
 26.10.2004: *) Option Merge durch Merge = TRUE/FALSE ersetzt
             *) Neuer Default ist Merge = TRUE
             *) Der Suchbereich x = a..b kann nun auch ein
                unendliches Intervall sein
             *) Suchbereich x = a..b ist nun optional,
                Default ist x = -infinity .. infinity
*/
/*++
  numeric::realroots(p(x), <x=a..b>, <eps>, <Merge = TRUE/FALSE> )

  p(x) : some arithmetical expression in x
  x    : DOM_IDENT or indexed identifier
  a, b : real numerical constants (including +- infinity)
  eps  : positive numerical constant >= 10^(-DIGITS)
  Merge: do merge resulting intervals whatever the length
         of the result is.

  Returns a list of intervals (=lists), in which real roots of the
  expression p(x) may exist. All intervals are inside [a,b]. All 
  intervals have length < eps. The default is eps = 0.01.

  Warning: if p(x) is identical to zero, then the entire
  search interval is returned. If the result violates
  b-a < eps, then a warning is given;

  Also equations p(x): lhs(x)=rhs(x) are permitted as 1st argument.

  It is garanteed that no real roots exist outside the intervals.
  However, it is not guaranteed that each interval does indeed
  contain a root.

>> numeric::realroots((x^2 - 1)^2,x=-6..6,0.01);

                      [[-1.0, -1.0], [1.0, 1.0]]

>> numeric::realroots(5*x-20*x^3+16*x^5, x=-6..6,0.01);

    [0.5859375, 0.58984375], [0.9453125, 0.953125]]

>> numeric::realroots(sin(x),x=-4..4, 0.1 );

   [[-3.146875, -3.0859375], [-0.1, 0.0], [0.0, 0.1], [3.0859375, 3.146875]]

++*/

numeric::realroots := proc(eq, var)
local p, x, a, b, ab, eps, R, OKTypes, 
      user_searchrange, options, user_merge, doMerge, Arg;
begin
   // set defaults
   doMerge:= TRUE; // default
   eps:= 0.01;     // default for length of subdivided intervals
   R:= 10^5;       // default for cutoff at +-infinity
                   // Beware: use R = 10^5, not 10.0^5 or 1e05, because floating 
                   // point values would be changed substantially by DOM_INTERVAL!

   if args(0)<1 then error("expecting at least one argument") end_if;
   if args(0)>4 then error("expecting at most four arguments") end_if;
   if eq::dom::hasProp(Cat::Matrix)=TRUE then
     eq:= expr(eq):
   end_if:
   case domtype(eq)
   of DOM_SET do
   of DOM_ARRAY do
      if nops(eq) <> 1 then
         error("expecting one expression or equation");
      end_if:
      eq:= op(eq, 1);
   end_case;
   p:= subs(eq, float(0)=0, EvalChanges);
   if type(p)="_equal" then p:= op(p,1)-op(p,2) end_if;
   if not testtype(p, Type::Arithmetical) and 
      not domtype(p)=DOM_POLY then
      error("the first argument must be an equation ".
            "or of domain type DOM_POLY ".
            "or of type 'Type::Arithmetical'")
   end_if;
   if args(0) > 1 then
      var := args(2);
      if (not has(var, Merge)) and 
         domtype(float(var)) <> DOM_FLOAT and
         (type(var)<>"_equal" or type(op(var,2))<>"_range")
      then error("second argument is not of the form x = a..b"); 
      end_if;
   end_if;
   if args(0) < 2 or not hastype(args(2), "_range") then
      user_searchrange:= FALSE;
      x:= numeric::indets(p):
      if nops(x) > 1 then
          error("the first argument must contain only one unknown");
      end_if;
      x:= op(x);
      a:= RD_NINF;
      b:= RD_INF;
   else
      user_searchrange:= TRUE;
      x:= op(var, 1);
      if domtype(x) <> DOM_IDENT and type(x) <> "_index" then
         error("in the second argument 'x = a..b', the unknown x ".
               "must be an identifier or an indexed identifier")
      end_if;
      if numeric::indets(p) minus {x} <> {} then
         error("the first argument must be a univariate expression or equation in ".expr2text(x)) 
      end_if; 
      // Do not (!!!) convert bounds of search interval to floats.
      // Otherwise there is danger of missing roots due to roundoff!!
      // Example: numeric::realroots(sin(PI*x),x=-1.0..1.0,0.1);
      a:= subs(op(var, [2,1]), [-infinity = RD_NINF, infinity = RD_INF]);
      if domtype(float(a))<>DOM_FLOAT then 
         error("illegal symbolic or complex left bound a in 2nd argument x = a..b") 
      end_if;
      b:= subs(op(var, [2,2]), [-infinity = RD_NINF, infinity = RD_INF]);
      if domtype(float(b))<>DOM_FLOAT then 
         error("illegal symbolic or complex right bound b in 2nd argument x = a..b") 
      end_if;
      if float(b)<float(a) 
         then error("illegal search interval x = a..b with a>b");
      end_if;
   end_if;

   // search for optional arguments:
   if user_searchrange then
      options:= [args(3..args(0))];
   else 
      options:= [args(2..args(0))];
   end_if;
   user_merge:= FALSE:
   for Arg in options do
       if has(Arg, Merge) then 
          user_merge:= TRUE:
          case Arg
          of Merge do
          of Merge = TRUE do
             doMerge:= TRUE; 
             break;
          of Merge = FALSE do
             doMerge:= FALSE; 
             break;
          otherwise
             error("specify merging of intervals by Merge, Merge = TRUE, or Merge = FALSE");
          end_case;
          next;
       end_if;
       eps:= float(Arg);
       if domtype(eps)<>DOM_FLOAT
          then error("illegal precision goal")
       end_if;
       if eps<0.999*10^(-DIGITS) 
          then error("precision goal too small. It must be >= 10^(-DIGITS)")
       end_if;
   end_for;

   // special case: p = 0
   if iszero(p) then 
      if specfunc::abs( float(b)-float(a))>eps then
         warning("zero function, returning search interval"); 
      end_if;
      return(subs([[float(a),float(b)]],
                  [RD_NINF = -infinity, RD_INF = infinity]));
   end_if;

   //Treat polynomial expressions via polylib::realroots.
   //This is rigorous and also much faster.
   if (domtype(p) = DOM_POLY or testtype(p, Type::PolyExpr(x))) and
       map(if domtype(p) = DOM_POLY then {coeff(p)} else {coeff(p, [x])} end_if, 
           domtype@float) = {DOM_FLOAT} then
        p:=polylib::realroots(p, eps);
        // intersect isolating intervals from polylib::realroots
        // with the search interval [a,b] of numeric::realroots
        OKTypes:= {DOM_INT,DOM_RAT,DOM_FLOAT}:
        if {domtype(a)} minus OKTypes<>{} then a:=float(a):end_if;
        if {domtype(b)} minus OKTypes<>{} then b:=float(b):end_if;
        p:=map(p,
               proc(x) begin  //x[1], x[2] are DOM_INT or DOM_RAT
                 if x[1]<a and x[2]< a then return(null()) end_if; 
                 if x[1]<a and x[2]>=a and x[2]<=b then return([a, x[2]]) end_if; 
                 if x[1]<a and x[2]> b then return([a, b]) end_if; 
                 if x[2]>b and x[1]> b then return(null()) end_if; 
                 if x[2]>b and x[1]>=a and x[1]<=b then return([x[1],b]) end_if; 
               //if x[2]>b and x[2]< a then return([a, b]) end_if; 
                 return(x);
               end_proc);
        p:= map(p, map, float);
        // the default for polynomials is Merge = FALSE unless requested
        // otherwise by the user. Hence: if user_merge and doMerge
        if user_merge and doMerge then
           // Now merge intervals with common bounds.
           a:=1; 
           while a < nops(p) do
              for b from a+1 to nops(p) do
                 if op(p[b-1],2) <> op(p[b],1) then break end_if
              end_for;
              if doMerge or float(op(p[b-1],2)-op(p[a],1))<=1.0001*eps
              then
                 p[a]:=[op(p[a],1), op(p[b-1],2)];
                 for b from a+1 to b-1 do delete p[a+1] end_for;
              end_if;
              a:=a+1
           end_while;
        end_if;
        return(p);
   else //treat non-polynomial expressions via interval arithmetic.
        // First, convert constant expressions in p into intervals:
        // Use expr(p), since p may be a complex DOM_POLY:
        p:= interval(expr(p)):
        // Enlarge the search interval a bit via interval arithmetic.
        ab:= a ... b;
        // Pass the interval to the routine doing the actual work
        p:= [numeric::realroots_(p, x, ab, eps, R)];
        if doMerge then
           p := _union(op(p));
           if op(p,0) = hold(_union) then
              p := [op(p)];
           elif p = {} then
              return([])
           else
              p := [p];
           end_if;
        end_if;
        return(subs(map(p, x->[op(x)]), 
               [RD_NINF = -infinity, RD_INF = infinity]));
   end_if;
end_proc:

//-----------------------------------------------------
// recursive utility realroots_. If 0 is not contained in the
// image p(ab), then ab is eliminated from the search
// range. If 0 is contained in p(ab), then bisectioning is
// used to investigate the 2 halfs of the interval. The search
// stops when DOM_INTERVAL::width(ab) < eps.
// An infinite interval such as a ... RD_INF is split
// into a ... 10^5 and 10^5 ... RD_INF. The second
// interval is returned without further subdivisions.
   
//-----------------------------------------------------

numeric::realroots_ := proc(p, x, ab, eps, R)
   local image, containszero, a, b, c;
begin
   userinfo(1, "investigating interval ".expr2text(ab));
   image:=subs(p, x = ab, EvalChanges);

   if not testtype(image, DOM_INTERVAL) then
       error("cannot handle the expression ".expr2text(p).
             "  via interval arithmetic");
   end_if:

   containszero:= bool(0 in image);
   if containszero = FALSE
   then // The interval ab definitely does not contain a root,
        // so eliminate this interval from the range that may 
        // contain roots:

        return(null());

   else // If 0 is in the image interval, then we really do not know
        // whether the interval [a, b] contains a root: interval arithmetic
        // may have enlarged the image interval p([a, b]) 'artificially' 
        // to contain zero. So, investigate further:

        //-----------------------------------------------
        // special case: infinite interval [-infinity, b]
        //-----------------------------------------------
        if op(ab, 1) = RD_NINF then
           if op(ab, 2) <= -R then
              return(ab) // do not investigate this interval any further
           else
              // split RD_NINF ... b into RD_NINF ... -R and -R ... b
              b:= min(-R, op(ab, 2));
              return(numeric::realroots_(p, x, subsop(ab, 2 = b), eps, R),
                     numeric::realroots_(p, x, subsop(ab, 1 = b), eps, R));
           end_if;
        end_if;
        //-----------------------------------------------
        // special case: infinite interval [a, infinity]
        //-----------------------------------------------
        if op(ab, 2) = RD_INF then
           if op(ab, 1) >= R then
              return(ab) // do not investigate this interval any further
           else
              // split a ... RD_INF into a ... R and R ... RD_INF.
              a:= max(op(ab, 1), R):
              return(numeric::realroots_(p, x, subsop(ab, 2 = a), eps, R),
                     numeric::realroots_(p, x, subsop(ab, 1 = a), eps, R));
           end_if;
        end_if;
 
        //-------------------------------
        // generic case: finite intervals
        //-------------------------------
        if DOM_INTERVAL::width(ab) < eps then return(ab) end_if;
        c := DOM_INTERVAL::center(ab);
        return(numeric::realroots_(p, x, subsop(ab, 2=c), eps, R),
               numeric::realroots_(p, x, subsop(ab, 1=c), eps, R));
   end_if;
end_proc:

// end of file 
