// 
//       
// 

/*++

  numeric::realroots2(p(x), x=a..b, <eps>, <Merge> )

  x    : DOM_IDENT or indexed identifier
  a, b : real numerical constants (not +- infinity)
  eps  : positive numerical constant >= 10^(-DIGITS)
  Merge: do merge resulting intervals whatever the length
         of the result is.

  Returns a list of intervals (=lists), in which real roots of the
  expression p(x) may exist. All intervals are inside [a,b]. All 
  intervals have length < eps. The default is eps = 0.01.

  Warning: if p(x) is identical to zero, then the entire
  search interval is returned. If the result violates
  b-a < eps, then a warning is given;

  Also equations p(x): lhs(x)=rhs(x) are permitted as 1st argument.

  It is garanteed that no real roots exist outside the intervals.
  However, it is not guaranteed that each interval does indeed
  contain a root.

>> numeric::realroots2((x^2 - 1)^2,x=-6..6,0.01);

                      [[-1.0, -1.0], [1.0, 1.0]]

>> numeric::realroots2(5*x-20*x^3+16*x^5, x=-6..6,0.01);



    [0.5859375, 0.58984375], [0.9453125, 0.953125]]

>> numeric::realroots2(sin(x),x=-4..4, 0.1 );

   [[-3.146875, -3.0859375], [-0.1, 0.0], [0.0, 0.1], [3.0859375, 3.146875]]

++*/

numeric::realroots2 := proc(eq,var)
local p, x, a, b, eps, OKTypes, doMerge, Arg;
begin
   if args(0)<2 then error("expecting at least two arguments") end_if;
   if args(0)>4 then error("expecting at most four arguments") end_if;

   if eq::dom::hasProp(Cat::Matrix)=TRUE then
     eq:= expr(eq):
   end_if:
   case domtype(eq)
   of DOM_SET do
   of DOM_ARRAY do
      if nops(eq) <> 1 then
         error("expecting one expression or equation");
      end_if:
      eq:= op(eq, 1);
   end_case;
   p:= subs(eq, float(0)=0, EvalChanges);
   if type(p)="_equal" then p:= op(p,1)-op(p,2) end_if;
   if not testtype(p, Type::Arithmetical) and 
      not domtype(p)=DOM_POLY then
      error("first argument must be an equation".
            "or of domain type DOM_POLY ".
            "or of type 'Type::Arithmetical'")
   end_if;
   if type(var)<>"_equal" or type(op(var,2))<>"_range"
      then error("second argument is not of the form x=a..b"); 
   end_if;
   x:= op(var, 1);
   if numeric::indets(p) minus {x} <> {}
      then error("only univariate expressions allowed") 
   end_if; 
   // Do not (!!!) convert bounds of search interval to floats.
   // Otherwise there is danger of missing roots due to roundoff!!
   // Example: numeric::realroots2(sin(PI*x),x=-1.0..1.0,0.1);
   a:= op(var, [2,1]);
   if domtype(float(a))<>DOM_FLOAT 
      then error("illegal symbolic or complex left bound") 
   end_if;
   b:= op(var, [2,2]);
   if domtype(float(b))<>DOM_FLOAT 
      then error("illegal symbolic or complex right bound") 
   end_if;
   if float(b)<float(a) 
      then error("illegal search interval x=a..b with a>b");
   end_if;

   // search for optional arguments:
   doMerge:= FALSE; // default
   eps:= 0.01;      // default
   for Arg in [args(3..args(0))] do
       if Arg = hold(Merge)
          then doMerge:= TRUE; break;
       end_if;
       eps:= float(Arg);
       if domtype(eps)<>DOM_FLOAT
          then error("illegal precision goal")
       end_if;
       if eps<0.999*10^(-DIGITS) 
          then error("illegal precision goal < 10^(-DIGITS)")
       end_if;
   end_for;
   if iszero(p) then 
      if specfunc::abs( float(b)-float(a))>eps then
         warning("zero function, returning search interval"); 
      end_if;
      return([[float(a),float(b)]])
   end_if;

   //Treat polynomial expressions via polylib::realroots.
   //This is rigorous and also much faster.
   if domtype(p)=DOM_POLY or testtype(p, Type::PolyExpr(x))
   then p:=polylib::realroots(p, eps);
        // intersect isolating intervals from polylib::realroots
        // with the search interval [a,b] of numeric::realroots2
        OKTypes:= {DOM_INT,DOM_RAT,DOM_FLOAT}:
        if {domtype(a)} minus OKTypes<>{} then a:=float(a):end_if;
        if {domtype(b)} minus OKTypes<>{} then b:=float(b):end_if;
        p:=map(p,
               proc(x) begin  //x[1], x[2] are DOM_INT or DOM_RAT
                 if x[1]<a and x[2]< a then return(null()) end_if; 
                 if x[1]<a and x[2]>=a and x[2]<=b then return([a, x[2]]) end_if; 
                 if x[1]<a and x[2]> b then return([a, b]) end_if; 
                 if x[2]>b and x[1]> b then return(null()) end_if; 
                 if x[2]>b and x[1]>=a and x[1]<=b then return([x[1],b]) end_if; 
               //if x[2]>b and x[2]< a then return([a, b]) end_if; 
                 return(x);
               end_proc);
   else //treat non-polynomial expressions:
       p:=[numeric::realroots2_(p, x, a, b, eps)];
   end_if;

// Now merge intervals with common bounds.
// In most cases nothing should happen because most
// subintervals satisfy eps/2< length <=eps. Only a few
// small subintervals should exist from exact treatment
// of boundaries in numeric::realroots2_;
   a:=1; 
   while a<nops(p) do
      for b from a+1 to nops(p) do
         if op(p[b-1],2)<>op(p[b],1) then break end_if
      end_for;
      if doMerge or float(op(p[b-1],2)-op(p[a],1))<=1.0001*eps
      then
         p[a]:=op(p[a],1)..op(p[b-1],2);
         for b from a+1 to b-1 do delete p[a+1] end_for;
      end_if;
      a:=a+1
   end_while;
   map(p, range -> [float(op(range,1)),float(op(range,2))]);
end_proc:

numeric::realroots2_ := proc(p,x,a,b,eps)
   local fa, fb, tmp, c,i;
begin
   userinfo(1, "investigating interval ".expr2text([a,b]));
   // Speed up interval arithmetic by using floats.
   // However, there is danger of loosing roots at the
   // boundaries due to roundoff. 
   fa:= float(a): fb:= float(b):
   // Treat boundaries with exact arithmetic:
   // If a is a singularity, then there is no need for this:
   if traperror((tmp:=float(subs(p,x=a,EvalChanges))))=0 
   then if a<>fa and iszero(tmp) then 
         // Note that float boundaries fa+eps are ok here,
         // because a root interval a..fa+eps is returned
         // to the user. So loosing a root a+eps in fa+eps..b
         // by roundoff is acceptable.
         if fa+eps>=fb 
            then return(fa..fb)
            else return(a..fa+eps, numeric::realroots2_(p,x,fa+eps,b,eps));
         end_if;
      end_if;
   end_if;
   // If b is a singularity, then there is no need for this:
   if traperror((tmp:=float(subs(p,x=b,EvalChanges))))=0 
   then if b<>fb and iszero(float(subs(p,x=b,EvalChanges))) then 
         // Note that float boundaries fb-eps are ok here,
         // because a root interval fb-eps..b is returned
         // to the user. So loosing a root b-eps in a..fb-eps
         // by roundoff is acceptable.
         if fb-eps<=fa
            then return(fa..fb)
            else return(numeric::realroots2_(p,x,a,fb-eps,eps),fb-eps..b);
         end_if;
      end_if;
    end_if;
   // Now switch to float arithmetic
   i:=eval(subs(float(p),x=Dom::Interval([fa,fb]), Unsimplified));

/*  -- old version of test whether the interval i contains 0 -----

   // Note that Dom::Interval([a,b]) can produce various
   // output types. In principle, sign should be defined
   // for all output types. If not, then need to fix this
   // at the appropriate point of the library, not here!!
   // So the following code is obsolete:
   //
   //case domtype(i)
   //  of Dom::Interval do i:= sign(i);
   //                      // For Dom::Interval sign(i) can bei 
   //                      // either 1, -1, {1, 0}, {-1,0}, {1,-1,0}
   //                      if domtype(i)<>DOM_SET 
   //                        then i:= {i};
   //                      end_if;
   //                      break;
   // // i can be Dom::Interval, but also {a} (for a=b):
   //  of DOM_SET do i:= map(i, sign); break;
   //  otherwise error("something is wrong with Dom::Interval");
   //end_case;
   //
   // if i minus {0,-1,1} <> {} then error("cannot determine sign") end_if;
   // If i={1} or i={-1} then there is definitely no root in the interval:

   i:= sign(i);

   if type(i) = DOM_SET and not contains(i, 0) then
      return(null())
   end_if;

   if type(i) = "sign" then
      error("Interval arithmetic failed"):
   end_if;

   if not (0 in i) then
      return(null());
   end_if;
*/

   //  new version of test whether the interval i contains 0
   case is(0 in i)
   of FALSE do return(null());
   of UNKNOWN do 
      error("interval arithmetic failed"):
   end_case;

   //----------------------------------------------------------------
   // If 0 in i, we do not really know whether to accept this interval
   // or not: interval arithmetic may have overestimated the interval 
   // to contain zero. So, investigate further:
   //----------------------------------------------------------------
   if abs(float(b-a))<eps then return(a..b) end_if;
   c:=(a+b)/2; 
   return(numeric::realroots2_(p,x,a,c,eps),
          numeric::realroots2_(p,x,c,b,eps));
end_proc:

// end of file 
