//------   file  rotationMatrix.mu  28.1.08   -------------------
/*  Help file:

 rotationMatrix - orthogonal matrix describing the rotation around an axis

 Call:

 numeric::rotationMatrix(angle, axis, <, Symbolic> <, ReturnType = t> <, NoWarning>)
                                                                   
 Parameter:
  angle -- an arithmetical expression
  axis  -- a list or a matrix (DOM_ARRAY, DOM_HFARRAY or category Cat::Matrix)
  t     -- either DOM_ARRAY, DOM_HFARRAY, Dom::DenseMatrix() or Dom::Matrix()
                                                                   
*/
//-----------------------------------------------------------------
numeric::rotationMatrix:=
proc(angle, axis)
local symbolic, dowarn, i, returnType, 
      x, y, z, n, s, c, t, Q;
begin

  if args(0) < 2 then error("expecting at least two arguments") end_if;

  //----------------------------------------------
  // Check the axis and set default return types
  //----------------------------------------------
  case domtype(axis) 
  of DOM_LIST do
     returnType:= Dom::Matrix(): break;
  of DOM_ARRAY do
  of DOM_HFARRAY do
     returnType:= axis::dom:
     axis:= [op(axis)];
     break;
  otherwise
     if axis::dom::hasProp(Cat::Matrix)=TRUE then
        returnType:= Dom::Matrix();
        if axis::dom::constructor = Dom::DenseMatrix then
           returnType:= Dom::DenseMatrix();
        end_if;
        axis:= [op(axis)];
     else
        error("expecting the rotation axis as a list, an array, ".
              "an hfarray, or a matrix of category 'Cat::Matrix'"):
     end_if:
  end_case;

  //---------------------------------------------
  // Check the options
  //---------------------------------------------
  symbolic:= FALSE; // default
  dowarn:= TRUE;    // default //dowarn is presently not used
  for i from 3 to args(0) do
      case args(i)
      of Symbolic do symbolic:= TRUE;
                     break;
      of NoWarning do
                     dowarn:= FALSE;
                     break;
      otherwise
         if type(args(i)) = "_equal" and
            lhs(args(i)) = ReturnType then
              returnType:= rhs(args(i));
              if {returnType} minus {DOM_ARRAY, DOM_HFARRAY, Dom::DenseMatrix(), Dom::Matrix()}
                 <> {} then
                 error("illegal return type ".expr2text(args(i)).
                       " specified. Choose between DOM_ARRAY, DOM_HFARRAY, ".
                       "Dom::Matrix(), or Dom::DenseMatrix()"):
              end_if;
         else error("unknown option ".expr2text(args(i)));
         end_if;
      end_case;
  end_for:

  //---------------------------------------------
  // Do the work
  //---------------------------------------------

  if symbolic then
     [x, y, z]:= axis;
     n:= sqrt(x^2 + y^2 + z^2);
  else
     [x, y, z]:= float(axis);
     n:= specfunc::sqrt(x^2+y^2+z^2);
  end_if:

  if iszero(n) then
     return("3D rotation requires a non-zero axis");
  end_if;

  // normalize the axis [x,y,z]
  x:= x/n; 
  y:= y/n; 
  z:= z/n;

  // the rotation matrix, as found in graphics gems, p. 466:
  if symbolic then 
     s:= sin(angle);
     c:= cos(angle);
     t:= 1 - c; // = 2*sin(angle/2)^2
  else
     s:= specfunc::sin(angle);
     c:= specfunc::cos(angle);
     t:= 1 - c; // = 2*specfunc::sin(angle/2)^2;
  end_if;

  Q := array(1..3, 1..3, [[t*x^2 + c,   t*x*y - s*z, t*x*z + s*y],
                          [t*x*y + s*z, t*y^2 + c,   t*y*z - s*x],
                          [t*x*z - s*y, t*y*z + s*x, t*z^2 + c]]);

  case returnType
    // mind the ordering: process DOM_{HF}ARRAY first to
    // avoid unnecessary loading of the DOMAIN package
  of DOM_ARRAY do
     return(Q);
  of DOM_HFARRAY do
     return(hfarray(1..3, 1..3, [op(Q)]));
  of Dom::Matrix() do
     return(matrix(3, 3, Q));
  of Dom::DenseMatrix() do
     return(returnType::create(Q));
  otherwise
       error("unexpected return type"):
  end_case;

end_proc:
