/*
numeric::solve -- numercal roots of equations

numeric::solve is the interface function for the float attribute of solve,
i.e., 
>> float(hold(solve)(eqs,vars))
calls
>> numeric::solve(eqs,vars);

Call:  numeric::solve(eq);
       numeric::solve(eq, x);
       numeric::solve(eq, x=a);    // starting point for numerical search
       numeric::solve(eq, x=a..b); // search interval
       numeric::solve(eqs);
       numeric::solve(eqs, vars);
       numeric::solve(eqs, {x=a, y=a..b, z, u=.., ..});
       numeric::solve(eqs, [x=a, y=a..b, z, u=.., ..]);
       numeric::solve(eqs, vars, RestrictedSearch, Random, NoWarning, MultiSolutions)

Parameters:  eq   -- a single univariate equation
                     or a univariate expression
                     or a univariate DOM_POLY polynomial
                     or a matrix with 1 operand
                     or an array with 1 operand
             eqs  -- a set or list of multivariate equations,
                     expressions or DOM_POLY polynomials
                     or a matrix or an array
             vars -- a set or list of indeterminates

Options: Multiple   -- only to be used for polynomial equations
         RestrictedSearch -- only to be used for non-polynomial equations
         Random           -- only to be used for non-polynomial equations
         NoWarning -- only has an effect for polynomial equations
        
         MultiSolutions,
         RestrictedSearch and Random are passed to numeric::fsolve 
         (see numeric::fsolve for details)

Return value: a set of numerical roots

Details: numeric::solve checks the type of the equations.

         For a univariate polynomial equation it calls numeric::polyroots.

         For a multivariate polynomial system it calls numeric::polysysroots.

         Rational equations/systems are treated like polynomial equations
         (the numerator(s) is(are) extracted).

         For a non-polynomial (system of) equation(s) it calls
         numeric::fsolve. In this case only one solution is returned.

         Specification of starting points and/or search intervals
         only has an effect for non-polynomial equations.

         If the equations contain inequalities such as
         tan(x) < 0, then an unevaluated call of solve is returned. 

Examples:
>> numeric::solve(x^2=PI), numeric::solve(x^2-PI),
   numeric::solve(poly(x^2-PI, [x])),
   numeric::solve(x^2=PI, x), numeric::solve(x^2-PI, x),
   numeric::solve(poly(x^2-PI, [x]), x),
   numeric::solve(x^2=PI, x=1), numeric::solve(x^2-PI, x=1),
   numeric::solve(poly(x^2-PI, [x]), x=1),
   numeric::solve(x^2=PI, x=0..1), numeric::solve(x^2-PI, x=0..1),
   numeric::solve(poly(x^2-PI, [x]), x=0..1),
   numeric::solve({x^2=PI}, [x=0]), numeric::solve(x^2-PI, {x=0..1});

{[x = -1.77245385], [x = 1.77245385]},
   {[x = -1.77245385], [x = 1.77245385]},
   {[x = -1.77245385], [x = 1.77245385]}, {-1.77245385, 1.77245385},
   {-1.77245385, 1.77245385}, {-1.77245385, 1.77245385},
   {-1.77245385, 1.77245385}, {-1.77245385, 1.77245385},
   {-1.77245385, 1.77245385}, {-1.77245385, 1.77245385},
   {-1.77245385, 1.77245385}, {-1.77245385, 1.77245385},
   {[x = -1.77245385], [x = 1.77245385]},
   {[x = -1.77245385], [x = 1.77245385]}

>> DIGITS:=3:
   numeric::solve({x^2+y+1, y^2+x+1=2}),
   numeric::solve({x^2+y+1, y^2+x+1=2}, {x,y}),
   numeric::solve({x^2+y+1, y^2+x+1=2}, [y,x]),
   numeric::solve({x^2+y+1, y^2+x+1=2}, [x,y]),
   numeric::solve({x^2+y+1, y^2+x+1=2}, [x=2,y=3..4]);

{[y = -1.0, x = 0.0], [y = 1.1 + 0.665 I, x = 0.226 - 1.46 I],
   [y = 1.1 - 0.665 I, x = 0.226 + 1.46 I], [y = -1.2, x = -0.453]}, 
   { [y = -1.0, x = 0.0], [y = 1.1 + 0.665 I, x = 0.226 - 1.46 I],
     [y = 1.1 - 0.665 I, x = 0.226 + 1.46 I], [y = -1.2, x = -0.453]}, 
   { [y = -1.0, x = 0.0], [y = 1.1 + 0.665 I, x = 0.226 - 1.46 I],
     [y = 1.1 - 0.665 I, x = 0.226 + 1.46 I], [y = -1.2, x = -0.453]},
   { [x = 0.0, y = -1.0], [x = 0.226 - 1.46 I, y = 1.1 + 0.665 I],
     [x = - 0.453 + 0.0 I, y = - 1.2 + 1.16e-10 I],
     [x = 0.226 + 1.46 I, y = 1.1 - 0.665 I]},
   { [x = 0.0, y = -1.0], [x = 0.226 - 1.46 I, y = 1.1 + 0.665 I],
     [x = - 0.453 + 0.0 I, y = - 1.2 + 1.16e-10 I],
     [x = 0.226 + 1.46 I, y = 1.1 - 0.665 I]}

>> numeric::solve(x^2 = cos(x) ),
   numeric::solve(x^2 = cos(x) , x),
   numeric::solve(x^2 = cos(x) , x=1),
   numeric::solve(x^2 = cos(x) , x=-1..0),
   numeric::solve({x^2 = cos(x)} , x),
   numeric::solve({x^2 = cos(x)} , [x=0..3]);

   {[x = -0.825]}, {-0.824}, {0.824}, {-0.825}, {0.825}, {0.824}

>> numeric::solve([x^2 = cos(y), y^2 = cos(x)]),
   numeric::solve([x^2 = cos(y), y^2 = cos(x)], [x,y]),
   numeric::solve([x^2 = cos(y), y^2 = cos(x)], [x=-1,y=0..1]),
   numeric::solve([x^2 = cos(y), y^2 = cos(x)], {x=-1,y=0..1});

   {[y = -0.824, x = -0.825]}, {[x = 0.824, y = -0.824]},
   {[x = -0.824, y = 0.824]}, {[x = -0.824, y = 0.824]}

>> numeric::solve([x^2 = 1], Multiple);

   {[[x = -1.0], 1], [[x = 1.0], 1]}

>> numeric::solve(x^2 = sin(x), x= 0.1..5, RestrictedSearch);
                            {0.8767262154}
// deterministic search in the following 2 calls
>> numeric::solve(x^2 = exp(-x), [x= 0..1, y], RestrictedSearch);
             {[x = 0.7034674235, y = 100.5430479]}
>> numeric::solve(x^2 = exp(-x), [x= 0..1, y], RestrictedSearch);
             {[x = 0.7034674235, y = 100.5430479]}
// random search in the following 3 calls
>> numeric::solve(x^2 = exp(-x), [x= 0..1, y], RestrictedSearch, Random);
             {[x = 0.7034674225, y = 23.21783262]}
>> numeric::solve(x^2 = exp(-x), [x= 0..1, y], RestrictedSearch, Random);
             {[x = 0.7034674228, y = 10.596933]}
>> numeric::solve(x^2 = exp(-x), [x= 0..1, y], RestrictedSearch, Random);
             {[x = 0.7034674225, y = -0.963182675]}



*/

numeric::solve:= proc(eqs)
   local orig_args,
         var, vars, uvars, user_vars, ivar, ivars,
         i, nonlinear, is_rational, is_polynomial,
         is_scalar, is_scalar1,
         fsolve_options, polyroots_options, 
         polysysroots_options, allrealroots_options,
         use_multiple, use_allrealroots, Arg,
         eqs_are_numeric, s, r;
 begin
   userinfo(10, "numeric::solve called with args ".expr2text(args()));
   if args(0)<1 then error("expecting at least one argument") end_if;
   if args(0)>4 then error("expecting at most four arguments") end_if;

   orig_args:= [args(1..args(0))];

   if eqs::dom::hasProp(Cat::Matrix)=TRUE then
     eqs:= expr(eqs):
   end_if:
   if domtype(eqs) = DOM_ARRAY then
     eqs:= [op(eqs)];
   end_if:

   // determine all symbolic objects before eliminating trivial
   // equations, because this is, what the symbolic solver does, too.
   // vars:= [op(numeric::indets(eqs))];
   vars:= DOM_SET::sort(numeric::indets(eqs));
   // convert equation(s) to list of equation(s)
   is_scalar := FALSE;
   case domtype(eqs) 
   of DOM_SET do eqs:= [op(eqs)];
                 break;
   of DOM_LIST do break;
   otherwise is_scalar:= TRUE;
   end_case;

   if domtype(eqs)<>DOM_LIST then eqs:= [eqs]; end_if;
   // convert equations to expressions:
   eqs:= subs(eqs, float(0)=0);
   eqs:= map(eqs, proc(eq) begin 
                   case type(eq)
                   of "_equal" do return(op(eq,1)-op(eq,2));
                   of "_less" do
                   of "_leequal" do return(FAIL);
                   otherwise return(eq) 
                   end_case;
                   end_proc);
   // eliminate trivial equations
   if has(eqs, FAIL) then
      //return(solve(op(map(orig_args, float))));
      return(hold(solve)(op(map(orig_args, float))));
   end_if:
   eqs:= select(eqs, _not@iszero);

   user_vars:= FALSE;  // did the user specify vars ?
   uvars:= null():     // the vars specified by the user
   fsolve_options:= null():
   polyroots_options:= null();
   polysysroots_options:= null();
   use_multiple:= FALSE;
   use_allrealroots:= FALSE;
   allrealroots_options:= null();
   for Arg in [args(2..args(0))] do
       if Arg = Multiple then
                 use_multiple:= TRUE;
                 next;
       end_if;
       if Arg= RestrictedSearch then
                 fsolve_options:= fsolve_options, Arg;
                 next;
       end_if;
       if Arg= UnrestrictedSearch then
                 fsolve_options:= fsolve_options, Arg;
                 next;
       end_if;
       if Arg= MultiSolutions then
                 fsolve_options:= fsolve_options, Arg;
                 next;
       end_if;
       if Arg = Random then
                 fsolve_options:= fsolve_options, Arg;
                 next;
       end_if;
       if Arg = FixedPrecision then
                 polyroots_options:= polyroots_options, Arg;
                 next;
       end_if;
       if Arg = SquareFree then
                 polyroots_options:= polyroots_options, Arg;
                 next;
       end_if;
       if Arg = Factor then
                 polyroots_options:= polyroots_options, Arg;
                 next;
       end_if;
       if Arg = NoWarning then
                 polyroots_options:= polyroots_options, Arg;
                 polysysroots_options:= polysysroots_options, Arg;
                 allrealroots_options:= allrealroots_options, Arg;
                 next;
       end_if;
       if Arg = AllRealRoots then
                 use_allrealroots:= TRUE;
                 next;
       end_if;
       if has(Arg, VectorFormat) or has(Arg, Real) then
                 // May turn up in a symbolic return of a symbolic
                 // solve command. Ignore this.
                 next;
       end_if;
       if user_vars then  // uvars have been set before
                 error("unknown option");
       end_if:
       user_vars:= TRUE;
       uvars:= Arg;
   end_for;

   eqs_are_numeric:= TRUE: // are there symbolic parameters?
                           // Can handle this for polynomial systems!

   // convert unknowns and search ranges to list of unknowns:
   is_scalar1 := FALSE;
   if user_vars then 
     case domtype(uvars)
     of DOM_SET do uvars := DOM_SET::sort(uvars);
                   break;
     of DOM_LIST do break;
     otherwise //univariate case
               uvars:= [uvars];
               is_scalar1:= TRUE;
     end_case;
     if domtype(uvars)=DOM_SET  then uvars:= DOM_SET::sort(uvars); end_if;
     if domtype(uvars)<>DOM_LIST then uvars:= DOM_SET::sort(uvars); end_if;
     // strip off starting points or search ranges for numeric::fsolve
     s:= map(uvars, proc(eq) begin if type(eq)= "_equal" then 
             op(eq,1) else eq end_if; end_proc);
     if {op(vars)} minus {op(s)} <> {} then
        eqs_are_numeric:= FALSE;
     end_if;
     vars:= s;
   end_if;

   // now vars=[x1,x2,..], uvars=[x1=a1,x2=a1..b1,..]

   if vars=[] then error("no indeterminate(s)"); end_if;

   // is_scalar = TRUE  -> output is {value}
   // is_scalar = FALSE -> output is {[x=value]}

   is_scalar:= user_vars /*and is_scalar*/ and is_scalar1;

   if nops(eqs)=0 then
      if is_scalar
         then return(C_);
         else // return({[(var = var) $ var in vars]});
              return({[]});
      end_if;
   end_if;

   //------- call the appropriate method ----------------------------
   // convert a rational system to a polynomial system
   is_polynomial:= TRUE;
   if has(map(eqs,testtype,Type::PolyExpr(vars)), FALSE) then
      is_rational:= TRUE;
      for var in vars do
        if has(map({op(eqs)}, testtype, Type::RatExpr(var)), FALSE) then
           is_rational:= FALSE;
           break;
        end_if;
      end_for:
      if is_rational then
        userinfo(1, "rational equation(s), extracting numerator(s)");
        eqs:= map(eqs, numer);
        // need to double check the new equations
        if has(map(eqs,testtype,Type::PolyExpr(vars)), FALSE) then
           is_polynomial:= FALSE;
        end_if;
      else // not rational -> not polynomial (without need to double check)
           is_polynomial:= FALSE;
      end_if:
   end_if;

   if use_allrealroots then
        userinfo(1, "heuristic search for all real solutions");
        if nops(eqs) > 1 then
           error("expecting only one equation with option 'AllRealRoots'");
        end_if;
        if user_vars then
           if  type(uvars[1]) = "_equal" then
               s:= numeric::allRealRoots(eqs[1], uvars[1], allrealroots_options);
           else
               s:= numeric::allRealRoots(eqs[1], allrealroots_options);
           end_if;
           s:= op(s);
        else
           s:= numeric::allRealRoots(eqs[1], allrealroots_options);
           s:= [vars[1] = r] $ r in s;
        end_if;
   elif not is_polynomial then
        userinfo(1, "non-polynomial system, searching for one solution");
        userinfo(1, "calling numeric::fsolve(".expr2text(args()).")"); 
        if nops(eqs)>nops(vars) then 
           error("can handle case 'more equations than unknowns' only for polynomial systems")
        end_if;
        if not eqs_are_numeric then
           error("symbolic parameters not allowed in non-polynomial equations");
        end_if;
        if use_multiple then
           error("option 'Multiple' not allowed in non-polynomial equations")
        end_if;
        if user_vars 
          then s:=numeric::fsolve(eqs, uvars, fsolve_options);
               // for univariate problems: s=[x=0.123] -> s = 0.123
               if is_scalar then 
                  if type(s) =  "_exprseq" then
                       // with the option 'MultiSolutions', numeric::fsolve returns
                       // the sequence [x = solution1], [x = solution2], ...
                       s:= op(s, [i, 1, 2]) $ i = 1..nops(s);
                  else s:= op(s, [1,2]); 
                  end_if;
               end_if;
          else s:=numeric::fsolve(eqs, vars, fsolve_options);
        end_if;
   else userinfo(1, "polynomial system, searching for all solutions");
        if nops(vars)=1 and nops(eqs)=1 // univariate problem
        and eqs_are_numeric
          then userinfo(1, "calling numeric::polyroots(".expr2text(eqs[1], polyroots_options).")"); 
               if is_scalar
                  then s:=op(numeric::polyroots(eqs[1], polyroots_options));
                  else // polyroots returns [r1,r2,...], convert this 
                       // to  s:= [vars[1]=r1], [vars[1]=r2],...
                       s:=op(map(numeric::polyroots(eqs[1], polyroots_options), r->[vars[1]=r]));
               end_if;
          else // we are dealing with a polynomial system.
               // If it is linear, call numeric::linsolve.
               // Otherwise, call numeric::polysysroots:
           
               nonlinear:= FALSE;
               for i from 1 to nops(eqs) do
                 if nonlinear then break end_if;
                 // important for sparse systems: only investigate
                 // linearity w.r.t. the variables ivars that are
                 // actually contained in each equation. Otherwise,
                 // checking diff(eqs[i], var) would be too expensive: 
                 ivars:= numeric::indets(eqs[i]) intersect {op(vars)};
                 for ivar in ivars do
                     if numeric::indets(diff(eqs[i], ivar)) intersect ivars <> {}
                     then nonlinear:= TRUE:
                          break;
                     end_if;
                  end_for:
                end_for:

                if nonlinear then
                     userinfo(1, "calling numeric::polysysroots"); 
                     s:= numeric::polysysroots(eqs, vars, polysysroots_options);

                     // special case: polysysroots may return a RootOf object,
                     // if a polynomial with symbolic paramaters is to be
                     // be solved for a single variable.
                     // In this case, return the RootOf object directly
                     // without the processing below:

                     if domtype(s) <> DOM_SET or map(s, domtype) <> {DOM_LIST} then 
                        return(s) 
                     end_if;

                     // Otherwise, polysysroots returns {[x = ...], [x = ...]}.
                     // To process this below, we need s = [x = ...], [x = ...], ...
   
                     if is_scalar then 
                       // s = {[x = sol1], [x = sol2], ..} -> {sol1, sol2, ..}
                       s:= map(s, op, [1,2]);
                     end_if;
                     s:= op(s);
 
                else userinfo(1, "calling numeric::linsolve"); 
                     // map float to eqs and use NoWarning, because
                     // numeric::linsolve may internally switch to
                     // its symbolic mode if there are symbolic parameters
                     s:= numeric::linsolve(map(eqs, float), vars, NoWarning);
                     // solve(x = 1) returns {[x = 1]}, but
                     // solve(x = 1, x) returns {1}:
                     if is_scalar then s:= op(s, [1,2]); end_if;
                end_if;
        end_if;
   end_if;

   if has([s], FAIL) then
      // if is_scalar 
      // then return({})
      // else return({[]})
      // end_if;
      return({});
   end_if;
   //------------------- post processing -------------------
   if has([args()], hold(Multiple))
           then return(Dom::Multiset(s));
           else return({s});
   end_if;
end_proc:

// end of file 
