
/*-------------------------------------------------------

numlib::factorGaussInt(n)

n - complex number of the form a+b*I, where a, b are integers
    (b=0 allowed)

returns the factorization of the Gaussian integer n as a list
[u, p1, a1, p2, a2, ..., pk, ak]
such that
- n = u* p1^a1 * .... * pk^ak
- u is a Gaussian unit (one of 1, -1, I, -I)
- each p_j is a Gaussian prime with both real and imag. part being non-negative
- each a_j is a positive integer

--------------------------------------------------------*/


numlib::factorGaussInt:=
proc(n)
  local a, b,
  p: DOM_INT,
  gc: DOM_INT,
  Norm: DOM_INT,
  gcfactors: DOM_LIST,
  gunit,
  l: DOM_LIST,
  i: DOM_INT,
  Normfactors: DOM_LIST,
  factortab: DOM_TABLE;
  
begin
  if args(0) <> 1 then
    error("wrong number of arguments");
  end_if;
  case domtype(n)
    of DOM_COMPLEX do
      a:= op(n, 1);
      b:= op(n, 2);
      if domtype(a) <> DOM_INT or domtype(b) <> DOM_INT then
        error("Illegal argument")
      end_if;
      break
    of DOM_INT do
      a:= n;
      b:= 0;
      break
    otherwise
      error("Illegal argument")
  end_case;

  gc:= igcd(a,b);
  assert(gc > 0);
  a:= a/gc;
  b:= b/gc;
  n:= n/gc;
  
  factortab:= table();  
  gcfactors:= stdlib::ifactor(gc);
  // there are three kinds of prime factors of gc:
  // primes p= 4k+1 are of the form x^2 + y^2 and thus reducible in Z[i]
  // primes p= 4k+3 remain prime
  // p=2 is (1+I)^2 up to a unit factor
  for i from 2 to nops(gcfactors) step 2 do
    case modp((p:= gcfactors[i]), 4)
      of 1 do
        // solve x^2 + y^2 = p; then p= (x+Iy)(x-Iy)
        l:= op(numlib::cornacchia(1, 1, p), 1);
        assert(domtype(l) = DOM_LIST);
        factortab[l[1] + l[2]*I]:= gcfactors[i+1];
        factortab[l[2] + l[1]*I]:= gcfactors[i+1];
        break
      of 2 do
        factortab[1+I] := 2*gcfactors[i+1];
        break
      of 3 do
        factortab[gcfactors[i]]:= gcfactors[i+1]
    end_case;
  end_for;

  Norm := a^2 + b^2;
  Normfactors:= stdlib::ifactor(Norm);
  assert(Normfactors[1] = 1);


  // a+b*i has no integer factor:
  // all of its prime factors must be (1+I) or divisors of a rational
  // prime of the form 4k+1
  
  for i from 2 to nops(Normfactors) step 2 do
    p:= Normfactors[i];
    // no Gaussian integer can have a norm congruent 3 (mod 4)
    assert(p mod 4 <> 3); 
    if p = 2 then
      // the multiplicity of 1+I must be 1, since otherwise
      // 2 would divide a+b*I
      assert(Normfactors[i+1] = 1);
      if contains(factortab, 1+I) then
        factortab[1+I]:= factortab[1+I] + 1
      else
        factortab[1+I]:= 1
      end_if
    else
      l:= op(numlib::cornacchia(1, 1, p), 1);
      assert(domtype(l) = DOM_LIST);
      // we know that a+b*I must be a multiple of l[1] + l[2]*I or
      // of l[2] + l[1] * I, but do not know which is the case,
      // so we have to try
      // But we know that their product p*I does not divide a+b*I
      
      if map([Re, Im](op(n/(l[1] + l[2]*I), 1)), domtype) = [DOM_INT $2] then
        if contains(factortab, l[1] + l[2]*I) then
          factortab[l[1] + l[2]*I] := factortab[l[1] + l[2]*I] + Normfactors[i+1];
        else  
          factortab[l[1] + l[2]*I] := Normfactors[i+1];
        end_if
      else
        if contains(factortab, l[2] + l[1]*I) then
          factortab[l[2] + l[1]*I] := factortab[l[2] + l[1]*I] + Normfactors[i+1];
        else
          factortab[l[2] + l[1]*I] := Normfactors[i+1];
        end_if
      end_if
    end_if;
  end_for;

  // compute the correct unit. 
  l:= [op(factortab)];
  gunit:= n*gc/_mult(op(l[i], 1)^op(l[i],2) $i=1..nops(l));
  [gunit].map(l, op);

end_proc:

numlib::factorGaussInt(0):= [0]: