
/** numlib::invphi - computes the inverse of the Euler phi function

  Call:  numlib::invphi(n : Type::PosInt)

  Return value:  {i | numlib::phi(i) = n}

  About the algorithm:

    phi(num) = phi( p1^n1 * p2^n2 * ... * pm^nm)
             = p1^n1 * ... * pm^nm * (1 - 1/p1) * ... (1 - 1/pm)
             = product(pi^(ni-1) * (pi-1), i=1..m)               (*)
             = n   (the given number)

  Observation:    Apart from (2-1) all pi-1 are even.  

  Step 1: factor n and find all even divisors j of n;
          these are the candidates for the pi-1

  Step 2: select all j+1(=pi) which are prime

  Step 3: build all factors f[i,k] = pi^k * (pi-1) with 
          f[i,k] | n  (this is done by allFactors)

  Step 4: find all products of the f[i,k]'s with value n;
          these are matchings of equation (*)

  Step 5: build the products of the powers pi^(k+1) corresponding
          to the factors found in the matching in step 4;
          these products are the numbers num with phi(num)=n
*/

numlib::invphi :=
proc(n : Type::PosInt) : DOM_LIST
  local Factors, compose, allFactors, s1;
begin
  if n=1 then
    [1, 2]
  elif n mod 2 = 1 then
    []
  else

    /*
     *  allFactors:   find all divisors p^j*(p-1) of n  with p in l
     *  return value: a sorted list of lists of the type
     *                [p^j*(p-1), p, p^(j+1)]  with  p^j*(p-1) | n
     */
    allFactors := 
      proc(n : Type::PosInt, l : Type::ListOf(Type::Prime)) 
                                              : Type::ListOf(DOM_LIST)
        local i, j, f1, res, power;
      begin
        res := [];
        for i from 1 to nops(l) do
          j := if i = 1 and l[i] = 2 then 1 else 0 end_if;
          power := l[i]^j;
          f1 := (l[i]-1)*power;
          power := power * l[i];
          /* now is f1 = (p-1)*p^j and power = p^(j+1) */

          while n mod f1  = 0 do
            res := res . [[f1, l[i], power]];
            j := j + 1;
            f1 := f1*l[i];
            power := power*l[i];
          end_while;
        end_for;
        sort(res, (i,j) -> i[1] > j[1]);
      end_proc;

    /*
     *  compose:  finds all products of numbers from l which equal
     *            to n and returns a list of the corresponding 
     *            products of the p^(j+1)'s
     *            The list exclude contains all prime numbers of
     *            which a factor p^j*(p-1) has already been processed.
     *            All other powers of p are excluded.
     *
     *  return value: a list of positive integers
     */
    compose := 
      proc(n : Type::PosInt, 
           l : Type::ListOf(DOM_LIST),
           exclude : Type::ListOf(DOM_INT)) : DOM_LIST
        local i, j, f1, res, tmp;
      begin
        if l = [] then return([]) else
          res := [];
          for i from 1 to nops(l) do
            j := l[i];
            f1 := j[1];

            // since f1 is no divisor of n we can skip this value;
            // if we have already processed a value (p^j)*(p-1)
            // we have to skip all following powers of p
            if f1 > n or n mod f1 <> 0 or 
               contains(exclude, j[2]) <> 0 then next end_if;
 
            if n = f1 then
              // Bingo, we have a match
              res := res . [j[3]];
            else
              // recursive call with all bigger factors;
              // we have to exclude the current p
              tmp := n/f1;
              // since all factors contain a factor prime-1, prime > 2
              // they are all even -> forget odd remaining values
              if tmp mod 2 <> 1 then
                tmp := compose(tmp, [op(l, i+1..nops(l))], 
                               exclude.[j[2]]);
                if tmp <> [] then 
                  res := res. zip([j[3] $ nops(tmp)], 
                                  tmp, _mult);
                end_if;
              end_if;
            end_if;
          end_for;
        end_if;
        res
      end_proc;

    // all primes i with i-1 | n in a sorted list
    Factors := [2] . select(map(select(numlib::divisors(n), 
                                       i -> i mod 2 = 0), 
                                _plus, 1), 
                            isprime);

    // ... and now the heavy part 
    s1 := compose(n, allFactors(n, Factors), []);

    // the above algorithm does not find values with 
    // n = 2 * i : Type::Odd; we add them now, since
    // phi(2 * i : Type::Odd) = phi(i)
    s1 := s1 . map(select(s1, x -> x mod 2 = 1), _mult, 2);
    sort(s1)
  end_if;
end_proc:
