/* 
   ode::exact_first_order(eq,y,x) 
 

   DESCRIPTION: 

   Input: eq is a first-order expression in y(x).
   Output: FAIL or a set of solutions.


   REFERENCES: [1] D. Zwillinger: "Handbook of Differential Equations", 
                   Section 62, pp. 258

   DETAILS: 
  
   ode::exact_first_order(eq,y,x) tries to solve the first-order 
   equation eq=0 with respect to y(x) by using the method described
   in [1]. 

   Namely, if the equation has the form 
    
                        y'(x) = N(x,y)/M(x,y) 

   and 
                        diff(M,x)+diff(N,y) = 0, 

   it is said to be an exact ODE.

 
   EXAMPLES:
 
     > ode::solve((exp(y(x))+2*x*y(x)+1)*diff(y(x),x)-(3*x^2-y(x)^2-7),y(x));
     > ode::exact_first_order(p^2*(diff(x(p),p)+2*x/p-3*a*p),x,p);
*/

ode::exact_first_order:= proc(eq,y,x,solveOptions,odeOptions) 
  local N,M,yp,g,f,intOptions,optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  userinfo(2,"trying to recognize an exact first order equation");
  yp:= genident();
  g:= genident();
  f:= genident();
  eq:= subs(ode::normal(eq,Expand=FALSE),diff(y(x),x)=yp,y(x)=y,EvalChanges);
  if testtype(eq,Type::PolyExpr(yp)) then
    eq:= poly(eq,[yp]);
    if degree(eq)=1 then
         /* do not call normal to get N and M, not to destroy exact equations
            when called from ode::int_factors */
      N:= -coeff(eq,0);
      M:= coeff(eq,1);
      // Fix for MMA Testsuite
      if testeq(diff(M,x)+diff(N,y),0,Steps=0,NumberOfRandomTests=5) = FALSE or 
         not iszero(ode::normal(simplify(diff(M,x)+diff(N,y),optIgnoreAnalyticConstraints))) then
        [N,M]:= ode::normal(-coeff(eq,0)/coeff(eq,1),List);
      end_if;
      // End of fix for MMA Testsuite
      if ode::odeIszero(diff(M,x)+diff(N,y)) and 
         iszero(ode::normal(simplify(diff(M,x)+diff(N,y), optIgnoreAnalyticConstraints))) then
        userinfo(1,"exact first order equation");
        N:= -int(N,x,intOptions);
        M:= expand(int(M,y,intOptions),optIgnoreAnalyticConstraints);
        // the solution is N+f(y)=M+g(x)
        userinfo(3,"solution is",N+f(y)=M+g(x));
        f:= select((if type(M)="_plus" then [op(M)] else [M] end_if),
                    proc()
                    begin
                      has(args(1),y) and not has(args(1),x)
                    end_proc );
        N:= N+op(f);
        M:= combine(M,exp,optIgnoreAnalyticConstraints); 
        N:= combine(N,exp,optIgnoreAnalyticConstraints);
        if has(ode::normal(N-M),y) then
          return(FAIL)
        end_if;
        eq:= solve(N+genident("C"),y,op(solveOptions));
        return(eq);
      end_if
    end_if
  end_if;
  if contains(odeOptions, Type = ExactFirstOrder) then 
    ode::odeWarning("cannot detect exact first order equation");
  end_if;  
  
  return(FAIL);
end_proc:

