/* 
   ode::exact_second_order(eq,y,x) 


   DESCRIPTION: 

   Input: eq is a second-order equation in y(x).
   Output: FAIL or a list of solutions.


   REFERENCES: [1] D. Zwillinger: "Handbook of Differential Equations", 
                   Section 63, pp. 260


   DETAILS:  
   ode::exact_second_order(eq,y,x) tries to solve the second order ODE eq=0
   with respect to y(x) using the method described in [1].

   Namely, if eq is of the form 

                         f(x,y,y') y'' + g(x,y,y') = 0, 

   with the two conditions

          diff(f,x,x)+2*p*diff(f,x,y)+p^2*diff(f,y,y) =
          diff(g,x,p)+p*diff(g,y,p)-diff(g,y)

          diff(f,x,p)+p*diff(f,y,p)+2*diff(f,y) = diff(g,p,p)

   then eq is the total differential of some function.


   EXAMPLE: 

    > ode::solve(x*y(x)*diff(y(x),x,x)+x*diff(y(x),x)^2+y(x)*diff(y(x),x),y(x));
*/

ode::exact_second_order:= proc(eq,y,x,solveOptions,odeOptions)
  local eq0,f,g,p,ypp,cf,cond1,cond2,firint,res,optIgnoreAnalyticConstraints,found;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  p:= `#p`;
  ypp:= `#ypp`;
  eq0:= eq;
  eq:= subs(eq,diff(y(x),x)=p,diff(y(x),x,x)=ypp,y(x)=y);
  found:= FALSE;
  misc::maprec(eq,
               {"sign"} = proc(elem)
                              begin
                                if has(op(elem,1),[y,p,ypp]) then 
                                  found:= misc::breakmap();
                                end_if;
                                elem;
                              end_proc);  
  if found then 
    return(FAIL);
  end_if;    
  if testtype(eq,Type::PolyExpr(ypp)) then
    if degree(eq,[ypp])=1 then
      f:= ode::normal(coeff(eq,[ypp],1),Expand=FALSE);
      cf:= coerce(factor(f), DOM_LIST)[1]; 
      assert(not iszero(cf));
      f:= f/cf;
      g:= ode::normal(coeff(eq,[ypp],0)/cf,Expand=FALSE); 
      // check second condition first
      cond2:= ode::normal(diff(f,x,p)+p*diff(f,y,p)+2*diff(f,y)-diff(g,p,p));
      if has(cond2,{cos,sin}) then
        cond2:= simplify(rewrite(cond2,sincos),optIgnoreAnalyticConstraints)
      end_if;
      if iszero(cond2) then
        cond1:= ode::normal(diff(f,x,x)+2*p*diff(f,x,y)+p^2*diff(f,y,y)-
                       (diff(g,x,p)+p*diff(g,y,p)-diff(g,y)));
        if has(cond1,{cos,sin}) then
          cond1:= simplify(rewrite(cond1,sincos),optIgnoreAnalyticConstraints)
        end_if;
        if iszero(cond1) then
          userinfo(1,"exact second order equation");
          if not has((firint:=
                      ode::firint_second_order(eq0,y,x,solveOptions,odeOptions)),
                      FAIL) and 
             not hastype(firint,piecewise) and 
             ode::checkIntegration(firint,y,x,solveOptions,odeOptions) <> FAIL 
             // needed to fix g701091
          then
            if has((res:=ode::solve_eq(firint,y,x,{},solveOptions,{})),
                   FAIL) then
              userinfo(2,"exact second order method failed but found a first".
                          " order integral", firint);
              return(FAIL);
            else
              assert(not has(res, {`#ypp`, `#p`}));
              return(res);
            end_if
          end_if
        end_if
      else 
        if contains(odeOptions, Type = ExactSecondOrder) then 
          ode::odeWarning("cannot detect exact second order equation");
          return(FAIL);
        end_if;  
      end_if
    end_if
  end_if;
  if contains(odeOptions, Type = ExactSecondOrder) then 
    warning("cannot detect exact second order equation");
  end_if;  
  
  return(FAIL);
end_proc:


// This function tries to compute a first integral for the 2nd order ODE eq
// which is supposed (!!!) to be exact 
ode::firint_second_order:= proc(eq,y,x,solveOptions,odeOptions)
  local eq0,p,ypp,f,cf,g,h,f0,f1,intf0,firint,intOptions,optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;  
  eq0:= eq;
  p:= `#p`;
  ypp:= `#ypp`;
  eq:= subs(eq, diff(y(x),x)=p, diff(y(x),x,x)=ypp, y(x)=y);
  if testtype(eq,Type::PolyExpr(ypp)) and degree(eq,[ypp])=1 then
    f:= ode::normal(coeff(eq,[ypp],1),Expand=FALSE);
    // Anpassung an Umstellung von Factored::_index (Walter 29.12.04)
    cf:= coerce(factor(f), DOM_LIST)[1];
    f:= f/cf;
    g:= ode::normal(coeff(eq,[ypp],0)/cf,Expand=FALSE); 
    f:= int(f,p,intOptions);
    // solution is phi=constant where phi=int(f,p)+h(x,y)
    // we should also have g = diff(phi,x)+diff(phi,y)*y'
    g:= ode::normal((g-diff(f,x)-diff(f,y)*p));
    // thus now we should have g = diff(h,x)+diff(h,y)*p
    /* we deal only with the three following easy cases:
                  (1) no y in g ==> h=int(g,x)
                  (2) no x in g ==> h=int(g/p,y)
                  (3) g=f0(x)+f1(y)*p ==> h=int(f0,x)+int(f1,y) */
    if not has(g,y) and not has(g,p) then
      h:= int(g,x,intOptions)
    elif not has(g,x) and not has(g,p) then
      h:= int(g/p,y,intOptions)
    elif testtype(g,Type::PolyExpr(p)) and degree(g,[p])<=1 then
      f1:= ode::normal(coeff(g,[p],1)); 
      f0:= ode::normal(coeff(g,[p],0));
      if not has(f1,x) and not has(f0,y) then
        h:= int(f1,y,intOptions)+int(f0,x,intOptions)
      else
        if type(f0)="_plus" then
          intf0:= _plus(map(op(f0), int, x))
        else
          intf0:= int(f0,x,intOptions)
        end_if;
        h:= intf0 + int(f1-diff(intf0,y),y,intOptions)
      end_if
    else
      firint:= int(eq0,x,intOptions);
      if not has(firint,int) then 
        return(firint + genident("C"))
      else   
        userinfo(2,"cannot find a first order integral");
        return(FAIL);
      end_if;
    end_if;

    if type(f)="_plus" then
      // remove unnecessary constants
      f:= expand(f-select(f,testtype,Type::Constant),optIgnoreAnalyticConstraints)
    end_if:

    if hastype(h,"int") and has(h,p) then
      userinfo(2,"find a first order integral but cannot solve it");
      return(FAIL);
    end_if:
    
    // now we obtain a first integral
    if type(f)="ln" or type(solvelib::combineln(f))="ln" then
      // to avoid to solve eq:=ln(diff(y(x),x))+....
      firint:= expand(exp(f+h),optIgnoreAnalyticConstraints)-genident("C")
    elif type(f)="arctan" or type(combine::arctan(f))="arctan" then
      // to avoid to solve eq:=arctan(diff(y(x),x))+....
      firint:= expand(tan(combine(f+h,arctan,optIgnoreAnalyticConstraints)),
                      optIgnoreAnalyticConstraints)-genident("C")
    elif hastype(f, "int") then
      userinfo(2,"find a first order integral but cannot solve it");
      return(FAIL)
    else
      firint:= f+h-genident("C");
    end_if; 
    if traperror((firint:= evalAt(firint,y=y(x),p=diff(y(x),x)))) = 0 then 
      // The input ODE is supposed to be exact. This is why we do not apply 
      // an additional test to check the integration. 
      userinfo(2,"first order integral is",firint);
      return(firint);
    end_if;  
  end_if:
  
  return(FAIL);
end_proc:

