/* 
   ode::exact_nth_order(l,x,n)

   DESCRIPTION: 
 
   Input: l is a list of (n+2) coefficients, n is the order
   Output: FAIL or a solution 

   REFERENCES: [1] D. Zwillinger: "Handbook of Differential Equations", 
                   Section 64, pp. 263


   DETAILS:

   ode::exact_nth_order(l,x,n) tries to solve the linear homogeneous n-th 
   order ODE

     l[1]*y(x)+l[2]*diff(y(x),x)+...+l[n+1]*diff(y(x),x$n)+l[n+2]=0

   with respect to y(x) using the method [1]. 


   EXAMPLES: 

     > ode::exact_nth_order([0,6,3+6*x,1+x+x^2,-6*x],x,3);

   which is the solution of the 3rd order equation 

        (1+x+x^2)*diff(y(x),x$3)+(3+6*x)*diff(y(x),x$2)+6*diff(y(x),x) = 6*x
*/

ode::exact_nth_order :=
proc(l,x,n,solveOptions, odeOptions) 
  local i,j, y,intOptions, optIgnoreAnalyticConstraints;
  // save y;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;  
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  
  // check if exact using condition (52.3) 
  // delete y;
  y:= genident();
  if _plus((-1)^i*diff(l[i+1],x$i) $ i=0..n)=0 then
    userinfo(1,"exact n-th order equation");
      // applying formula (52.4) 
    l:=[_plus((-1)^j*diff(l[i+1+j],x$j) $ j=0..n-i) $ i=1..n,
        int(l[n+2],x,intOptions)+genident("C")];
    userinfo(2,"new equation is",
             _plus(l[i+1]*diff(y(x),x$i)$i=0..n-1)+l[n+1]=0);
    sysassign(ode::depth,ode::depth+1);
    l:=ode::solve_linear(l,x,n-1,solveOptions,odeOptions);
    sysassign(ode::depth,ode::depth-1);
    if l=FAIL then
      userinfo(1,"exact n-th order failed")
    else
      userinfo(1,"exact n-th order worked")
    end_if;
    return(l);
  else
    return(FAIL);
  end_if;
  
end_proc:


/*
   ode::exact_nth_order_nonlinear(eq,y,x)

   DESCRIPTION: 

   Input: eq is a nth-order nonlinear equation in y(x).
   Output: FAIL or a list of solutions.

   REFERENCES: [1] D. Zwillinger: "Handbook of Differential Equations", 
                   Section 64, pp. 263

   DETAILS: 
 
   ode::exact_nth_order_nonlinear(eq,y,x) tries to solve the nth order nonlinear 
   ODE eq=0 with respect to y(x) using the method described in [1]. 


   EXAMPLE: 

    > ode::exact_nth_order_nonlinear(9*x^2*diff(y(x),x)^2+3*y(x)^2+
            3*x^3*diff(y(x),x)*diff(y(x),x$2)+18*x*y(x)*diff(y(x),x)+
            9*x^2*y(x)*diff(y(x),x$2)+x^3*y(x)*diff(y(x),x$3), y,x);

   COMMENT:

   I think it's better to use ode::exact_first_order (resp. ode::exact_second_order)
   for first (resp. second) order ODEs, even if ode::exact_nth_order_nonlinear
   should also succeed.
*/


ode::exact_nth_order_nonlinear :=
proc(eq,y,x,solveOptions,odeOptions)
  local n,Y,eqx,lfi,lfix,lDfi,i,j,k,cond,t,firint,res,
        de,intOptions,optIgnoreAnalyticConstraints;
begin
  optIgnoreAnalyticConstraints:= if has(solveOptions, IgnoreAnalyticConstraints) then 
                IgnoreAnalyticConstraints;
              else
                null();
              end_if;  
  intOptions:= null();            
  if has(solveOptions, IgnoreSpecialCases) then 
    intOptions:= intOptions,IgnoreSpecialCases;
  end_if;
  if has(solveOptions, IgnoreAnalyticConstraints) then   
    intOptions:= intOptions,IgnoreAnalyticConstraints;
  end_if;   
  de:= eq;
  n:= ode::order(eq,{y},solveOptions,odeOptions);
  Y:= genident("y"); 
  t:= genident("t");
  eq:= subs(eq, {diff(y(x),x$i)=Y[i] $ i=0..n});
  eqx:= subs(eq, {Y[i]=Y[i](x) $ i=0..n});
  lfi:= [diff(eq,Y[i]) $ i=0..n];
  lfix:= subs(lfi, {Y[i]=Y[i](x) $ i=0..n});
  lDfi:= [];
  for i from 1 to n do
    lDfi:=append(lDfi, subs(diff(lfix[i+1],x$i),
                            {diff(Y[j](x),x$k)=Y[j+k](x) $ k=1..i $ j=0..n}));
  end_for:
  cond:= lfix[1]+ _plus((-1)^i*lDfi[i] $ i=1..n);
  if ode::odeIszero(cond) and 
     iszero((cond:= expand(cond,optIgnoreAnalyticConstraints))) then
    userinfo(1, "exact n-th order nonlinear equation");
    firint:= int(de,x,IgnoreSpecialCases,intOptions);
    if not hastype(firint,"int") and 
       not hastype(firint, piecewise) then 
      firint:= firint - genident("C");
      userinfo(2, "first integral is", firint);
      if has((res:=ode::solve_eq(firint,y,x,{},solveOptions,odeOptions)),
             FAIL) then
        userinfo(1,"exact n-th order nonlinear method failed " .
                   " but found a first order integral", firint);
        return(FAIL)
      else
        userinfo(1,"exact n-th order nonlinear worked");
        return(res)
      end_if
    else 
      userinfo(2,"exact n-th order nonlinear equation " .
                 " but didn't find a first integral");
    end_if;
  end_if:

  return(FAIL);
end_proc:

